/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: fsutils.c
 */

#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "widget.h"
#include "task.h"
#include "fsutils.h"
#include "thing.h"
#include "readable.h"
#include "plugin_cb.h"
#include "logging.h"
#include "help.h"

/*
 *
 *   inline void add_fsim_name_to_filesys_op_clist (GtkCList *, gint, object_handle_t)
 *
 *   Description:
 *      This routine updates the current row with the name of the
 *      FSIM associated with the given volume.
 *
 *   Entry:
 *      clist  - address of the selections GtkCList widget
 *      row    - the row in the list to update
 *      volume - the volume's object handle
 *
 *   Exit:
 *      The name of the FSIM for the given volume is added to the given row
 *
 */
inline void add_fsim_name_to_filesys_op_clist(GtkCList * clist, gint row, object_handle_t volume)
{
	gint rc;
	handle_object_info_t *object;

	rc = evms_get_info(volume, &object);

	if (rc != SUCCESS) {
		log_warning("%s: evms_get_info() returned error code %d.\n", __FUNCTION__, rc);
	} else {
		gchar *fsim_name;

		fsim_name = get_volume_fsim_name(&(object->info.volume));

		gtk_clist_set_text(clist, row, SL_SIZE_COLUMN, fsim_name);

		g_free(fsim_name);
		evms_free(object);
	}
}

/*
 *
 *   inline void set_filesys_clist_properties (GtkCList *)
 *
 *   Description:
 *      This routine sets some common properties about the
 *      file system operations clist.
 *
 *   Entry:
 *      clist - address of the selections GtkCList widget
 *
 *   Exit:
 *      Nothing returned
 *
 */
inline void set_filesys_clist_properties(GtkCList * clist)
{
	gtk_clist_set_column_auto_resize(clist, SL_SIZE_COLUMN, TRUE);
	set_selection_window_clist_column_titles(clist, _("FSIM"),
						 _("Logical Volume Containing File System"), NULL);
}

/*
 *
 *   inline void add_volume_to_filesys_clist (GtkCList *, object_handle_t, gboolean)
 *
 *   Description:
 *      This routine appends a new row to the file system operations
 *      clist with the name of a volume and its associated FSIM name.
 *
 *   Entry:
 *      clist       - address of the selections GtkCList widget
 *      volume      - the volume's object handle
 *      is_selected - TRUE if the item to add should be automatically selected
 *
 *   Exit:
 *      A new row is added to the clist
 *
 */
inline void add_volume_to_filesys_clist(GtkCList * clist, object_handle_t volume,
					gboolean is_selected)
{
	gint row;

	row = add_thing_to_selection_list(clist, volume, is_selected);

	if (row != -1)
		add_fsim_name_to_filesys_op_clist(clist, row, volume);
}

/*
 *
 *   void on_filesys_operation_clist_realize (GtkCList *, gpointer)
 *
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of volumes that an FSIM associated with the volume says
 *      can either be fsck'd, or obliterated.
 *
 *   Entry:
 *      clist     - address of the selections GtkCList widget
 *      user_data - contains the filesys_op_t FSCK, or UNMKFS
 *
 *   Exit:
 *      Selection list populated with volumes acceptable to the operation
 *
 */
void on_filesys_operation_clist_realize(GtkCList * clist, gpointer user_data)
{
	gint rc;
	filesys_op_t opcode = GPOINTER_TO_UINT(user_data);
	handle_array_t *volumes;

	set_filesys_clist_properties(clist);

	rc = evms_get_volume_list(0, 0, 0, &volumes);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_volume_list() returned error code %d.\n", __FUNCTION__, rc);
	} else {
		gint i;
		gboolean op_allowed;
		gboolean is_selected = (volumes->count == 1);

		for (i = 0; i < volumes->count; i++) {
			switch (opcode) {
			case FSCK:
				op_allowed = evms_can_fsck(volumes->handle[i]) == 0;
				break;
			case UNMKFS:
				op_allowed = evms_can_unmkfs(volumes->handle[i]) == 0;
				break;

			default:
				log_warning("%s: Unknown filesys_op_t code %d.\n", __FUNCTION__,
					    opcode);
				op_allowed = FALSE;
				break;
			}

			if (op_allowed)
				add_volume_to_filesys_clist(clist, volumes->handle[i], is_selected);
		}

		if (clist->rows == 1)
			gtk_clist_select_row(clist, 0, 0);

		evms_free(volumes);
	}
}

/*
 *
 *   void on_fsim_clist_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of file system interface modules to choose from.
 *
 *   Entry:
 *      widget    - address of the selections GtkCList widget
 *      user_data - contains task action code for action to use list
 *
 *   Exit:
 *      Selection list populated with available FSIM plugins
 *
 */
void on_fsim_clist_realize(GtkWidget * widget, gpointer user_data)
{
	set_selection_window_clist_column_titles(GTK_CLIST(widget), NULL,
						 _("File System Interface Module"), NULL);
	on_plugin_selection_clist_realize(GTK_CLIST(widget), EVMS_FILESYSTEM_INTERFACE_MODULE, 0,
					  GPOINTER_TO_UINT(user_data));
}

/*
 *
 *   void on_initiate_filesys_operation_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine initiates a checking/repairing operation on a file system
 *      located on the volume selected.  It is also used by the context popup
 *      menu item that initiates a make file system on a volume.
 *
 *      In the case of fsck operations, our selection window has a list of
 *      volumes and the current selection is used to start the task.  For a
 *      mkfs operation, the selection window has a list of FSIM plugins and
 *      the current selection is used to initiate the task.  After the mkfs is
 *      started, we set the volume that initiated the operation as the
 *      selected object.
 *
 *   Entry:
 *      button    - the button that was clicked
 *      user_data - contains filesys_op_t value MKFS, or FSCK
 *
 *   Exit:
 *      Nothing is returned but hopefully the options notebook
 *      is displayed.
 *
 */
void on_initiate_filesys_operation_button_clicked(GtkButton * button, gpointer user_data)
{
	GList *window_list;
	GtkWidget *clist;
	GtkWidget *next_window;
	GtkWidget *selection_window;
	object_handle_t volume = 0;
	object_handle_t object;
	object_handle_t prev_object;
	task_action_t filesys_task;
	filesys_op_t opcode = GPOINTER_TO_UINT(user_data);

	if (opcode == MKFS)
		filesys_task = EVMS_Task_mkfs;
	else
		filesys_task = EVMS_Task_fsck;

	/*
	 * Hide the selection window and either redisplay the existing options window
	 * if it exists and the selection did not change. If the options window
	 * does not exist, create it. However, if the selection changed and
	 * the options window existed then destroy the current options window and create
	 * a new one.
	 */

	selection_window = gtk_widget_get_toplevel(GTK_WIDGET(button));
	clist = lookup_widget(GTK_WIDGET(button), "selection_window_clist");
	object = GPOINTER_TO_UINT(get_single_select_current_row_data(GTK_CLIST(clist)));
	window_list = get_window_list(selection_window);
	next_window = gtk_object_get_data(GTK_OBJECT(selection_window), "next_window_id");
	prev_object =
	    GPOINTER_TO_UINT(gtk_object_get_data(GTK_OBJECT(selection_window), "previous_object"));

	/*
	 * For a mkfs operation, the handle of the volume we want to prepare will be
	 * stored as data associated with this selection window containing the FSIMs.
	 */

	if (opcode == MKFS)
		volume =
		    GPOINTER_TO_UINT(gtk_object_get_data
				     (GTK_OBJECT(selection_window), "volume_handle"));

	if (window_list == NULL)
		window_list = g_list_append(window_list, selection_window);

	if (object != prev_object) {
		gint rc;
		task_handle_t handle;

		rc = evms_create_task(object, filesys_task, &handle);

		if (rc != SUCCESS) {
			log_error("%s: evms_create_task() returned error code %d.\n", __FUNCTION__,
				  rc);

			display_results_window(rc, NULL, _("Unable to create the task context."),
					       NULL, FALSE, selection_window);
		} else {
			if (opcode == MKFS) {
				gchar mem[sizeof(handle_array_t) + sizeof(object_handle_t)];
				handle_array_t * ha = (handle_array_t *) mem;

				ha->count = 1;
				ha->handle[0] = volume;

				rc = evms_set_selected_objects(handle, ha, NULL, NULL);
			}

			if (rc != SUCCESS) {
				log_error
				    ("%s: evms_set_selected_objects() returned error code %d.\n",
				     __FUNCTION__, rc);

				display_results_window(rc, NULL,
						       _
						       ("File System Interface Module did not accept the volume."),
						       NULL, FALSE, selection_window);
			} else {
				gchar *button_text;
				gchar *window_title;

				/*
				 * If we already created the next window for options,
				 * make sure we destroy the window since our selection
				 * has apperently changed and so the panel needs to be
				 * recreated.
				 */

				if (next_window != NULL)
					destroy_window_list(g_list_find(window_list, next_window));

				button_text = get_task_action_string(handle);

				if (opcode == MKFS)
					window_title = _("Make File System - Options");
				else
					window_title = _("Check/Repair File System - Options");

				next_window = create_standard_task_window(window_title,
									  button_text,
									  NULL,
									  on_button_clicked_display_prev_window,
									  on_button_clicked_destroy_window_list,
									  handle);

				window_list = g_list_append(window_list, next_window);

				set_window_list(next_window, window_list);
				set_window_list(selection_window, window_list);

				bind_options_callout_to_toplevel_widget(next_window,
									options_window_callout);

				gtk_object_set_data(GTK_OBJECT(selection_window), "next_window_id",
						    next_window);
				gtk_object_set_data(GTK_OBJECT(selection_window), "previous_object",
						    GUINT_TO_POINTER(object));

				/*
				 * Connect the destroy signal to the next window. If the window is
				 * destroyed, so is the task context.
				 */

				gtk_signal_connect(GTK_OBJECT(next_window), "destroy",
						   GTK_SIGNAL_FUNC(on_task_window_destroy),
						   GUINT_TO_POINTER(handle));

				gtk_widget_show(next_window);
				gtk_widget_hide(selection_window);

				g_free(button_text);
			}
		}
	} else {
		gtk_widget_show(next_window);
		gtk_widget_hide(selection_window);
	}
}

/*
 *
 *   void on_remove_filesys_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine initiates the removal of a file system located
 *      on the volume selected.
 *
 *   Entry:
 *      button    - the button that was clicked
 *      user_data - not used
 *
 *   Exit:
 *      Invokes evms_unmkfs and displays a results window.
 *
 */
void on_remove_filesys_button_clicked(GtkButton * button, gpointer user_data)
{
	GtkCList *clist;
	object_handle_t handle;

	clist = GTK_CLIST(lookup_widget(GTK_WIDGET(button), "selection_window_clist"));

	handle = GPOINTER_TO_UINT(get_single_select_current_row_data(clist));

	if (handle != 0) {
		gint rc;

		rc = evms_unmkfs(handle);

		if (rc != SUCCESS)
			log_error("%s: evms_unmkfs() returned error code %d.\n", __FUNCTION__, rc);

		display_selection_window_results(GTK_WIDGET(button), rc,
						 _
						 ("An error was encountered removing the file system."),
						 _
						 ("The operation has been scheduled for completion when changes are committed."));
	}
}

/*
 *
 *   inline void init_filesys_single_item_clist (GtkWidget *, object_handle_t)
 *
 *   Description:
 *      This routine will display one volume in the filesys operation
 *      window's selection list.
 *
 *   Entry:
 *      window - the filesys operation window id
 *      volume - the volume's object handle
 *
 *   Exit:
 *      See description.
 *
 */
inline void init_filesys_single_item_clist(GtkWidget * window, object_handle_t volume)
{
	GtkCList *clist;

	clist = GTK_CLIST(gtk_object_get_data(GTK_OBJECT(window), "selection_window_clist"));

	set_filesys_clist_properties(clist);

	add_volume_to_filesys_clist(clist, volume, TRUE);
}

/*
 *
 *   inline void populate_clist_with_fsims_for_volume (GtkWidget *, object_handle_t)
 *
 *   Description:
 *      This routine will display the names of the FSIMs that have
 *      agreed they can make a file system on the volume supplied.
 *
 *   Entry:
 *      window - the FSIM selection window id
 *      volume - the volume's object handle
 *
 *   Exit:
 *      See description.
 *
 */
inline void populate_clist_with_fsims_for_volume(GtkWidget * window, object_handle_t volume)
{
	gint rc;
	GtkCList *clist;
	handle_array_t *plugins;

	clist = GTK_CLIST(gtk_object_get_data(GTK_OBJECT(window), "selection_window_clist"));

	set_selection_window_clist_column_titles(clist, NULL, _("File System Interface Module"),
						 NULL);

	rc = evms_get_plugin_list(EVMS_FILESYSTEM_INTERFACE_MODULE, 0, &plugins);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_plugin_list() returned error code %d.\n", __FUNCTION__, rc);
	} else {
		gint i;

		for (i = 0; i < plugins->count; i++) {
			if (evms_can_mkfs(volume, plugins->handle[i]) == 0)
				add_thing_to_selection_list(clist, plugins->handle[i], FALSE);
		}

		if (clist->rows == 1)
			gtk_clist_select_row(clist, 0, 0);

		evms_free(plugins);
	}

	gtk_clist_set_column_visibility(clist, SL_SIZE_COLUMN, FALSE);
}

/*
 *
 *   void on_fsck_popup_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine will display the volume containing the file system
 *      we intend to fsck and allow the user to confirm the checking of
 *      the file system.
 *
 *   Entry:
 *      menuitem  - the menuitem that initiated the action
 *      user_data - volume's object handle
 *
 *   Exit:
 *      See description.
 *
 */
void on_fsck_popup_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Check/Repair File System"),
						  NULL,
						  get_fsck_help_text(),
						  NULL,
						  on_initiate_filesys_operation_button_clicked,
						  NULL, NULL, NULL, NULL, GUINT_TO_POINTER(FSCK));

	init_filesys_single_item_clist(window, GPOINTER_TO_UINT(user_data));

	gtk_widget_show(window);
}

/*
 *
 *   void on_mkfs_popup_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine will display the FSIMs that agreed that they could
 *      make a file system on the volume that initiated this action.
 *
 *   Entry:
 *      menuitem  - the menuitem that initiated the action
 *      user_data - volume's object handle
 *
 *   Exit:
 *      See description.
 *
 */
void on_mkfs_popup_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Make File System"),
						  NULL,
						  get_mkfs_help_text(),
						  NULL,
						  on_initiate_filesys_operation_button_clicked,
						  NULL, NULL, NULL, NULL, GUINT_TO_POINTER(MKFS));

	/*
	 * Save the handle of the volume that initiated this operation. It
	 * will be retrieved by the on_initiate_filesys_operation_button_clicked()
	 * signal handler to allow setting the volume as the selected object after
	 * it creates the EVMS_Task_mkfs task.
	 */

	gtk_object_set_data(GTK_OBJECT(window), "volume_handle", user_data);

	populate_clist_with_fsims_for_volume(window, GPOINTER_TO_UINT(user_data));

	gtk_widget_show(window);
}

/*
 *
 *   void on_unmkfs_popup_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine will display the volume containing the file system
 *      we intend to unmkfs and allow the user to confirm the removal
 *      of the file system.
 *
 *   Entry:
 *      menuitem  - the menuitem that initiated the action
 *      user_data - volume's object handle
 *
 *   Exit:
 *      See description.
 *
 */
void on_unmkfs_popup_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Remove File System"),
						  _("Remove"),
						  get_unmkfs_help_text(),
						  NULL,
						  on_remove_filesys_button_clicked,
						  NULL, NULL, NULL, NULL, NULL);

	init_filesys_single_item_clist(window, GPOINTER_TO_UINT(user_data));

	gtk_widget_show(window);
}

/*
 *
 *   gboolean volume_has_fsim (object_handle_t)
 *
 *   Description:
 *      This routine checks to see if the volume has an associated
 *      FSIM and returns TRUE if it does.
 *
 *   Entry:
 *      volume - volume's object handle
 *
 *   Exit:
 *      See Description.
 *
 */
gboolean volume_has_fsim(object_handle_t volume)
{
	gint rc;
	gboolean result = FALSE;
	handle_object_info_t *object;

	rc = evms_get_info(volume, &object);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);
	} else {
		result = object->info.volume.file_system_manager != 0;
		evms_free(object);
	}

	return result;
}

/*
 *
 *   gboolean can_volume_be_formatted (object_handle_t)
 *
 *   Description:
 *      This routine checks to see if any FSIM can mkfs it
 *      and if so returns TRUE.
 *
 *   Entry:
 *      volume - volume's object handle
 *
 *   Exit:
 *      See Description.
 *
 */
gboolean can_volume_be_formatted(object_handle_t volume)
{
	gint rc;
	gboolean result = FALSE;
	handle_array_t *plugins;

	rc = evms_get_plugin_list(EVMS_FILESYSTEM_INTERFACE_MODULE, 0, &plugins);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_plugin_list() returned error code %d.\n", __FUNCTION__, rc);
	} else {
		gint i;

		for (i = 0; i < plugins->count && result == FALSE; i++) {
			result = evms_can_mkfs(volume, plugins->handle[i]) == 0;
		}

		evms_free(plugins);
	}

	return result;
}
