#! /usr/bin/env python

# Copyright (C) 2004-2005 Ross Burton <ross@burtonini.com>
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc., 59 Temple
# Place, Suite 330, Boston, MA 02111-1307 USA

import os

import gettext
_ = gettext.gettext

import pygtk; pygtk.require("2.0")
from gtk import gdk
import gtk
import gobject
import xdg.Menu
import thread
import time
import apt_pkg

def xmlescape(s):
    from xml.sax.saxutils import escape
    if s: return escape(s)

class AppInstall(gtk.Window):

    # Row types
    (TYPE_HEADER,
     TYPE_GROUP,
     TYPE_PACKAGE) = range(0, 3)

    # Column enumeration
    (COL_TYPE,
     COL_WAS_INSTALLED,
     COL_TO_INSTALL,
     COL_NAME,
     COL_DESC,
     COL_ICON,
     COL_PACKAGE) = range(0, 7)

    # Values representing the users choice from the "unapplied changes" dialog
    (CHANGES_NONE,
     CHANGES_IGNORE,
     CHANGES_APPLY,
     CHANGES_CANCEL) = range(0,4)

    # The base path to the .desktop files
    datadir = None
    
    # The current icon theme
    icons = None

    # The listview store of packages
    store = None

    # The actual treeview
    treeview = None

    # The package install/remove worker to use
    packageWorker = None
    
    def __init__(self, datadir):
        import gtk.glade        
        gtk.Window.__init__(self, gtk.WINDOW_TOPLEVEL)

        self.datadir = datadir
        
        self.icons = gtk.icon_theme_get_default()
        self.icons.prepend_search_path(self.datadir)

        # Setup the window
        self.set_title(_("Application Installer"))
        self.set_default_size (450, 500)
        self.connect("delete-event", self.deleteEvent)
        self.connect("destroy", gtk.main_quit)
        gtk.window_set_default_icon(self.icons.load_icon("gnome-settings-default-applications", 32, 0))

        # Grab the top-level container and magically reparent it
        glade = gtk.glade.XML(os.path.join(self.datadir, "gnome-app-install.glade"))
        old_window = glade.get_widget("main_window")
        box = old_window.get_child()
        old_window.remove(box)
        self.add(box)

        # Set the images
        glade.get_widget("icon_image").set_from_pixbuf(self.icon("gnome-settings-default-applications", 48))

        # Hack to move the Help button to the beginning of the container
        help = glade.get_widget("help_button")
        help.get_parent().child_set_property(help, "secondary", True)

        # Connect the buttons. TODO: use glade autoconnect
        glade.get_widget("advanced_button").connect("clicked", self.advancedClicked)
        glade.get_widget("apply_button").connect("clicked", self.applyClicked)
        glade.get_widget("close_button").connect("clicked", self.closeClicked)
        
        # Create the treeview store
        self.store = gtk.TreeStore(gobject.TYPE_INT, gobject.TYPE_BOOLEAN, gobject.TYPE_BOOLEAN, gobject.TYPE_STRING, gobject.TYPE_STRING, gdk.Pixbuf.__gtype__, gobject.TYPE_STRING)

        # Setup the treeview
        # TODO: set background colour of scrolledwindow. test8.py behaves differently here
        self.treeview = glade.get_widget("treeview")
        self.setup_treeview()
        self.treeview.set_sensitive(False)

        # TODO: doesn't work, annoyingly
        #self.connect("style-set", lambda parent, old, widget: widget.set_property ("background-gdk", parent.get_style().bg[gtk.STATE_SELECTED]), glade.get_widget("scrolledwindow"))
        
        # Create an idle callback to load the APT cache and setup the installed
        # states when the UI starts
        def idle():
            self.window.set_cursor(gdk.Cursor(gdk.WATCH)) ; gtk.main_iteration()
            self.updateCache()
            self.update_installed_states()
            self.window.set_cursor(None)
            return False
        gobject.idle_add(idle)
        
        self.show()

        self.packageWorker = PackageWorker()
        
    def updateCache(self):
        self.treeview.set_sensitive(False)
        apt_pkg.init()
        self.cache = apt_pkg.GetCache()
        self.depcache = apt_pkg.GetDepCache(self.cache)
        self.depcache.ReadPinFile()
        self.depcache.Init()
        self.treeview.set_sensitive(True)
        self.store.clear()
        self.populate_menu()

    def setup_treeview(self):
        self.treeview.get_selection().set_mode(gtk.SELECTION_NONE)
        self.treeview.set_search_column(self.COL_NAME)
        
        def visibility_magic(cell_layout, renderer, model, iter, visible_types):
            row_type = model.get_value(iter, self.COL_TYPE)
            if isinstance(visible_types, int):
                renderer.set_property("visible", row_type == visible_types)
            else:
                renderer.set_property("visible", row_type in visible_types)

        def package_view_func(cell_layout, renderer, model, iter):
            visibility_magic(cell_layout, renderer, model, iter, self.TYPE_PACKAGE)
            name, desc, current, future = model.get(iter, self.COL_NAME, self.COL_DESC, self.COL_WAS_INSTALLED, self.COL_TO_INSTALL)
            if current != future:
                renderer.set_property("markup", "<b>%s</b>\n<small>%s</small>" % (name, desc))
            else:
                renderer.set_property("markup", "%s\n<small>%s</small>" % (name, desc))

        def on_install_toggle(renderer, path, store):
            i = store.get_iter(path)
            store.set_value (i, self.COL_TO_INSTALL, not store.get_value (i, self.COL_TO_INSTALL))

        column = gtk.TreeViewColumn("")
        
        render = gtk.CellRendererToggle()
        render.connect('toggled', on_install_toggle, self.store)
        column.pack_start(render, False)
        column.add_attribute(render, "active", self.COL_TO_INSTALL)
        column.set_cell_data_func (render, visibility_magic, self.TYPE_PACKAGE)
        
        render = gtk.CellRendererPixbuf()
        column.pack_start(render, False)
        column.add_attribute(render, "pixbuf", self.COL_ICON)
        column.set_cell_data_func (render, visibility_magic, (self.TYPE_GROUP, self.TYPE_PACKAGE))

        render = gtk.CellRendererText()
        render.set_property("scale", 1.0)
        render.set_property("weight", 700)
        column.pack_start(render, True)
        column.add_attribute(render, "markup", self.COL_NAME)
        column.set_cell_data_func (render, visibility_magic, self.TYPE_GROUP)

        render = gtk.CellRendererText()
        column.pack_start(render, True)
        column.add_attribute(render, "markup", self.COL_NAME)
        column.set_cell_data_func (render, package_view_func)
        
        render = gtk.CellRendererText()
        render.set_property("xpad", 4)
        render.set_property("ypad", 4)        
        render.set_property("scale", 1.3)
        render.set_property("weight", 700)
        self.connect("style-set", lambda parent, old, widget: widget.set_property ("foreground-gdk", parent.get_style().fg[gtk.STATE_SELECTED]), render)
        self.connect("style-set", lambda parent, old, widget: widget.set_property ("background-gdk", parent.get_style().bg[gtk.STATE_SELECTED]), render)
        column.pack_start(render, True)
        column.add_attribute(render, "text", self.COL_NAME)
        column.set_cell_data_func (render, visibility_magic, self.TYPE_HEADER)

        render = gtk.CellRendererText()
        render.set_property("width", self.treeview.style_get_property("expander-size") + self.treeview.style_get_property("horizontal-separator"))
        column.pack_start(render, False)
        column.set_cell_data_func (render, visibility_magic, self.TYPE_HEADER)

        self.treeview.append_column(column)

    def populate_from_entry(self, store, node, parent = None):
        for entry in node.getEntries():
            if isinstance(entry, xdg.Menu.Menu):
                i = store.append(parent)
                store.set(i, self.COL_TYPE, self.TYPE_GROUP, self.COL_NAME, xmlescape(entry.getName()), self.COL_ICON, self.icon(entry.getIcon(), 32))
                self.populate_from_entry(store, entry, i)
            elif isinstance(entry, xdg.Menu.MenuEntry):
                if entry.DesktopEntry.hasKey("X-AppInstall-Package"):
                    try:
                        pkgname = entry.DesktopEntry.get("X-AppInstall-Package")
                        pkg = self.cache[pkgname]
                        # we ignore packages we can't install (no candidate)
                        if self.depcache.GetCandidateVer(pkg) == None: 
                            continue
                        i = store.append(parent)
                        store.set(i, self.COL_TYPE, self.TYPE_PACKAGE, self.COL_NAME, xmlescape(entry.DesktopEntry.getName()), self.COL_DESC, xmlescape(entry.DesktopEntry.getComment()), self.COL_ICON, self.icon(entry.DesktopEntry.get("X-AppInstall-Icon", "") or entry.DesktopEntry.getIcon(), 24), self.COL_PACKAGE, pkgname)
                    except KeyError:
                        pass          # ignore unkown packages
                else:
                    print "Got non-package menu entry %s" % entry.DesktopEntry.getName()
            elif isinstance(entry, xdg.Menu.Header):
                print "got header"
                
    def populate_menu(self):
        def add_header(store, name):
            store.set (store.append(None), self.COL_TYPE, self.TYPE_HEADER, self.COL_NAME, xmlescape(name), self.COL_ICON, None)
        
        menu = xdg.Menu.parse(os.path.join(self.datadir, "applications.menu"))
        # TODO: this should be done in xdg.Menu
        try:
            import locale ; menu.setLocale(locale.getlocale(locale.LC_MESSAGES)[0])
        except ValueError: 
            pass            # do nothing for unkown locales
        add_header(self.store, _("Applications"))
        self.populate_from_entry(self.store, menu)
        # mvo: disabled for now, nothing in it (ubuntu #7311)
        #add_header(self.store, _("System Services")) # TODO. Somehow. Not sure.
        self.treeview.set_model(self.store)

    def update_installed_states(self):
        def each(model, path, iter, self):
            type, name = model.get(iter, self.COL_TYPE, self.COL_PACKAGE)
            if type == self.TYPE_PACKAGE:
                try:
                    installed = self.cache[name].CurrentVer is not None
                    model.set(iter, self.COL_WAS_INSTALLED, installed, self.COL_TO_INSTALL, installed)
                except KeyError:
                    pass                # ignore stuff that is not in the cache
        self.store.foreach(each, self)
    
    def icon(self, name, size):
        if name is None or name == "":
            print "Using dummy icon"
            name = "gnome-other"
        if name.startswith("/"):
            print "Doesn't handle absolute paths"
            name = "gnome-other"
        if name.find(".") != -1:
            import os.path
            name = os.path.splitext(name)[0]
        if not self.icons.has_icon(name):
            print "Icon %s is not in theme" % name
            name = "gnome-other"
        # FIXME: mvo: this try: except is a hack to work around 
        #             ubuntu #6858 (icon is no longer in cache after removal)
        #             of a pkg. correct is probably to reload the icontheme
        #             (or something)
        try:
            icon = self.icons.load_icon(name, size, 0)
        except gobject.GError:
            icon = self.icons.load_icon("gnome-other", size, 0)
        if icon.get_width() != size:
            print "Got badly sized icon for %s" % name
            icon = icon.scale_simple(size, size, gdk.INTERP_BILINEAR)
        return icon

    def ignoreChanges(self):
        if (len(self.get_changes()) == 0):
            return self.CHANGES_NONE
        dialog = gtk.MessageDialog (self, gtk.DIALOG_MODAL, gtk.MESSAGE_QUESTION, gtk.BUTTONS_NONE, "")
        dialog.set_markup(_("<big><b>Changes Pending</b></big>\n\nYou have selected to add or remove applications, but these changes have not been applied.  "
                          "Do you want to ignore them, or do them now?"))
        dialog.add_button(_("Ignore"), self.CHANGES_IGNORE)
        dialog.add_button(_("Apply"), self.CHANGES_APPLY)
        res = dialog.run()
        dialog.destroy()
        if res == gtk.RESPONSE_DELETE_EVENT:
            res = self.CHANGES_CANCEL
        return res
    
    def get_changes(self):
        selections = []
        def each(store, path, i):
            previous, future, name = store.get (i, self.COL_WAS_INSTALLED, self.COL_TO_INSTALL, self.COL_PACKAGE)
            if previous == future: return
            if previous and not future:
                selections.append("%s\tuninstall" % name)
            if not previous and future:
                selections.append("%s\tinstall" % name)
        self.store.foreach(each)
        return selections
    
    def advancedClicked(self, button):
        res = self.ignoreChanges()
        # If we should cancel, return now
        if res == self.CHANGES_CANCEL:
            return
        # If we need to apply, do it now
        if res == self.CHANGES_APPLY:
            self.applyClicked(None)
        # Here we are either ignoring or have applied, so run Synaptic
        os.spawnl(os.P_NOWAIT, "/usr/sbin/synaptic", "synaptic") # TODO: activate startup notification (needs wrapping)
        gtk.main_quit()

    def applyClicked(self, button):
        # Set a busy cursor
        self.window.set_cursor(gdk.Cursor(gdk.WATCH)) ; gtk.main_iteration()
        # Get the selections delta for the changes
        # Apply them
        self.packageWorker.processSelections(self,self.get_changes()) 
        # Reload the APT cache
        self.updateCache()
        # Update the treeview
        self.update_installed_states()
        # And reset the cursor
        self.window.set_cursor(None)

    def closeClicked(self, button):
        res = self.ignoreChanges()
        if res == self.CHANGES_NONE or res == self.CHANGES_IGNORE:
            gtk.main_quit()
        elif res == self.CHANGES_APPLY:
            self.applyClicked(None)
            gtk.main_quit()

    def deleteEvent(self, window, event):
        res = self.ignoreChanges()
        if res == self.CHANGES_NONE or res == self.CHANGES_IGNORE:
            return False
        elif res == self.CHANGES_APPLY:
            self.applyClicked(None)
            return False
        return True


"""
A class which does the actual package installing/removing.
"""
class PackageWorker:
    def run_synaptic(self, id, lock, selections):
        #print "run_synaptic(%s,%s,%s)" % (id, lock, selections)
        cmd = "/usr/sbin/synaptic --hide-main-window --set-selections --non-interactive --plug-progress-into %s " % (id)
        f = os.popen(cmd, "w")
        for s in selections:
            f.write("%s\tinstall\n" % s)
        f.close()
        lock.release()

    def plug_removed(self, w, (win,socket)):
        # plug was removed, but we don't want to get it removed, only hiden
        # unti we get more
        win.hide()
        return True

    def plug_added(self, sock, win):
        win.show()
        while gtk.events_pending():
            gtk.main_iteration()

    def processSelections(self, main_window, selections):
        main_window.set_sensitive(False)
        win = gtk.Window()
        win.set_border_width(6)
        #print main_window
        win.set_transient_for(main_window)
        win.set_position(gtk.WIN_POS_CENTER_ON_PARENT)
        win.resize(400,200)
        win.set_title("")
        win.set_resizable(False)
        # prevent the window from closing with the delete button (there is
        # a cancel button in the window)
        win.connect("delete_event", lambda e,w: True);
    
        # create the socket
        socket = gtk.Socket()
        socket.show()
        win.add(socket)
        
        socket.connect("plug-added", self.plug_added, win)
        socket.connect("plug-removed", self.plug_removed, (win,socket))
        lock = thread.allocate_lock()
        lock.acquire()
        t = thread.start_new_thread(self.run_synaptic,(socket.get_id(),lock,selections))
        while lock.locked():
            while gtk.events_pending():
                gtk.main_iteration()
            time.sleep(0.05)
        win.destroy()
        main_window.set_sensitive(True)

# Entry point for testing in source tree
if __name__ == '__main__':
    app = AppInstall("data")
    gtk.main()
