# Copyright (C) 2004-2005 Ross Burton <ross@burtonini.com>
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc., 59 Temple
# Place, Suite 330, Boston, MA 02111-1307 USA

import os, sys

import gettext
_ = gettext.gettext

import pygtk; pygtk.require("2.0")
from gtk import gdk
import LaunchpadIntegration
import gtk
import gobject
import xdg.Menu
import thread
import time
from BrowserView import GtkHtml2BrowserView as BrowserView
import gnomevfs
import re
import subprocess
import tempfile
import warnings
from warnings import warn
warnings.filterwarnings("ignore", "ICON:.*", UserWarning)
warnings.filterwarnings("ignore", "apt API not stable yet", FutureWarning)
import apt, apt_pkg

DISTRO_VERSION = "breezy"

def xmlescape(s):
    from xml.sax.saxutils import escape
    if s: return escape(s)

class OpProgressWindow(apt.progress.OpProgress):
    def __init__(self, glade, parent):
        self.progress_window = glade.get_widget("progress_window")
        self.progress_window.hide()
        self.progress_window.set_transient_for(parent)
        #self.progress_window.set_position(gtk.WIN_POS_CENTER_ON_PARENT)
        self.progress_window.set_position(gtk.WIN_POS_CENTER_ALWAYS)
        self.progressbar_cache = glade.get_widget("progressbar_cache")
    def update(self, percent):
        #if not self.progress_window.visible():
        self.progress_window.show()
        self.progressbar_cache.set_fraction(percent/100.0)
        self.progressbar_cache.set_text("%s" % self.subOp)
        while gtk.events_pending():
            gtk.main_iteration()
    # using __del__ here sucks (because of eventual GC lazines)
    # but there is no "complete" callback in apt yet and "Done"
    # is called for each subOp
    def __del__(self):
        self.progress_window.hide()
        
class AppInstall(gtk.Window):

    # Row types
    (TYPE_HEADER,
     TYPE_GROUP,
     TYPE_PACKAGE) = range(0, 3)

    # Column enumeration
    (COL_TYPE,
     COL_WAS_INSTALLED,
     COL_TO_INSTALL,
     COL_NAME,
     COL_DESC,
     COL_ICON,
     COL_ICON_FILE,
     COL_PACKAGE,
     COL_MIME,
     COL_EXEC,
     COL_TERMINAL,
     COL_AVAILABLE,
     COL_SECTION,
     COL_PATH) = range(0, 14)
     
    # Columns for newly installed packages
    (NEWP_NAME,
     NEWP_DESC,
     NEWP_ICON,
     NEWP_PACKAGE,
     NEWP_EXEC,
     NEWP_TERMINAL,
     NEWP_MENU) = range(0, 7)

    # Values representing the users choice from the "unapplied changes" dialog
    (CHANGES_NONE,
     CHANGES_IGNORE,
     CHANGES_APPLY,
     CHANGES_CANCEL) = range(0,4)

    # The base path to the .desktop files
    menudir = None
    
    # The base path to other data (glade, html)
    datadir = None
    
    # The current icon theme
    icons = None

    # The listview store of packages
    store = None
    
    # The listview store for search results
    search_store = None
    
    # The listview store for newly added programs
    new_store = None
    add_store = None
    remove_store = None
    
    # The listview store for programs with multiple entries
    multiple_store = None

    # The actual treeview
    treeview = None
    
    # The mozembed
    browser = None

    # The package install/remove worker to use
    packageWorker = None
    
    # filter and filtered cache for searching
    filter = None
    filtered_cache = None
    
    # search widgets
    entry, search_button, clear_button = (None, None, None)
    
    # window and listview for newly installed programs
    installed_win = None
    installed_view = None
    
    # window and listview for packages with multiple entries
    multiple_win = None
    multiple_view = None
    
    pending_win = None
    pending_add_tree = None
    pending_remove_tree = None
    
    unavailable_win = None
    needed_repo = None
    
    # filenames of info pages
    info_pages = None
    
    # dict of packages with multiple menu entries
    multiple_entry_pkgs = {}
    search_multiples = {}
    
    def __init__(self, menudir, datadir, arguments=None):
        import gtk.glade        
        gtk.Window.__init__(self, gtk.WINDOW_TOPLEVEL)

        self.menudir = menudir
        self.datadir = datadir
        self.info_pages = {}
        
        self.icons = gtk.icon_theme_get_default()
        self.icons.prepend_search_path(os.path.join(self.menudir, "icons"))

        # Setup the window
        self.set_title(_("Add Applications"))
        self.set_default_size (750, 500)
        self.set_position(gtk.WIN_POS_CENTER)
        self.connect("delete-event", self.deleteEvent)
        self.connect("destroy", self.destroyEvent)
        gtk.window_set_default_icon(self.icons.load_icon("gnome-settings-default-applications", 32, 0))
        
        # Grab the top-level container and magically reparent it
        self.glade = glade = gtk.glade.XML(os.path.join(self.datadir, "gnome-app-install.glade"))
        old_window = glade.get_widget("main_window")
        box = old_window.get_child()
        old_window.remove(box)
        self.add(box)

        # XXX: LaunchpadIntegration support, needs better sepearation
        widget = glade.get_widget ("menuitem_help_menu");
        LaunchpadIntegration.set_sourcepackagename("gnome-app-install")
        LaunchpadIntegration.add_items (widget, -1, False, True);
 
        # Set the images
        #glade.get_widget("icon_image").set_from_pixbuf(self.icon("gnome-settings-default-applications", 48)[0])

        # Hack to move the Help button to the beginning of the container
        help = glade.get_widget("help_button")
        help.get_parent().child_set_property(help, "secondary", True)

        # Connect the buttons. TODO: use glade autoconnect
        glade.get_widget("advanced_item").connect("activate", self.advancedClicked)
        glade.get_widget("advanced_button").hide()
        glade.get_widget("repositories_item").connect("activate", self.repositoriesClicked)
        glade.get_widget("quit_item").connect("activate", self.closeClicked)
        self.apply_button = glade.get_widget("apply_button")
        self.apply_button.connect("clicked", self.applyClicked)
        glade.get_widget("close_button").connect("clicked", self.closeClicked)
        
        self.apply_button.set_sensitive(False)

        # about window
        self.about_dialog = glade.get_widget("about_dialog")
        glade.get_widget("about_item").connect("activate", self.aboutClicked)

        # Connect search widgets
        self.left_side = glade.get_widget("left_vbox")
        self.search_entry = glade.get_widget("search_entry")
        self.search_button = glade.get_widget("search_button")
        self.clear_button = glade.get_widget("clear_button")
        self.search_entry.connect("activate", self.searchClicked)
        self.search_button.connect("clicked", self.searchClicked)
        self.clear_button.connect("clicked", self.clearClicked)
        
        self.clear_button.set_sensitive(False)
                
        # Create the treeview store
        base_column_types = (gobject.TYPE_INT, gobject.TYPE_BOOLEAN, gobject.TYPE_BOOLEAN, gobject.TYPE_STRING, gobject.TYPE_STRING, gdk.Pixbuf.__gtype__, gobject.TYPE_STRING, gobject.TYPE_STRING, object, gobject.TYPE_STRING, gobject.TYPE_BOOLEAN, gobject.TYPE_BOOLEAN, gobject.TYPE_STRING)
        self.store = gtk.TreeStore(*base_column_types)
        self.multiple_store = gtk.ListStore(*base_column_types)
        self.add_store = gtk.ListStore(*base_column_types)
        self.remove_store = gtk.ListStore(*base_column_types)
        
        # Create the search results store
        self.search_store = gtk.ListStore(*(base_column_types + (object,)))
        
        featured_file = open(os.path.join(datadir, "featured.txt"))
        self.featured_list = featured_file.read().split()
        
        # Setup the treeview
        # TODO: set background colour of scrolledwindow. test8.py behaves differently here
        self.treeview = glade.get_widget("treeview")
        self.setupTreeview()
        self.treeview.connect("cursor-changed", self.showInfo)
        self.treeview.connect("row-activated", self.rowActivated)
        self.treescroll = glade.get_widget("scrolled_window")

        
        # Create an idle callback to load the APT cache and setup the installed
        # states when the UI starts
        def idle():
            self.updateCache()
            self.isInfoPage = True
            self.isInfoPage = False
            # handle arguments
            if arguments is not None and len(arguments) > 1:
                for arg in arguments[1:]:
                    if arg.startswith("--mime-type="):
                        self.search_entry.set_text("mime-type:%s" % arg[len("--mime-type="):])
                        self.searchClicked(None)
                        
            return False
        gobject.idle_add(idle)
        
        self.intro_page = self.generateIntro()
        
        # Setup gtkmozembed
        self.browser = BrowserView()
        self.isInfoPage = True
	#self.browser.connect("open-uri", self.openWebpage)
        glade.get_widget("browser_box").pack_start(self.browser)
        self.browser.show()

      
        # set up installed window
        # TODO: set up the window on demand
        self.installed_win = glade.get_widget("installed_window")
        self.installed_win.hide()
        self.installed_win.set_transient_for(self)
        self.installed_win.connect("delete-event", self.deleteInstalledWin)
        self.installed_win.connect("destroy", self.destroyInstalledWin)
        self.installed_view = glade.get_widget("installed_view")
        # The program needs to be executed as a normal user, not root. See comment in execPackage
        #self.installed_view.connect("row-activated", self.execPackage)
        glade.get_widget("installed_close").connect("clicked", self.closeInstalledWin)
        glade.get_widget("installed_run_label").hide()
        self.setupInstalledView()
        
        # set up multiple notification window
        self.multiple_win = glade.get_widget("multiple_window")
        self.multiple_win.hide()
        self.multiple_win.set_transient_for(self)
        self.multiple_view = glade.get_widget("multiple_view")
        self.setupSecondaryView(self.multiple_view)
        glade.get_widget("multiple_close").connect("clicked", self.closeMultipleWin)
        
        # set up unavailable program dialog
        self.unavailable_win = glade.get_widget("unavailable_window")
        self.unavailable_win.hide()
        self.unavailable_win.set_transient_for(self)
        self.unavailable_label = glade.get_widget("unavailable_label")
        glade.get_widget("unavailable_ok").connect("clicked", self.enableRepository)
        glade.get_widget("unavailable_cancel").connect("clicked", lambda x: self.unavailable_win.hide()) 
        
        # set up changes pending window
        self.pending_win = glade.get_widget("pending_window")
        self.pending_win.hide()
        self.pending_win.connect("destroy", self.pending_win.hide)
        self.pending_win.set_transient_for(self)
        self.pending_add_tree = glade.get_widget("pending_add_tree")
        self.pending_add_tree.set_model(self.add_store)
        self.setupSecondaryView(self.pending_add_tree)
        self.pending_remove_tree = glade.get_widget("pending_remove_tree")
        self.pending_remove_tree.set_model(self.remove_store)
        self.setupSecondaryView(self.pending_remove_tree)
        self.pending_label = glade.get_widget("pending_label")
        self.pending_ignore = glade.get_widget("ignore_change_button")
        
        self.show()

        self.packageWorker = PackageWorker()
        
    def icon(self, name, size):
        if name is None or name == "":
            warn("ICON: Using dummy icon")
            name = "gnome-other"
        if name.startswith("/"):
            warn("ICON: Doesn't handle absolute paths: '%s'" % name)
            name = "gnome-other"
        if name.find(".") != -1:
            import os.path
            # splitting off extensions in all cases broke evolution's icon
            # hopefully no common image extensions will ever have len != 3
            if len(os.path.splitext(name)[1]) == (3 + 1): # extension includes '.'
                name = os.path.splitext(name)[0]
        if not self.icons.has_icon(name):
            warn("ICON: Icon '%s' is not in theme" % name)
            name = "gnome-other"
        # FIXME: mvo: this try: except is a hack to work around 
        #             ubuntu #6858 (icon is no longer in cache after removal)
        #             of a pkg. correct is probably to reload the icontheme
        #             (or something)
        try:
            icon = self.icons.load_icon(name, size, 0)
        except gobject.GError:
            icon = self.icons.load_icon("gnome-other", size, 0)
            name = "gnome-other"
        if icon.get_width() != size:
            warn("ICON: Got badly sized icon for %s" % name)
            icon = icon.scale_simple(size, size, gdk.INTERP_BILINEAR)
        
        info = self.icons.lookup_icon(name, size, gtk.ICON_LOOKUP_NO_SVG | gtk.ICON_LOOKUP_USE_BUILTIN)
        if info is None:
            info = self.icons.lookup_icon("gnome-other", size, gtk.ICON_LOOKUP_NO_SVG | gtk.ICON_LOOKUP_USE_BUILTIN)
        filename = info.get_filename()
        
        return icon, filename
    
    def setupTreeview(self):
        self.treeview.set_search_column(self.COL_NAME)
        
        def visibility_magic(cell_layout, renderer, model, iter, visible_types):
            row_type = model.get_value(iter, self.COL_TYPE)
            if isinstance(visible_types, int):
                renderer.set_property("visible", row_type == visible_types)
            else:
                renderer.set_property("visible", row_type in visible_types)

        def package_view_func(cell_layout, renderer, model, iter):
            visibility_magic(cell_layout, renderer, model, iter, self.TYPE_PACKAGE)
            name, desc, current, future, available = model.get(iter, self.COL_NAME, self.COL_DESC, self.COL_WAS_INSTALLED, self.COL_TO_INSTALL, self.COL_AVAILABLE)
            if not available:
                markup = "<span foreground=\"gray\"><i>%s\n<small>%s</small></i></span>" % (name, desc)
            elif current != future:
                markup = "<b>%s</b>\n<small>%s</small>" % (name, desc)
            else:
                markup = "%s\n<small>%s</small>" % (name, desc)
            renderer.set_property("markup", markup)


        def on_install_toggle(renderer, path, store, self):
            i = store.get_iter(path)
            name, available, section = store.get(i, self.COL_PACKAGE, self.COL_AVAILABLE, self.COL_SECTION)
            if not available:
                # check if we have seen the component
                for it in self.cache._cache.FileList:
                    if it.Component != "" and it.Component == section:
                        # warn that this app is not available on this plattform
                        d = gtk.MessageDialog(parent=self,
                                              flags=gtk.DIALOG_MODAL,
                                              type=gtk.MESSAGE_INFO,
                                              buttons=gtk.BUTTONS_OK)
                        d.set_markup("<big><b>%s</b></big>\n\n%s" % (
                            _("Application '%s' not available" % name),
                            _("The application can not be found in your "
                              "archive. This usually means that it is not "
                              "available for your hardware plattform.")))
                        d.run()
                        d.destroy()
                        return
                self.needed_repo = section
                self.unavailable_label.set_text(_("This program is currently not installable, but should be available in the '%s' repository. Would you like to enable this repository?" % self.needed_repo))
                self.unavailable_win.show()
                return

            # first check if the operation is save
            pkg = name
            if self.cache[pkg].isInstalled:
                # check if it can be removed savly
                self.cache[pkg].markDelete(autoFix=False)
                if self.cache._depcache.BrokenCount > 0:
                    d = gtk.MessageDialog(parent=self,
                                          flags=gtk.DIALOG_MODAL,
                                          type=gtk.MESSAGE_INFO,
                                          buttons=gtk.BUTTONS_OK)
                    d.set_markup("<big><b>%s</b></big>\n\n%s" % (
                                 _("Cannot remove '%s'" % pkg),
                                 _("There are other applications depending "
                                   "on this one. If you really want to remove "
                                   "it, please use the \"Advanced\" mode from "
                                   "the menu.")))
                    d.run()
                    d.destroy()
                    self.cache[pkg].markKeep()
                    assert self.cache._depcache.BrokenCount == 0
                    assert self.cache._depcache.DelCount == 0
                    return
                self.cache[pkg].markKeep()
                # FIXME: those assert may be a bit too strong,
                # we may just rebuild the cache if something is
                # wrong
                assert self.cache._depcache.BrokenCount == 0
                assert self.cache._depcache.DelCount == 0
            else:
                # check if it can be installed savely
                apt_error = False
                try:
                    self.cache[pkg].markInstall(autoFix=True)
                except SystemError:
                    apt_error = True
                if self.cache._depcache.BrokenCount > 0 or \
                   self.cache._depcache.DelCount > 0 or apt_error:
                    d = gtk.MessageDialog(parent=self, flags=gtk.DIALOG_MODAL,
                                          type=gtk.MESSAGE_INFO,
                                          buttons=gtk.BUTTONS_OK)
                    d.set_markup("<big><b>%s</b></big>\n\n%s" % (
                                 _("Cannot install '%s'" % pkg),
                                 _("Installing this application would mean "
                                   "that something else needs to be removed. "
                                   "Please use the \"Advanced\" mode to "
                                   "install '%s'." % pkg)))
                    d.run()
                    d.destroy()
                    # reset the cache
                    # FIXME: a "pkgSimulateInstall,remove"  thing would
                    # be nice
                    for p in self.cache.keys():
                        self.cache[p].markKeep()
                    # FIXME: those assert may be a bit too strong,
                    # we may just rebuild the cache if something is
                    # wrong
                    assert self.cache._depcache.BrokenCount == 0
                    assert self.cache._depcache.DelCount == 0
                    return
            
            if name in self.multiple_entry_pkgs.keys():
                # if the user hasn't been notified, notify
                if not self.multiple_entry_pkgs[name][0]:
                    self.notifyMultiple(name)
            if store == self.store and name in self.multiple_entry_pkgs.keys():
                for pair in self.multiple_entry_pkgs[name][1]:
                    store.set_value (store.get_iter(pair[1].get_path()), self.COL_TO_INSTALL, not store.get_value (store.get_iter(pair[1].get_path()), self.COL_TO_INSTALL))
            elif store == self.search_store and name in self.search_multiples.keys():
                for mpath in self.search_multiples[name]:
                    store.set_value (store.get_iter(mpath), self.COL_TO_INSTALL, not store.get_value (store.get_iter(mpath), self.COL_TO_INSTALL))
            else:
                store.set_value (i, self.COL_TO_INSTALL, not store.get_value (i, self.COL_TO_INSTALL))
                
            self.apply_button.set_sensitive(self.isChanged())


        column = gtk.TreeViewColumn("")
        check_column = gtk.TreeViewColumn("Installed")
        
        # check boxes
        self.toggle_render = gtk.CellRendererToggle()
        self.store_toggle_id = self.toggle_render.connect('toggled', on_install_toggle, self.store, self)
        self.search_toggle_id = self.toggle_render.connect('toggled', on_install_toggle, self.search_store, self)
        self.toggle_render.handler_block(self.search_toggle_id)
        self.toggle_render.set_property("xalign", 0.3)
        check_column.pack_start(self.toggle_render, False)
        check_column.add_attribute(self.toggle_render, "active", self.COL_TO_INSTALL)
        check_column.set_cell_data_func (self.toggle_render, visibility_magic, self.TYPE_PACKAGE)
        
        # program icons
        render = gtk.CellRendererPixbuf()
        column.pack_start(render, False)
        column.add_attribute(render, "pixbuf", self.COL_ICON)
        column.set_cell_data_func (render, visibility_magic, (self.TYPE_GROUP, self.TYPE_PACKAGE))

        # menu group names
        render = gtk.CellRendererText()
        render.set_property("scale", 1.0)
        render.set_property("weight", 700)
        column.pack_start(render, True)
        column.add_attribute(render, "markup", self.COL_NAME)
        column.set_cell_data_func (render, visibility_magic, self.TYPE_GROUP)

        # package names
        render = gtk.CellRendererText()
        column.pack_start(render, True)
        column.add_attribute(render, "markup", self.COL_NAME)
        column.set_cell_data_func (render, package_view_func)
        
        # headers
        render = gtk.CellRendererText()
        render.set_property("xpad", 4)
        render.set_property("ypad", 4)        
        render.set_property("scale", 1.3)
        render.set_property("weight", 700)
        self.connect("style-set", lambda parent, old, widget: widget.set_property ("foreground-gdk", parent.get_style().fg[gtk.STATE_SELECTED]), render)
        self.connect("style-set", lambda parent, old, widget: widget.set_property ("background-gdk", parent.get_style().bg[gtk.STATE_SELECTED]), render)
        column.pack_start(render, False)
        column.add_attribute(render, "text", self.COL_NAME)
        column.set_cell_data_func (render, visibility_magic, self.TYPE_HEADER)

        # indent headers (i think)
        render = gtk.CellRendererText()
        render.set_property("width", self.treeview.style_get_property("expander-size") + self.treeview.style_get_property("horizontal-separator"))
        column.pack_start(render, False)
        column.set_cell_data_func (render, visibility_magic, self.TYPE_HEADER)

        self.treeview.append_column(check_column)
        self.treeview.append_column(column)
        
        
        
    # --------------------------------------
    # Multiple notification window functions
    # --------------------------------------
    
    def notifyMultiple(self, pkgname):
        if self.multiple_entry_pkgs[pkgname][0]:
            return
        self.multiple_store.clear()
        for row_pair in self.multiple_entry_pkgs[pkgname][1]:
            i = self.multiple_store.append()
            self.multiple_store.set(i, *row_pair[0])
        self.multiple_view.set_model(self.multiple_store)
        self.multiple_win.show()
        self.multiple_entry_pkgs[pkgname][0] = True
    
    def setupSecondaryView(self, view):
        def package_view_func(cell_layout, renderer, model, iter):
            name, desc, current, future = model.get(iter, self.COL_NAME, self.COL_DESC, self.COL_WAS_INSTALLED, self.COL_TO_INSTALL)
            renderer.set_property("markup", "%s\n<small>%s</small>" % (name, desc))
        
        view.set_search_column(self.COL_NAME)
        view.get_selection().set_mode(gtk.SELECTION_NONE)
        
        column = gtk.TreeViewColumn("")
        
        render = gtk.CellRendererPixbuf()
        column.pack_start(render, False)
        column.add_attribute(render, "pixbuf", self.COL_ICON)
        
        render = gtk.CellRendererText()
        render.set_property("xpad", 4)
        column.pack_start(render, True)
        column.add_attribute(render, "markup", self.COL_NAME)
        column.set_cell_data_func (render, package_view_func)
        
        view.append_column(column)
        
    def closeMultipleWin(self, button):
        self.multiple_win.hide()

    # -------------------------------
    # Menu entry processing functions
    # -------------------------------
    
    temp_dupes = {}
    
    def populateFromEntry(self, store, node, parent = None, more_node = None):
        for entry in node.getEntries():
            if isinstance(entry, xdg.Menu.Menu):
                i = store.append(parent)
                icon, icon_file = self.icon(entry.getIcon(), 32)
                store.set(i, self.COL_TYPE, self.TYPE_GROUP, self.COL_NAME, xmlescape(entry.getName()), self.COL_ICON, icon, self.COL_ICON_FILE, icon_file)
                more = store.append(i)
                store.set(more, self.COL_TYPE, self.TYPE_GROUP, self.COL_NAME, _("More programs..."))
                self.populateFromEntry(store, entry, i, more)
            elif isinstance(entry, xdg.Menu.MenuEntry):
                if entry.DesktopEntry.hasKey("X-AppInstall-Package"):
                    pkgname = entry.DesktopEntry.get("X-AppInstall-Package")
                    section = entry.DesktopEntry.get("X-AppInstall-Section")
                    if self.cache.has_key(pkgname):
                        available = True
                    else:
                        available = False
                    if pkgname in self.featured_list:
                        i = store.append(parent)
                    else:
                        i = store.append(more_node)
                    icon, icon_file = self.icon(entry.DesktopEntry.get("X-AppInstall-Icon", "") or entry.DesktopEntry.getIcon(), 24)
                    columns = (self.COL_TYPE, self.TYPE_PACKAGE, self.COL_NAME, xmlescape(entry.DesktopEntry.getName()), self.COL_DESC, xmlescape(entry.DesktopEntry.getComment()) or "", self.COL_ICON, icon, self.COL_ICON_FILE, icon_file, self.COL_PACKAGE, pkgname, self.COL_MIME, entry.DesktopEntry.getMimeType(), self.COL_EXEC, entry.DesktopEntry.getExec(), self.COL_TERMINAL, entry.DesktopEntry.getTerminal(), self.COL_AVAILABLE, available, self.COL_SECTION, section)
                    store.set(i, *columns)
                    if self.temp_dupes.has_key(pkgname):
                        self.temp_dupes[pkgname] = self.temp_dupes[pkgname] + [(columns, gtk.TreeRowReference(store, store.get_path(i)))]
                    else:
                        self.temp_dupes[pkgname] = [(columns, gtk.TreeRowReference(store, store.get_path(i)))]
                else:
                    print "Got non-package menu entry %s" % entry.DesktopEntry.getName()
            elif isinstance(entry, xdg.Menu.Header):
                print "got header"
                
            if more_node is not None and store.iter_n_children(parent) > 1:

                last_iter = store.iter_nth_child(parent, store.iter_n_children(parent) - 1)
                store.move_after(more_node, last_iter)
                
    def populateMenu(self):
        def add_header(store, name):
            store.set (store.append(None), self.COL_TYPE, self.TYPE_HEADER, self.COL_NAME, xmlescape(name), self.COL_ICON, None)
        
        menu = xdg.Menu.parse(os.path.join(self.menudir, "applications.menu"))
        # TODO: this should be done in xdg.Menu
        #add_header(self.store, _("Applications"))
        self.populateFromEntry(self.store, menu)
        if self.multiple_entry_pkgs == {}:
            for key in self.temp_dupes.keys():
                if len(self.temp_dupes[key]) > 1:
                    self.multiple_entry_pkgs[key] = [False, self.temp_dupes[key]]
        self.temp_dupes.clear()
        # mvo: disabled for now, nothing in it (ubuntu #7311)
        #add_header(self.store, _("System Services")) # TODO. Somehow. Not sure.
        self.treeview.set_model(self.store)

    # ----------------
    # Update functions
    # ----------------
    
    def updateCache(self):
        self.set_sensitive(False)

        # get the lock
        try:
            apt_pkg.PkgSystemLock()
        except SystemError:
            d = gtk.MessageDialog(parent=self,
                                  flags=gtk.DIALOG_MODAL,
                                  type=gtk.MESSAGE_ERROR,
                                  buttons=gtk.BUTTONS_OK)
            d.set_markup("<big><b>%s</b></big>\n\n%s" % (
                _("Unable to get exclusive lock"),
                _("This usually means that another package management "
                  "application (like apt-get or aptitude) already running. "
                  "Please close that application first")))
            res = d.run()
            d.destroy()
            sys.exit()

        self.window.set_cursor(gdk.Cursor(gdk.WATCH)) ; gtk.main_iteration()
        self.cache = apt.Cache(OpProgressWindow(self.glade, self))
        self.store.clear()
        # the new store invalidates TreeRowReferences, so make new ones.
        self.multiple_entry_pkgs.clear()
        
        # clear search cache
        self.filtered_cache = None
        
        self.populateMenu()
        self.updateInstalledStates()
        self.browser.loadUri("file://" + self.intro_page)
        
        for file in self.info_pages.values():
            os.remove(file)
        self.info_pages.clear()
        
        adj = self.treescroll.get_vadjustment()
        adj.set_value(0)
        
        self.window.set_cursor(None)
        self.set_sensitive(True)
    
    def updateInstalledStates(self):
        def each(model, path, iter, self):
            type, name = model.get(iter, self.COL_TYPE, self.COL_PACKAGE)
            if type == self.TYPE_PACKAGE:
                try:
                    installed = self.cache[name].isInstalled
                    model.set(iter, self.COL_WAS_INSTALLED, installed, self.COL_TO_INSTALL, installed)
                except KeyError:
                    pass                # ignore stuff that is not in the cache
        self.store.foreach(each, self)
    
    def getChanges(self, get_paths=False):
        # If search results are being displayed, copy back any changes
        if self.treeview.get_model() is not self.store:
            self.clearClicked(None)
        self.updateChangeStores()
        selections = []
        packages = []
        def each(store, path, i):
            previous, future, name = store.get (i, self.COL_WAS_INSTALLED, self.COL_TO_INSTALL, self.COL_PACKAGE)
            if name in packages and not get_paths: return # filter out multiples
            if previous == future: return
            if previous and not future:
                selections.append(("%s\tuninstall" % name, path))
                packages.append(name)
            if not previous and future:
                selections.append(("%s\tinstall" % name, path))
                packages.append(name)
        self.store.foreach(each)
        if get_paths:
            return selections
        else:
            return [x[0] for x in selections]
        
    def isChanged(self):
        self.temp_changed = False
        self.temp_changed_paths = []
        def each(store, path, i):
            previous, future = store.get(i, self.COL_WAS_INSTALLED, self.COL_TO_INSTALL)
            original_path = None
            if store is self.search_store:
                original_path = store.get_value(i, self.COL_PATH)
                self.temp_changed_paths.append(original_path)
            if previous != future and not path in self.temp_changed_paths:
                self.temp_changed = True
                
        if self.treeview.get_model() is self.search_store:
            self.search_store.foreach(each)
            if self.temp_changed:
                return True
            
        self.store.foreach(each)
        return self.temp_changed
    
    def reapplyChanges(self, selections):
        for item in selections:
            # if the updated store and the old store don't have the same paths, this will break.
            # that shouldn't happen.
            pkgname, available = self.store.get(self.store.get_iter(item[1]), self.COL_PACKAGE, self.COL_AVAILABLE)
            # don't copy back changes to an uninstallable package
            if not available:
                continue
            if not pkgname in item[0]:
                print "ERROR: Package name mismatch while reapplying changes"
            if "uninstall" in item[0]:
                to_install = False
            elif "install" in item[0]:
                to_install = True
            else: # this should never happen.
                continue
            self.store.set(self.store.get_iter(item[1]), self.COL_TO_INSTALL, to_install)
            
    def printChanges(self, selections):
        for package in selections:
            print package.split()[0]
    
    def buttonHBox(self, label_text, stock_id):
        image = gtk.image_new_from_stock(stock_id, gtk.ICON_SIZE_BUTTON)
        label = gtk.Label(label_text)
        hbox = gtk.HBox(False, 0)
        hbox.set_border_width(2)
        hbox.pack_start(image, False, False, 3)
        hbox.pack_start(label, False, False, 3)
        
        image.show()
        label.show()
        
        return hbox
    
    def ignoreChanges(self):
        """
        If any changes have been made, ask the user to apply them and return a value based on the status.
        Returns self.CHANGES_NONE, self.CHANGES_IGNORE, self.CHANGES_APPLY, or self.CHANGES_CANCEL.
        """
        if (len(self.getChanges()) == 0):
            return self.CHANGES_NONE
        
        box = self.buttonHBox(_("Ignore"), gtk.STOCK_CANCEL)
        self.pending_ignore.remove(self.pending_ignore.get_child())
        self.pending_ignore.add(box)
        box.show()
        self.pending_label.set_markup(_("<big><b>Changes Pending</b></big>\n\nYou have selected to add or remove the following applications, but these changes have not been applied. Would you like to apply them?"))
        
        self.adjustPendingWin()
        
        res = self.pending_win.run()
        self.pending_win.hide()
        if res == gtk.RESPONSE_DELETE_EVENT:
            res = self.CHANGES_CANCEL
        return res
        
    def updateChangeStores(self):
        # If search results are being displayed, copy back any changes
        if self.treeview.get_model() is not self.store:
            self.clearClicked(None)
        if self.new_store == None:
            self.new_store = gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_STRING, gdk.Pixbuf.__gtype__, gobject.TYPE_STRING, gobject.TYPE_STRING, gobject.TYPE_BOOLEAN, object)
        self.add_store.clear()
        self.remove_store.clear()
        self.header = _("Applications")
        def each(store, path, i):
            if store.get_value(i, self.COL_TYPE) == self.TYPE_HEADER:
                self.header = store.get_value(i, self.COL_NAME)
            previous, future = store.get (i, self.COL_WAS_INSTALLED, self.COL_TO_INSTALL)
            if previous and not future:
                self.remove_store.append(tuple(store[i]))
            elif not previous and future:
                menu_path = [self.header]
                for j in range(1, len(path)):
                    if store[ path[0:j] ][self.COL_NAME] != _("More programs..."):
                        menu_path.append(store[ path[0:j] ][self.COL_NAME])
                self.new_store.append(store.get(i, self.COL_NAME, self.COL_DESC, self.COL_ICON, self.COL_PACKAGE, self.COL_EXEC, self.COL_TERMINAL) + (menu_path,))
                self.add_store.append(tuple(store[i]))
        self.store.foreach(each)
        
    def checkNewStore(self):
        to_remove = []
        def each(store, path, i):
            name = store.get_value(i, self.COL_NAME)
            if not self.cache[name].isInstalled:
                to_remove.append(gtk.TreeRowReference(store, path))
        self.new_store.foreach(each)
        for ref in to_remove:
            self.new_store.remove(self.new_store.get_iter(ref.get_path()))
        
    # -------------------------------
    # "New programs" dialog functions
    # -------------------------------
    
    def setupInstalledView(self):
        def package_view_func(cell_layout, renderer, model, iter):
            name, menu_path = model.get(iter, self.NEWP_NAME, self.NEWP_MENU)
            menu_text = "</b> &gt; <b>".join(menu_path)
            menu_text = "<b>" + menu_text + "</b>"
            renderer.set_property("markup", "%s\n<small>Menu: %s</small>" % (name, menu_text))
        
        self.installed_view.set_search_column(self.NEWP_NAME)
        self.installed_view.get_selection().set_mode(gtk.SELECTION_NONE)
        
        column = gtk.TreeViewColumn("")
        
        render = gtk.CellRendererPixbuf()
        column.pack_start(render, False)
        column.add_attribute(render, "pixbuf", self.NEWP_ICON)
        
        render = gtk.CellRendererText()
        render.set_property("xpad", 4)
        column.pack_start(render, True)
        column.add_attribute(render, "markup", self.NEWP_NAME)
        column.set_cell_data_func (render, package_view_func)
        
        self.installed_view.append_column(column)
        
    def execPackage(self, treeview, path, view_column):
        # PROBLEM: since gnome-app-install is run as root, the executed program is as well.
        # g-a-i needs to be run as a normal user and pipe into a gksudo'd synaptic. somehow.
        # see gksu module in gnome-python-extras devel branch
        treeiter = self.new_store.get_iter(path)
        command, terminal = self.new_store.get(treeiter, self.NEWP_EXEC, self.NEWP_TERMINAL)
        cmd_parts = []
        if command == "": return
        for part in command.split():
            while True:
                # two consecutive '%' characters represent an actual '%'
                if len(part) >= 2 and part[:2] == '%%':
                    cmd_parts.append('%')
                    part = part[2:]
                    continue
                # we're running the command without any options, so strip out placeholders
                if part[0] == '%': break
                # if the last part was an actual '%', we don't want to join it with a space, so do it by hand
                if cmd_parts[-1:] == '%':
                    part = '%' + part
                    cmd_parts[-1:] = part
                    break
                cmd_parts.append(part)
                break
        
        if terminal:
            command = " ".join(cmd_parts)
            command = "gnome-terminal --command=\"" + command + "\""
            cmd_parts = command.split()
        
        # run program
        os.spawnvp(os.P_NOWAIT, cmd_parts[0], cmd_parts)
        
    
    # ----------------------------
    # Main window button callbacks
    # ----------------------------
    
    def advancedClicked(self, button):
        res = self.ignoreChanges()
        # If we should cancel, return now
        if res == self.CHANGES_CANCEL:
            return
        # If we need to apply, do it now
        if res == gtk.RESPONSE_APPLY:
            self.applyClicked(None)
        # Here we are either ignoring or have applied, so run Synaptic
        os.spawnl(os.P_NOWAIT, "/usr/sbin/synaptic", "synaptic") # TODO: activate startup notification (needs wrapping)
        self.quit()
        
    def repositoriesClicked(self, widget):
        """ start gnome-software preferences """
        # args: "-n" means we take care of the reloading of the
        # package list ourself
        args = ['/usr/bin/gnome-software-properties', '-n']
        child = subprocess.Popen(args)
        self.set_sensitive(False)
        res = None
        while res == None:
            res = child.poll()
            time.sleep(0.05)
            while gtk.events_pending():
                gtk.main_iteration()
        # repository information changed, call "reload"
        if res > 0:
            self.reloadSources()
        self.set_sensitive(True)
        gtk.main_iteration()

    def applyClicked(self, button):
        selections = self.getChanges()
        
        # if the signal didn't come from a button, the calling function already talked to the user
        if button != None:
            box = self.buttonHBox(_("Cancel"), gtk.STOCK_CANCEL)
            self.pending_ignore.remove(self.pending_ignore.get_child())
            self.pending_ignore.add(box)
            box.show()
            self.pending_label.set_markup(_("<big><b>Changes Pending</b></big>\n\nThe following changes will be applied. Are you sure you want to continue?"))

            # FIXME: dynamically make apply sensitive depending if
            # something was selected or not
            if len(self.add_store) == len(self.remove_store) == 0:
                return

            self.adjustPendingWin()

            res = self.pending_win.run()
            self.pending_win.hide()
            
            if res == self.CHANGES_NONE or res == gtk.RESPONSE_REJECT:
                return
            elif res == gtk.RESPONSE_APPLY:
                pass
        
        # Set a busy cursor
        self.window.set_cursor(gdk.Cursor(gdk.WATCH)) ; gtk.main_iteration()
        
        # Get the selections delta for the changes and apply them
        self.packageWorker.perform_action(self,selections)
        
        # Print out list of changed packages (useful for generated featured list and blacklist)
        #self.printChanges(self.getChanges())
        
        # Reload the APT cache and treeview
        self.updateCache()
        
        # Show window with newly installed programs
        self.checkNewStore() # only show things that successfully installed
        if len(self.new_store) > 0:
            self.installed_view.set_model(self.new_store)
            self.installed_win.set_position(gtk.WIN_POS_CENTER_ON_PARENT)
            self.installed_win.show()
        
        # And reset the cursor
        self.window.set_cursor(None)
        
    def adjustPendingWin(self):
        # see what to show/hide
        new_height = 500
        for action in ["add", "remove"]:
            w = self.glade.get_widget("vbox_%s" % action)
            if len(getattr(self,"%s_store" % action)) == 0:
                w.hide()
                new_height -= 225
            else:
                w.show()
                
        width, height = self.pending_win.get_size()
        self.pending_win.resize(width, new_height)
            
    def searchClicked(self, widget, query=None):
        self.set_sensitive(False) ; gtk.main_iteration()
        self.window.set_cursor(gdk.Cursor(gdk.WATCH)) ; gtk.main_iteration()
        self.clearInfo()
        
        if query == None:
            query = self.search_entry.get_text()
        #print "Searching for \"" + query + "\""

        self.copySearchChanges()
        self.search_store.clear()
        #self.clearClicked(self.clear_button)
        
        if query.lower().startswith("mime-type:"):
            mime_query = query[len("mime-type:"):].strip()
            self.mimeSearch(mime_query)
        else:
            self.aptSearch(query)
        
        # populate list of multiples and their new paths in the search_store
        temp_multiples = {}
        for row in self.search_store:
            pkgname = row[self.COL_PACKAGE]
            if not pkgname in self.multiple_entry_pkgs:
                continue
            if temp_multiples.has_key(pkgname):
                temp_multiples[pkgname] = temp_multiples[pkgname] + [row.path]
            else:
                temp_multiples[pkgname] = [row.path]
        
        for key in temp_multiples.keys():
            if len(temp_multiples[key]) > 1:
                self.search_multiples[key] = temp_multiples[key]
        
        # If no results are found, add a placeholder
        if len(self.search_store) == 0:
            self.search_store.set(self.search_store.append(None), self.COL_TYPE, self.TYPE_HEADER, self.COL_NAME, xmlescape(_("No results found")), self.COL_ICON, None, self.COL_PATH, None)
        
        if self.treeview.get_model() is self.store:
            # Process toggle events using the search store
            self.toggle_render.handler_block(self.store_toggle_id)
            self.toggle_render.handler_unblock(self.search_toggle_id)
        
        # Show the search results in the tree view
        self.treeview.set_model(self.search_store)
        self.treeview.set_rules_hint(True)
        
        self.window.set_cursor(None)
        self.set_sensitive(True)
        self.clear_button.set_sensitive(True)

    def copySearchChanges(self):
        # Copy any changes in the search store back to the original store
        for row in self.search_store:
            if row[self.COL_PATH] == None:
                print "No path found. If the header doesn't say \"No results found\", this is bad."
                continue
            
            name = row[self.COL_PACKAGE]
            
            if name in self.multiple_entry_pkgs.keys():
                for pair in self.multiple_entry_pkgs[name][1]:
                    self.store.set_value (self.store.get_iter(pair[1].get_path()), self.COL_TO_INSTALL, row[self.COL_TO_INSTALL])
            else:
                treeiter = self.store.get_iter(row[self.COL_PATH])
                self.store.set_value(treeiter, self.COL_TO_INSTALL, row[self.COL_TO_INSTALL])
        

        
    def clearClicked(self, button):
        self.search_entry.set_text("")

        self.copySearchChanges()
        self.search_multiples = {}

        
        # Redirect toggle events back to the original store
        self.toggle_render.handler_block(self.search_toggle_id)
        self.toggle_render.handler_unblock(self.store_toggle_id)
        
        # Point the treeview back to the original store
        self.treeview.set_model(self.store)
        self.treeview.set_rules_hint(False)
        
        self.clear_button.set_sensitive(False)

    def aboutClicked(self, button):
        self.about_dialog.run()
        self.about_dialog.hide()
        
    def closeClicked(self, button):
        res = self.ignoreChanges()
        if res == self.CHANGES_NONE or res == gtk.RESPONSE_REJECT:
            if self.installed_win.get_property("visible") == False:
                self.quit()
            else:
                self.hide()
        elif res == gtk.RESPONSE_APPLY:
            self.applyClicked(None)
            if self.installed_win.get_property("visible") == False:
                self.quit()
            else:
                self.hide()
                
    def reloadSources(self):
        self.set_sensitive(False)
        
        # set the cursor to the previously selected program after reloading.
        # this only works if the path stays the same in between, which should
        # always happen.
        path = self.treeview.get_cursor()[0]
        if self.treeview.get_model() is not self.store:
            path = self.treeview.get_model()[path][self.COL_PATH]
            
        changes = self.getChanges(True)
        self.packageWorker.perform_action(self, action=PackageWorker.UPDATE)
        self.updateCache()
        self.reapplyChanges(changes)
        
        if path != None:
            self.treeview.expand_to_path(path)
            self.treeview.set_cursor(path)
        
        self.set_sensitive(True)
                
    def enableRepository(self, button):
        self.unavailable_win.hide()
        # sanity check
        if self.needed_repo == "":
            print "no repo found in enableRepository"
            return
        sources = open("/etc/apt/sources.list")
        lines = sources.readlines()
        mirror = "archive.ubuntu.com"
        for line in lines:
            if line.startswith("deb"):
                parts = line.split()
                try:
                    new_mirror = parts[1].split("/")[2]
                except:
                    continue
                if mirror in new_mirror:
                    mirror = new_mirror
                    break
        sources.close()
        sources = open("/etc/apt/sources.list", "a")
        newline = "deb http://%s/ubuntu/ %s %s" % (mirror, DISTRO_VERSION, self.needed_repo)
        sources.write("\n" + newline + "\n")
        sources.close()
        
        self.reloadSources()
            
        
    # ----------------
    # Search functions
    # ----------------
    
    def aptSearch(self, query):
        fcache = SimpleFilteredCache(cache=self.cache)
        fcache.setFilter(SearchFilter(query))
        fcache.runFilter()
        results = fcache.keys()

        # then search the desktop file information
        def match(row):
            if row[self.COL_TYPE] == self.TYPE_PACKAGE and\
               (row[self.COL_NAME] and query in row[self.COL_NAME].lower()) or\
               (row[self.COL_DESC] and query in row[self.COL_DESC].lower()):
                if not row[self.COL_PACKAGE] in results:
                    results.append(row[self.COL_PACKAGE])
        def recurse(row):
            match(row)
            for child in row.iterchildren():
                recurse(child)
        for row in self.store:
            recurse(row)
        
        # build a store of the results, adding a path column to each
        # row for later use
        def treeSearch(row):
            if not row: return
            if self.store.get_value(row, self.COL_PACKAGE) in results:
                values = list(self.store.get(row, *range(0,13)))
                self.search_store.append(values + [self.store.get_path(row)])
            treeSearch(self.store.iter_children(row))
            treeSearch(self.store.iter_next(row))
            
        treeSearch(self.store.get_iter_first())
    
    def mimeSearch(self, mime_type):
        
        def match(row):
            matched = []
            patterns = row[self.COL_MIME]
            if patterns is not None:
                for repattern in patterns:
                    # mvo: we get a list of regexp from
                    # pyxdg.DesktopEntry.getMimeType, but it does not
                    # use any special pattern at all, so we use the plain
                    # pattern (e.g. text/html, audio/mp3 here)
                    pattern = repattern.pattern
                    if mime_type in pattern:
                        self.search_store.append(list(row) + [row.path])
        def recurse(row):
            match(row)
            for child in row.iterchildren():
                recurse(child)
        for row in self.store:
            recurse(row)
    
    
    # ---------------------------
    # Window management functions
    # ---------------------------
    
    def closeInstalledWin(self, button):
        self.installed_win.hide()
        self.new_store = None
        if self.get_property("visible") == False:
            self.quit()
    
    def destroyInstalledWin(self, data=None):
        if self.get_property("visible") == False:
            self.quit()
    
    def deleteInstalledWin(self, window, event):
        self.installed_win.hide()
        self.new_store = None
        return False

    def deleteEvent(self, window, event):
        res = self.ignoreChanges()
        if res == self.CHANGES_NONE or res == gtk.RESPONSE_REJECT:
            return False
        elif res == gtk.RESPONSE_APPLY:
            self.applyClicked(None)
            return False
        return True
        
    def destroyEvent(self, data=None):
        if self.installed_win.get_property("visible") == False:
            self.quit()
            
    def quit(self):
        for file in self.info_pages.values():
            os.remove(file)
        os.remove(self.intro_page)
        gtk.main_quit()
            
    # -------------------
    # Info page functions
    # -------------------
    
    def generateInfoPage(self, program_name, package_name, icon_filename, summary, available, section):
        data = {"name" : program_name or "", "summary" : summary or "", "bgcolor" : "white"}
        
        if not icon_filename.endswith(".xpm"):
            data["icon"] = '<img src=\"' + icon_filename + '\" align=\"right\" />'
        else:
            data["icon"] = ""
        
        if available:
            pkg = self.cache[package_name]
            rough_desc = pkg.description
            
            # Replace URLs with links
            # Uncomment this when the gtkmozembed bug is fixed.
            #rough_desc = re.sub('(^|[\\s.:;?\\-\\]<])' + 
            #          '(http://[-\\w;/?:@&=+$.!~*\'()%,#]+[\\w/])' +
            #          '(?=$|[\\s.:;?\\-\\[\\]>])',
            #          '\\1<a href="\\2">\\2</a>', rough_desc)
                    
            lines = rough_desc.split('\n')
            #print rough_desc
                                    
            # remove wrapped newlines, omit first line synopsis
            if len(lines) == 0: clean_desc = ""
            elif len(lines) == 1: clean_desc = lines[0]
            else: clean_desc = lines[1]
            for i in range(2, len(lines)):
                if lines[i].split() == []:
                    continue
                
                # preserve non-wrapping newlines and newlines before a bullet
                first_chunk = lines[i].split()[0]
                # this is ugly. if any wrapping bugs pop up, simplify this.
                # the len != 1 test is supposed to avoid checking if every string is in the bullet list
                if len(lines[i-1] + " " + first_chunk) > 65 and (len(first_chunk) != 1 or first_chunk not in ['*','o','-']):
                    clean_desc = clean_desc + " " + lines[i]
                else:
                    clean_desc = clean_desc + "<br />" + lines[i]
                    
            data["description"] = clean_desc
            #print 'Removed \"' + lines[0] + '\" from the description.'
        else:
            data['description'] = _("<hr>This program is not currently installable. It should be available in the \"%s\" section of the repository. Enable that section in the <b>Repositories</b> dialog in the <b>Settings</b> menu to install it." % section)
        
        if package_name in self.multiple_entry_pkgs.keys():
            other_names = []
            for pair in self.multiple_entry_pkgs[package_name][1]:
                other_names.append(self.store.get_value(self.store.get_iter(pair[1].get_path()), self.COL_NAME))
            other_names.remove(program_name)
            data["multiple_info"] = _("The package that provides this program also provides the following programs: ") + ", ".join(other_names)
        else:
            data["multiple_info"] = ""
            
        
        template_file = open(os.path.join(self.datadir, "template.html"))
        template = template_file.read()
        template_file.close()
        for key in data.keys():
            template = template.replace('$' + key, data[key])
        
        filename = tempfile.NamedTemporaryFile().name
        info_page = open(filename, 'w')
        info_page.write(template)
        info_page.close()
        self.info_pages[program_name] = filename
        
        return filename
    
    def clearInfo(self):
        self.browser.loadUri("about:blank")

    def rowActivated(self, treeview, path, view_column):
        iter = treeview.get_model().get_iter(path)
        type = treeview.get_model().get_value(iter, self.COL_TYPE)
        if type == self.TYPE_GROUP:
            if treeview.row_expanded(path):
                treeview.collapse_row(path)
            else:
                treeview.expand_row(path, False)
        elif type == self.TYPE_PACKAGE:
            # emitting the toggled signal with a tuple path doesn't work
            # this works around that bug
            path_str = str(path)[1:-1]
            parts = path_str.split(",")
            stripped = []
            for part in parts:
                stripped.append(part.strip())
            path_str = ":".join(stripped)
            
            self.toggle_render.emit("toggled", path_str)
        
        
    
    def showInfo(self, treeview):
        path = treeview.get_cursor()[0]
        iter = treeview.get_model().get_iter(path)
        
        if treeview.get_model().get_value(iter, self.COL_TYPE) != self.TYPE_PACKAGE:
            filename = self.intro_page
        else:
            program_name, package_name, icon_filename, summary, available, section = treeview.get_model().get(iter, self.COL_NAME, self.COL_PACKAGE, self.COL_ICON_FILE, self.COL_DESC, self.COL_AVAILABLE, self.COL_SECTION)
            if not self.info_pages.has_key(program_name):
                filename = self.generateInfoPage(program_name, package_name, icon_filename, summary, available, section)
            else:
                filename = self.info_pages[program_name]
        
        self.isInfoPage = True
        self.browser.loadUri("file://" + filename)
        self.isInfoPage = False
        
    def openWebpage(self, mozembed, uri):
        if not self.isInfoPage:
            # A bug in  gtkmozembed causes it to pass a gobject.GPointer as the uri argument, so this doesn't work.
            #gnomevfs.url_show(uri)
            return True
        else:
            return False
        
    def generateIntro(self):
        text = """<html>
<head>
    <meta http-equiv="content-type" content="text/html; charset=UTF-8">
    <style type="text/css">
    body {
        background: white;
        font-size: 10pt;
        font-family: Sans;
    }
    </style>
</head>
<body>"""
        icon_filename = self.icon("gnome-settings-default-applications", 24)[1]
        text += '<img src=\"' + icon_filename + '\" align=\"right\" />\n'
        text += '<h2>' + _("Add/Remove Programs") + '</h2>\n'
        text += '<p>' + _("""This program allows you to add or remove programs with a
click of a button. Browse through the categories to find programs, or search
through all of the programs using the search box. Check the box next to a program to
install it, and uncheck the box to remove it.""") + '</p>\n'
        text += '<p>' + _("""Only simple additions and removals can be performed with
this program. For more complicated needs, use the Synaptic Package Manager,
which can be run by clicking "Advanced" in the "File" menu.""") + '</p>\n'
        text += "</body></html>"
        
        filename = tempfile.NamedTemporaryFile().name
        file = open(filename, 'w')
        file.write(text)
        file.close()
        
        return filename
        
class SimpleFilteredCache(apt.cache.FilteredCache):
    """ a simpler version of the filtered cache that will not react to
        cache changed (no need, we are only interessted in text)
    """
    def filterCachePostChange(self):
        pass
    def runFilter(self):
        self._reapplyFilter()

class SearchFilter(apt.cache.Filter):
    """ a filter class that just searchs insensitive in name/description """
    def SetSearchTerm(self, term):
        self._term = term.lower()
    def apply(self, pkg):
        if self._term in pkg.name.lower() or \
               self._term in pkg.description.lower():
            return True
        else:
            return False
    def __init__(self, query=None):
        if query != None:
            self.SetSearchTerm(query)
        

"""
A class which does the actual package installing/removing.
"""
class PackageWorker:
    # synaptic actions
    (INSTALL, UPDATE) = range(2)
    
    def run_synaptic(self, id, lock, selections=None, action=INSTALL):
        apt_pkg.PkgSystemUnLock()
        #print "run_synaptic(%s,%s,%s)" % (id, lock, selections)
        cmd = ["/usr/sbin/synaptic", "--hide-main-window",
               "--non-interactive",
               "--plug-progress-into", "%s" % (id) ]
        
        if action == self.INSTALL:
            cmd.append("--set-selections")
            f = os.popen(" ".join(cmd), "w")
            for s in selections:
                f.write("%s\n" % s)
            f.close()
        elif action == self.UPDATE:
            print "Updating..."
            cmd.append("--update-at-startup")
            subprocess.call(cmd)
            
        lock.release()

    def plug_removed(self, w, (win,socket)):
        # plug was removed, but we don't want to get it removed, only hiden
        # unti we get more
        win.hide()
        return True

    def plug_added(self, sock, win):
        win.show()
        while gtk.events_pending():
            gtk.main_iteration()

    def get_plugged_win(self, main_window):
        win = gtk.Window()
        win.set_border_width(6)
        #print main_window
        win.set_transient_for(main_window)
        win.set_position(gtk.WIN_POS_CENTER_ON_PARENT)
        win.resize(400,200)
        win.set_title("")
        win.set_resizable(False)
        # prevent the window from closing with the delete button (there is
        # a cancel button in the window)
        win.connect("delete_event", lambda e,w: True);
    
        # create the socket
        socket = gtk.Socket()
        socket.show()
        win.add(socket)
        
        socket.connect("plug-added", self.plug_added, win)
        socket.connect("plug-removed", self.plug_removed, (win,socket))
        
        return win, socket
    
    def perform_action(self, main_window, selections=None, action=INSTALL):
        main_window.set_sensitive(False)
        plug_win, socket = self.get_plugged_win(main_window)
        
        lock = thread.allocate_lock()
        lock.acquire()
        t = thread.start_new_thread(self.run_synaptic,(socket.get_id(),lock,selections,action))
        while lock.locked():
            while gtk.events_pending():
                gtk.main_iteration()
            time.sleep(0.05)
        plug_win.destroy()
        main_window.set_sensitive(True)
        

# Entry point for testing in source tree
if __name__ == '__main__':
    app = AppInstall(os.path.abspath("menu-data"), os.path.abspath("data"), sys.argv)
    gtk.main()
