
/* GnomeMeeting -- A Video-Conferencing application
 * Copyright (C) 2000-2004 Damien Sandras
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 * GnomeMeeting is licensed under the GPL license and as a special exception,
 * you have permission to link or otherwise combine this program with the
 * programs OpenH323 and Pwlib, and distribute the combination, without
 * applying the requirements of the GNU GPL to the OpenH323 program, as long
 * as you do follow the requirements of the GNU GPL for all the rest of the
 * software thus combined.
 */


/*
 *                         lid.cpp  -  description
 *                         -----------------------
 *   begin                : Sun Dec 1 2002
 *   copyright            : (C) 2000-2004 by Damien Sandras
 *   description          : This file contains LID functions.
 *
 */


#include "../config.h"

#include "lid.h"
#include "endpoint.h"
#include "gnomemeeting.h"
#include "urlhandler.h"
#include "misc.h"
#include "log_window.h"
#include "main_window.h"
#include "pref_window.h"
#include "callbacks.h"

#include "dialog.h"
#include "gm_conf.h"



#ifdef HAS_IXJ
GMLid::GMLid (PString d)
  :PThread (1000, NoAutoDeleteThread)
{
  stop = FALSE;
  soft_codecs = FALSE;
  
  dev_name = d;
  
  Open ();
  
  this->Resume ();
  thread_sync_point.Wait ();
} 


GMLid::~GMLid ()
{
  stop = TRUE;
  
  Close ();

  PWaitAndSignal m(quit_mutex);
}


BOOL
GMLid::Open ()
{
  GtkWidget *main_window = NULL;
  GtkWidget *history_window = NULL;
  
  GMH323EndPoint *ep = NULL;

  gchar *lid_country = NULL;

  int lid_aec = 0;
  int lid_odt = 0;

  BOOL return_val = FALSE;

  PWaitAndSignal m(device_access_mutex);
  
  ep = GnomeMeeting::Process ()->Endpoint ();
  
  main_window = GnomeMeeting::Process ()->GetMainWindow ();
  history_window = GnomeMeeting::Process ()->GetHistoryWindow ();
  
  if (!IsOpen ()) {

    /* We use the default settings, but the device name provided as argument */
    lid_country = gm_conf_get_string (AUDIO_DEVICES_KEY "lid_country_code");
    lid_aec = gm_conf_get_int (AUDIO_DEVICES_KEY "lid_echo_cancellation_level");
    lid_odt = gm_conf_get_int (AUDIO_DEVICES_KEY "lid_output_device_type");

    if (OpalIxJDevice::Open (dev_name)) {

      gm_history_window_insert (history_window,
				_("Opened Quicknet device %s"), 
				(const char *) GetName ()); 
      
      if (lid_country)
	SetCountryCodeName(lid_country);
      
      SetAEC (0, (OpalLineInterfaceDevice::AECLevels) lid_aec);
      StopTone (0);
      SetLineToLineDirect (0, 1, FALSE);
      if (lid_odt == 0) // POTS
	EnableAudio (0, TRUE);
      else 
	EnableAudio (0, FALSE);

      soft_codecs =
	GetMediaFormats ().GetValuesIndex (OpalMediaFormat (OPAL_PCM16)) 
	!= P_MAX_INDEX;
      return_val = TRUE;
    }
    else {
      
      gnomemeeting_error_dialog (GTK_WINDOW (main_window), _("Error while opening the Quicknet device."), _("Please check that your driver is correctly installed and that the device is working correctly."));

      return_val = FALSE;
    }
  }

  g_free (lid_country);
  
  return return_val;
}


BOOL
GMLid::Close ()
{
  GtkWidget *history_window = NULL;
  
  PWaitAndSignal m(device_access_mutex);

  history_window = GnomeMeeting::Process ()->GetHistoryWindow ();
  
  
  if (IsOpen ()) {

    OpalIxJDevice::Close ();

    gm_history_window_insert (history_window,
			      _("Closed Quicknet device %s"), 
			      (const char *) GetName ());
  }

  return TRUE;
}


void
GMLid::Main ()
{
  GtkWidget *main_window = NULL;
  GtkWidget *history_window = NULL;
  GtkWidget *prefs_window = NULL;

  GMH323EndPoint *endpoint = NULL;

  BOOL off_hook = TRUE;
  BOOL last_off_hook = TRUE;
  BOOL do_not_connect = TRUE;

  char c = 0;
  char old_c = 0;
  const char *url = NULL;
  
  PTime now;
  PTime last_key_press;

  PString call_token;

  BOOL is_on_hold = FALSE;

  unsigned int input_vol = 0;
  unsigned int output_vol = 0;
  int lid_odt = 0;
  GMH323EndPoint::CallingState calling_state = GMH323EndPoint::Standby;


  PWaitAndSignal m(quit_mutex);
  thread_sync_point.Signal ();

  
  endpoint = GnomeMeeting::Process ()->Endpoint ();

  main_window = GnomeMeeting::Process ()->GetMainWindow ();
  history_window = GnomeMeeting::Process ()->GetHistoryWindow ();
  prefs_window = GnomeMeeting::Process ()->GetPrefsWindow ();


  /* Check the initial hook status. */
  off_hook = last_off_hook = IsLineOffHook (OpalIxJDevice::POTSLine);

  gnomemeeting_threads_enter ();
  lid_odt = gm_conf_get_int (AUDIO_DEVICES_KEY "lid_output_device_type");
  gnomemeeting_threads_leave ();
  

  /* Update the mixers if the lid is used */
  GetPlayVolume (0, output_vol);
  GetRecordVolume (0, input_vol);
  
    
  gnomemeeting_threads_enter ();
  gm_main_window_set_volume_sliders_values (main_window, output_vol, input_vol);
  gnomemeeting_threads_leave ();

  while (IsOpen() && !stop) {

    calling_state = endpoint->GetCallingState ();

    off_hook = IsLineOffHook (0);
    now = PTime ();


    /* If there is a DTMF, trigger the dialpad event */
    c = ReadDTMF (0);
    if (c) {

      gnomemeeting_threads_enter ();
      gm_main_window_press_dialpad (main_window, c);
      gnomemeeting_threads_leave ();

      last_key_press = PTime ();
      do_not_connect = FALSE;
    }


    /* If there is a state change: the phone is put off hook */
    if (off_hook == TRUE && last_off_hook == FALSE) {

      gnomemeeting_threads_enter ();
      gm_history_window_insert (history_window, _("Phone is off hook"));
      gm_main_window_flash_message (main_window, _("Phone is off hook"));
      gnomemeeting_threads_leave ();


      /* We answer the call */
      gnomemeeting_threads_enter ();
      if (calling_state == GMH323EndPoint::Called)
	GnomeMeeting::Process ()->Connect ();
      gnomemeeting_threads_leave ();


      /* We are in standby */
      if (calling_state == GMH323EndPoint::Standby) {

	if (lid_odt == 0) { /* POTS: Play a dial tone */

	  PlayTone (0, OpalLineInterfaceDevice::DialTone);
	  EnableAudio (0, TRUE);
	}
	else
	  EnableAudio (0, FALSE);

	/* Get the URL from the GUI and call it */
	gnomemeeting_threads_enter ();
	url = gm_main_window_get_call_url (main_window);
	
	/* Automatically connect */
	if (url && !GMURL (url).IsEmpty ())
	  GnomeMeeting::Process ()->Connect (url);
	gnomemeeting_threads_leave ();
      }
    }

    
    /* if phone is put on hook */
    if (off_hook == FALSE && last_off_hook == TRUE) {

      gnomemeeting_threads_enter ();
      gm_history_window_insert (history_window, _("Phone is on hook"));
      gm_main_window_flash_message (main_window, _("Phone is on hook"));
      gm_main_window_set_call_url (main_window, GMURL ().GetDefaultURL ());
      gnomemeeting_threads_leave ();

      RingLine (0, 0);

      /* If we are in a call, or calling, we disconnect */
      if (calling_state == GMH323EndPoint::Connected
	  || calling_state == GMH323EndPoint::Calling) {
	
	gnomemeeting_threads_enter ();
	GnomeMeeting::Process ()->Disconnect ();
	gnomemeeting_threads_leave ();
      }
    }


    /* Examine the DTMF history for special actions */
    if (off_hook == TRUE) {

      PTimeInterval t = now - last_key_press;

      /* Trigger the call after 3 seconds, or on # pressed if
       we are in standby */
      if ((t.GetSeconds () > 3 && !do_not_connect)
	  || c == '#') {

	if (calling_state == GMH323EndPoint::Standby) {
	  
	  gnomemeeting_threads_enter ();
	  url = gm_main_window_get_call_url (main_window);
	
	  /* Automatically connect */
	  if (url && !GMURL (url).IsEmpty ())
	    GnomeMeeting::Process ()->Connect (url);
	  gnomemeeting_threads_leave ();
	}
	do_not_connect = TRUE;
      }


      /* *1 to hold the current call */
      if (old_c == '*' && c == '1'
	  && calling_state == GMH323EndPoint::Connected) {

	call_token = endpoint->GetCurrentCallToken ();
	is_on_hold = endpoint->IsCallOnHold (call_token);
	if (endpoint->SetCallOnHold (call_token, !is_on_hold))
	  is_on_hold = !is_on_hold; /* It worked */


	/* Update the GUI */
	gnomemeeting_threads_enter ();
	gm_main_window_set_call_hold (GTK_WIDGET (main_window), is_on_hold);
	gnomemeeting_threads_leave ();
      }
    }


    /* Save some history */
    last_off_hook = off_hook;
    if (c)
      old_c = c;

    
    /* We must poll to read the hook state */
    PThread::Sleep (50);
  }
}


OpalMediaFormat::List
GMLid::GetAvailableAudioCapabilities ()
{
  OpalMediaFormat::List list;

  list = GetMediaFormats ();

  return list;
}


void
GMLid::UpdateState (GMH323EndPoint::CallingState i)
{
  int lid_odt = 0;

  lid_odt = gm_conf_get_int (AUDIO_DEVICES_KEY "lid_output_device_type");
  
  if (IsOpen ()) {

    switch (i) {

    case GMH323EndPoint::Calling:
      RingLine (0, 0);
      PlayTone (0, OpalLineInterfaceDevice::RingTone);
      SetRemoveDTMF (0, TRUE);
      
      break;

    case GMH323EndPoint::Called: 

      if (lid_odt == 0)
	RingLine (OpalIxJDevice::POTSLine, 0x33);
      else
	RingLine (0, 0);
      break;

    case GMH323EndPoint::Standby: /* Busy */
      RingLine (0, 0);
      PlayTone (0, OpalLineInterfaceDevice::BusyTone);
      if (lid_odt == 1) {
	
	PThread::Current ()->Sleep (2800);
	StopTone (0);
      }
      break;

    case GMH323EndPoint::Connected:

      RingLine (0, 0);
      StopTone (0);
    }    

    
    if (lid_odt == 0) // POTS
      EnableAudio (0, TRUE);
    else 
      EnableAudio (0, FALSE);
  }
}


void
GMLid::SetVolume (BOOL is_encoding, 
                  int x)
{
  if (IsOpen ()) {

    if (!is_encoding)
      SetPlayVolume (0, x);
    else
      SetRecordVolume (0, x);
  }
}


void
GMLid::PlayDTMF (const char *digits)
{
  if (IsOpen () && digits) {

    StopTone (0);
    OpalIxJDevice::PlayDTMF (0, digits);
  }
}


BOOL
GMLid::areSoftwareCodecsSupported ()
{
  if (IsOpen ())
    return soft_codecs;
  else
    return FALSE;
}


void
GMLid::Lock ()
{
  device_access_mutex.Wait ();
}


void
GMLid::Unlock ()
{
  device_access_mutex.Signal ();
}
#endif
