/*
 * Copyright (C) 2011 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.

 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Ken VanDine <ken.vandine@canonical.com>
 */

using Dee;
using Gee;
using Config;
using Unity;
using Gwibber;

namespace UnityGwibber {

  /* DBus name for the place. Must match out .place file */
  const string BUS_NAME = "com.canonical.Unity.Lens.Gwibber";
  const string ICON_PATH = Config.DATADIR + "/icons/unity-icon-theme/places/svg/";

  /**
   * The Daemon class implements all of the logic for the place.
   *
   */
  public class Daemon : GLib.Object
  {
    private Unity.Lens lens;
    private Unity.Scope scope;
    private Unity.PreferencesManager preferences = Unity.PreferencesManager.get_default ();
    private Gwibber.Streams streams_service;
    private Gwibber.Service service;
    private Gwibber.Accounts accounts;
    private Gwibber.Utils utils;
    private Dee.Model? _model = null;
    private Dee.Model? _streams_model = null;
    private Dee.Filter _sort_filter;
    /* Keep track of the previous search, so we can determine when to
     * filter down the result set instead of rebuilding it */
    private unowned Dee.ModelIter _stream_iter_first = null;
    private unowned Dee.ModelIter _stream_iter_last = null;

    private Dee.Analyzer _analyzer;
    private Dee.Index _index;
    private Dee.ICUTermFilter _ascii_filter;
    private Ag.Manager _account_manager;
    private bool _has_accounts = false;

    construct
    {
      lens = new Unity.Lens("/com/canonical/unity/lens/gwibber", "gwibber");
      lens.search_in_global = false;
      lens.search_hint = _("Enter name or content you would like to search for");
      lens.visible = false;
      try
      {
        lens.export ();
      } catch (GLib.IOError e)
      {
        warning ("failed to export lens: %s", e.message);
      }


      // Check for accounts
      _account_manager = new Ag.Manager.for_service_type("microblogging");
      GLib.List<Ag.AccountService> accts = _account_manager.get_enabled_account_services();
      foreach (Ag.AccountService account_service in accts) {
        Ag.Account account = account_service.get_account();
        account.set_enabled (false);
        message ("ACCOUNT PROVIDER: %s", account.get_provider_name());
      }

      // We only want to trigger starting gwibber-service if there are accounts
      if (accts.length() > 0)
      {
        _has_accounts = true;
        setup ();
      }

      _account_manager.enabled_event.connect ((id) =>
      {
        accts = _account_manager.get_enabled_account_services();
        if (accts.length() > 0 && !_has_accounts)
        {
          _has_accounts = true;
          setup ();
        }
        else if (accts.length() == 0)
        {
          lens.visible = false;
          _has_accounts = false;
        }
      });
    }


    void setup ()
    {
      scope = new Unity.Scope ("/com/canonical/unity/scope/gwibber");
      scope.search_in_global = false;
      scope.preview_uri.connect (preview);

      lens.visible = true;

      lens.add_local_scope (scope);

      /* Listen for filter changes */
      scope.notify["active"].connect(() =>
      {
        if (scope.active)
        {
          scope.queue_search_changed (SearchType.DEFAULT);
        }
      });

      scope.generate_search_key.connect ((lens_search) =>
      {
        return lens_search.search_string.strip ();
      });
      scope.search_changed.connect ((lens_search, search_type, cancellable) =>
      {
        if (search_type == SearchType.DEFAULT)
          update_scope_search.begin (lens_search, cancellable);
        else
          update_global_search.begin (lens_search, cancellable);
      });

      lens.notify["active"].connect ((obj, pspec) => {
        if (lens.active && scope.active)
        {
          if (_stream_iter_first != _model.get_first_iter () || _stream_iter_last != _model.get_last_iter ())
          {
            if (scope.active)
            {
              scope.queue_search_changed (SearchType.DEFAULT);
            }
          }
        }
      });

      scope.filters_changed.connect (() =>
      {
        scope.queue_search_changed (SearchType.DEFAULT);
      });

      preferences.notify["remote-content-search"].connect ((obj, pspec) =>
      {
        scope.queue_search_changed (SearchType.DEFAULT);
      });

    }

    private void setup_gwibber ()
    {
      streams_service = new Gwibber.Streams();
      service = new Gwibber.Service();
      utils = new Gwibber.Utils();
      accounts = new Gwibber.Accounts();

      populate_categories ();
      populate_filters();

      _streams_model = streams_service.stream_model;
      Intl.setlocale(LocaleCategory.COLLATE, "C");
      _sort_filter = Dee.Filter.new_collator_desc (StreamModelColumn.TIMESTAMP);
      _model = new Dee.FilterModel (_streams_model, _sort_filter);

      _ascii_filter = new Dee.ICUTermFilter.ascii_folder ();
      _analyzer = new Dee.TextAnalyzer ();
      _analyzer.add_term_filter ((terms_in, terms_out) =>
      {
        for (uint i = 0; i < terms_in.num_terms (); i++)
        {
          unowned string term = terms_in.get_term (i);
          var folded = _ascii_filter.apply (term);
          terms_out.add_term (term);
          if (folded != term) terms_out.add_term (folded);
        }
      });
      var reader = Dee.ModelReader.new ((model, iter) =>
      {
        var sender_col = StreamModelColumn.SENDER;
        var msg_col = StreamModelColumn.MESSAGE;
        return "%s\n%s".printf (model.get_string (iter, sender_col), 
                                model.get_string (iter, msg_col));
      });
      _index = new Dee.TreeIndex (_model, _analyzer, reader);
    }

    private void populate_filters ()
    {
      var filters = new GLib.List<Unity.Filter> ();

      /* Stream filter */
      {
        var filter = new CheckOptionFilter ("stream", _("Stream"));

        filter.add_option ("messages", _("Messages"));
        filter.add_option ("replies", _("Replies"));
        filter.add_option ("images", _("Images"));
        filter.add_option ("videos", _("Videos"));
        filter.add_option ("links", _("Links"));
        filter.add_option ("private", _("Private"));
        filter.add_option ("public", _("Public"));

        filters.append (filter);
      }

      /* Account filter */
      {
        var filter = create_account_filter ();
        filters.append (filter);
      }

      lens.filters = filters;

      accounts.created.connect((source) => {
        /* FIXME: we need a way to add an option 
        var filter = scope.get_filter ("account_id") as CheckOptionFilter;
        filter.add_option (source.id, source.service + "/" + source.username);
        */
      });

      /* FIXME: we need a way to remove an option or remove and re-add the
       * filter on account deletion */
    }

    private Unity.CheckOptionFilter create_account_filter ()
    {
      GLib.Icon icon;
      GLib.File icon_file;
      var filter = new CheckOptionFilter ("account_id", _("Account"));
      foreach (var _acct in accounts.list ())
      {
        /* create a service icon, which isn't used yet as of libunity 4.24
           when it is used, we can drop _acct.service from the option name 
           and rely on the service icon and username to disambiguate accounts */
        icon_file = GLib.File.new_for_path (GLib.Path.build_filename (Config.PKGDATADIR + "/plugins/" + _acct.service + "/ui/icons/16x16/" + _acct.service + ".png"));
        icon = new GLib.FileIcon (icon_file);

        filter.add_option (_acct.id, _acct.service + "/" + _acct.username, icon);
      }
      return filter;
    }


    private void populate_categories ()
    {
      var categories = new GLib.List<Unity.Category> ();
      Icon icon;

      var icon_dir = File.new_for_path ("/usr/share/gwibber/ui/icons/hicolor/scalable/places/");

      icon = new FileIcon (icon_dir.get_child ("group-messages.svg"));
      var cat = new Unity.Category (_("Messages"), icon, Unity.CategoryRenderer.HORIZONTAL_TILE);
      categories.append (cat);

      icon = new FileIcon (icon_dir.get_child ("group-replies.svg"));
      cat =  new Unity.Category (_("Replies"), icon, Unity.CategoryRenderer.HORIZONTAL_TILE);
      categories.append (cat);

      icon = new FileIcon (icon_dir.get_child ("group-images.svg"));
      cat =  new Unity.Category (_("Images"), icon, Unity.CategoryRenderer.FLOW);
      categories.append (cat);

      icon = new FileIcon (icon_dir.get_child ("group-videos.svg"));
      cat =  new Unity.Category (_("Videos"), icon, Unity.CategoryRenderer.FLOW);
      categories.append (cat);

      icon = new FileIcon (icon_dir.get_child ("group-links.svg"));
      cat =  new Unity.Category (_("Links"), icon, Unity.CategoryRenderer.HORIZONTAL_TILE);
      categories.append (cat);

      icon = new FileIcon (icon_dir.get_child ("group-private.svg"));
      cat =  new Unity.Category (_("Private"), icon, Unity.CategoryRenderer.HORIZONTAL_TILE);
      categories.append (cat);

      icon = new FileIcon (icon_dir.get_child ("group-public.svg"));
      cat =  new Unity.Category (_("Public"), icon, Unity.CategoryRenderer.HORIZONTAL_TILE);
      categories.append (cat);

      lens.categories = categories;
    }


    private bool is_empty_search (LensSearch search)
    {
      return search.search_string.strip () == "";
    }

    private async void update_global_search (LensSearch search, Cancellable cancellable)
    {
      /**
       * only perform the request if the user has not disabled
       * online/commercial suggestions. That will hide the category as well.
       */
      if (preferences.remote_content_search != Unity.PreferencesManager.RemoteContent.ALL)
      {
        search.results_model.clear ();
        search.finished ();
        return;
      }

      var results_model = scope.global_results_model;

      // FIXME: no results for home screen of the dash?
      if (is_empty_search (search))
      {
        search.finished ();
        return;
      }

      update_results_model (results_model, search.search_string, null);

      search.finished ();
    }

    private async void update_scope_search  (LensSearch search, Cancellable cancellable)
    {
      /**
       * only perform the request if the user has not disabled
       * online/commercial suggestions. That will hide the category as well.
       */
      if (preferences.remote_content_search != Unity.PreferencesManager.RemoteContent.ALL)
      {
        search.results_model.clear ();
        search.finished ();
        return;
      }

      var results_model = search.results_model;

      update_results_model (results_model, search.search_string, null);
      debug ("%u results", results_model.get_n_rows ());
      search.finished ();
    }


    /* Generic method to update a results model. We do it like this to minimize
     * code dup between updating the global- and the entry results model */
    private void update_results_model (Dee.Model results_model,
                                       string? search, Categories? category)
    {
      unowned Dee.ModelIter iter, end;

      var stream_ids = new Gee.ArrayList<string> ();
      var filter = scope.get_filter("stream") as CheckOptionFilter;
      if (filter.filtering)
      {
        foreach (Unity.FilterOption option in filter.options)
        {
          if (option.active)
          {
            stream_ids.add (option.id);
          }
        }
      }

      var account_ids = new Gee.ArrayList<string> ();
      filter = scope.get_filter("account_id") as CheckOptionFilter;
      if (filter.filtering)
      {
        foreach (Unity.FilterOption option in filter.options)
        {
          if (option.active)
          {
            account_ids.add (option.id);
          }
        }
      }

      results_model.clear ();

      if (_model == null)
        setup_gwibber ();

      iter = _model.get_first_iter ();
      end = _model.get_last_iter ();

      _stream_iter_first = _model.get_first_iter ();
      _stream_iter_last = end;
     
      var term_list = Object.new (typeof (Dee.TermList)) as Dee.TermList;
      // search only the folded terms, FIXME: is that a good idea?
      _analyzer.tokenize (_ascii_filter.apply (search), term_list);

      var matches = new Sequence<Dee.ModelIter> ();
      for (uint i = 0; i < term_list.num_terms (); i++)
      {
        // FIXME: use PREFIX search only for the last term?
        var result_set = _index.lookup (term_list.get_term (i),
                                        Dee.TermMatchFlag.PREFIX);
        bool first_pass = i == 0;
        CompareDataFunc<Dee.ModelIter> cmp_func = (a, b) =>
        {
          return a == b ? 0 : ((void*) a > (void*) b ? 1 : -1);
        };
        // intersect the results (cause we want to AND the terms)
        var remaining = new Sequence<Dee.ModelIter> ();
        foreach (var item in result_set)
        {
          if (first_pass)
            matches.insert_sorted (item, cmp_func);
          else if (matches.lookup (item, cmp_func) != null)
            remaining.insert_sorted (item, cmp_func);
        }
        if (!first_pass) matches = (owned) remaining;
        // final result set empty already?
        if (matches.get_begin_iter () == matches.get_end_iter ()) break;
      }

      matches.sort ((a, b) =>
      {
        var col = StreamModelColumn.TIMESTAMP;
        return _model.get_string (b, col).collate (_model.get_string (a, col));
      });

      var match_iter = matches.get_begin_iter ();
      var match_end_iter = matches.get_end_iter ();
      while (match_iter != match_end_iter)
      {
        iter = match_iter.get ();

        if (matches_filters (_model, iter, stream_ids, account_ids))
        {
          add_result (_model, iter, results_model);
        }

        match_iter = match_iter.next ();
      }

      if (term_list.num_terms () > 0) return;

      /* Go over the whole model if we had empty search */
      while (iter != end)
      {
        if (matches_filters (_model, iter, stream_ids, account_ids))
        {
          add_result (_model, iter, results_model);
        }
        iter = _model.next (iter);
      }

      //debug ("Results has %u rows", results_model.get_n_rows());
    }

    private bool matches_filters (Dee.Model model, Dee.ModelIter iter,
                                  Gee.List<string> stream_ids,
                                  Gee.List<string> account_ids)
    {
      bool stream_match = true;
      bool account_match = true;
      if (stream_ids.size > 0)
      {
        stream_match = model.get_string (iter, StreamModelColumn.STREAM) in stream_ids;
      }
      if (account_ids.size > 0)
      {
        string[] _accounts_array = (string[])model.get_value (iter, StreamModelColumn.ACCOUNTS);
        foreach (var a in _accounts_array)
        {
          string _account = a.split(":")[0];
          if (!(_account in account_ids))
            account_match = false;
        }
      }
      return account_match && stream_match;
    }

    private void add_result (Dee.Model model, Dee.ModelIter iter, Dee.Model results_model)
    {
      Categories group = Categories.MESSAGES;
      string _img_uri = null;

      unowned string stream_id = 
        model.get_string (iter, StreamModelColumn.STREAM);
      switch (stream_id)
      {
        case "messages": group = Categories.MESSAGES; break;
        case "replies": group = Categories.REPLIES; break;
        case "images": group = Categories.IMAGES; break;
        case "videos": group = Categories.VIDEOS; break;
        case "links": group = Categories.LINKS; break;
        case "private": group = Categories.PRIVATE; break;
        case "public": group = Categories.PUBLIC; break;
      }
      
      string _icon_uri = model.get_string (iter, StreamModelColumn.ICON_URI);
      if (stream_id == "images")
        _img_uri = model.get_string (iter, StreamModelColumn.IMG_SRC);
      else if (stream_id == "videos")
      {
        _img_uri = model.get_string (iter, StreamModelColumn.VIDEO_PIC);
        if (_img_uri.length < 1)
          _img_uri = get_avatar_path (_icon_uri);
      }
      else
        _img_uri = get_avatar_path (_icon_uri);

      results_model.append (model.get_string(iter, StreamModelColumn.URL),
                            _img_uri,
                            group,
                            "text/html",
                            _model.get_string(iter, StreamModelColumn.SENDER),
                            _model.get_string(iter, StreamModelColumn.MESSAGE));
    }

    private string get_avatar_path (string uri)
    {
      var _avatar_cache_image = utils.avatar_path (uri);
      if (_avatar_cache_image == null)
      {
        try
        {
          _avatar_cache_image = service.avatar_path (uri);
        } catch (GLib.Error e)
        {
        }
        if (_avatar_cache_image == null)
          _avatar_cache_image = uri;
      }

      return _avatar_cache_image;
    }

    public Unity.Preview preview (string uri)
    {
      debug ("Previewing: %s",  uri);
      Unity.SocialPreview preview = null;
      var icon_dir = File.new_for_path (ICON_PATH);
      Icon icon;
      string retweet_str, like_str, _img_uri = null;

      if (_streams_model == null)
        setup_gwibber ();

      var model = _streams_model;
      unowned Dee.ModelIter iter, end;
      iter = model.get_first_iter ();
      end = model.get_last_iter ();

      while (iter != end)
      {
        var url = model.get_string (iter, StreamModelColumn.URL);
        if (url == uri)
        {
          debug ("Found %s", url);
          var icon_uri = model.get_string (iter, StreamModelColumn.ICON_URI);
          if ("_normal." in icon_uri)
          {
            //If it looks like a twitter avatar, try to get the original
            var ss = icon_uri.split("_normal.");
            icon_uri = ss[0] + "." + ss[1];
          }
          var avatar = Icon.new_for_string (icon_uri);

          //var avatar = new FileIcon (File.new_for_path (get_avatar_path (icon_uri)));
          var likes = model.get_double (iter, StreamModelColumn.LIKES);
          var sender = model.get_string (iter, StreamModelColumn.SENDER);
          var sender_nick = model.get_string (iter, StreamModelColumn.SENDER_NICK);
          var timestring = model.get_string (iter, StreamModelColumn.TIMESTRING);
          if (sender_nick.length > 0)
            sender_nick = "@" + sender_nick;
          else
            sender_nick = sender;
          var content = model.get_string (iter, StreamModelColumn.MESSAGE);
          string title = sender_nick + " " + timestring;
          var comments_json = model.get_string (iter, StreamModelColumn.COMMENTS);

          preview = new Unity.SocialPreview (sender, title, content, avatar);
          // work around bug 1058198
          preview.title = sender;
          preview.subtitle = title;
          // end work around
          preview.add_info (new InfoHint ("likes", _("Favorites"), null, likes.to_string()));
          var stream_id = model.get_string (iter, StreamModelColumn.STREAM);
          if (stream_id == "images")
            _img_uri = model.get_string (iter, StreamModelColumn.IMG_SRC);
          else if (stream_id == "videos" && _img_uri != null && _img_uri.length < 1)
            _img_uri = model.get_string (iter, StreamModelColumn.VIDEO_PIC);
          if (_img_uri != null && _img_uri.length > 0)
          {
            Icon img = Icon.new_for_string (_img_uri);
            preview.image = img;
          }
          parse_comments (preview, comments_json);
          break;
        }
        iter = model.next (iter);
      }

      string[] _accounts_array = (string[])model.get_value (iter, StreamModelColumn.ACCOUNTS);
      string[] _seen = null;
      foreach (var a in _accounts_array)
      {
        var _account_id = a.split(":")[0];
        if (_account_id in _seen)
          continue;
        _seen += _account_id;
        var _account_service = a.split(":")[1];
        var _status_id = a.split(":")[2];
        var icon_str = icon_dir.get_child ("service-" + _account_service + ".svg");
        if (icon_str.query_exists())
          icon = new FileIcon (icon_str);
        else
          icon = null;

        var view_action = new Unity.PreviewAction ("view", _("View"), icon);
        preview.add_action (view_action);

        view_action.activated.connect ((source) => {
          try
          {
            if (GLib.AppInfo.launch_default_for_uri (uri, null))
            {
              return new Unity.ActivationResponse(Unity.HandledType.HIDE_DASH);
            }
          }
          catch (GLib.Error e)
          {
            warning ("Failed to launch default application for uri '%s': %s", uri, e.message);
          }
          return new Unity.ActivationResponse(Unity.HandledType.NOT_HANDLED);
        });

        if (_account_service == "twitter" || _account_service == "identica" || 
            _account_service == "statusnet" || _account_service == "sina" ||
            _account_service == "sohu") 
        {
          if (_account_service == "twitter")
            retweet_str = _("Retweet");
          else 
            retweet_str = _("Share");
          var retweet_action = new Unity.PreviewAction ("retweet", retweet_str, icon);
          preview.add_action (retweet_action);
          retweet_action.activated.connect ((source) => {
            service.retweet(_status_id, _account_id);
            return new Unity.ActivationResponse(Unity.HandledType.SHOW_PREVIEW);
          });
        }
        
        bool from_me = model.get_bool (iter, StreamModelColumn.FROM_ME);
        /* REPLY
        if ((_account_service != "flicker" && _account_service != "foursquare" && _account_service != "digg") && !from_me)
        {
          var reply_action = new Unity.PreviewAction ("reply", _("Reply"), icon);
          preview.add_action (reply_action);
          reply_action.activated.connect ((source) => {
            return new Unity.ActivationResponse(Unity.HandledType.HIDE_DASH);
          });
        }
        */

        bool liked = model.get_bool (iter, StreamModelColumn.LIKED);
        if ((_account_service != "flicker" && _account_service != "foursquare" && _account_service != "digg") && !from_me)
        {
          like_str = liked ? _("Unlike") : _("Like");
          var like_action = new Unity.PreviewAction ("like", like_str, icon);
          preview.add_action (like_action);
          like_action.activated.connect ((source) => {
            var ret = liked ? service.unlike (_status_id, _account_id) : service.like (_status_id, _account_id);
            if (ret)
            {
              model.set_value (iter, StreamModelColumn.LIKED, !liked);
            }
            Unity.Preview new_preview = this.preview (uri);
            return new Unity.ActivationResponse.with_preview(new_preview);
          });

        }
      }

      return preview;
    }

    private void parse_comments (Unity.SocialPreview preview, string _comments)
    {
      if (_comments != null)
      {
        string cname = "";
        string ctext = "";
        string time_string = "";

        var parser = new Json.Parser();
        try
        {
          parser.load_from_data(_comments, -1);
        }
        catch (Error e)
        {
        }
        unowned Json.Node comments_node = null;
        comments_node = parser.get_root();
        if (comments_node != null)
        {
          Json.Object comments_obj = comments_node.get_object ();
          if (comments_obj != null)
          {
            if (comments_obj.has_member ("comments"))
            {
              Json.Array comments = comments_obj.get_array_member ("comments");
              if (comments.get_length() > 0)
              {
                for(int i = 0; i < comments.get_length(); i++)
                {
                  var obj = comments.get_element(i).get_object();
                  if (obj != null)
                  {
                    if (obj.has_member ("text"))
                    {
                      ctext = obj.get_string_member ("text");
                      if (ctext != null)
                        ctext = GLib.Markup.escape_text (ctext);
                    }
                    if (obj.has_member ("time"))
                    {
                      var ctime = obj.get_int_member ("time");
                      time_string = utils.generate_time_string ((uint)ctime);
                      debug ("ctime: %s", time_string);
                    }
                    if (obj.has_member ("sender"))
                    {
                      var _sender_obj = obj.get_object_member ("sender");
                      if (_sender_obj != null)
                      {
                        if (_sender_obj.has_member ("name"))
                        {
                          cname = _sender_obj.get_string_member ("name");
                          if (cname != null)
                            cname = GLib.Markup.escape_text (cname);
                        }
                      }
                    }
                    preview.add_comment (new Unity.SocialPreview.Comment (cname, cname, ctext, time_string));
                  }
                }
              }  
            }  
          }
        }
      }
    }

  } /* End Daemon class */
} /* end Gwibber namespace */
