/*
 * Copyright(C) 2010 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Ken VaDine <ken.vandine@canonical.com>
 */

namespace Gwibber
{

    public class Account : GLib.Object {
        private Ag.Account _account = null;
        [Description(id = "Gwibber account id", blurb = "This is id Gwibber uses to identify the account")]
        public string id {
            get { return unique_id; }
        }
        [Description(username = "username associated with an account", blurb = "This is the account's username assigned by the service")]
        public string username {
            get { return _account.get_display_name(); }
        }
        public string service {
            get { return _account.get_provider_name(); }
        }

        private string _color;
        public string color { 
            get {
              GLib.Value t = "";
              _account.get_value("gwibber/color", ref t);
              _color = t.get_string();
              return _color;
            }
            set { 
              GLib.Value t = value;
              _account.set_value("gwibber/color", t); 
              _account.store_blocking();
            }
        }
        private string? _send_enabled;
        public string? send_enabled {
            get {
              GLib.Value t = false;
              var res = _account.get_value("gwibber/send_enabled", ref t);
              if (res == Ag.SettingSource.NONE) {
                _send_enabled = null;
                return _send_enabled;
              }
              if (t.get_boolean())
                _send_enabled = "1";
              else
                _send_enabled = "0";
              return _send_enabled;
            }
            set { 
              GLib.Value t = false;
              if (value == "1")
                t = true;
              _account.set_value("gwibber/send_enabled", t);
              _account.store_blocking();
            }
        }
        private string _receive_enabled;
        public string receive_enabled {
            get {
              GLib.Value t = "";
              _account.get_value("gwibber/receive_enabled", ref t);
              _receive_enabled = t.get_string();
              return _receive_enabled;
            }
            set { 
              GLib.Value t = value;
              _account.set_value("gwibber/receive_enabled", t);
              _account.store_blocking();
            }
        }

        private Ag.AccountService account_service;
        private string unique_id;

        /** 
            Account::updated:

            Emitted when the account has changed
        */
        public signal void updated();

        /** 
            Account::deleted:

            Emitted when the account has been deleted
        */
        public signal void deleted();

        public Account (Ag.AccountService account_service) {
            this.account_service = account_service;
            _account = account_service.get_account();
            unique_id =
                "%u/%s".printf(_account.id,
                               account_service.get_service().get_name());

            account_service.changed.connect(() => {
                updated();
            });
            _account.deleted.connect(() => {
                deleted();
            });
        }

        /**
            toggle_send_enabled:

            Toggle the value of send_enabled for the account.
        */
        public void toggle_send_enabled()
        {
            if (send_enabled == "1")
              send_enabled = "0";
            else
              send_enabled = "1";
        }

    }
}
