/*
 * Line6 PODxt Pro USB driver - 0.5
 *
 * Copyright (C) 2004, 2005 Markus Grabner (grabner@icg.tu-graz.ac.at)
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation, version 2.
 *
 */

#include <linux/config.h>

#ifdef CONFIG_USB_DEBUG
#define DEBUG 1
#endif

#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/string.h>
#include <linux/types.h>
#include <linux/usb.h>

#include <sound/driver.h>
#include <sound/core.h>
#include <sound/minors.h>
#include <sound/rawmidi.h>

#include "audio.h"
#include "control.h"
#include "driver.h"
#include "midi.h"


#define podxtpro_rawmidi_substream_midi(substream) ((snd_podxtpro_pcm_t *)((substream)->rmidi->private_data))


static int send_midi_async(struct usb_podxtpro *podxtpro, unsigned char *data, int length);


/*
	Pass data received via USB to MIDI.
*/
void podxtpro_midi_receive(struct usb_podxtpro *podxtpro, unsigned char *data, int length)
{
	if(podxtpro->midi->substream_receive)
		snd_rawmidi_receive(podxtpro->midi->substream_receive, data, length);
}

/*
	Read data from MIDI buffer and transmit them via USB.
*/
static void podxtpro_midi_transmit(snd_rawmidi_substream_t *substream)
{
	unsigned long flags;
	unsigned char data[PODXTPRO_RAW_CHUNK_SIZE];
	usb_podxtpro_t *podxtpro = podxtpro_rawmidi_substream_midi(substream)->podxtpro;
	spin_lock_irqsave(&podxtpro->midi->midi_transmit_lock, flags);

	for(;;) {
		int done = snd_rawmidi_transmit_peek(substream, data, PODXTPRO_RAW_CHUNK_SIZE);
		int err = 0;

		if(done == 0)
			break;

		if(!((done == 1) && (data[0] == 0xfe)))  // skip "Active Sense" messages
			err = send_midi_async(podxtpro, data, done);

		if(err == 0)
			snd_rawmidi_transmit_ack(substream, done);

		if(done != sizeof(data))
			break;
	}

	spin_unlock_irqrestore(&podxtpro->midi->midi_transmit_lock, flags);
}

/*
	Notification of completion of MIDI transmission.
*/
static void midi_sent(struct urb *urb, struct pt_regs *regs)
{
	unsigned long flags;
	int status;
	int num;
	struct usb_podxtpro *podxtpro = (struct usb_podxtpro *)urb->context;

	status = urb->status;
	kfree(urb->transfer_buffer);
	usb_free_urb(urb);

	if(status == -ESHUTDOWN)
		return;

	spin_lock_irqsave(&podxtpro->midi->send_urb_lock, flags);
	num = --podxtpro->midi->num_active_send_urbs;

	if(num == 0) {
		podxtpro_midi_transmit(podxtpro->midi->substream_transmit);
		num = podxtpro->midi->num_active_send_urbs;
	}

	if(num == 0)
		wake_up_interruptible(&podxtpro->midi->send_wait);

	spin_unlock_irqrestore(&podxtpro->midi->send_urb_lock, flags);
}

/*
	Send an asynchronous MIDI message.
	Assumes that podxtpro->midi->send_urb_lock is held
	(i.e., this function is serialized).
*/
static int send_midi_async(struct usb_podxtpro *podxtpro, unsigned char *data, int length)
{
	struct urb *urb;
	int retval, i;
	unsigned char *transfer_buffer;

	urb = usb_alloc_urb(0, GFP_ATOMIC);

	if(urb == 0) {
		dev_err(podxtpro->msgdev, "Out of memory\n");
		return -ENOMEM;
	}

#if DO_URB_DUMP_SEND
	podxtpro_write_hexdump(podxtpro, 'S', data, length);
#endif

	transfer_buffer = (unsigned char *)kmalloc(length, GFP_ATOMIC);

	if(transfer_buffer == 0) {
		usb_free_urb(urb);
		dev_err(podxtpro->msgdev, "Out of memory\n");
		return -ENOMEM;
	}

	memcpy(transfer_buffer, data, length);
	usb_fill_bulk_urb(urb,
										podxtpro->usbdev,
										usb_sndbulkpipe(podxtpro->usbdev, PODXTPRO_EP_WRITE_MIDI),
										transfer_buffer, length, midi_sent, podxtpro);
	urb->actual_length = 0;
	retval = usb_submit_urb(urb, GFP_ATOMIC);

	if(retval < 0) {
		dev_err(podxtpro->msgdev, "usb_submit_urb failed\n");
		usb_free_urb(urb);
		return -EINVAL;
	}

	++podxtpro->midi->num_active_send_urbs;

	/* Detect some cases that require a channel dump.
		 Important notes:
		 *) The actual dump request can not be sent here since we are not allowed to
		 wait for the completion of the first message in this context, and sending
		 the dump request before completion of the previous message leaves the POD
		 in an undefined state. The dump request will be sent when the echoed
		 commands are received.
		 *) This method fails if a param change message is "chopped" after the first
		 byte.
	*/
	for(i = 0; i < length; ++i) {
		if(data[i] == PODXTPRO_PROGRAM_CHANGE) {
			podxtpro_invalidate_current(podxtpro);
			break;
		}
		else if((data[i] == PODXTPRO_PARAM_CHANGE) && (i < length - 1))
			if((data[i + 1] == PODXTPRO_amp_model_setup) || (data[i + 1] == PODXTPRO_effect_setup)) {
				podxtpro_invalidate_current(podxtpro);
				break;
			}
	}

	return 0;
}

static int podxtpro_midi_output_open(snd_rawmidi_substream_t *substream)
{
	return 0;
}

static int podxtpro_midi_output_close(snd_rawmidi_substream_t *substream)
{
	return 0;
}

static void podxtpro_midi_output_trigger(snd_rawmidi_substream_t *substream, int up)
{
	unsigned long flags;
	usb_podxtpro_t *podxtpro = podxtpro_rawmidi_substream_midi(substream)->podxtpro;

	podxtpro->midi->substream_transmit = substream;
	spin_lock_irqsave(&podxtpro->midi->send_urb_lock, flags);

	if(podxtpro->midi->num_active_send_urbs == 0)
		podxtpro_midi_transmit(substream);

	spin_unlock_irqrestore(&podxtpro->midi->send_urb_lock, flags);
}

static void podxtpro_midi_output_drain(snd_rawmidi_substream_t *substream)
{
	static int count = 0;
	usb_podxtpro_t *podxtpro = podxtpro_rawmidi_substream_midi(substream)->podxtpro;
	wait_queue_head_t *head = &podxtpro->midi->send_wait;
	DECLARE_WAITQUEUE(wait, current);
	add_wait_queue(head, &wait);
	current->state = TASK_INTERRUPTIBLE;

	while(podxtpro->midi->num_active_send_urbs > 0)
		if(signal_pending(current))
			break;
		else
			schedule();

	current->state = TASK_RUNNING;
	remove_wait_queue(head, &wait);
	++count;
}

static int podxtpro_midi_input_open(snd_rawmidi_substream_t *substream)
{
	return 0;
}

static int podxtpro_midi_input_close(snd_rawmidi_substream_t *substream)
{
	return 0;
}

static void podxtpro_midi_input_trigger(snd_rawmidi_substream_t *substream, int up)
{
	usb_podxtpro_t *podxtpro = podxtpro_rawmidi_substream_midi(substream)->podxtpro;

	if(up)
		podxtpro->midi->substream_receive = substream;
	else
		podxtpro->midi->substream_receive = 0;
}

static snd_rawmidi_ops_t podxtpro_midi_output_ops = {
	.open = podxtpro_midi_output_open,
	.close = podxtpro_midi_output_close,
	.trigger = podxtpro_midi_output_trigger,
	.drain = podxtpro_midi_output_drain,
};

static snd_rawmidi_ops_t podxtpro_midi_input_ops = {
	.open = podxtpro_midi_input_open,
	.close = podxtpro_midi_input_close,
	.trigger = podxtpro_midi_input_trigger,
};

/*
	Cleanup the PODxt Pro MIDI device.
*/
static void podxtpro_cleanup_midi(snd_rawmidi_t *rmidi)
{
}

/* create a MIDI device */
static int snd_podxtpro_new_midi(struct snd_podxtpro_midi *midi)
{
	snd_rawmidi_t *rmidi;
	int err;

	if((err = snd_rawmidi_new(midi->podxtpro->card, "PODxt Pro MIDI", 0, 1, 1, &rmidi)) < 0)
		return err;

	rmidi->private_data = midi;
	rmidi->private_free = podxtpro_cleanup_midi;
	strcpy(rmidi->name, midi->podxtpro->devname);

	rmidi->info_flags =
		SNDRV_RAWMIDI_INFO_OUTPUT |
		SNDRV_RAWMIDI_INFO_INPUT |
		SNDRV_RAWMIDI_INFO_DUPLEX;

	snd_rawmidi_set_ops(rmidi, SNDRV_RAWMIDI_STREAM_OUTPUT, &podxtpro_midi_output_ops);
	snd_rawmidi_set_ops(rmidi, SNDRV_RAWMIDI_STREAM_INPUT, &podxtpro_midi_input_ops);
	return 0;
}

/* MIDI device destructor */
static int snd_podxtpro_midi_free(snd_device_t *device)
{
	return 0;
}

/*
	Initialize the PODxt Pro USB audio system.
	Create and register the sound card and mixer entries.
	Create URBs for playback and capture.
*/
int podxtpro_init_midi(struct usb_podxtpro *podxtpro)
{
	static snd_device_ops_t midi_ops = {
		.dev_free = snd_podxtpro_midi_free,
	};

	int err;
	snd_podxtpro_midi_t *midi;

	midi = kmalloc(sizeof(snd_podxtpro_midi_t), GFP_KERNEL);

	if(midi == NULL)
		return -ENOMEM;

	memset(midi, 0, sizeof(snd_podxtpro_midi_t));
	midi->podxtpro = podxtpro;
	podxtpro->midi = midi;

	if((err = snd_device_new(podxtpro->card, SNDRV_DEV_RAWMIDI, podxtpro, &midi_ops)) < 0)
		return err;

	if((err = snd_podxtpro_new_midi(midi)) < 0)
		return err;

	init_waitqueue_head(&midi->send_wait);
	spin_lock_init(&midi->send_urb_lock);
	spin_lock_init(&midi->midi_transmit_lock);
	return 0;
}
