/*
 * IEEE 802.11 station / management functionality
 *
 * Copyright (c) 2003, Jouni Malinen <jkmaline@cc.hut.fi>
 * Copyright (c) 2004-2005, Michael Wu <flamingice@sourmilk.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation. See README and COPYING for
 * more details.
 */

#include <linux/wireless.h>
#include <linux/if_ether.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/random.h>
#include <net/ieee80211.h>
#include <asm/types.h>

#include "ieee80211_sta.h"

#define IEEE80211_SCAN_INTERVAL (5 * HZ)
#define IEEE80211_SCAN_LISTEN (HZ / 10)
#define IEEE80211_AUTH_TIMEOUT (1 * HZ)
#define IEEE80211_AUTH_MAX_TRIES 3
#define IEEE80211_ASSOC_TIMEOUT (1 * HZ)
#define IEEE80211_ASSOC_MAX_TRIES 3
#define IEEE80211_MONITORING_INTERVAL (30 * HZ)
#define IEEE80211_LINKCHECK_INTERVAL (3 * HZ)

static void ieee80211_timer(void *ptr);

ParseRes ieee802_11_parse_elems(struct ieee80211_info_element *start, size_t len,
				struct ieee802_11_elems *elems)
{
	size_t left = len;
	struct ieee80211_info_element *info = start;
	int unknown = 0;

	memset(elems, 0, sizeof(*elems));

	while (left >= 2) {
		left -= 2;

		if (info->len > left) {
#if 0
			if (net_ratelimit())
				printk(KERN_DEBUG "IEEE 802.11 element parse "
				       "failed (id=%d elen=%d left=%d)\n",
				       id, elen, left);
#endif
			return ParseFailed;
		}

		switch (info->id) {
		case MFIE_TYPE_SSID:
			elems->ssid = info->data;
			elems->ssid_len = info->len;
			break;
		case MFIE_TYPE_RATES:
			elems->supp_rates = info->data;
			elems->supp_rates_len = info->len;
			break;
		case MFIE_TYPE_FH_SET:
			elems->fh_params = info->data;
			elems->fh_params_len = info->len;
			break;
		case MFIE_TYPE_DS_SET:
			elems->ds_params = info->data;
			elems->ds_params_len = info->len;
			break;
		case MFIE_TYPE_CF_SET:
			elems->cf_params = info->data;
			elems->cf_params_len = info->len;
			break;
		case MFIE_TYPE_TIM:
			elems->tim = info->data;
			elems->tim_len = info->len;
			break;
		case MFIE_TYPE_IBSS_SET:
			elems->ibss_params = info->data;
			elems->ibss_params_len = info->len;
			break;
		case MFIE_TYPE_CHALLENGE:
			elems->challenge = info->data;
			elems->challenge_len = info->len;
			break;
		case MFIE_TYPE_RSN:
			break;
		case MFIE_TYPE_RATES_EX:
			break;
		case MFIE_TYPE_GENERIC:
			break;
		default:
#if 0
			printk(KERN_DEBUG "IEEE 802.11 element parse ignored "
				      "unknown element (id=%d elen=%d)\n",
				      id, elen);
#endif
			unknown++;
			break;
		}

		left -= info->len;
		info  = (void *)info + info->len + 2;
	}

	if (left)
		return ParseFailed;

	return unknown ? ParseUnknown : ParseOK;
}

static char * ieee80211_reason_code(int reason)
{
	switch (reason) {
	case WLAN_REASON_UNSPECIFIED:
		return "Reason Unspecified";
	case WLAN_REASON_PREV_AUTH_NOT_VALID:
		return "Previous authentication no longer valid";
	case WLAN_REASON_DEAUTH_LEAVING:
		return "Deauthenticated: STA Leaving";
	case WLAN_REASON_DISASSOC_DUE_TO_INACTIVITY:
		return "Disassociated due to inactivity";
	case WLAN_REASON_DISASSOC_AP_BUSY:
		return "Disassociated: AP busy";
	case WLAN_REASON_CLASS2_FRAME_FROM_NONAUTH_STA:
		return "Class 2 frame from nonauthenticated STA";
	case WLAN_REASON_CLASS3_FRAME_FROM_NONASSOC_STA:
		return "Class 3 frame from nonassociated STA";
	case WLAN_REASON_DISASSOC_STA_HAS_LEFT:
		return "Disassociation: STA has left";
	case WLAN_REASON_STA_REQ_ASSOC_WITHOUT_AUTH:
		return "STA requesting association not authenticated";

	/* 802.11i */
	case WLAN_REASON_INVALID_IE:
		return "Invalid IE";
	case WLAN_REASON_MIC_FAILURE:
		return "MIC failure";
	case WLAN_REASON_4WAY_HANDSHAKE_TIMEOUT:
		return "4 way handshake timeout";
	case WLAN_REASON_GROUP_KEY_HANDSHAKE_TIMEOUT:
		return "Group key handshake timeout";
	case WLAN_REASON_IE_DIFFERENT:
		return "IE different";
	case WLAN_REASON_INVALID_GROUP_CIPHER:
		return "Invalid group cipher";
	case WLAN_REASON_INVALID_PAIRWISE_CIPHER:
		return "Invalid pairwise cipher";
	case WLAN_REASON_INVALID_AKMP:
		return "Invalid AKMP";
	case WLAN_REASON_UNSUPP_RSN_VERSION:
		return "Unsupported RSN version";
	case WLAN_REASON_INVALID_RSN_IE_CAP:
		return "Invalid RSN IE capability";
	case WLAN_REASON_IEEE8021X_FAILED:
		return "IEEE 802.1x failed";
	case WLAN_REASON_CIPHER_SUITE_REJECTED:
		return "Cipher suite rejected";

	default:
		return "Unknown reason";
	}
}

static char * ieee80211_status_code(int status)
{
	switch (status) {
	case WLAN_STATUS_SUCCESS:
		return "Success";
	case WLAN_STATUS_UNSPECIFIED_FAILURE:
		return "Unspecified failure";
	case WLAN_STATUS_CAPS_UNSUPPORTED:
		return "Capabilities unsupported";
	case WLAN_STATUS_REASSOC_NO_ASSOC:
		return "Reassociation failure; no association";
	case WLAN_STATUS_ASSOC_DENIED_UNSPEC:
		return "Association denied";
	case WLAN_STATUS_NOT_SUPPORTED_AUTH_ALG:
		return "Authentication algorithm not supported";
	case WLAN_STATUS_UNKNOWN_AUTH_TRANSACTION:
		return "Unknown authentication transaction";
	case WLAN_STATUS_CHALLENGE_FAIL:
		return "Challenge failed";
	case WLAN_STATUS_AUTH_TIMEOUT:
		return "Authentication timeout";
	case WLAN_STATUS_AP_UNABLE_TO_HANDLE_NEW_STA:
		return "AP unable to handle new STA";
	case WLAN_STATUS_ASSOC_DENIED_RATES:
		return "Association denied: Rates not supported";

	/* 802.11b */
	case WLAN_STATUS_ASSOC_DENIED_NOSHORTPREAMBLE:
		return "Association denied: Short preamble not supported";
	case WLAN_STATUS_ASSOC_DENIED_NOPBCC:
		return "Association denied: PBCC not supported";
	case WLAN_STATUS_ASSOC_DENIED_NOAGILITY:
		return "Association denied: Agility not supported";
	default:
		return "Unknown status code";
	}
}


static void ieee80211_set_associated(struct ieee80211_data *data, int assoc)
{
	struct ieee80211_device *ieee = data->ieee;
	
	if (assoc && ieee->state != IEEE80211_ASSOCIATED) {
		ieee->state = IEEE80211_ASSOCIATED;
		memcpy(ieee->bssid, data->bssid, ETH_ALEN);

		if (data->link_change)
			data->link_change(data->dev, assoc);
	
		netif_carrier_on(data->dev);
		if (netif_queue_stopped(data->dev))
			netif_wake_queue(data->dev);
		else
			netif_start_queue(data->dev);
	} else if (!assoc && ieee->state == IEEE80211_ASSOCIATED) {
		memset(ieee->bssid, 0, ETH_ALEN);

		if (data->link_change)
			data->link_change(data->dev, assoc);

		netif_carrier_off(data->dev);
		netif_stop_queue(data->dev);
	} else
		return;

	if (data->ieee->iw_mode == IW_MODE_INFRA) {
		union iwreq_data wrqu;
		wrqu.ap_addr.sa_family = ARPHRD_ETHER;
		memcpy(wrqu.ap_addr.sa_data, ieee->bssid, ETH_ALEN);
		wireless_send_event(data->dev, SIOCGIWAP, &wrqu, NULL);
	}
}

static int ieee80211_network_score(struct ieee80211_data *data,
				   struct ieee80211_network *network)
{
	struct ieee80211_device *ieee = data->ieee;
	int score;

	if (ieee->scan_age != 0 &&
	    !time_after(network->last_scanned + ieee->scan_age, jiffies))
		return -300;

	score = network->stats.rssi;

	/* detect hidden SSIDs and avoid them */
	if (network->flags & NETWORK_EMPTY_ESSID)
		score -= 50;

	if (data->pref_channel == network->channel)
		score += 100;

	return score;
}

/* removes all networks from list */
static void ieee80211_free_network(struct ieee80211_data *data)
{
	struct ieee80211_device *ieee = data->ieee;
	unsigned long flags;

	/* need irqsave as opposed to bh so this can be run during suspend */
	spin_lock_irqsave(&data->ieee->lock, flags);

	list_splice_init(&ieee->network_list, &ieee->network_free_list);

	spin_unlock_irqrestore(&data->ieee->lock, flags);
}

static int ieee80211_match_network(struct ieee80211_data *data, struct ieee80211_network *network)
{
	if (data->flags & ANY_SSID)
		return 1;

	if (!memcmp(&network->ssid, data->ssid, data->ssid_len))
		return 1;

	if (network->flags & NETWORK_EMPTY_ESSID &&
	    (network->ssid_len == data->ssid_len ||
	    (network->ssid_len == 1 && network->ssid[0] == ' ')))
		return 1;

	return 0;
}

static int ieee80211_select_bssid(struct ieee80211_data *data)
{
	struct ieee80211_device *ieee = data->ieee;
	struct ieee80211_network *network, *selected = NULL;
	int best_score = 0, score;
	u64 oldest_timestamp = 0;

	spin_lock_bh(&data->ieee->lock);

	/* Select first network that matches with current configuration */
	list_for_each_entry(network, &ieee->network_list, list) {
		score = ieee80211_network_score(data, network);
		if (score < -200)
			continue;

		if (ieee->iw_mode == IW_MODE_INFRA && 
		    ((network->capability & WLAN_CAPABILITY_IBSS) || 
		    !(network->capability & WLAN_CAPABILITY_ESS) ))
			continue;
		if (ieee->iw_mode == IW_MODE_ADHOC &&
		    ((network->capability & WLAN_CAPABILITY_ESS) ||
		    !(network->capability & WLAN_CAPABILITY_IBSS) ))
			continue;

		printk(KERN_DEBUG "%s: CHAN=%d BSSID=" MAC_FMT " SSID=%s"
			" RSSI=%d score=%d\n",
		       data->dev->name, network->channel, MAC_ARG(network->bssid),
		       escape_essid(network->ssid, network->ssid_len),
		       network->stats.rssi, score);

		if (data->flags & PREF_BSSID_SET &&
		    memcmp(&network->bssid, data->pref_bssid, ETH_ALEN) == 0) {
			selected = network;
			break;
		} else if (ieee80211_match_network(data, network)) {
			if (ieee->iw_mode == IW_MODE_INFRA &&
			    (!selected || score > best_score) ) {
				best_score = score;
				selected = network;
			} else if (ieee->iw_mode == IW_MODE_ADHOC &&
				   *((u64 *)network->time_stamp) > oldest_timestamp) {
				oldest_timestamp = *((u64 *)network->time_stamp);
				selected = network;
			}
		}
	}

	if (selected) {
		if (memcmp(data->bssid, &selected->bssid, ETH_ALEN) != 0) {
			printk(KERN_DEBUG "%s: new BSSID " MAC_FMT "\n",
			       data->dev->name, MAC_ARG(selected->bssid));
		}
		memcpy(data->bssid, selected->bssid, ETH_ALEN);
		if (data->flags & ANY_SSID) {
			data->ssid_len = selected->ssid_len;
			memcpy(data->ssid, selected->ssid, selected->ssid_len);
		}
		data->channel = selected->channel;
		data->beacon_interval = (selected->beacon_interval ? selected->beacon_interval : 100);
		data->timestamp = *((u64 *)network->time_stamp);
		data->bss.network = selected;
		if (selected->capability & WLAN_CAPABILITY_PRIVACY &&
		    selected->wpa_ie_len == 0 && selected->rsn_ie_len == 0)
			data->bss.auth_algorithm = WLAN_AUTH_SHARED_KEY;
		else
			data->bss.auth_algorithm = WLAN_AUTH_OPEN;
		data->bss.auth_state = 0;
	}

	spin_unlock_bh(&data->ieee->lock);

	return selected ? 0 : -1;
}

static void ieee80211_send_auth(struct ieee80211_data *data)
{
	struct sk_buff *skb;
	struct ieee80211_auth *auth;
	struct ieee80211_info_element *info;
	u16 fc;

	if (!data->tx)
		return;

	skb = dev_alloc_skb(data->ieee->tx_headroom + sizeof(*auth) +
			    sizeof(struct ieee80211_info_element) +
			    WLAN_AUTH_CHALLENGE_LEN + 8 /* WEP */);
	if (skb == NULL) {
		printk(KERN_DEBUG "%s: failed to allocate buffer for auth "
		       "frame\n", data->dev->name);
		return;
	}
	skb_reserve(skb, data->ieee->tx_headroom);
	
	auth = (struct ieee80211_auth *) skb_put(skb, sizeof(*auth));
	fc = IEEE80211_FTYPE_MGMT | IEEE80211_STYPE_AUTH;

	data->bss.auth_state++;
	if (data->bss.auth_algorithm == WLAN_AUTH_OPEN) {
		auth->algorithm = __constant_cpu_to_le16(WLAN_AUTH_OPEN);
		auth->transaction = cpu_to_le16(data->bss.auth_state);
		auth->status = WLAN_STATUS_SUCCESS;

		if (data->bss.auth_state != 1) {
			printk(KERN_DEBUG "%s: Unable to handle authentication transaction: %d\n",
			       data->dev->name, data->bss.auth_state);
			goto fail;
		}
	} else if (data->bss.auth_algorithm == WLAN_AUTH_SHARED_KEY) {
		auth->algorithm = cpu_to_le16(WLAN_AUTH_SHARED_KEY);
		auth->transaction = cpu_to_le16(data->bss.auth_state);
		auth->status = WLAN_STATUS_SUCCESS;

		switch (data->bss.auth_state) {
		case 1:
			break;
		case 3:
			info = (struct ieee80211_info_element *)skb_put(skb, sizeof(struct ieee80211_info_element) + WLAN_AUTH_CHALLENGE_LEN);
			info->id = MFIE_TYPE_CHALLENGE;
			info->len = WLAN_AUTH_CHALLENGE_LEN;
			if (data->bss.challenge) {
				memcpy(info->data, data->bss.challenge, WLAN_AUTH_CHALLENGE_LEN);
			} else {
				printk(KERN_DEBUG "%s: No challenge to return\n", data->dev->name);
				goto fail;
			}

			fc |= IEEE80211_FCTL_PROTECTED;
			break;
		default:
			printk(KERN_DEBUG "%s: Unable to handle authentication transaction: %d\n",
			       data->dev->name, data->bss.auth_state);
			goto fail;
		}
	} else {
		printk(KERN_DEBUG "%s: Unknown authentication algorithm: %d\n", data->dev->name, data->bss.auth_algorithm);
		goto fail;
	}

	printk(KERN_DEBUG "%s: TX authentication (alg=%d transaction=%d)\n",
	       data->dev->name, data->bss.auth_algorithm, data->bss.auth_state);

	auth->header.frame_ctl = cpu_to_le16(fc);
	memcpy(auth->header.addr1, data->bssid, ETH_ALEN);
	data->tx(data->dev, skb);
	return;

fail:
	dev_kfree_skb(skb);
}

void ieee80211_send_deauth(struct ieee80211_data *data, u16 reason, u8 *addr)
{
	struct sk_buff *skb;
	struct ieee80211_deauth *deauth;
	u16 fc;

	if (!data->tx)
		return;

	skb = dev_alloc_skb(data->ieee->tx_headroom + sizeof(*deauth));
	if (skb == NULL) {
		printk(KERN_DEBUG "%s: failed to allocate buffer for auth "
		       "frame\n", data->dev->name);
		return;
	}
	skb_reserve(skb, data->ieee->tx_headroom);

	deauth = (struct ieee80211_deauth *) skb_put(skb, sizeof(*deauth));
	fc = IEEE80211_FTYPE_MGMT | IEEE80211_STYPE_DEAUTH;
	deauth->reason = cpu_to_le16(reason);

	printk(KERN_DEBUG "%s: TX deauthentication (reason=%d %s)\n", data->dev->name, reason, ieee80211_reason_code(reason));

	//ieee80211_set_associated(data, 0);
	deauth->header.frame_ctl = cpu_to_le16(fc);
	memcpy(deauth->header.addr1, addr, ETH_ALEN);
	spin_lock_bh(&data->ieee->lock);
	data->tx(data->dev, skb);
	spin_unlock_bh(&data->ieee->lock);
}

static void ieee80211_authenticate(struct ieee80211_data *data)
{
	if (data->auth_tries++ > IEEE80211_AUTH_MAX_TRIES) {
		printk(KERN_DEBUG "%s: authentication with AP " MAC_FMT
		       " CHAN=%d timed out\n",
		       data->dev->name, MAC_ARG(data->bssid), data->channel);

		ieee80211_start_scan(data);
		return;
	}

	if (data->associate)
		data->associate(data->dev);

	spin_lock_bh(&data->ieee->lock);
	if (data->bss.auth_state != 5) {
		data->ieee->state = IEEE80211_AUTHENTICATING;
		printk(KERN_DEBUG "%s: authenticate with AP " MAC_FMT " CHAN=%d\n",
		       data->dev->name, MAC_ARG(data->bssid), data->channel);

		data->bss.auth_state = 0;
		ieee80211_send_auth(data);
	}
	spin_unlock_bh(&data->ieee->lock);

	schedule_delayed_work(&data->work, IEEE80211_AUTH_TIMEOUT);
}

static void ieee80211_send_assoc_req(struct ieee80211_data *data)
{
	struct ieee80211_device *ieee = data->ieee;
	struct sk_buff *skb;
	struct ieee80211_reassoc_request *reassoc_req;
	struct ieee80211_info_element *info;
	struct ieee80211_network *network;
	u16 fc;
	unsigned int reassoc = 0;

	if (!data->tx)
		return;

	skb = dev_alloc_skb(ieee->tx_headroom + sizeof(*reassoc_req) +
			    sizeof(struct ieee80211_info_element) + data->ssid_len +
			    sizeof(struct ieee80211_info_element) + data->num_supp_rates +
			    ieee->wpa_ie_len);
	if (!skb) {
		printk(KERN_DEBUG "%s: failed to allocate buffer for assoc req "
		       "frame\n", data->dev->name);
		return;
	}
	skb_reserve(skb, data->ieee->tx_headroom);
	
	spin_lock_bh(&data->ieee->lock);
	network = data->bss.network;
	/* TODO: check how prev_bssid_set is suppose to work */
	if (data->flags & PREV_BSSID_SET) {
		reassoc_req = (struct ieee80211_reassoc_request *) skb_put(skb, sizeof(*reassoc_req));
		fc = IEEE80211_FTYPE_MGMT | IEEE80211_STYPE_REASSOC_REQ;
		reassoc_req->capability = cpu_to_le16(network->capability);
		reassoc_req->listen_interval = cpu_to_le16(data->listen_interval);
		memcpy(reassoc_req->current_ap, data->prev_bssid, ETH_ALEN);
		reassoc = 1;
	} else {
		reassoc_req = (struct ieee80211_reassoc_request *) skb_put(skb, sizeof(struct ieee80211_assoc_request));
		fc = IEEE80211_FTYPE_MGMT | IEEE80211_STYPE_ASSOC_REQ;
		reassoc_req->capability = cpu_to_le16(network->capability);
		reassoc_req->listen_interval = cpu_to_le16(data->listen_interval);
	}

	/* SSID */
	info = (struct ieee80211_info_element *)skb_put(skb, sizeof(struct ieee80211_info_element) + data->ssid_len);
	info->id = MFIE_TYPE_SSID;
	info->len = data->ssid_len;
	memcpy(info->data, data->ssid, data->ssid_len);

	info = (struct ieee80211_info_element *)skb_put(skb, sizeof(struct ieee80211_info_element) + data->num_supp_rates);
	/* All supported rates */
	info->id = MFIE_TYPE_RATES;
	info->len = data->num_supp_rates;
	memcpy(info->data, data->supp_rates, data->num_supp_rates);

	/* WPA/RSN/WMM */
	if (ieee->wpa_ie_len)
		memcpy(skb_put(skb, ieee->wpa_ie_len), ieee->wpa_ie, ieee->wpa_ie_len);

	printk(KERN_DEBUG "%s: TX %sssocReq (capab=0x%x gen_ie_len=0x%x)\n",
	       data->dev->name, reassoc ? "Rea" : "A", network->capability,
	       ieee->wpa_ie_len);

	reassoc_req->header.frame_ctl = cpu_to_le16(fc);
	memcpy(reassoc_req->header.addr1, data->bssid, ETH_ALEN);
	data->tx(data->dev, skb);
	spin_unlock_bh(&data->ieee->lock);
}

void ieee80211_send_disassoc(struct ieee80211_data *data, u16 reason, u8 *addr)
{
	struct sk_buff *skb;
	struct ieee80211_disassoc *disassoc;
	u16 fc;

	if (!data->tx)
		return;

	skb = dev_alloc_skb(data->ieee->tx_headroom + sizeof(*disassoc));
	if (!skb) {
		printk(KERN_DEBUG "%s: failed to allocate buffer for assoc req "
		       "frame\n", data->dev->name);
		return;
	}
	skb_reserve(skb, data->ieee->tx_headroom);

	fc = IEEE80211_FTYPE_MGMT | IEEE80211_STYPE_DISASSOC;
	disassoc = (struct ieee80211_disassoc *) skb_put(skb, sizeof(*disassoc));
	disassoc->reason = cpu_to_le16(reason);
	
	printk(KERN_DEBUG "%s: TX Disassoc (reason=%d %s)\n", data->dev->name, reason, ieee80211_reason_code(reason));

	//ieee80211_set_associated(data, 0);
	disassoc->header.frame_ctl = cpu_to_le16(fc);
	memcpy(disassoc->header.addr1, addr, ETH_ALEN);
	spin_lock_bh(&data->ieee->lock);
	data->tx(data->dev, skb);
	spin_unlock_bh(&data->ieee->lock);
}

/* this function dampens the effect of the link bouncing off and on */
void ieee80211_linkcheck(struct ieee80211_data *data)
{
	if (data->ieee->state == IEEE80211_ASSOCIATED)
		schedule_delayed_work(&data->work, IEEE80211_LINKCHECK_INTERVAL);
}

static void ieee80211_associated(struct ieee80211_data *data)
{
	//struct ieee80211_bss *bss, *tmp, *prev = NULL;

	/* TODO: start monitoring current AP signal quality and number of
	 * missed beacons. Scan other channels every now and then and search
	 * for better APs. */

#if 0
	spin_lock_bh(&data->lock);

	/* remove expired BSSes */
	bss = data->bss;
	while (bss) {
		if (time_after(jiffies, bss->last_rx + IEEE80211_MONITORING_INTERVAL) &&
		     memcmp(data->bssid, bss->bssid, ETH_ALEN)) {
			if (prev)
				prev->next = bss->next;
			else
				data->bss = bss->next;

			tmp = bss;
			bss = bss->next;
			kfree(tmp);
		} else {
			prev = bss;
			bss = bss->next;
		}
	}

	spin_unlock_bh(&data->lock);
#endif

	if (data->flags & LINK_ON) {
		if (data->link_monitor)
			data->link_monitor(data->dev);
		schedule_delayed_work(&data->work, IEEE80211_MONITORING_INTERVAL);
	} else if (data->flags & AUTO_ASSOCIATE)
		ieee80211_start_scan(data);
	else
		ieee80211_set_associated(data, 0);
}

static void ieee80211_associate(struct ieee80211_data *data)
{
	if (data->assoc_tries++ > IEEE80211_ASSOC_MAX_TRIES) {
		printk(KERN_DEBUG "%s: association with AP " MAC_FMT
		       " CHAN=%d timed out\n",
		       data->dev->name, MAC_ARG(data->bssid), data->channel);
		ieee80211_start_scan(data);
		return;
	}

	data->ieee->state = IEEE80211_ASSOCIATING;
	printk(KERN_DEBUG "%s: associate with AP " MAC_FMT "\n",
	       data->dev->name, MAC_ARG(data->bssid));

	ieee80211_send_assoc_req(data);

	schedule_delayed_work(&data->work, IEEE80211_ASSOC_TIMEOUT);
}

static void ieee80211_send_probe_req(struct ieee80211_data *data)
{
	struct sk_buff *skb;
	struct ieee80211_probe_request *probe_req;
	struct ieee80211_info_element *info;
	u16 fc;
	u8 dst[ETH_ALEN];

	if (!data->tx)
		return;

	skb = dev_alloc_skb(data->ieee->tx_headroom + sizeof(*probe_req) +
			    sizeof(struct ieee80211_info_element) +
			    sizeof(struct ieee80211_info_element) + data->num_supp_rates);
	if (skb == NULL) {
		printk(KERN_DEBUG "%s: failed to allocate buffer for probe "
		       "request\n", data->dev->name);
		return;
	}

	skb_reserve(skb, data->ieee->tx_headroom);
	probe_req = (struct ieee80211_probe_request *) skb_put(skb, sizeof(*probe_req));
	memset(probe_req, 0, skb->len);
	fc = IEEE80211_FTYPE_MGMT | IEEE80211_STYPE_PROBE_REQ;
	memset(dst, ~0, ETH_ALEN);
	info = (struct ieee80211_info_element *) skb_put(skb, sizeof(struct ieee80211_info_element));

	/* Broadcast SSID */
	info->id = MFIE_TYPE_SSID;
	info->len = 0;

	/* All supported rates */
	info = (struct ieee80211_info_element *) skb_put(skb, sizeof(struct ieee80211_info_element) + data->num_supp_rates);
	info->id = MFIE_TYPE_RATES;
	info->len = data->num_supp_rates;
	memcpy(info->data, data->supp_rates, data->num_supp_rates);

	probe_req->header.frame_ctl = cpu_to_le16(fc);
	memcpy(probe_req->header.addr1, dst, ETH_ALEN);
	spin_lock_bh(&data->ieee->lock);
	data->tx(data->dev, skb);
	spin_unlock_bh(&data->ieee->lock);
}

static void ieee80211_scan_complete(struct ieee80211_data *data)
{
	if (data->ieee->iw_mode == IW_MODE_INFRA) {
		if (!ieee80211_select_bssid(data))
			ieee80211_authenticate(data);
		else {
			ieee80211_free_network(data);

			schedule_delayed_work(&data->work, IEEE80211_SCAN_INTERVAL);
		}
	} else if (data->ieee->iw_mode == IW_MODE_ADHOC) {
		int start = 0;
		if (ieee80211_select_bssid(data)) {
			memcpy(data->bssid, data->dev->dev_addr, 3);
			get_random_bytes(&data->bssid[3], 3);
			data->beacon_interval = 100;
			if (data->pref_channel)
				data->channel = data->pref_channel;
			else
				data->channel = data->chan_range.min;

			if (!data->ssid) {
				printk(KERN_WARNING "%s: No SSID set. SSID set to \"default\"\n", data->dev->name);
				memcpy(data->ssid, "default", 7);
				data->ssid_len = 7;
			}
			printk(KERN_DEBUG "%s: No matching adhoc sta found. Creating IBSS " MAC_FMT " CHAN=%d\n",
			       data->dev->name, MAC_ARG(data->bssid), data->channel);
			start = 1;
		} else
			printk(KERN_DEBUG "%s: joining IBSS " MAC_FMT " CHAN=%d\n",
			       data->dev->name, MAC_ARG(data->bssid), data->channel);

		if (data->setup_ibss)
			data->setup_ibss(data->dev, start);

		ieee80211_set_associated(data, 1);
		schedule_delayed_work(&data->work, IEEE80211_MONITORING_INTERVAL);
	}
}

static void ieee80211_scan(struct ieee80211_data *data)
{
	if (data->scan_channel == 0) {
		printk(KERN_DEBUG "%s: scanning..\n", data->dev->name);
		ieee80211_increment_scans(data->ieee);
		
		data->scan_channel = data->chan_range.min;

		/*if (data->flags & ANY_SSID)
			data->ssid_len = 0;*/

		data->bss.network = NULL;
		data->auth_tries = data->assoc_tries = 0;
		data->beacon_interval = 0;

		if (data->scan)
			data->scan(data->dev);
	}/* if (data->scan.num_channels) {
	
	}*/ else
		data->scan_channel++;
	
	if (data->scan_channel > data->chan_range.max) {
		if (data->flags & SCAN_NOTIFY) {
			union iwreq_data wrqu;
			data->flags &= ~SCAN_NOTIFY;
			wrqu.data.length = 0;
			wrqu.data.flags = 0;
			wireless_send_event(data->dev, SIOCGIWSCAN, &wrqu, NULL);
		}

		data->scan_param.flags = 0;
		data->scan_param.num_channels = 0;
		data->scan_param.scan_type = IW_SCAN_TYPE_ACTIVE;
		memset(data->scan_param.bssid.sa_data, ~0, ETH_ALEN);

		data->scan_channel = 0;
		if (data->flags & AUTO_ASSOCIATE)
			ieee80211_scan_complete(data);
		else
			data->ieee->state = IEEE80211_SHUTDOWN;
	} else {
		if (data->set_channel)
			data->set_channel(data->dev, data->scan_channel);
		if (data->scan_param.scan_type == IW_SCAN_TYPE_ACTIVE)
			ieee80211_send_probe_req(data);
		schedule_delayed_work(&data->work, IEEE80211_SCAN_LISTEN);
	}
}

void ieee80211_start_scan(struct ieee80211_data *data)
{
	if (data->ieee->iw_mode != IW_MODE_INFRA &&
	    data->ieee->iw_mode != IW_MODE_ADHOC)
		return;

	/* don't do anything if we're already scanning */
	if (data->ieee->state == IEEE80211_INITIALIZED)
		return;

	data->ieee->state = IEEE80211_INITIALIZED;
	ieee80211_set_associated(data, 0);
	data->scan_channel = 0;

	schedule_work(&data->work);
}

void ieee80211_init(struct ieee80211_data *data)
{
	INIT_WORK(&data->work, ieee80211_timer, data);

	data->listen_interval = 10;
	data->ieee->state = IEEE80211_UNINITIALIZED;
	data->flags = AUTO_ASSOCIATE | AUTO_RATE | ANY_SSID;
	data->wpa_version = IW_AUTH_WPA_VERSION_DISABLED;
	data->cipher_pairwise = IW_AUTH_CIPHER_NONE;
	data->cipher_group = IW_AUTH_CIPHER_NONE;
	data->auth_alg = IW_AUTH_ALG_OPEN_SYSTEM | IW_AUTH_ALG_SHARED_KEY;

	if (!data->supp_rates)
		data->supp_rates = (u8 *)IEEE80211_B_RATES;

	data->num_supp_rates = strlen(data->supp_rates);
	data->rate = data->supp_rates[data->num_supp_rates-1];

	data->scan_param.flags = 0;
	data->scan_param.num_channels = 0;
	data->scan_param.scan_type = IW_SCAN_TYPE_ACTIVE;
	memset(data->scan_param.bssid.sa_data, ~0, ETH_ALEN);
}

void ieee80211_stop(struct ieee80211_data *data)
{
	cancel_delayed_work(&data->work);
	flush_scheduled_work();
	data->ieee->state = IEEE80211_SHUTDOWN;
	ieee80211_set_associated(data, 0);

	ieee80211_free_network(data);

	memset(data->bssid, 0, ETH_ALEN);

	data->scan_param.flags = 0;
	data->scan_param.num_channels = 0;
	data->scan_param.scan_type = IW_SCAN_TYPE_ACTIVE;
	memset(data->scan_param.bssid.sa_data, ~0, ETH_ALEN);
}

static void ieee80211_rx_mgmt_auth(struct ieee80211_data *data,
				   struct ieee80211_auth *auth,
				   size_t len,
				   struct ieee80211_rx_stats *rx_status)
{
	u16 auth_alg, auth_transaction, status_code;
	struct ieee802_11_elems elems;
	size_t baselen;

	if (data->ieee->state != IEEE80211_AUTHENTICATING) {
		printk(KERN_DEBUG "%s: authentication frame received from "
		       MAC_FMT ", but not in authenticate state - ignored\n",
		       data->dev->name, MAC_ARG(auth->header.addr2));
		return;
	}

	if (len < sizeof(struct ieee80211_auth)) {
		printk(KERN_DEBUG "%s: too short (%d) authentication frame "
		       "received from " MAC_FMT " - ignored\n",
		       data->dev->name, len, MAC_ARG(auth->header.addr2));
		return;
	}

	if (memcmp(data->bssid, auth->header.addr2, ETH_ALEN) != 0) {
		printk(KERN_DEBUG "%s: authentication frame received from "
		       "unknown AP (SA=" MAC_FMT " BSSID=" MAC_FMT ") - "
		       "ignored\n", data->dev->name, MAC_ARG(auth->header.addr2),
		       MAC_ARG(auth->header.addr3));
		return;
	}

	baselen = (u8 *) auth->info_element - (u8 *) auth;
	if (baselen > len) {
		printk(KERN_DEBUG "%s: Auth underflow %d > %d\n",
		       data->dev->name, baselen, len);
		return;
	}

	if (ieee802_11_parse_elems(auth->info_element, len - baselen,
				   &elems) == ParseFailed) {
		printk(KERN_DEBUG "%s: Parse failed.\n", data->dev->name);
		return;
	}

	auth_alg = le16_to_cpu(auth->algorithm);
	auth_transaction = le16_to_cpu(auth->transaction);
	status_code = le16_to_cpu(auth->status);

	printk(KERN_DEBUG "%s: RX authentication (alg=%d "
	       "transaction=%d status=%d %s)\n",
	       data->dev->name, auth_alg,
	       auth_transaction, status_code,
	       ieee80211_status_code(status_code));

	spin_lock_bh(&data->ieee->lock);

	if (auth_transaction != data->bss.auth_state + 1) {
		printk(KERN_DEBUG "%s: unexpected authentication "
		       "transaction number. expected %d, got %d\n",
		       data->dev->name, data->bss.auth_state + 1, auth_transaction);
		data->bss.auth_state = 0;
		spin_unlock_bh(&data->ieee->lock);
		return;
	}
	data->bss.auth_algorithm = auth_alg;
	data->bss.auth_state = auth_transaction;

	if ((auth_transaction == 2 || auth_transaction == 4)
	    && status_code != WLAN_STATUS_SUCCESS) {
		data->auth_tries++;
		data->bss.auth_state = 0;
		if (auth_transaction == 0x2) {
			data->bss.auth_algorithm = WLAN_AUTH_OPEN;
			printk(KERN_DEBUG "%s: Falling back on open authentication...\n", data->dev->name);
		}
		spin_unlock_bh(&data->ieee->lock);
		return;
	}

	switch (auth_alg) {
	case WLAN_AUTH_OPEN:
		if (auth_transaction == 2)
			data->bss.auth_state = 5;
		else
			printk(KERN_DEBUG "%s: invalid authentication "
			       "transaction number: %d\n",
			       data->dev->name, auth_transaction);
		break;
	case WLAN_AUTH_SHARED_KEY:
		switch (auth_transaction) {
		case 2:
			if (elems.challenge_len != WLAN_AUTH_CHALLENGE_LEN) {
				printk(KERN_DEBUG "%s: invalid challenge length: %d\n",
				       data->dev->name, elems.challenge_len);
				return;
			}
			data->bss.challenge = elems.challenge;
			ieee80211_send_auth(data);
			break;
		case 4:
			data->bss.auth_state = 5;
			break;
		default:
			printk(KERN_DEBUG "%s: invalid authentication"
			       "transaction number: %d\n",
			       data->dev->name, auth_transaction);
			break;
		}
		break;
	}

	if (data->bss.auth_state == 5) {
		data->auth_tries = 0;
		printk(KERN_DEBUG "%s: authenticated\n", data->dev->name);

		spin_unlock_bh(&data->ieee->lock);
		ieee80211_associate(data);
	} else
		spin_unlock_bh(&data->ieee->lock);
}

static void ieee80211_rx_mgmt_deauth(struct ieee80211_data *data,
				     struct ieee80211_deauth *deauth,
				     size_t len,
				     struct ieee80211_rx_stats *rx_status)
{
	u16 reason_code;

	if (len < sizeof(struct ieee80211_deauth)) {
		printk(KERN_DEBUG "%s: too short (%d) deauthentication frame "
		       "received from " MAC_FMT " - ignored\n",
		       data->dev->name, len, MAC_ARG(deauth->header.addr2));
		return;
	}

	if (memcmp(data->bssid, deauth->header.addr2, ETH_ALEN) != 0) {
		printk(KERN_DEBUG "%s: deauthentication frame received from "
		       "unknown AP (SA=" MAC_FMT " BSSID=" MAC_FMT ") - "
		       "ignored\n", data->dev->name, MAC_ARG(deauth->header.addr2),
		       MAC_ARG(deauth->header.addr2));
		return;
	}

	reason_code = le16_to_cpu(deauth->reason);

	printk(KERN_DEBUG "%s: RX deauthentication from " MAC_FMT
	       " (reason=%d %s)\n",
	       data->dev->name, MAC_ARG(deauth->header.addr2), reason_code, ieee80211_reason_code(reason_code));

	spin_lock_bh(&data->ieee->lock);
	if (data->bss.auth_state == 5)
		printk(KERN_DEBUG "%s: deauthenticated\n", data->dev->name);

	data->bss.auth_state = 0;
	spin_unlock_bh(&data->ieee->lock);

	ieee80211_set_associated(data, 0);
	ieee80211_authenticate(data);
}


static void ieee80211_rx_mgmt_disassoc(struct ieee80211_data *data,
				       struct ieee80211_disassoc *disassoc,
				       size_t len,
				       struct ieee80211_rx_stats *rx_status)
{
	u16 reason_code;

	if (len < sizeof(struct ieee80211_disassoc)) {
		printk(KERN_DEBUG "%s: too short (%d) disassociation frame "
		       "received from " MAC_FMT " - ignored\n",
		       data->dev->name, len, MAC_ARG(disassoc->header.addr2));
		return;
	}

	if (memcmp(data->bssid, disassoc->header.addr2, ETH_ALEN) != 0) {
		printk(KERN_DEBUG "%s: disassociation frame received from "
		       "unknown AP (SA=" MAC_FMT " BSSID=" MAC_FMT ") - "
		       "ignored\n", data->dev->name, MAC_ARG(disassoc->header.addr2),
		       MAC_ARG(disassoc->header.addr3));
		return;
	}

	reason_code = le16_to_cpu(disassoc->reason);

	printk(KERN_DEBUG "%s: RX disassociation from " MAC_FMT
	       " (reason=%d %s)\n",
	       data->dev->name, MAC_ARG(disassoc->header.addr2), reason_code, ieee80211_reason_code(reason_code));

	data->assoc_tries++;
	
	if (data->ieee->state == IEEE80211_ASSOCIATED)
		printk(KERN_DEBUG "%s: disassociated\n", data->dev->name);

	ieee80211_set_associated(data, 0);
	ieee80211_associate(data);
}

static void ieee80211_rx_mgmt_assoc_resp(struct ieee80211_data *data,
					 struct ieee80211_assoc_response *assoc_resp,
					 size_t len,
					 struct ieee80211_rx_stats *rx_status,
					 int reassoc)
{
	struct ieee80211_network *network;
	struct ieee802_11_elems elems;
	size_t baselen;
	u16 capab_info, status_code, aid;

	/* AssocResp and ReassocResp have identical structure, so process both
	 * of them in this function. */

	if (data->ieee->state != IEEE80211_ASSOCIATING) {
		printk(KERN_DEBUG "%s: association frame received from "
		       MAC_FMT ", but not in associate state - ignored\n",
		       data->dev->name, MAC_ARG(assoc_resp->header.addr2));
		return;
	}

	if (len < sizeof(struct ieee80211_assoc_response)) {
		printk(KERN_DEBUG "%s: too short (%d) association frame "
		       "received from " MAC_FMT " - ignored\n",
		       data->dev->name, len, MAC_ARG(assoc_resp->header.addr2));
		return;
	}

	if (memcmp(data->bssid, assoc_resp->header.addr2, ETH_ALEN) != 0) {
		printk(KERN_DEBUG "%s: association frame received from "
		       "unknown AP (SA=" MAC_FMT " BSSID=" MAC_FMT ") - "
		       "ignored\n", data->dev->name, MAC_ARG(assoc_resp->header.addr2),
		       MAC_ARG(assoc_resp->header.addr3));
		return;
	}

	capab_info = le16_to_cpu(assoc_resp->capability);
	status_code = le16_to_cpu(assoc_resp->status);
	aid = le16_to_cpu(assoc_resp->aid);

	printk(KERN_DEBUG "%s: RX %sssocResp (capab=0x%x "
	       "aid=%d status=%d %s)\n",
	       data->dev->name, reassoc ? "Rea" : "A",
	       capab_info, aid & 0x3FFF, status_code,
	       ieee80211_status_code(status_code));

	spin_lock_bh(&data->ieee->lock);
	network = data->bss.network;
	if (status_code != WLAN_STATUS_SUCCESS) {
		printk(KERN_DEBUG "%s: AP denied association\n",
		       data->dev->name);
		data->assoc_tries++;
		spin_unlock_bh(&data->ieee->lock);
		return;
	}

	if ((aid & ((1<<15) | (1<<14))) != ((1<<15) | (1<<14)))
		printk(KERN_DEBUG "%s: invalid aid value %d; bits 15:14 not "
		       "set\n", data->dev->name, aid);
	aid &= ~((1<<15) | (1<<14));

	baselen = (u8 *) assoc_resp->info_element - (u8 *) assoc_resp;
        if (ieee802_11_parse_elems(assoc_resp->info_element, len - baselen, &elems) != ParseFailed
	    && elems.supp_rates) {
		memset(network->rates, 0, MAX_RATES_LENGTH);
		memcpy(network->rates, elems.supp_rates,
		       min(elems.supp_rates_len, MAX_RATES_LENGTH));
	}

	spin_unlock_bh(&data->ieee->lock);

	printk(KERN_DEBUG "%s: associated\n", data->dev->name);
	data->aid = aid;
	data->assoc_tries = 0;

	ieee80211_set_associated(data, 1);
	schedule_delayed_work(&data->work, IEEE80211_SCAN_INTERVAL);
}


void ieee80211_rx_mgmt(struct ieee80211_data *data, struct sk_buff *skb,
		       struct ieee80211_rx_stats *rx_status)
{
	struct ieee80211_hdr_4addr *hdr;
	u16 fc;

	if (skb->len < 24)
		goto fail;

	hdr = (struct ieee80211_hdr_4addr *) skb->data;
	fc = le16_to_cpu(hdr->frame_ctl);

	switch (WLAN_FC_GET_STYPE(fc)) {
	case IEEE80211_STYPE_PROBE_REQ:
		/* TODO */
		break;
	case IEEE80211_STYPE_PROBE_RESP:
	case IEEE80211_STYPE_BEACON:
		ieee80211_rx_mgt(data->ieee, hdr, rx_status);
		break;
	case IEEE80211_STYPE_AUTH:
		ieee80211_rx_mgmt_auth(data, (struct ieee80211_auth *)hdr, skb->len, rx_status);
		break;
	case IEEE80211_STYPE_ASSOC_RESP:
		ieee80211_rx_mgmt_assoc_resp(data, (struct ieee80211_assoc_response *)hdr, skb->len, rx_status, 0);
		break;
	case IEEE80211_STYPE_REASSOC_RESP:
		/* FIXME: stupid cast */
		ieee80211_rx_mgmt_assoc_resp(data, (struct ieee80211_assoc_response *)hdr, skb->len, rx_status, 1);
		break;
	case IEEE80211_STYPE_DEAUTH:
		ieee80211_rx_mgmt_deauth(data, (struct ieee80211_deauth *)hdr, skb->len, rx_status);
		break;
	case IEEE80211_STYPE_DISASSOC:
		ieee80211_rx_mgmt_disassoc(data, (struct ieee80211_disassoc *)hdr, skb->len, rx_status);
		break;
	default:
		printk(KERN_DEBUG "%s: received unknown management frame - "
		       "stype=%d\n", data->dev->name, WLAN_FC_GET_STYPE(fc));
		break;
	}

 fail:
	if (skb)
		dev_kfree_skb(skb);
}


static void ieee80211_timer(void *ptr)
{
	struct ieee80211_data *data = (struct ieee80211_data *) ptr;

	switch (data->ieee->state) {
	case IEEE80211_INITIALIZED:
		ieee80211_scan(data);
		break;
	case IEEE80211_AUTHENTICATING:
		ieee80211_authenticate(data);
		break;
	case IEEE80211_AUTHENTICATED:
	case IEEE80211_ASSOCIATING:
		ieee80211_associate(data);
		break;
	case IEEE80211_ASSOCIATED:
		ieee80211_associated(data);
		break;
	case IEEE80211_SHUTDOWN:
		break;
	default:
		printk(KERN_DEBUG "ieee80211_timer: Unknown state %d\n",
		       data->ieee->state);
		break;
	}
}

int ieee80211_filter_duplicates(struct ieee80211_dup_cache *cache, struct ieee80211_hdr_3addr *hdr)
{
	struct ieee80211_dup_cache *entry;
	unsigned int i, oldest_idx = 0;
	unsigned long oldest = ~0;
	u16 seq_ctl, fc;

	/* don't check multicast/broadcast frames */
	if (hdr->addr1[0] & 0x01)
		return 0;

	fc = le16_to_cpu(hdr->frame_ctl);
	if (WLAN_FC_GET_TYPE(fc) == IEEE80211_FTYPE_MGMT &&
	    WLAN_FC_GET_STYPE(fc) == IEEE80211_STYPE_ATIM)
		return 0;

	seq_ctl = le16_to_cpu(hdr->seq_ctl);

	for (i = 0; i < IEEE80211_DUP_CACHE_SIZE; i++) {
		entry = &cache[i];

		/* look for matching addr & seq */
		if (WLAN_GET_SEQ_SEQ(seq_ctl) == WLAN_GET_SEQ_SEQ(entry->seq_ctl) &&
		    !memcmp(entry->addr, hdr->addr2, ETH_ALEN))
			break;

		/* keep track of which tuple is the oldest */
		if (entry->last_rx < oldest) {
			oldest_idx = i;
			oldest = entry->last_rx;
		}
	}

	if (i == IEEE80211_DUP_CACHE_SIZE) {
		/* no match: replace oldest tuple with new tuple */
		entry = &cache[oldest_idx];
		memcpy(entry->addr, hdr->addr2, ETH_ALEN);
		entry->seq_ctl = seq_ctl;
		entry->last_rx = jiffies;
		return 0;
	}

	entry->last_rx = jiffies;

	if (entry->seq_ctl != seq_ctl) {
		/* addr2 & seq matched: update frag */
		entry->seq_ctl = seq_ctl;
		return 0;
	}

	if (hdr->frame_ctl & __constant_cpu_to_le16(IEEE80211_FCTL_RETRY))
		return 1;

	return 0;
}

/* returns a rate appropriate for the destination, if it can be determined */
/* TODO: probably needs to be implemented properly for adhoc mode */
u8 ieee80211_get_rate(struct ieee80211_data *data, u8 *addr)
{
/*	struct ieee80211_network *network;*/
	u8 rate;

/*	network = ieee80211_get_network(data->ieee, addr);

	if (bss)
		rate = bss->rate;
	else*/
		rate = data->rate;

	return rate;
}
