/*
 *  Copyright (C) 2002 Intersil Americas Inc.
 *  Copyright (C) 2003 Herbert Valerio Riedel <hvr@gnu.org>
 *  Copyright (C) 2003 Luis R. Rodriguez <mcgrof@ruslug.rutgers.edu>
 *  Copyright (C) 2003 Aurelien Alleaume <slts@free.fr>
 *  Copyright (C) 2004-2005 Jean-Baptiste Note <jean-baptiste.note@m4x.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef _ISLPCI_DEV_H
#define _ISLPCI_DEV_H

#include <linux/version.h>
#include <linux/list.h>
#include <linux/interrupt.h>

#include "isl_38xx.h"

#define DRV_NAME	"islsm_pci"
#define DRV_VERSION	"0"

/* General driver definitions */
#define PCIDEVICE_LATENCY_TIMER_MIN 		0x40
#define PCIDEVICE_LATENCY_TIMER_VAL 		0x50

/* some states might not be superflous and may be removed when
   design is finalized (hvr) */
typedef enum {
	PRV_STATE_OFF = 0,	/* this means hw_unavailable is != 0 */
	PRV_STATE_PREBOOT,	/* we are in a pre-boot state (empty RAM) */
	PRV_STATE_BOOT,		/* boot state (fw upload, run fw) */
	PRV_STATE_POSTBOOT,	/* after boot state, need reset now */
	PRV_STATE_PREINIT,	/* pre-init state */
	PRV_STATE_INIT,		/* init state (restore MIB backup to device) */
	PRV_STATE_READY,	/* driver&device are in operational state */
	PRV_STATE_SLEEP		/* device in sleep mode */
} islpci_state_t;

struct islpci_membuf {
	unsigned                length;
	struct sk_buff         *skb;	/* the skb itself */
	void                   *mem;	/* address of memory as seen by CPU */
	dma_addr_t              pci_addr;	/* address of memory as seen by device */
};

	/* Each queue has three indexes:
	 *   free/index_mgmt/data_rx/tx (called index, see below),
	 *   driver_curr_frag, and device_curr_frag (in the control block)
	 * All indexes are ever-increasing, but interpreted modulo the
	 * device queue size when used.
	 *   index <= device_curr_frag <= driver_curr_frag  at all times
	 * For rx queues, [index, device_curr_frag) contains fragments
	 * that the interrupt processing needs to handle (owned by driver).
	 * [device_curr_frag, driver_curr_frag) is the free space in the
	 * rx queue, waiting for data (owned by device).  The driver
	 * increments driver_curr_frag to indicate to the device that more
	 * buffers are available.
	 * If device_curr_frag == driver_curr_frag, no more rx buffers are
	 * available, and the rx DMA engine of the device is halted.
	 * For tx queues, [index, device_curr_frag) contains fragments
	 * where tx is done; they need to be freed (owned by driver).
	 * [device_curr_frag, driver_curr_frag) contains the frames
	 * that are being transferred (owned by device).  The driver
	 * increments driver_curr_frag to indicate that more tx work
	 * needs to be done.
	 */

struct islpci_queue {
	struct islpci_membuf   *ringbuf;
	int                     len;	/* the length of the ringbufer */

	isl38xx_fragment       *cb_queue;

	/* queue management */
	unsigned int            index;
	int                     full;

	size_t                  buffer_size;	/* the size of the buffer to allocate */
};

typedef struct {
	spinlock_t              slock;	/* generic spinlock */

	/* PCI bus allocation & configuration members */
	struct pci_dev         *pdev;	/* PCI structure information */

	/* to be gotten rid of */
	u32                     pci_state[16];	/* used for suspend/resume */
//      char *firmware;

	void __iomem           *device_base;	/* ioremapped device base address */

	/* consistent DMA region */
	void                   *driver_mem_address;	/* base DMA address */
	dma_addr_t              device_host_address;	/* base DMA address (bus address) */
//      dma_addr_t device_psm_buffer;   /* host memory for PSM buffering (bus address) */

	/* device queue interface members */
	struct isl38xx_cb      *control_block;	/* device control block */
	struct islpci_queue     queues[ISL38XX_CB_QCOUNT];
	/* set by an update in the CB */
	int                     needs_trigger;

	/* rx tasklet */
	struct tasklet_struct   rx_task;
	/* rx queue spinlock */
	spinlock_t              rxqueue_lock;

	/* wait for a reset interrupt */
	wait_queue_head_t       reset_done;

	// Be inspired and do the same for the mgmt management in softmac ?
	/* used by islpci_mgt_transaction */
//      struct semaphore mgmt_sem; /* serialize access to mailbox and wqueue */
//      struct islpci_mgmtframe *mgmt_received;   /* mbox for incoming frame */
//      wait_queue_head_t mgmt_wqueue;            /* waitqueue for mbox */

	/* state machine */
	islpci_state_t          state;
	int                     state_off;	/* enumeration of off-state, if 0 then
						 * we're not in any off-state */

	struct work_struct      reset_task;
	int                     reset_task_pending;

	/* spin lock for enforcing delays for
	   access to the pci registers */
	spinlock_t              pcireg_lock;

	/* our network_device structure  */
	struct net_device      *ndev;
} islpci_private;

islpci_state_t          islpci_get_state(islpci_private *priv);
islpci_state_t          islpci_set_state(islpci_private *priv,
					 islpci_state_t new_state);

#define ISLPCI_TX_TIMEOUT               (2*HZ)

#define ISLPCI_OF_NETDEV(x) (islsm_priv(x))
#define ISLPCI_OF_ISLSM(x)  ((islpci_private *) (x)->priv)
#define NETDEV_OF_ISLPCI(x) ((x)->ndev)

irqreturn_t             islpci_interrupt(int, void *, struct pt_regs *);

//int prism54_post_setup(islpci_private *, int);
int                     islpci_reset(islpci_private *, int);

int                     islpci_free_memory(islpci_private *);
struct net_device       *islpci_alloc(struct pci_dev *);
void                    islpci_free(struct net_device *);
int                     islpci_romboot(islpci_private *priv);

static inline void add_le32p(u32 *le_number, u32 add)
{
	*le_number = cpu_to_le32(le32_to_cpup(le_number) + add);
}

#endif				/* _ISLPCI_DEV_H */
