/*
*******************************************************************************
**        O.S   : Linux
**   FILE NAME  : arcmsr_attr.c
**        BY    : Erich Chen
**   Description: attributes exported to sysfs and device host
*******************************************************************************
** Copyright (C) 2002 - 2005, Areca Technology Corporation All rights reserved
**
**     Web site: www.areca.com.tw
**       E-mail: erich@areca.com.tw
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License version 2 as
** published by the Free Software Foundation.
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
*******************************************************************************
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 3. The name of the author may not be used to endorse or promote products
**    derived from this software without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
** IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
** OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
** IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
** INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES(INCLUDING,BUT
** NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION)HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE)ARISING IN ANY WAY OUT OF THE USE OF
** THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************
** For history of changes, see Documentation/scsi/ChangeLog.arcmsr
**     Firmware Specification, see Documentation/scsi/arcmsr_spec.txt
*******************************************************************************
*/
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/pci.h>

#include <scsi/scsi_cmnd.h>
#include <scsi/scsi_device.h>
#include <scsi/scsi_host.h>
#include <scsi/scsi_transport.h>
#include "arcmsr.h"

static ssize_t
arcmsr_sysfs_iop_message_read(struct kobject *kobj, char *buf, loff_t off,
    size_t count)
{
	struct class_device *cdev = container_of(kobj,struct class_device,kobj);
	struct Scsi_Host *host = class_to_shost(cdev);
	struct AdapterControlBlock *pACB = (struct AdapterControlBlock *) host->hostdata;
	struct MessageUnit __iomem *reg = pACB->pmu;
	uint8_t *pQbuffer,*ptmpQbuffer;
	int32_t allxfer_len = 0;

	if (!capable(CAP_SYS_ADMIN) || (off + count) > 1032 || (count && off) == 0)
		return 0;

	/* do message unit read. */
	ptmpQbuffer = (uint8_t *)buf;
	while ((pACB->rqbuf_firstindex != pACB->rqbuf_lastindex)
		&& (allxfer_len < 1031)) {
		pQbuffer = &pACB->rqbuffer[pACB->rqbuf_firstindex];
		memcpy(ptmpQbuffer, pQbuffer, 1);
		pACB->rqbuf_firstindex++;
		pACB->rqbuf_firstindex %= ARCMSR_MAX_QBUFFER;
		ptmpQbuffer++;
		allxfer_len++;
	}
	if (pACB->acb_flags & ACB_F_IOPDATA_OVERFLOW) {
		struct QBUFFER __iomem * prbuffer = (struct QBUFFER __iomem *)
					&reg->message_rbuffer;
		uint8_t __iomem * iop_data = (uint8_t __iomem *)prbuffer->data;
		int32_t iop_len;

		pACB->acb_flags &= ~ACB_F_IOPDATA_OVERFLOW;
		iop_len = readl(&prbuffer->data_len);
		while (iop_len > 0) {
			pACB->rqbuffer[pACB->rqbuf_lastindex] = readb(iop_data);
			pACB->rqbuf_lastindex++;
			pACB->rqbuf_lastindex %= ARCMSR_MAX_QBUFFER;
			iop_data++;
			iop_len--;
		}
		writel(ARCMSR_INBOUND_DRIVER_DATA_READ_OK,
				&reg->inbound_doorbell);
	}
	return (allxfer_len);
}

static ssize_t
arcmsr_sysfs_iop_message_write(struct kobject *kobj, char *buf, loff_t off,
    size_t count)
{
	struct class_device *cdev = container_of(kobj,struct class_device,kobj);
	struct Scsi_Host *host = class_to_shost(cdev);
	struct AdapterControlBlock *pACB = (struct AdapterControlBlock *) host->hostdata;
	int32_t my_empty_len, user_len, wqbuf_firstindex, wqbuf_lastindex;
	uint8_t *pQbuffer, *ptmpuserbuffer;

	if (!capable(CAP_SYS_ADMIN) || (off + count) > 1032 || (count && off) == 0)
		return 0;

	/* do message unit write. */
	ptmpuserbuffer = (uint8_t *)buf;
	user_len = (int32_t)count;
	wqbuf_lastindex = pACB->wqbuf_lastindex;
	wqbuf_firstindex = pACB->wqbuf_firstindex;
	if (wqbuf_lastindex != wqbuf_firstindex) {
		arcmsr_post_Qbuffer(pACB);
		return 0;
	} else {
		my_empty_len = (wqbuf_firstindex-wqbuf_lastindex - 1)
				&(ARCMSR_MAX_QBUFFER - 1);
		if (my_empty_len >= user_len) {
			while (user_len > 0) {
				pQbuffer =
				&pACB->wqbuffer[pACB->wqbuf_lastindex];
				memcpy(pQbuffer, ptmpuserbuffer, 1);
				pACB->wqbuf_lastindex++;
				pACB->wqbuf_lastindex %= ARCMSR_MAX_QBUFFER;
				ptmpuserbuffer++;
				user_len--;
			}
			if (pACB->acb_flags & ACB_F_MESSAGE_WQBUFFER_CLEARED) {
				pACB->acb_flags &=
					~ACB_F_MESSAGE_WQBUFFER_CLEARED;
				arcmsr_post_Qbuffer(pACB);
			}
			return count;
		} else {
			return 0;
		}
	}
}

static struct bin_attribute arcmsr_sysfs_message_transfer_attr = {
	.attr = {
		.name = "iop_message_transfer",
		.mode = S_IRUSR | S_IWUSR,
		.owner = THIS_MODULE,
	},
	.size = 0,
	.read = arcmsr_sysfs_iop_message_read,
	.write = arcmsr_sysfs_iop_message_write,
};

void
arcmsr_alloc_sysfs_attr(struct AdapterControlBlock *pACB) {
	struct Scsi_Host *host = pACB->host;

	sysfs_create_bin_file(&host->shost_classdev.kobj,
							&arcmsr_sysfs_message_transfer_attr);
}

void
arcmsr_free_sysfs_attr(struct AdapterControlBlock *pACB) {
	struct Scsi_Host *host = pACB->host;

	sysfs_remove_bin_file(&host->shost_gendev.kobj, &arcmsr_sysfs_message_transfer_attr);
}

static ssize_t
arcmsr_attr_host_driver_version(struct class_device *cdev, char *buf) {
	return snprintf(buf, PAGE_SIZE,
			"ARCMSR: %s\n",
			ARCMSR_DRIVER_VERSION);
}

static ssize_t
arcmsr_attr_host_driver_state(struct class_device *cdev, char *buf) {
	struct Scsi_Host *host = class_to_shost(cdev);
	struct AdapterControlBlock *pACB = (struct AdapterControlBlock *) host->hostdata;
	int len = 0;

	len = snprintf(buf, PAGE_SIZE,
				"=================================\n"
				"Current commands posted:     %4d\n"
				"Max commands posted:         %4d\n"
				"Max sgl length:              %4d\n"
				"Max sector count:            %4d\n"
				"SCSI Host Resets:            %4d\n"
				"SCSI Aborts/Timeouts:        %4d\n"
				"=================================\n",
				atomic_read(&pACB->ccboutstandingcount),
				ARCMSR_MAX_OUTSTANDING_CMD,
				ARCMSR_MAX_SG_ENTRIES,
				ARCMSR_MAX_XFER_SECTORS,
				pACB->num_resets,
				pACB->num_aborts);
	return len;
}

static ssize_t
arcmsr_attr_host_fw_model(struct class_device *cdev, char *buf) {
    struct Scsi_Host *host = class_to_shost(cdev);
	struct AdapterControlBlock *pACB = (struct AdapterControlBlock *) host->hostdata;
	return snprintf(buf, PAGE_SIZE,
			"Adapter Model: %s\n",
			pACB->firm_model);
}

static ssize_t
arcmsr_attr_host_fw_version(struct class_device *cdev, char *buf) {
	struct Scsi_Host *host = class_to_shost(cdev);
	struct AdapterControlBlock *pACB = (struct AdapterControlBlock *) host->hostdata;

	return snprintf(buf, PAGE_SIZE,
			"Firmware Version:  %s\n",
			pACB->firm_version);
}

static ssize_t
arcmsr_attr_host_fw_request_len(struct class_device *cdev, char *buf) {
	struct Scsi_Host *host = class_to_shost(cdev);
	struct AdapterControlBlock *pACB = (struct AdapterControlBlock *) host->hostdata;

	return snprintf(buf, PAGE_SIZE,
			"Reguest Lenth: %4d\n",
			pACB->firm_request_len);
}

static ssize_t
arcmsr_attr_host_fw_numbers_queue(struct class_device *cdev, char *buf) {
	struct Scsi_Host *host = class_to_shost(cdev);
	struct AdapterControlBlock *pACB = (struct AdapterControlBlock *) host->hostdata;

	return snprintf(buf, PAGE_SIZE,
			"Numbers of Queue: %4d\n",
			pACB->firm_numbers_queue);
}

static ssize_t
arcmsr_attr_host_fw_sdram_size(struct class_device *cdev, char *buf) {
	struct Scsi_Host *host = class_to_shost(cdev);
	struct AdapterControlBlock *pACB = (struct AdapterControlBlock *) host->hostdata;

	return snprintf(buf, PAGE_SIZE,
			"SDRAM Size: %4d\n",
			pACB->firm_sdram_size);
}

static ssize_t
arcmsr_attr_host_fw_hd_channels(struct class_device *cdev, char *buf) {
	struct Scsi_Host *host = class_to_shost(cdev);
	struct AdapterControlBlock *pACB = (struct AdapterControlBlock *) host->hostdata;

	return snprintf(buf, PAGE_SIZE,
			"Hard Disk Channels: %4d\n",
			pACB->firm_hd_channels);
}

static ssize_t
arcmsr_attr_host_pci_info(struct class_device *cdev, char *buf) {
	struct Scsi_Host *host = class_to_shost(cdev);
	struct AdapterControlBlock *pACB = (struct AdapterControlBlock *) host->hostdata;
	int pci_type;

	pci_type = pci_find_capability(pACB->pdev, PCI_CAP_ID_EXP);
	return snprintf(buf, PAGE_SIZE,
			"=================================\n"
			"pci type: %s"
			"pci name: %s\n"
			"bus number: %d\n"
			"function number: %4d\n"
			"slot number: %d\n"
			"irq: %d\n"
			"=================================\n",
			pci_type ? "PCI EXPRESS \n" : "PCI \n",
			pci_name(pACB->pdev),
			pACB->pdev->bus->number,
			PCI_FUNC(pACB->pdev->devfn),
			PCI_SLOT(pACB->pdev->devfn),
			pACB->pdev->irq
		);
}

static CLASS_DEVICE_ATTR(host_driver_version, S_IRUGO, arcmsr_attr_host_driver_version, NULL);
static CLASS_DEVICE_ATTR(host_driver_state, S_IRUGO, arcmsr_attr_host_driver_state, NULL);
static CLASS_DEVICE_ATTR(host_fw_model, S_IRUGO, arcmsr_attr_host_fw_model, NULL);
static CLASS_DEVICE_ATTR(host_fw_version, S_IRUGO, arcmsr_attr_host_fw_version, NULL);
static CLASS_DEVICE_ATTR(host_fw_request_len, S_IRUGO, arcmsr_attr_host_fw_request_len, NULL);
static CLASS_DEVICE_ATTR(host_fw_numbers_queue, S_IRUGO, arcmsr_attr_host_fw_numbers_queue, NULL);
static CLASS_DEVICE_ATTR(host_fw_sdram_size, S_IRUGO, arcmsr_attr_host_fw_sdram_size, NULL);
static CLASS_DEVICE_ATTR(host_fw_hd_channels, S_IRUGO, arcmsr_attr_host_fw_hd_channels, NULL);
static CLASS_DEVICE_ATTR(host_pci_info, S_IRUGO, arcmsr_attr_host_pci_info, NULL);

struct class_device_attribute *arcmsr_host_attrs[] = {
	&class_device_attr_host_driver_version,
	&class_device_attr_host_driver_state,
	&class_device_attr_host_fw_model,
	&class_device_attr_host_fw_version,
	&class_device_attr_host_fw_request_len,
	&class_device_attr_host_fw_numbers_queue,
	&class_device_attr_host_fw_sdram_size,
	&class_device_attr_host_fw_hd_channels,
	&class_device_attr_host_pci_info,
	NULL,
};
