/*
 * linux/arch/i386/kernel/sysenter.c
 *
 * (C) Copyright 2002 Linus Torvalds
 * Portions based on the vdso-randomization code from exec-shield:
 * Copyright(C) 2005-2006, Red Hat, Inc., Ingo Molnar
 *
 * This file contains the needed initializations to support sysenter.
 */

#include <linux/init.h>
#include <linux/smp.h>
#include <linux/thread_info.h>
#include <linux/sched.h>
#include <linux/gfp.h>
#include <linux/string.h>
#include <linux/elf.h>
#include <linux/mm.h>
#include <linux/module.h>

#include <asm/cpufeature.h>
#include <asm/msr.h>
#include <asm/pgtable.h>
#include <asm/unistd.h>

/*
 * Should the kernel map a VDSO page into processes and pass its
 * address down to glibc upon exec()?
 */
unsigned int __read_mostly vdso_enabled = 1;

EXPORT_SYMBOL_GPL(vdso_enabled);

static int __init vdso_setup(char *s)
{
	vdso_enabled = simple_strtoul(s, NULL, 0);

	return 1;
}

__setup("vdso=", vdso_setup);

extern asmlinkage void sysenter_entry(void);

void enable_sep_cpu(void)
{
	int cpu = get_cpu();
	struct tss_struct *tss = &per_cpu(init_tss, cpu);

	if (!boot_cpu_has(X86_FEATURE_SEP)) {
		put_cpu();
		return;
	}

	tss->ss1 = __KERNEL_CS;
	tss->esp1 = sizeof(struct tss_struct) + (unsigned long) tss;
	wrmsr(MSR_IA32_SYSENTER_CS, __KERNEL_CS, 0);
	wrmsr(MSR_IA32_SYSENTER_ESP, tss->esp1, 0);
	wrmsr(MSR_IA32_SYSENTER_EIP, (unsigned long) sysenter_entry, 0);
	put_cpu();	
}

/*
 * These symbols are defined by vsyscall.o to mark the bounds
 * of the ELF DSO images included therein.
 */
extern const char vsyscall_int80_start, vsyscall_int80_end;
extern const char vsyscall_sysenter_start, vsyscall_sysenter_end;
static void *syscall_page;

#ifdef CONFIG_COMPAT_VDSO
static void fixup_vsyscall_elf(char *page)
{
	Elf32_Ehdr *hdr;
	Elf32_Shdr *sechdrs;
	Elf32_Phdr *phdr;
	char *secstrings;
	int i, j, n;

	hdr = (Elf32_Ehdr *)page;
	
	printk("Remapping vsyscall page to %08x\n", (unsigned int)VDSO_HIGH_BASE);

	/* Sanity checks against insmoding binaries or wrong arch,
           weird elf version */
	if (memcmp(hdr->e_ident, ELFMAG, 4) != 0 ||
		!elf_check_arch(hdr) ||
		hdr->e_type != ET_DYN)
		panic("Bogus ELF in vsyscall DSO\n");

	hdr->e_entry += VDSO_HIGH_BASE;
	sechdrs = (void *)hdr + hdr->e_shoff;
	secstrings = (void *)hdr + sechdrs[hdr->e_shstrndx].sh_offset;

	for (i = 1; i < hdr->e_shnum; i++) {
		if (!(sechdrs[i].sh_flags & SHF_ALLOC))
			continue;

		sechdrs[i].sh_addr += VDSO_HIGH_BASE;
		if (strcmp(secstrings+sechdrs[i].sh_name, ".dynsym") == 0) {
			Elf32_Sym  *sym =  (void *)hdr + sechdrs[i].sh_offset;
			n = sechdrs[i].sh_size / sizeof(*sym);
			for (j = 1; j < n;  j++) {
				int ndx = sym[j].st_shndx;
				if (ndx == SHN_UNDEF || ndx == SHN_ABS)
					continue;
				sym[j].st_value += VDSO_HIGH_BASE;
			}
		} else if (strcmp(secstrings+sechdrs[i].sh_name, ".dynamic") == 0) {
			Elf32_Dyn *dyn = (void *)hdr + sechdrs[i].sh_offset;
			int tag;
			while ((tag = (++dyn)->d_tag) != DT_NULL) {
				if (tag == DT_PLTGOT || tag == DT_HASH ||
				    tag == DT_STRTAB || tag == DT_SYMTAB ||
				    tag == DT_RELA || tag == DT_INIT ||
				    tag == DT_FINI || tag == DT_REL ||
				    tag == DT_JMPREL || tag == DT_VERSYM ||
				    tag == DT_VERDEF || tag == DT_VERNEED)
					dyn->d_un.d_val += VDSO_HIGH_BASE;
			}
		} else if (strcmp(secstrings+sechdrs[i].sh_name, ".useless") == 0) {
			/* This is demonic; see vsyscall.lds.S; it puts the
			 * .got in a section named .useless */
			uint32_t *got = (void *)hdr + sechdrs[i].sh_offset;
			*got += VDSO_HIGH_BASE;
		}
	}
	phdr = (void *)hdr + hdr->e_phoff;
	for (i = 0; i < hdr->e_phnum; i++) {
		phdr[i].p_vaddr += VDSO_HIGH_BASE;
		phdr[i].p_paddr += VDSO_HIGH_BASE;
	}

#if 0
/* 
 * To verify the binary image in memory is identical, linked in the VDSO page
 * from a COMPAT_VDSO compile without this patch; then diff the two.  For a
 * non-relocated fixmap, the VDSO image is identical.
 */
{
	extern const char vsyscall_orig_start, vsyscall_orig_end;
	int *l1 = (int *)page, *l2 = (int *)&vsyscall_orig_start;
	int foo = vsyscall_orig_end - vsyscall_orig_start / 4;
	for (i = 0; i < foo; i++) {
		if (l1[i] != l2[i]) {
			printk("vsyscall - delta [%03x] orig %08x new %08x\n",
				i, l2[i], l1[i]);
		}
	}
}
#endif
}
#endif

int __init sysenter_setup(void)
{
	syscall_page = (void *)get_zeroed_page(GFP_ATOMIC);

#ifdef CONFIG_COMPAT_VDSO
	__set_fixmap(FIX_VDSO, __pa(syscall_page), PAGE_READONLY_EXEC);
	printk("Compat vDSO mapped to %08lx.\n", __fix_to_virt(FIX_VDSO));
#endif

	if (!boot_cpu_has(X86_FEATURE_SEP)) {
		memcpy(syscall_page,
		       &vsyscall_int80_start,
		       &vsyscall_int80_end - &vsyscall_int80_start);
#ifdef CONFIG_COMPAT_VDSO
		fixup_vsyscall_elf((char *)syscall_page);
#endif
		return 0;
	}

	memcpy(syscall_page,
	       &vsyscall_sysenter_start,
	       &vsyscall_sysenter_end - &vsyscall_sysenter_start);
#ifdef CONFIG_COMPAT_VDSO
	fixup_vsyscall_elf((char *)syscall_page);
#endif

	return 0;
}

#ifndef CONFIG_COMPAT_VDSO
static struct page *syscall_nopage(struct vm_area_struct *vma,
				unsigned long adr, int *type)
{
	struct page *p = virt_to_page(adr - vma->vm_start + syscall_page);
	get_page(p);
	return p;
}

/* Prevent VMA merging */
static void syscall_vma_close(struct vm_area_struct *vma)
{
}

static struct vm_operations_struct syscall_vm_ops = {
	.close = syscall_vma_close,
	.nopage = syscall_nopage,
};

const char *arch_vma_name(struct vm_area_struct *vma)
{
	if (vma->vm_mm && vma->vm_start == (long)vma->vm_mm->context.vdso)
		return "[vdso]";
	return NULL;
}

struct vm_area_struct *get_gate_vma(struct task_struct *tsk)
{
	return NULL;
}

int in_gate_area(struct task_struct *task, unsigned long addr)
{
	return 0;
}

int in_gate_area_no_task(unsigned long addr)
{
	return 0;
}
#endif

/* Defined in vsyscall-sysenter.S */
extern void SYSENTER_RETURN;

/* Setup a VMA at program startup for the vsyscall page */
int arch_setup_additional_pages(struct linux_binprm *bprm, int exstack)
{
#ifndef CONFIG_COMPAT_VDSO
	struct vm_area_struct *vma;
	struct mm_struct *mm = current->mm;
	unsigned long addr;
	int ret;

	down_write(&mm->mmap_sem);
	addr = get_unmapped_area(NULL, 0, PAGE_SIZE, 0, 0);
	if (IS_ERR_VALUE(addr)) {
		ret = addr;
		goto up_fail;
	}

	vma = kmem_cache_zalloc(vm_area_cachep, GFP_KERNEL);
	if (!vma) {
		ret = -ENOMEM;
		goto up_fail;
	}

	vma->vm_start = addr;
	vma->vm_end = addr + PAGE_SIZE;
	/* MAYWRITE to allow gdb to COW and set breakpoints */
	vma->vm_flags = VM_READ|VM_EXEC|VM_MAYREAD|VM_MAYEXEC|VM_MAYWRITE;
	/*
	 * Make sure the vDSO gets into every core dump.
	 * Dumping its contents makes post-mortem fully interpretable later
	 * without matching up the same kernel and hardware config to see
	 * what PC values meant.
	 */
	vma->vm_flags |= VM_ALWAYSDUMP;
	vma->vm_flags |= mm->def_flags;
	vma->vm_page_prot = protection_map[vma->vm_flags & 7];
	vma->vm_ops = &syscall_vm_ops;
	vma->vm_mm = mm;

	ret = insert_vm_struct(mm, vma);
	if (unlikely(ret)) {
		kmem_cache_free(vm_area_cachep, vma);
		goto up_fail;
	}

	current->mm->context.vdso = (void *)addr;
	current_thread_info()->sysenter_return =
				    (void *)VDSO_SYM(&SYSENTER_RETURN);
	mm->total_vm++;
up_fail:
	up_write(&mm->mmap_sem);
	return ret;
#else
	/*
	 * If COMPAT_VDSO is not enabled, simply set vdso to point to
	 * fixmap page.
	 */
	current->mm->context.vdso = (void *)VDSO_HIGH_BASE;
	current_thread_info()->sysenter_return =
				    (void *)VDSO_SYM(&SYSENTER_RETURN);

	return 0;
#endif
}
