/* zd_mac.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/usb.h>
#include <linux/jiffies.h>
#include <net/ieee80211_radiotap.h>

#include "zd_def.h"
#include "zd_chip.h"
#include "zd_mac.h"
#include "zd_ieee80211.h"
#include "zd_rf.h"
#include "zd_util.h"

static void housekeeping_init(struct zd_mac *mac);
static void housekeeping_enable(struct zd_mac *mac);
static void housekeeping_disable(struct zd_mac *mac);

int zd_mac_init_hw(struct ieee80211_hw *dev, u8 device_type)
{
	int r;
	struct zd_mac *mac = zd_dev_mac(dev);
	struct zd_chip *chip = &mac->chip;
	u8 addr[ETH_ALEN];
	u8 default_regdomain;

	r = zd_chip_enable_int(chip);
	if (r)
		goto out;
	r = zd_chip_init_hw(chip, device_type);
	if (r)
		goto disable_int;

	zd_get_e2p_mac_addr(chip, addr);
	r = zd_write_mac_addr(chip, addr);
	if (r)
		goto disable_int;
	ZD_ASSERT(!irqs_disabled());
	spin_lock_irq(&mac->lock);
	SET_IEEE80211_PERM_ADDR(dev, addr);
	spin_unlock_irq(&mac->lock);

	r = zd_read_regdomain(chip, &default_regdomain);
	if (r)
		goto disable_int;
	spin_lock_irq(&mac->lock);
	mac->regdomain = mac->default_regdomain = default_regdomain;
	spin_unlock_irq(&mac->lock);

	/* We must inform the device that we are doing encryption/decryption in
	 * software at the moment. */
	r = zd_set_encryption_type(chip, ENC_SNIFFER);
	if (r)
		goto disable_int;

	/* TODO: waiting for regulatory domain support in mac80211 */
	/*r = zd_geo_init(zd_mac_to_ieee80211(mac), mac->regdomain);
	if (r)
		goto disable_int;*/

	r = 0;
disable_int:
	zd_chip_disable_int(chip);
out:
	return r;
}

void zd_mac_clear(struct zd_mac *mac)
{
	flush_workqueue(zd_workqueue);
	zd_chip_clear(&mac->chip);
	ZD_ASSERT(!spin_is_locked(&mac->lock));
	ZD_MEMCLEAR(mac, sizeof(struct zd_mac));
}

static int reset_mode(struct zd_mac *mac)
{
	struct zd_ioreq32 ioreqs[] = {
		{ CR_RX_FILTER, STA_RX_FILTER },
		{ CR_SNIFFER_ON, 0U },
	};

	if (mac->mode == IEEE80211_IF_TYPE_MNTR) {
		ioreqs[0].value = 0xffffffff;
		ioreqs[1].value = 0x1;
	}

	return zd_iowrite32a(&mac->chip, ioreqs, ARRAY_SIZE(ioreqs));
}

static int zd_mac_open(struct ieee80211_hw *dev)
{
	struct zd_mac *mac = zd_dev_mac(dev);
	struct zd_chip *chip = &mac->chip;
	int r;

	r = zd_chip_enable_int(chip);
	if (r < 0)
		goto out;

	r = zd_chip_set_basic_rates(chip, CR_RATES_80211B | CR_RATES_80211G);
	if (r < 0)
		goto disable_int;
	r = reset_mode(mac);
	if (r)
		goto disable_int;
	r = zd_chip_switch_radio_on(chip);
	if (r < 0)
		goto disable_int;
	r = zd_write_mac_addr(chip, mac->hwaddr);
	if (r)
		goto disable_radio;
	r = zd_chip_enable_rx(chip);
	if (r < 0)
		goto disable_radio;
	r = zd_chip_enable_hwint(chip);
	if (r < 0)
		goto disable_rx;

	housekeeping_enable(mac);
	return 0;
disable_rx:
	zd_chip_disable_rx(chip);
disable_radio:
	zd_chip_switch_radio_off(chip);
disable_int:
	zd_chip_disable_int(chip);
out:
	return r;
}

static int zd_mac_stop(struct ieee80211_hw *dev)
{
	struct zd_mac *mac = zd_dev_mac(dev);
	struct zd_chip *chip = &mac->chip;
	struct sk_buff *skb;

	/*
	 * The order here deliberately is a little different from the open()
	 * method, since we need to make sure there is no opportunity for RX
	 * frames to be processed by softmac after we have stopped it.
	 */

	zd_chip_disable_rx(chip);
	housekeeping_disable(mac);

	zd_chip_disable_hwint(chip);
	zd_chip_switch_radio_off(chip);
	zd_chip_disable_int(chip);

	while ((skb = skb_dequeue(&mac->tx_queue))) {
		struct ieee80211_tx_control *control =
			*(struct ieee80211_tx_control **)skb->cb;
		kfree(control);
		kfree_skb(skb);
	}

	return 0;
}

static int zd_calc_tx_length_us(u8 *service, u8 cs_rate, u16 tx_length)
{
	static const u8 rate_divisor[] = {
		[ZD_CS_CCK_RATE_1M]	=  1,
		[ZD_CS_CCK_RATE_2M]	=  2,
		[ZD_CS_CCK_RATE_5_5M]	= 11, /* bits must be doubled */
		[ZD_CS_CCK_RATE_11M]	= 11,
		[ZD_OFDM_RATE_6M]	=  6,
		[ZD_OFDM_RATE_9M]	=  9,
		[ZD_OFDM_RATE_12M]	= 12,
		[ZD_OFDM_RATE_18M]	= 18,
		[ZD_OFDM_RATE_24M]	= 24,
		[ZD_OFDM_RATE_36M]	= 36,
		[ZD_OFDM_RATE_48M]	= 48,
		[ZD_OFDM_RATE_54M]	= 54,
	};

	u32 bits = (u32)tx_length * 8;
	u32 divisor;

	divisor = rate_divisor[cs_rate];
	if (divisor == 0)
		return -EINVAL;

	switch (cs_rate) {
	case ZD_CS_CCK_RATE_5_5M:
		bits = (2*bits) + 10; /* round up to the next integer */
		break;
	case ZD_CS_CCK_RATE_11M:
		if (service) {
			u32 t = bits % 11;
			*service &= ~ZD_PLCP_SERVICE_LENGTH_EXTENSION;
			if (0 < t && t <= 3) {
				*service |= ZD_PLCP_SERVICE_LENGTH_EXTENSION;
			}
		}
		bits += 10; /* round up to the next integer */
		break;
	}

	return bits/divisor;
}

static void cs_set_control(struct zd_mac *mac, struct zd_ctrlset *cs,
	                   struct ieee80211_hdr *header, u32 flags)
{
	u16 fctl = le16_to_cpu(header->frame_control);

	/*
	 * CONTROL:
	 * - start at 0x00
	 * - if fragment 0, enable bit 0
	 * - if backoff needed, enable bit 0
	 * - if burst (backoff not needed) disable bit 0
	 * - if multicast, enable bit 1
	 * - if PS-POLL frame, enable bit 2
	 * - if in INDEPENDENT_BSS mode and zd1205_DestPowerSave, then enable
	 *   bit 4 (FIXME: wtf)
	 * - if frag_len > RTS threshold, set bit 5 as long if it isnt
	 *   multicast or mgt
	 * - if bit 5 is set, and we are in OFDM mode, unset bit 5 and set bit
	 *   7
	 */

	cs->control = 0;

	/* First fragment */
	if (flags & IEEE80211_TXCTL_FIRST_FRAGMENT)
		cs->control |= ZD_CS_NEED_RANDOM_BACKOFF;

	/* Multicast */
	if (is_multicast_ether_addr(header->addr1))
		cs->control |= ZD_CS_MULTICAST;

	/* PS-POLL */
	if ((fctl & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_PSPOLL)
		cs->control |= ZD_CS_PS_POLL_FRAME;

	if (flags & IEEE80211_TXCTL_USE_RTS_CTS)
		cs->control |= ZD_CS_RTS;

	if (flags & IEEE80211_TXCTL_USE_CTS_PROTECT)
		cs->control |= ZD_CS_SELF_CTS;

	/* FIXME: Management frame? */
}

static int fill_ctrlset(struct zd_mac *mac,
			struct sk_buff *skb,
			struct ieee80211_tx_control *control)
{
	int r;
	struct ieee80211_hdr *hdr = (struct ieee80211_hdr *) skb->data;
	unsigned int frag_len = skb->len + FCS_LEN;
	unsigned int packet_length;
	struct zd_ctrlset *cs = (struct zd_ctrlset *)
		skb_push(skb, sizeof(struct zd_ctrlset));

	ZD_ASSERT(frag_len <= 0xffff);

	cs->modulation = control->tx_rate;

	cs->tx_length = cpu_to_le16(frag_len);

	cs_set_control(mac, cs, hdr, control->flags);

	packet_length = frag_len + sizeof(struct zd_ctrlset) + 10;
	ZD_ASSERT(packet_length <= 0xffff);
	/* ZD1211B: Computing the length difference this way, gives us
	 * flexibility to compute the packet length.
	 */
	cs->packet_length = cpu_to_le16(mac->chip.is_zd1211b ?
			packet_length - frag_len : packet_length);

	/*
	 * CURRENT LENGTH:
	 * - transmit frame length in microseconds
	 * - seems to be derived from frame length
	 * - see Cal_Us_Service() in zdinlinef.h
	 * - if macp->bTxBurstEnable is enabled, then multiply by 4
	 *  - bTxBurstEnable is never set in the vendor driver
	 *
	 * SERVICE:
	 * - "for PLCP configuration"
	 * - always 0 except in some situations at 802.11b 11M
	 * - see line 53 of zdinlinef.h
	 */
	cs->service = 0;
	r = zd_calc_tx_length_us(&cs->service, ZD_CS_RATE(cs->modulation),
		                 le16_to_cpu(cs->tx_length));
	if (r < 0)
		return r;
	cs->current_length = cpu_to_le16(r);
	cs->next_frame_length = 0;

	return 0;
}

static int zd_mac_tx(struct ieee80211_hw *dev, struct sk_buff *skb,
		     struct ieee80211_tx_control *control)
{
	struct zd_mac *mac = zd_dev_mac(dev);
	struct ieee80211_tx_control *control_copy;
	int r;

	r = fill_ctrlset(mac, skb, control);
	if (r)
		return r;
	r = zd_usb_tx(&mac->chip.usb, skb->data, skb->len);
	if (r)
		return r;

	if (control->flags & IEEE80211_TXCTL_NO_ACK) {
		kfree_skb(skb);
		return 0;
	}

	control_copy = kmalloc(sizeof(*control_copy), GFP_ATOMIC);
	if (control_copy)
		memcpy(control_copy, control, sizeof(*control_copy));

	*(struct ieee80211_tx_control **)skb->cb = control_copy;
	skb_pull(skb, sizeof(struct zd_ctrlset));
	skb_queue_tail(&mac->tx_queue, skb);
	return 0;
}

void zd_mac_tx_failed(struct ieee80211_hw *dev)
{
	struct zd_mac *mac = zd_dev_mac(dev);
	struct ieee80211_tx_control *control;
	struct sk_buff *skb;

	skb = skb_dequeue(&mac->tx_queue);
	if (!skb)
		return;

	control = *(struct ieee80211_tx_control **)skb->cb;
	if (control) {
		struct ieee80211_tx_status status = {{0}};
		memcpy(&status.control, control, sizeof(status.control));
		ieee80211_tx_status_irqsafe(dev, skb, &status);
		kfree(control);
	} else
		kfree_skb(skb);

	return;
}

struct zd_rt_hdr {
	struct ieee80211_radiotap_header rt_hdr;
	u8  rt_flags;
	u8  rt_rate;
	u16 rt_channel;
	u16 rt_chbitmask;
} __attribute__((packed));

static void fill_rt_header(void *buffer, struct zd_mac *mac,
	                   const struct ieee80211_rx_status *stats,
			   const struct rx_status *status)
{
	struct zd_rt_hdr *hdr = buffer;

	hdr->rt_hdr.it_version = PKTHDR_RADIOTAP_VERSION;
	hdr->rt_hdr.it_pad = 0;
	hdr->rt_hdr.it_len = cpu_to_le16(sizeof(struct zd_rt_hdr));
	hdr->rt_hdr.it_present = cpu_to_le32((1 << IEEE80211_RADIOTAP_FLAGS) |
		                 (1 << IEEE80211_RADIOTAP_CHANNEL) |
				 (1 << IEEE80211_RADIOTAP_RATE));

	hdr->rt_flags = 0;
	if (status->decryption_type & (ZD_RX_WEP64|ZD_RX_WEP128|ZD_RX_WEP256))
		hdr->rt_flags |= IEEE80211_RADIOTAP_F_WEP;

	hdr->rt_rate = stats->rate / 5;

	/* FIXME: 802.11a */
	hdr->rt_channel = cpu_to_le16(ieee80211chan2mhz(
		                             _zd_chip_get_channel(&mac->chip)));
	hdr->rt_chbitmask = cpu_to_le16(IEEE80211_CHAN_2GHZ |
		((status->frame_status & ZD_RX_FRAME_MODULATION_MASK) ==
		ZD_RX_OFDM ? IEEE80211_CHAN_OFDM : IEEE80211_CHAN_CCK));
}

static int fill_rx_stats(struct ieee80211_rx_status *stats,
	                 const struct rx_status **pstatus,
		         struct zd_mac *mac,
			 const u8 *buffer, unsigned int length)
{
	const struct rx_status *status;

	*pstatus = status = zd_tail(buffer, length, sizeof(struct rx_status));
	if (status->frame_status & ZD_RX_ERROR) {
		/* FIXME: update? */
		return -EINVAL;
	}
	memset(stats, 0, sizeof(*stats));

	stats->channel = _zd_chip_get_channel(&mac->chip);
	stats->freq = zd_channels[stats->channel - 1].freq;
	stats->phymode = MODE_IEEE80211G;
	stats->ssi = zd_rx_strength_percent(status->signal_strength);
	stats->signal = zd_rx_qual_percent(buffer,
		                          length - sizeof(struct rx_status),
		                          status);
	stats->rate = zd_rx_rate(buffer, status);

	return 0;
}

static int filter_ack(struct ieee80211_hw *dev, struct ieee80211_hdr *rx_hdr,
		      struct ieee80211_rx_status *stats)
{
	struct zd_mac *mac = zd_dev_mac(dev);
	u16 fc = le16_to_cpu(rx_hdr->frame_control);
	struct sk_buff *skb;
	struct ieee80211_hdr *tx_hdr;
	struct ieee80211_tx_control *control;
	struct ieee80211_tx_status status = {{0}};

	if ((fc & (IEEE80211_FCTL_FTYPE | IEEE80211_FCTL_STYPE)) !=
	    (IEEE80211_FTYPE_CTL | IEEE80211_STYPE_ACK))
		return 0;

	spin_lock(&mac->tx_queue.lock);

	skb = skb_peek(&mac->tx_queue);
	if (!skb) {
		spin_unlock(&mac->tx_queue.lock);
		return 1;
	}

	tx_hdr = (struct ieee80211_hdr *) skb->data;

	if (!memcmp(tx_hdr->addr2, rx_hdr->addr1, ETH_ALEN))
		skb = __skb_dequeue(&mac->tx_queue);
	else {
		spin_unlock(&mac->tx_queue.lock);
		return 1;
	}

	spin_unlock(&mac->tx_queue.lock);

	control = *(struct ieee80211_tx_control **)skb->cb;
	if (control) {
		memcpy(&status.control, control, sizeof(status.control));
		status.flags = IEEE80211_TX_STATUS_ACK;
		status.ack_signal = stats->ssi;
		ieee80211_tx_status_irqsafe(dev, skb, &status);
		kfree(control);
	} else
		kfree_skb(skb);

	return 1;
}

int zd_mac_rx(struct ieee80211_hw *dev, const u8 *buffer, unsigned int length)
{
	int r;
	struct zd_mac *mac = zd_dev_mac(dev);
	struct ieee80211_rx_status stats;
	const struct rx_status *status;
	struct sk_buff *skb;

	if (length < ZD_PLCP_HEADER_SIZE + 10 /* IEEE80211_1ADDR_LEN */ +
	             FCS_LEN + sizeof(struct rx_status))
		return -EINVAL;

	r = fill_rx_stats(&stats, &status, mac, buffer, length);
	if (r)
		return r;

	length -= ZD_PLCP_HEADER_SIZE+
		  sizeof(struct rx_status);
	buffer += ZD_PLCP_HEADER_SIZE;

	if (length == (10 /* IEEE80211_1ADDR_LEN */ + FCS_LEN) &&
	    filter_ack(dev, (struct ieee80211_hdr *)buffer, &stats))
		return 0;

	skb = dev_alloc_skb(sizeof(struct zd_rt_hdr) + length);
	if (!skb)
		return -ENOMEM;
	if (mac->mode == IEEE80211_IF_TYPE_MNTR)
		fill_rt_header(skb_put(skb, sizeof(struct zd_rt_hdr)), mac,
			       &stats, status);
	memcpy(skb_put(skb, length), buffer, length);

	ieee80211_rx_irqsafe(dev, skb, &stats);
	return 0;
}

static int zd_mac_add_interface(struct ieee80211_hw *dev,
				struct ieee80211_if_init_conf *conf)
{
	struct zd_mac *mac = zd_dev_mac(dev);

	/* NOTE: using IEEE80211_IF_TYPE_MGMT to indicate no mode selected */
	if (mac->mode != IEEE80211_IF_TYPE_MGMT)
		return -1;

	switch (conf->type) {
	case IEEE80211_IF_TYPE_STA:
		mac->mode = conf->type;
		break;
	default:
		return -1;
	}

	mac->hwaddr = conf->mac_addr;

	return 0;
}

static void zd_mac_remove_interface(struct ieee80211_hw *dev,
				    struct ieee80211_if_init_conf *conf)
{
	struct zd_mac *mac = zd_dev_mac(dev);
	mac->mode = IEEE80211_IF_TYPE_MGMT;
}

static int zd_mac_config(struct ieee80211_hw *dev, struct ieee80211_conf *conf)
{
	struct zd_mac *mac = zd_dev_mac(dev);
	return zd_chip_set_channel(&mac->chip, conf->channel);
}

static int zd_mac_config_interface(struct ieee80211_hw *dev, int if_id,
				   struct ieee80211_if_conf *conf)
{
	struct zd_mac *mac = zd_dev_mac(dev);

	mac->associated = is_valid_ether_addr(conf->bssid);

	/* TODO: do hardware bssid filtering */
	return 0;
}

static void set_multicast_hash_handler(struct work_struct *work)
{
	struct zd_mac *mac =
		container_of(work, struct zd_mac, set_multicast_hash_work);
	struct zd_mc_hash hash;

	spin_lock_irq(&mac->lock);
	hash = mac->multicast_hash;
	spin_unlock_irq(&mac->lock);

	zd_chip_set_multicast_hash(&mac->chip, &hash);
}

static void zd_mac_set_multicast_list(struct ieee80211_hw *dev,
				      unsigned short dev_flags, int mc_count)
{
	struct zd_mc_hash hash;
	struct zd_mac *mac = zd_dev_mac(dev);
	unsigned long flags;

	if (dev_flags & (IFF_PROMISC|IFF_ALLMULTI)) {
		zd_mc_add_all(&hash);
	} else {
		struct dev_mc_list *mc = NULL;
		void *tmp = NULL;
		zd_mc_clear(&hash);
		while ((mc = ieee80211_get_mc_list_item(dev, mc, &tmp))) {
			dev_dbg_f(zd_mac_dev(mac), "mc addr " MAC_FMT "\n",
				  MAC_ARG(mc->dmi_addr));
			zd_mc_add_addr(&hash, mc->dmi_addr);
		}
	}

	spin_lock_irqsave(&mac->lock, flags);
	mac->multicast_hash = hash;
	spin_unlock_irqrestore(&mac->lock, flags);
	queue_work(zd_workqueue, &mac->set_multicast_hash_work);
}

static struct ieee80211_ops zd_ops = {
	.tx			= zd_mac_tx,
	.open			= zd_mac_open,
	.stop			= zd_mac_stop,
	.add_interface		= zd_mac_add_interface,
	.remove_interface	= zd_mac_remove_interface,
	.config			= zd_mac_config,
	.config_interface	= zd_mac_config_interface,
	.set_multicast_list	= zd_mac_set_multicast_list
};

struct ieee80211_hw *zd_mac_alloc(struct usb_interface *intf)
{
	struct zd_mac *mac;
	struct ieee80211_hw *dev;
	int i;

	dev = ieee80211_alloc_hw(sizeof(struct zd_mac), &zd_ops);
	if (!dev) {
		dev_dbg_f(&intf->dev, "out of memory\n");
		return NULL;
	}

	mac = zd_dev_mac(dev);

	memset(mac, 0, sizeof(*mac));
	spin_lock_init(&mac->lock);
	mac->dev = dev;

	mac->mode = IEEE80211_IF_TYPE_MGMT;
	mac->hwaddr = dev->wiphy->perm_addr;

	memcpy(mac->channels, zd_channels, sizeof(zd_channels));
	memcpy(mac->rates, zd_rates, sizeof(zd_rates));
	mac->modes[0].mode = MODE_IEEE80211G;
	mac->modes[0].num_rates = ARRAY_SIZE(zd_rates);
	mac->modes[0].rates = mac->rates;
	mac->modes[0].num_channels = ARRAY_SIZE(zd_channels);
	mac->modes[0].channels = mac->channels;
	mac->modes[1].mode = MODE_IEEE80211B;
	mac->modes[1].num_rates = 4;
	mac->modes[1].rates = mac->rates;
	mac->modes[1].num_channels = ARRAY_SIZE(zd_channels);
	mac->modes[1].channels = mac->channels;

	dev->flags = IEEE80211_HW_RX_INCLUDES_FCS |
		     IEEE80211_HW_WEP_INCLUDE_IV;
	dev->max_rssi = 100;
	dev->max_signal = 100;

	dev->queues = 1;
	dev->extra_tx_headroom = sizeof(struct zd_ctrlset);

	for (i = 0; i < 2; i++) {
		if (ieee80211_register_hwmode(dev, &mac->modes[i])) {
			dev_dbg_f(&intf->dev, "cannot register hwmode\n");
			ieee80211_free_hw(dev);
			return NULL;
		}
	}

	skb_queue_head_init(&mac->tx_queue);
	zd_chip_init(&mac->chip, dev, intf);
	housekeeping_init(mac);
	INIT_WORK(&mac->set_multicast_hash_work, set_multicast_hash_handler);

	SET_IEEE80211_DEV(dev, &intf->dev);
	return dev;
}

#define LINK_LED_WORK_DELAY HZ

static void link_led_handler(struct work_struct *work)
{
	struct zd_mac *mac =
		container_of(work, struct zd_mac, housekeeping.link_led_work.work);
	struct zd_chip *chip = &mac->chip;
	int is_associated;
	int r;

	spin_lock_irq(&mac->lock);
	is_associated = mac->associated;
	spin_unlock_irq(&mac->lock);

	r = zd_chip_control_leds(chip,
		                 is_associated ? LED_ASSOCIATED : LED_SCANNING);
	if (r)
		dev_err(zd_mac_dev(mac), "zd_chip_control_leds error %d\n", r);

	queue_delayed_work(zd_workqueue, &mac->housekeeping.link_led_work,
		           LINK_LED_WORK_DELAY);
}

static void housekeeping_init(struct zd_mac *mac)
{
	INIT_DELAYED_WORK(&mac->housekeeping.link_led_work, link_led_handler);
}

static void housekeeping_enable(struct zd_mac *mac)
{
	dev_dbg_f(zd_mac_dev(mac), "\n");
	queue_delayed_work(zd_workqueue, &mac->housekeeping.link_led_work,
			   0);
}

static void housekeeping_disable(struct zd_mac *mac)
{
	dev_dbg_f(zd_mac_dev(mac), "\n");
	cancel_rearming_delayed_workqueue(zd_workqueue,
		&mac->housekeeping.link_led_work);
	zd_chip_control_leds(&mac->chip, LED_OFF);
}
