/* zd_mac.h
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef _ZD_MAC_H
#define _ZD_MAC_H

#include <linux/kernel.h>
#include <net/mac80211.h>

#include "zd_chip.h"
#include "zd_ieee80211.h"

struct zd_ctrlset {
	u8     modulation;
	__le16 tx_length;
	u8     control;
	/* stores only the difference to tx_length on ZD1211B */
	__le16 packet_length;
	__le16 current_length;
	u8     service;
	__le16  next_frame_length;
} __attribute__((packed));

#define ZD_CS_RESERVED_SIZE	25

/* zd_crtlset field modulation */
#define ZD_CS_RATE_MASK		0x0f
#define ZD_CS_TYPE_MASK		0x10
#define ZD_CS_RATE(modulation) ((modulation) & ZD_CS_RATE_MASK)
#define ZD_CS_TYPE(modulation) ((modulation) & ZD_CS_TYPE_MASK)

#define ZD_CS_CCK		0x00
#define ZD_CS_OFDM		0x10

#define ZD_CS_CCK_RATE_1M	0x00
#define ZD_CS_CCK_RATE_2M	0x01
#define ZD_CS_CCK_RATE_5_5M	0x02
#define ZD_CS_CCK_RATE_11M	0x03
/* The rates for OFDM are encoded as in the PLCP header. Use ZD_OFDM_RATE_*.
 */

/* bit 5 is preamble (when in CCK mode), or a/g selection (when in OFDM mode) */
#define ZD_CS_CCK_PREA_LONG	0x00
#define ZD_CS_CCK_PREA_SHORT	0x20
#define ZD_CS_OFDM_MODE_11G	0x00
#define ZD_CS_OFDM_MODE_11A	0x20

/* zd_ctrlset control field */
#define ZD_CS_NEED_RANDOM_BACKOFF	0x01
#define ZD_CS_MULTICAST			0x02

#define ZD_CS_FRAME_TYPE_MASK		0x0c
#define ZD_CS_DATA_FRAME		0x00
#define ZD_CS_PS_POLL_FRAME		0x04
#define ZD_CS_MANAGEMENT_FRAME		0x08
#define ZD_CS_NO_SEQUENCE_CTL_FRAME	0x0c

#define ZD_CS_WAKE_DESTINATION		0x10
#define ZD_CS_RTS			0x20
#define ZD_CS_ENCRYPT			0x40
#define ZD_CS_SELF_CTS			0x80

/* Incoming frames are prepended by a PLCP header */
#define ZD_PLCP_HEADER_SIZE		5

struct rx_length_info {
	__le16 length[3];
	__le16 tag;
} __attribute__((packed));

#define RX_LENGTH_INFO_TAG		0x697e

struct rx_status {
	u8 signal_quality_cck;
	/* rssi */
	u8 signal_strength;
	u8 signal_quality_ofdm;
	u8 decryption_type;
	u8 frame_status;
} __attribute__((packed));

/* rx_status field decryption_type */
#define ZD_RX_NO_WEP	0
#define ZD_RX_WEP64	1
#define ZD_RX_TKIP	2
#define ZD_RX_AES	4
#define ZD_RX_WEP128	5
#define ZD_RX_WEP256	6

/* rx_status field frame_status */
#define ZD_RX_FRAME_MODULATION_MASK	0x01
#define ZD_RX_CCK			0x00
#define ZD_RX_OFDM			0x01

#define ZD_RX_TIMEOUT_ERROR		0x02
#define ZD_RX_FIFO_OVERRUN_ERROR	0x04
#define ZD_RX_DECRYPTION_ERROR		0x08
#define ZD_RX_CRC32_ERROR		0x10
#define ZD_RX_NO_ADDR1_MATCH_ERROR	0x20
#define ZD_RX_CRC16_ERROR		0x40
#define ZD_RX_ERROR			0x80

enum mac_flags {
	MAC_FIXED_CHANNEL = 0x01,
};

struct housekeeping {
	struct delayed_work link_led_work;
};

#define ZD_MAC_STATS_BUFFER_SIZE 16

struct zd_mac {
	struct zd_chip chip;
	spinlock_t lock;
	struct ieee80211_hw *dev;
	struct housekeeping housekeeping;
	struct work_struct set_multicast_hash_work;
	struct zd_mc_hash multicast_hash;
	u8 regdomain;
	u8 default_regdomain;
	int mode;
	int associated;
	u8 *hwaddr;
	struct sk_buff_head tx_queue;
	struct ieee80211_channel channels[14];
	struct ieee80211_rate rates[12];
	struct ieee80211_hw_mode modes[2];
};

static inline struct zd_mac *zd_dev_mac(struct ieee80211_hw *dev)
{
	return dev->priv;
}

static inline struct zd_mac *zd_chip_to_mac(struct zd_chip *chip)
{
	return container_of(chip, struct zd_mac, chip);
}

static inline struct zd_mac *zd_usb_to_mac(struct zd_usb *usb)
{
	return zd_chip_to_mac(zd_usb_to_chip(usb));
}

#define zd_mac_dev(mac) (zd_chip_dev(&(mac)->chip))

struct ieee80211_hw *zd_mac_alloc(struct usb_interface *intf);
void zd_mac_clear(struct zd_mac *mac);

int zd_mac_init_hw(struct ieee80211_hw *dev, u8 device_type);

int zd_mac_rx(struct ieee80211_hw *dev, const u8 *buffer, unsigned int length);
void zd_mac_tx_failed(struct ieee80211_hw *dev);

#ifdef DEBUG
void zd_dump_rx_status(const struct rx_status *status);
#else
#define zd_dump_rx_status(status)
#endif /* DEBUG */

/* TODO: remove this once we have a general modes/channels/rates filling func */
static const struct ieee80211_rate zd_rates[] = {
	{ .rate = 10,
	  .val = ZD_CS_CCK | ZD_CS_CCK_RATE_1M,
	  .flags = IEEE80211_RATE_CCK },
	{ .rate = 20,
	  .val = ZD_CS_CCK | ZD_CS_CCK_RATE_2M,
	  .val2 = ZD_CS_CCK | ZD_CS_CCK_RATE_2M | ZD_CS_CCK_PREA_SHORT,
	  .flags = IEEE80211_RATE_CCK_2 },
	{ .rate = 55,
	  .val = ZD_CS_CCK | ZD_CS_CCK_RATE_5_5M,
	  .val2 = ZD_CS_CCK | ZD_CS_CCK_RATE_5_5M | ZD_CS_CCK_PREA_SHORT,
	  .flags = IEEE80211_RATE_CCK_2 },
	{ .rate = 110,
	  .val = ZD_CS_CCK | ZD_CS_CCK_RATE_11M,
	  .val2 = ZD_CS_CCK | ZD_CS_CCK_RATE_11M | ZD_CS_CCK_PREA_SHORT,
	  .flags = IEEE80211_RATE_CCK_2 },
	{ .rate = 60,
	  .val = ZD_CS_OFDM | ZD_OFDM_RATE_6M,
	  .flags = IEEE80211_RATE_OFDM },
	{ .rate = 90,
	  .val = ZD_CS_OFDM | ZD_OFDM_RATE_9M,
	  .flags = IEEE80211_RATE_OFDM },
	{ .rate = 120,
	  .val = ZD_CS_OFDM | ZD_OFDM_RATE_12M,
	  .flags = IEEE80211_RATE_OFDM },
	{ .rate = 180,
	  .val = ZD_CS_OFDM | ZD_OFDM_RATE_18M,
	  .flags = IEEE80211_RATE_OFDM },
	{ .rate = 240,
	  .val = ZD_CS_OFDM | ZD_OFDM_RATE_24M,
	  .flags = IEEE80211_RATE_OFDM },
	{ .rate = 360,
	  .val = ZD_CS_OFDM | ZD_OFDM_RATE_36M,
	  .flags = IEEE80211_RATE_OFDM },
	{ .rate = 480,
	  .val = ZD_CS_OFDM | ZD_OFDM_RATE_48M,
	  .flags = IEEE80211_RATE_OFDM },
	{ .rate = 540,
	  .val = ZD_CS_OFDM | ZD_OFDM_RATE_54M,
	  .flags = IEEE80211_RATE_OFDM },
};

static const struct ieee80211_channel zd_channels[] = {
	{ .chan = 1,
	  .freq = 2412},
	{ .chan = 2,
	  .freq = 2417},
	{ .chan = 3,
	  .freq = 2422},
	{ .chan = 4,
	  .freq = 2427},
	{ .chan = 5,
	  .freq = 2432},
	{ .chan = 6,
	  .freq = 2437},
	{ .chan = 7,
	  .freq = 2442},
	{ .chan = 8,
	  .freq = 2447},
	{ .chan = 9,
	  .freq = 2452},
	{ .chan = 10,
	  .freq = 2457},
	{ .chan = 11,
	  .freq = 2462},
	{ .chan = 12,
	  .freq = 2467},
	{ .chan = 13,
	  .freq = 2472},
	{ .chan = 14,
	  .freq = 2484}
};

#endif /* _ZD_MAC_H */
