/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%     CCCC   OOO   N   N  SSSSS  TTTTT  IIIII  TTTTT  U   U  TTTTT  EEEEE     %
%    C      O   O  NN  N  SS       T      I      T    U   U    T    E         %
%    C      O   O  N N N  ESSS     T      I      T    U   U    T    EEE       %
%    C      O   O  N  NN     SS    T      I      T    U   U    T    E         %
%     CCCC   OOO   N   N  SSSSS    T    IIIII    T     UUU     T    EEEEE     %
%                                                                             %
%                                                                             %
%                       Methods to Consitute an Image                         %
%                                                                             %
%                             Software Design                                 %
%                               John Cristy                                   %
%                               October 1998                                  %
%                                                                             %
%                                                                             %
%  Copyright 1999-2004 ImageMagick Studio LLC, a non-profit organization      %
%  dedicated to making software imaging solutions freely available.           %
%                                                                             %
%  You may not use this file except in compliance with the License.  You may  %
%  obtain a copy of the License at                                            %
%                                                                             %
%    http://www.imagemagick.org/www/Copyright.html                            %
%                                                                             %
%  Unless required by applicable law or agreed to in writing, software        %
%  distributed under the License is distributed on an "AS IS" BASIS,          %
%  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.   %
%  See the License for the specific language governing permissions and        %
%  limitations under the License.                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%
*/

/*
  Include declarations.
*/
#include "magick/studio.h"
#include "magick/attribute.h"
#include "magick/blob.h"
#include "magick/error.h"
#include "magick/cache.h"
#include "magick/constitute.h"
#include "magick/delegate.h"
#include "magick/geometry.h"
#include "magick/list.h"
#include "magick/magick.h"
#include "magick/memory_.h"
#include "magick/monitor.h"
#include "magick/option.h"
#include "magick/resource_.h"
#include "magick/semaphore.h"
#include "magick/stream.h"
#include "magick/string_.h"
#include "magick/utility.h"

static SemaphoreInfo
  *constitute_semaphore = (SemaphoreInfo *) NULL;

/*
  Forward declarations.
*/
static Image
  *ReadImages(const ImageInfo *,ExceptionInfo *);

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   C o n s t i t u t e I m a g e                                             %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  ConstituteImage() returns an image from the the pixel data you supply.
%  The pixel data must be in scanline order top-to-bottom.  The data can be
%  char, short int, int, float, or double.  Float and double require the
%  pixels to be normalized [0..1], otherwise [0..MaxRGB].  For example, to
%  create a 640x480 image from unsigned red-green-blue character data, use
%
%      image=ConstituteImage(640,480,"RGB",CharPixel,pixels,&exception);
%
%  The format of the Constitute method is:
%
%      Image *ConstituteImage(const unsigned long columns,
%        const unsigned long rows,const char *map,const StorageType storage,
%        const void *pixels,ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o columns: width in pixels of the image.
%
%    o rows: height in pixels of the image.
%
%    o map:  This string reflects the expected ordering of the pixel array.
%      It can be any combination or order of R = red, G = green, B = blue,
%      A = alpha (0 is transparent), O = opacity (0 is opaque), C = cyan,
%      Y = yellow, M = magenta, K = black, I = intensity (for grayscale),
%      P = pad.
%
%    o storage: Define the data type of the pixels.  Float and double types are
%      expected to be normalized [0..1] otherwise [0..MaxRGB].  Choose from
%      these types: CharPixel, ShortPixel, IntegerPixel, LongPixel, FloatPixel,
%      or DoublePixel.
%
%    o pixels: This array of values contain the pixel components as defined by
%      map and type.  You must preallocate this array where the expected
%      length varies depending on the values of width, height, map, and type.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/
MagickExport Image *ConstituteImage(const unsigned long columns,
  const unsigned long rows,const char *map,const StorageType storage,
  const void *pixels,ExceptionInfo *exception)
{
  Image
    *image;

  MagickBooleanType
    status;

  /*
    Allocate image structure.
  */
  assert(map != (const char *) NULL);
  (void) LogMagickEvent(TraceEvent,GetMagickModule(),map);
  assert(pixels != (void *) NULL);
  assert(exception != (ExceptionInfo *) NULL);
  assert(exception->signature == MagickSignature);
  (void) SetExceptionInfo(exception,UndefinedException);
  image=AllocateImage((ImageInfo *) NULL);
  if (image == (Image *) NULL)
    return((Image *) NULL);
  if ((columns == 0) || (rows == 0))
    ThrowBinaryException(OptionError,"NonZeroWidthAndHeightRequired",map);
  image->columns=columns;
  image->rows=rows;
  SetImage(image,OpaqueOpacity);
  status=ImportImagePixels(image,0,0,columns,rows,map,storage,pixels);
  if (status == MagickFalse)
    {
      InheritException(exception,&image->exception);
      DestroyImage(image);
    }
  return(image);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   D e s t r o y C o n s t i t u t e                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  DestroyConstitute() destroys the constitute environment.
%
%  The format of the DestroyConstitute method is:
%
%      DestroyConstitute(void)
%
%
*/
MagickExport void DestroyConstitute(void)
{
  if (constitute_semaphore != (SemaphoreInfo *) NULL)
    DestroySemaphoreInfo(&constitute_semaphore);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   E x p o r t I m a g e P i x e l s                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  ExportImagePixels() extracts pixel data from an image and returns it to you.
%  The method returns MagickTrue on success otherwise MagickFalse if an error is
%  encountered.  The data is returned as char, short int, int, long, float,
%  or double in the order specified by map.
%
%  Suppose you want to extract the first scanline of a 640x480 image as
%  character data in red-green-blue order:
%
%      ExportImagePixels(image,0,0,640,1,"RGB",CharPixel,pixels,exception);
%
%  The format of the ExportImagePixels method is:
%
%      MagickBooleanType ExportImagePixels(const Image *image,
%        const long x_offset,const long y_offset,const unsigned long columns,
%        const unsigned long rows,const char *map,const StorageType type,
%        void *pixels,ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o x_offset, y_offset, columns, rows:  These values define the perimeter
%      of a region of pixels you want to extract.
%
%    o map:  This string reflects the expected ordering of the pixel array.
%      It can be any combination or order of R = red, G = green, B = blue,
%      A = alpha (0 is transparent), O = opacity (0 is opaque), C = cyan,
%      Y = yellow, M = magenta, K = black, I = intensity (for grayscale),
%      P = pad.
%
%    o type: Define the data type of the pixels.  Float and double types are
%      normalized to [0..1] otherwise [0..MaxRGB].  Choose from these types:
%      CharPixel, ShortPixel, IntegerPixel, LongPixel, FloatPixel, or
%      DoublePixel.
%
%    o pixels: This array of values contain the pixel components as defined by
%      map and type.  You must preallocate this array where the expected
%      length varies depending on the values of width, height, map, and type.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/
MagickExport MagickBooleanType ExportImagePixels(const Image *image,
  const long x_offset,const long y_offset,const unsigned long columns,
  const unsigned long rows,const char *map,const StorageType type,void *pixels,
  ExceptionInfo *exception)
{
  long
    y;

  QuantumType
    *quantum_map;

  register long
    i,
    x;

  register const PixelPacket
    *p;

  register IndexPacket
    *indexes;

  size_t
    length;

  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (image->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),image->filename);
  length=strlen(map);
  quantum_map=(QuantumType *) AcquireMagickMemory(length*sizeof(QuantumType));
  if (quantum_map == (QuantumType *) NULL)
    {
      (void) ThrowMagickException(exception,GetMagickModule(),
        ResourceLimitError,"MemoryAllocationFailed",image->filename);
      return(MagickFalse);
    }
  for (i=0; i < (long) length; i++)
  {
    switch (map[i])
    {
      case 'A':
      case 'a':
      {
        quantum_map[i]=AlphaQuantum;
        break;
      }
      case 'B':
      case 'b':
      {
        quantum_map[i]=BlueQuantum;
        break;
      }
      case 'C':
      case 'c':
      {
        quantum_map[i]=CyanQuantum;
        if (image->colorspace == CMYKColorspace)
          break;
        quantum_map=(QuantumType *) RelinquishMagickMemory(quantum_map);
        (void) ThrowMagickException(exception,GetMagickModule(),ImageError,
          "ColorSeparatedImageRequired",map);
        return(MagickFalse);
      }
      case 'g':
      case 'G':
      {
        quantum_map[i]=GreenQuantum;
        break;
      }
      case 'I':
      case 'i':
      {
        quantum_map[i]=IndexQuantum;
        break;
      }
      case 'K':
      case 'k':
      {
        quantum_map[i]=BlackQuantum;
        if (image->colorspace == CMYKColorspace)
          break;
        quantum_map=(QuantumType *) RelinquishMagickMemory(quantum_map);
        (void) ThrowMagickException(exception,GetMagickModule(),ImageError,
          "ColorSeparatedImageRequired",map);
        return(MagickFalse);
      }
      case 'M':
      case 'm':
      {
        quantum_map[i]=MagentaQuantum;
        if (image->colorspace == CMYKColorspace)
          break;
        quantum_map=(QuantumType *) RelinquishMagickMemory(quantum_map);
        (void) ThrowMagickException(exception,GetMagickModule(),ImageError,
          "ColorSeparatedImageRequired",map);
        return(MagickFalse);
      }
      case 'o':
      case 'O':
      {
        quantum_map[i]=OpacityQuantum;
        break;
      }
      case 'P':
      case 'p':
      {
        quantum_map[i]=UndefinedQuantum;
        break;
      }
      case 'R':
      case 'r':
      {
        quantum_map[i]=RedQuantum;
        break;
      }
      case 'Y':
      case 'y':
      {
        quantum_map[i]=YellowQuantum;
        if (image->colorspace == CMYKColorspace)
          break;
        quantum_map=(QuantumType *) RelinquishMagickMemory(quantum_map);
        (void) ThrowMagickException(exception,GetMagickModule(),ImageError,
          "ColorSeparatedImageRequired",map);
        return(MagickFalse);
      }
      default:
      {
        quantum_map=(QuantumType *) RelinquishMagickMemory(quantum_map);
        (void) ThrowMagickException(exception,GetMagickModule(),OptionError,
          "UnrecognizedPixelMap",map);
        return(MagickFalse);
      }
    }
  }
  switch (type)
  {
    case CharPixel:
    default:
    {
      register unsigned char
        *q;

      q=(unsigned char *) pixels;
      if (LocaleCompare(map,"BGR") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=ScaleQuantumToChar(p->blue);
              *q++=ScaleQuantumToChar(p->green);
              *q++=ScaleQuantumToChar(p->red);
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"BGRA") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=ScaleQuantumToChar(p->blue);
              *q++=ScaleQuantumToChar(p->green);
              *q++=ScaleQuantumToChar(p->red);
              *q++=ScaleQuantumToChar(MaxRGB-p->opacity);
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"BGRO") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=ScaleQuantumToChar(p->blue);
              *q++=ScaleQuantumToChar(p->green);
              *q++=ScaleQuantumToChar(p->red);
              *q++=ScaleQuantumToChar(p->opacity);
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"BGRP") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=ScaleQuantumToChar(p->blue);
              *q++=ScaleQuantumToChar(p->green);
              *q++=ScaleQuantumToChar(p->red);
              *q++=0;
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"I") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=ScaleQuantumToChar(PixelIntensityToQuantum(p));
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"RGB") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=ScaleQuantumToChar(p->red);
              *q++=ScaleQuantumToChar(p->green);
              *q++=ScaleQuantumToChar(p->blue);
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"RGBA") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=ScaleQuantumToChar(p->red);
              *q++=ScaleQuantumToChar(p->green);
              *q++=ScaleQuantumToChar(p->blue);
              *q++=ScaleQuantumToChar(MaxRGB-p->opacity);
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"RGBO") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=ScaleQuantumToChar(p->red);
              *q++=ScaleQuantumToChar(p->green);
              *q++=ScaleQuantumToChar(p->blue);
              *q++=ScaleQuantumToChar(p->opacity);
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"RGBP") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=ScaleQuantumToChar(p->red);
              *q++=ScaleQuantumToChar(p->green);
              *q++=ScaleQuantumToChar(p->blue);
              *q++=0;
              p++;
            }
          }
          break;
        }
      for (y=0; y < (long) rows; y++)
      {
        p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
        if (p == (const PixelPacket *) NULL)
          break;
        indexes=GetIndexes(image);
        for (x=0; x < (long) columns; x++)
        {
          for (i=0; i < (long) length; i++)
          {
            *q=0;
            switch (quantum_map[i])
            {
              case RedQuantum:
              case CyanQuantum:
              {
                *q=ScaleQuantumToChar(p->red);
                break;
              }
              case GreenQuantum:
              case MagentaQuantum:
              {
                *q=ScaleQuantumToChar(p->green);
                break;
              }
              case BlueQuantum:
              case YellowQuantum:
              {
                *q=ScaleQuantumToChar(p->blue);
                break;
              }
              case AlphaQuantum:
              {
                if (image->matte != MagickFalse)
                  *q=ScaleQuantumToChar(MaxRGB-p->opacity);
                break;
              }
              case OpacityQuantum:
              {
                if (image->matte != MagickFalse)
                  *q=ScaleQuantumToChar(p->opacity);
                break;
              }
              case BlackQuantum:
              {
                if (image->colorspace == CMYKColorspace)
                  *q=ScaleQuantumToChar(indexes[x]);
                break;
              }
              case IndexQuantum:
              {
                *q=ScaleQuantumToChar(PixelIntensityToQuantum(p));
                break;
              }
              default:
                break;
            }
            q++;
          }
          p++;
        }
      }
      break;
    }
    case ShortPixel:
    {
      register unsigned short
        *q;

      q=(unsigned short *) pixels;
      if (LocaleCompare(map,"BGR") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=ScaleQuantumToShort(p->blue);
              *q++=ScaleQuantumToShort(p->green);
              *q++=ScaleQuantumToShort(p->red);
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"BGRA") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=ScaleQuantumToShort(p->blue);
              *q++=ScaleQuantumToShort(p->green);
              *q++=ScaleQuantumToShort(p->red);
              *q++=ScaleQuantumToShort(MaxRGB-p->opacity);
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"BGRP") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=ScaleQuantumToShort(p->blue);
              *q++=ScaleQuantumToShort(p->green);
              *q++=ScaleQuantumToShort(p->red);
              *q++=0;
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"I") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=ScaleQuantumToShort(PixelIntensityToQuantum(p));
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"RGB") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=ScaleQuantumToShort(p->red);
              *q++=ScaleQuantumToShort(p->green);
              *q++=ScaleQuantumToShort(p->blue);
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"RGBA") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=ScaleQuantumToShort(p->red);
              *q++=ScaleQuantumToShort(p->green);
              *q++=ScaleQuantumToShort(p->blue);
              *q++=ScaleQuantumToShort(MaxRGB-p->opacity);
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"RGBP") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=ScaleQuantumToShort(p->red);
              *q++=ScaleQuantumToShort(p->green);
              *q++=ScaleQuantumToShort(p->blue);
              *q++=0;
              p++;
            }
          }
          break;
        }
      for (y=0; y < (long) rows; y++)
      {
        p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
        if (p == (const PixelPacket *) NULL)
          break;
        indexes=GetIndexes(image);
        for (x=0; x < (long) columns; x++)
        {
          for (i=0; i < (long) length; i++)
          {
            *q=0;
            switch (quantum_map[i])
            {
              case RedQuantum:
              case CyanQuantum:
              {
                *q=ScaleQuantumToShort(p->red);
                break;
              }
              case GreenQuantum:
              case MagentaQuantum:
              {
                *q=ScaleQuantumToShort(p->green);
                break;
              }
              case BlueQuantum:
              case YellowQuantum:
              {
                *q=ScaleQuantumToShort(p->blue);
                break;
              }
              case AlphaQuantum:
              {
                if (image->matte != MagickFalse)
                  *q=ScaleQuantumToShort(MaxRGB-p->opacity);
                break;
              }
              case OpacityQuantum:
              {
                if (image->matte != MagickFalse)
                  *q=ScaleQuantumToShort(p->opacity);
                break;
              }
              case BlackQuantum:
              {
                if (image->colorspace == CMYKColorspace)
                  *q=ScaleQuantumToShort(indexes[x]);
                break;
              }
              case IndexQuantum:
              {
                *q=ScaleQuantumToShort(PixelIntensityToQuantum(p));
                break;
              }
              default:
                break;
            }
            q++;
          }
          p++;
        }
      }
      break;
    }
    case IntegerPixel:
    {
      register unsigned int
        *q;

      q=(unsigned int *) pixels;
      if (LocaleCompare(map,"BGR") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=(unsigned int) ScaleQuantumToLong(p->blue);
              *q++=(unsigned int) ScaleQuantumToLong(p->green);
              *q++=(unsigned int) ScaleQuantumToLong(p->red);
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"BGRA") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=(unsigned int) ScaleQuantumToLong(p->blue);
              *q++=(unsigned int) ScaleQuantumToLong(p->green);
              *q++=(unsigned int) ScaleQuantumToLong(p->red);
              *q++=(unsigned int) ScaleQuantumToLong(MaxRGB-p->opacity);
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"BGRP") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=(unsigned int) ScaleQuantumToLong(p->blue);
              *q++=(unsigned int) ScaleQuantumToLong(p->green);
              *q++=(unsigned int) ScaleQuantumToLong(p->red);
              *q++=0U;
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"I") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=(unsigned int)
                ScaleQuantumToLong(PixelIntensityToQuantum(p));
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"RGB") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=(unsigned int) ScaleQuantumToLong(p->red);
              *q++=(unsigned int) ScaleQuantumToLong(p->green);
              *q++=(unsigned int) ScaleQuantumToLong(p->blue);
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"RGBA") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=(unsigned int) ScaleQuantumToLong(p->red);
              *q++=(unsigned int) ScaleQuantumToLong(p->green);
              *q++=(unsigned int) ScaleQuantumToLong(p->blue);
              *q++=(unsigned int) ScaleQuantumToLong(MaxRGB-p->opacity);
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"RGBP") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=(unsigned int) ScaleQuantumToLong(p->red);
              *q++=(unsigned int) ScaleQuantumToLong(p->green);
              *q++=(unsigned int) ScaleQuantumToLong(p->blue);
              *q++=0U;
              p++;
            }
          }
          break;
        }
      for (y=0; y < (long) rows; y++)
      {
        p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
        if (p == (const PixelPacket *) NULL)
          break;
        indexes=GetIndexes(image);
        for (x=0; x < (long) columns; x++)
        {
          for (i=0; i < (long) length; i++)
          {
            *q=0;
            switch (quantum_map[i])
            {
              case RedQuantum:
              case CyanQuantum:
              {
                *q=(unsigned int) ScaleQuantumToLong(p->red);
                break;
              }
              case GreenQuantum:
              case MagentaQuantum:
              {
                *q=(unsigned int) ScaleQuantumToLong(p->green);
                break;
              }
              case BlueQuantum:
              case YellowQuantum:
              {
                *q=(unsigned int) ScaleQuantumToLong(p->blue);
                break;
              }
              case AlphaQuantum:
              {
                if (image->matte != MagickFalse)
                  *q=(unsigned int) ScaleQuantumToLong(MaxRGB-p->opacity);
                break;
              }
              case OpacityQuantum:
              {
                if (image->matte != MagickFalse)
                  *q=(unsigned int) ScaleQuantumToLong(p->opacity);
                break;
              }
              case BlackQuantum:
              {
                if (image->colorspace == CMYKColorspace)
                  *q=(unsigned int) ScaleQuantumToLong(indexes[x]);
                break;
              }
              case IndexQuantum:
              {
                *q=(unsigned int)
                  ScaleQuantumToLong(PixelIntensityToQuantum(p));
                break;
              }
              default:
                *q=0;
            }
            q++;
          }
          p++;
        }
      }
      break;
    }
    case LongPixel:
    {
      register unsigned long
        *q;

      q=(unsigned long *) pixels;
      if (LocaleCompare(map,"BGR") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=ScaleQuantumToLong(p->blue);
              *q++=ScaleQuantumToLong(p->green);
              *q++=ScaleQuantumToLong(p->red);
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"BGRA") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=ScaleQuantumToLong(p->blue);
              *q++=ScaleQuantumToLong(p->green);
              *q++=ScaleQuantumToLong(p->red);
              *q++=ScaleQuantumToLong(MaxRGB-p->opacity);
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"BGRP") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=ScaleQuantumToLong(p->blue);
              *q++=ScaleQuantumToLong(p->green);
              *q++=ScaleQuantumToLong(p->red);
              *q++=0;
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"I") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=ScaleQuantumToLong(PixelIntensityToQuantum(p));
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"RGB") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=ScaleQuantumToLong(p->red);
              *q++=ScaleQuantumToLong(p->green);
              *q++=ScaleQuantumToLong(p->blue);
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"RGBA") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=ScaleQuantumToLong(p->red);
              *q++=ScaleQuantumToLong(p->green);
              *q++=ScaleQuantumToLong(p->blue);
              *q++=ScaleQuantumToLong(MaxRGB-p->opacity);
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"RGBP") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=ScaleQuantumToLong(p->red);
              *q++=ScaleQuantumToLong(p->green);
              *q++=ScaleQuantumToLong(p->blue);
              *q++=0;
              p++;
            }
          }
          break;
        }
      for (y=0; y < (long) rows; y++)
      {
        p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
        if (p == (const PixelPacket *) NULL)
          break;
        indexes=GetIndexes(image);
        for (x=0; x < (long) columns; x++)
        {
          for (i=0; i < (long) length; i++)
          {
            *q=0;
            switch (quantum_map[i])
            {
              case RedQuantum:
              case CyanQuantum:
              {
                *q=ScaleQuantumToLong(p->red);
                break;
              }
              case GreenQuantum:
              case MagentaQuantum:
              {
                *q=ScaleQuantumToLong(p->green);
                break;
              }
              case BlueQuantum:
              case YellowQuantum:
              {
                *q=ScaleQuantumToLong(p->blue);
                break;
              }
              case AlphaQuantum:
              {
                if (image->matte != MagickFalse)
                  *q=ScaleQuantumToLong(MaxRGB-p->opacity);
                break;
              }
              case OpacityQuantum:
              {
                if (image->matte != MagickFalse)
                  *q=ScaleQuantumToLong(p->opacity);
                break;
              }
              case BlackQuantum:
              {
                if (image->colorspace == CMYKColorspace)
                  *q=ScaleQuantumToLong(indexes[x]);
                break;
              }
              case IndexQuantum:
              {
                *q=ScaleQuantumToLong(PixelIntensityToQuantum(p));
                break;
              }
              default:
                break;
            }
            q++;
          }
          p++;
        }
      }
      break;
    }
    case FloatPixel:
    {
      register float
        *q;

      q=(float *) pixels;
      if (LocaleCompare(map,"BGR") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=(float) (QuantumScale*p->blue);
              *q++=(float) (QuantumScale*p->green);
              *q++=(float) (QuantumScale*p->red);
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"BGRA") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=(float) (QuantumScale*p->blue);
              *q++=(float) (QuantumScale*p->green);
              *q++=(float) (QuantumScale*p->red);
              *q++=(float) (QuantumScale*MaxRGB-p->opacity);
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"BGRP") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=(float) (QuantumScale*p->blue);
              *q++=(float) (QuantumScale*p->green);
              *q++=(float) (QuantumScale*p->red);
              *q++=0.0;
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"I") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=(float) (QuantumScale*PixelIntensityToQuantum(p));
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"RGB") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=(float) (QuantumScale*p->red);
              *q++=(float) (QuantumScale*p->green);
              *q++=(float) (QuantumScale*p->blue);
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"RGBA") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=(float) (QuantumScale*p->red);
              *q++=(float) (QuantumScale*p->green);
              *q++=(float) (QuantumScale*p->blue);
              *q++=(float) (QuantumScale*(MaxRGB-p->opacity));
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"RGBP") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=(float) (QuantumScale*p->red);
              *q++=(float) (QuantumScale*p->green);
              *q++=(float) (QuantumScale*p->blue);
              *q++=0.0;
              p++;
            }
          }
          break;
        }
      for (y=0; y < (long) rows; y++)
      {
        p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
        if (p == (const PixelPacket *) NULL)
          break;
        indexes=GetIndexes(image);
        for (x=0; x < (long) columns; x++)
        {
          for (i=0; i < (long) length; i++)
          {
            *q=0;
            switch (quantum_map[i])
            {
              case RedQuantum:
              case CyanQuantum:
              {
                *q=(float) (QuantumScale*p->red);
                break;
              }
              case GreenQuantum:
              case MagentaQuantum:
              {
                *q=(float) (QuantumScale*p->green);
                break;
              }
              case BlueQuantum:
              case YellowQuantum:
              {
                *q=(float) (QuantumScale*p->blue);
                break;
              }
              case AlphaQuantum:
              {
                if (image->matte != MagickFalse)
                  *q=(float) (QuantumScale*(MaxRGB-p->opacity));
                break;
              }
              case OpacityQuantum:
              {
                if (image->matte != MagickFalse)
                  *q=(float) (QuantumScale*p->opacity);
                break;
              }
              case BlackQuantum:
              {
                if (image->colorspace == CMYKColorspace)
                  *q=(float) (QuantumScale*indexes[x]);
                break;
              }
              case IndexQuantum:
              {
                *q=(float) (QuantumScale*PixelIntensityToQuantum(p));
                break;
              }
              default:
                *q=0;
            }
            q++;
          }
          p++;
        }
      }
      break;
    }
    case DoublePixel:
    {
      register double
        *q;

      q=(double *) pixels;
      if (LocaleCompare(map,"BGR") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=(QuantumScale*p->blue);
              *q++=(QuantumScale*p->green);
              *q++=(QuantumScale*p->red);
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"BGRA") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=(QuantumScale*p->blue);
              *q++=(QuantumScale*p->green);
              *q++=(QuantumScale*p->red);
              *q++=(QuantumScale*MaxRGB-p->opacity);
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"BGRP") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=(QuantumScale*p->blue);
              *q++=(QuantumScale*p->green);
              *q++=(QuantumScale*p->red);
              *q++=0.0;
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"I") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=QuantumScale*PixelIntensityToQuantum(p);
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"RGB") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=(QuantumScale*p->red);
              *q++=(QuantumScale*p->green);
              *q++=(QuantumScale*p->blue);
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"RGBA") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=(QuantumScale*p->red);
              *q++=(QuantumScale*p->green);
              *q++=(QuantumScale*p->blue);
              *q++=(QuantumScale*(MaxRGB-p->opacity));
              p++;
            }
          }
          break;
        }
      if (LocaleCompare(map,"RGBP") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
            if (p == (const PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              *q++=(QuantumScale*p->red);
              *q++=(QuantumScale*p->green);
              *q++=(QuantumScale*p->blue);
              *q++=0.0;
              p++;
            }
          }
          break;
        }
      for (y=0; y < (long) rows; y++)
      {
        p=AcquireImagePixels(image,x_offset,y_offset+y,columns,1,exception);
        if (p == (const PixelPacket *) NULL)
          break;
        indexes=GetIndexes(image);
        for (x=0; x < (long) columns; x++)
        {
          for (i=0; i < (long) length; i++)
          {
            *q=0;
            switch (quantum_map[i])
            {
              case RedQuantum:
              case CyanQuantum:
              {
                *q=QuantumScale*p->red;
                break;
              }
              case GreenQuantum:
              case MagentaQuantum:
              {
                *q=QuantumScale*p->green;
                break;
              }
              case BlueQuantum:
              case YellowQuantum:
              {
                *q=QuantumScale*p->blue;
                break;
              }
              case AlphaQuantum:
              {
                if (image->matte != MagickFalse)
                  *q=QuantumScale*(MaxRGB-p->opacity);
                break;
              }
              case OpacityQuantum:
              {
                if (image->matte != MagickFalse)
                  *q=QuantumScale*p->opacity;
                break;
              }
              case BlackQuantum:
              {
                if (image->colorspace == CMYKColorspace)
                  *q=QuantumScale*indexes[x];
                break;
              }
              case IndexQuantum:
              {
                *q=QuantumScale*PixelIntensityToQuantum(p);
                break;
              }
              default:
                *q=0;
            }
            q++;
          }
          p++;
        }
      }
      break;
    }
  }
  quantum_map=(QuantumType *) RelinquishMagickMemory(quantum_map);
  return(MagickTrue);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   I m p o r t I m a g e P i x e l s                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  ImportImagePixels() accepts pixel data and stores in the image at the
%  location you specify.  The method returns MagickTrue on success otherwise
%  MagickFalse if an error is encountered.  The pixel data can be either char,
%  short int, int, long, float, or double in the order specified by map.
%
%  Suppose your want want to upload the first scanline of a 640x480 image from
%  character data in red-green-blue order:
%
%      ImportImagePixels(image,0,0,640,1,"RGB",CharPixel,pixels);
%
%  The format of the ImportImagePixels method is:
%
%      MagickBooleanType ImportImagePixels(Image *image,const long x_offset,
%        const long y_offset,const unsigned long columns,
%        const unsigned long rows,const char *map,const StorageType type,
%        const void *pixels)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o x_offset, y_offset, columns, rows:  These values define the perimeter
%      of a region of pixels you want to define.
%
%    o map:  This string reflects the expected ordering of the pixel array.
%      It can be any combination or order of R = red, G = green, B = blue,
%      A = alpha (0 is transparent), O = opacity (0 is opaque), C = cyan,
%      Y = yellow, M = magenta, K = black, I = intensity (for grayscale),
%      P = pad.
%
%    o type: Define the data type of the pixels.  Float and double types are
%      normalized to [0..1] otherwise [0..MaxRGB].  Choose from these types:
%      CharPixel, ShortPixel, IntegerPixel, LongPixel, FloatPixel, or
%      DoublePixel.
%
%    o pixels: This array of values contain the pixel components as defined by
%      map and type.  You must preallocate this array where the expected
%      length varies depending on the values of width, height, map, and type.
%
%
*/
MagickExport MagickBooleanType ImportImagePixels(Image *image,
  const long x_offset,const long y_offset,const unsigned long columns,
  const unsigned long rows,const char *map,const StorageType type,
  const void *pixels)
{
  long
    y;

  PixelPacket
    *q;

  QuantumType
    *quantum_map;

  register IndexPacket
    *indexes;

  register long
    i,
    x;

  size_t
    length;

  /*
    Allocate image structure.
  */
  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (image->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),image->filename);
  length=strlen(map);
  quantum_map=(QuantumType *) AcquireMagickMemory(length*sizeof(QuantumType));
  if (quantum_map == (QuantumType *) NULL)
    ThrowBinaryException(ResourceLimitError,"MemoryAllocationFailed",
      image->filename);
  image->storage_class=DirectClass;
  for (i=0; i < (long) length; i++)
  {
    switch (map[i])
    {
      case 'a':
      case 'A':
      {
        quantum_map[i]=AlphaQuantum;
        image->matte=MagickTrue;
        break;
      }
      case 'B':
      case 'b':
      {
        quantum_map[i]=BlueQuantum;
        break;
      }
      case 'C':
      case 'c':
      {
        quantum_map[i]=CyanQuantum;
        image->colorspace=CMYKColorspace;
        break;
      }
      case 'g':
      case 'G':
      {
        quantum_map[i]=GreenQuantum;
        break;
      }
      case 'm':
      case 'M':
      {
        quantum_map[i]=MagentaQuantum;
        image->colorspace=CMYKColorspace;
        break;
      }
      case 'K':
      case 'k':
      {
        quantum_map[i]=BlackQuantum;
        image->colorspace=CMYKColorspace;
        break;
      }
      case 'I':
      case 'i':
      {
        quantum_map[i]=IndexQuantum;
        break;
      }
      case 'O':
      case 'o':
      {
        quantum_map[i]=OpacityQuantum;
        break;
      }
      case 'P':
      case 'p':
      {
        quantum_map[i]=UndefinedQuantum;
        break;
      }
      case 'R':
      case 'r':
      {
        quantum_map[i]=RedQuantum;
        break;
      }
      case 'Y':
      case 'y':
      {
        quantum_map[i]=YellowQuantum;
        image->colorspace=CMYKColorspace;
        break;
      }
      default:
      {
        quantum_map=(QuantumType *) RelinquishMagickMemory(quantum_map);
        (void) ThrowMagickException(&image->exception,GetMagickModule(),
          OptionError,"UnrecognizedPixelMap",map);
        return(MagickFalse);
      }
    }
  }
  /*
    Transfer the pixels from the pixel data array to the image.
  */
  switch (type)
  {
    case CharPixel:
    {
      register unsigned char
        *p;

      p=(unsigned char *) pixels;
      if (LocaleCompare(map,"BGR") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->blue=ScaleCharToQuantum(*p++);
              q->green=ScaleCharToQuantum(*p++);
              q->red=ScaleCharToQuantum(*p++);
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"BGRA") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->blue=ScaleCharToQuantum(*p++);
              q->green=ScaleCharToQuantum(*p++);
              q->red=ScaleCharToQuantum(*p++);
              q->opacity=MaxRGB-ScaleCharToQuantum(*p++);
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"BGRO") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->blue=ScaleCharToQuantum(*p++);
              q->green=ScaleCharToQuantum(*p++);
              q->red=ScaleCharToQuantum(*p++);
              q->opacity=ScaleCharToQuantum(*p++);
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"BGRP") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->blue=ScaleCharToQuantum(*p++);
              q->green=ScaleCharToQuantum(*p++);
              q->red=ScaleCharToQuantum(*p++);
              p++;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"I") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->red=ScaleCharToQuantum(*p++);
              q->green=q->red;
              q->blue=q->red;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"RGB") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->red=ScaleCharToQuantum(*p++);
              q->green=ScaleCharToQuantum(*p++);
              q->blue=ScaleCharToQuantum(*p++);
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"RGBA") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->red=ScaleCharToQuantum(*p++);
              q->green=ScaleCharToQuantum(*p++);
              q->blue=ScaleCharToQuantum(*p++);
              q->opacity=MaxRGB-ScaleCharToQuantum(*p++);
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"RGBO") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->red=ScaleCharToQuantum(*p++);
              q->green=ScaleCharToQuantum(*p++);
              q->blue=ScaleCharToQuantum(*p++);
              q->opacity=ScaleCharToQuantum(*p++);
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"RGBP") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->red=ScaleCharToQuantum(*p++);
              q->green=ScaleCharToQuantum(*p++);
              q->blue=ScaleCharToQuantum(*p++);
              p++;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      for (y=0; y < (long) rows; y++)
      {
        q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
        if (q == (PixelPacket *) NULL)
          break;
        indexes=GetIndexes(image);
        for (x=0; x < (long) columns; x++)
        {
          for (i=0; i < (long) length; i++)
          {
            switch (quantum_map[i])
            {
              case RedQuantum:
              case CyanQuantum:
              {
                q->red=ScaleCharToQuantum(*p);
                break;
              }
              case GreenQuantum:
              case MagentaQuantum:
              {
                q->green=ScaleCharToQuantum(*p);
                break;
              }
              case BlueQuantum:
              case YellowQuantum:
              {
                q->blue=ScaleCharToQuantum(*p);
                break;
              }
              case AlphaQuantum:
              {
                q->opacity=MaxRGB-ScaleCharToQuantum(*p);
                break;
              }
              case OpacityQuantum:
              {
                q->opacity=ScaleCharToQuantum(*p);
                break;
              }
              case BlackQuantum:
              {
                indexes[x]=ScaleCharToQuantum(*p);
                break;
              }
              case IndexQuantum:
              {
                q->red=ScaleCharToQuantum(*p);
                q->green=q->red;
                q->blue=q->red;
                break;
              }
              default:
                break;
            }
            p++;
          }
          q++;
        }
        if (SyncImagePixels(image) == MagickFalse)
          break;
      }
      break;
    }
    case ShortPixel:
    {
      register unsigned short
        *p;

      p=(unsigned short *) pixels;
      if (LocaleCompare(map,"BGR") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->blue=ScaleShortToQuantum(*p++);
              q->green=ScaleShortToQuantum(*p++);
              q->red=ScaleShortToQuantum(*p++);
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"BGRA") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->blue=ScaleShortToQuantum(*p++);
              q->green=ScaleShortToQuantum(*p++);
              q->red=ScaleShortToQuantum(*p++);
              q->opacity=MaxRGB-ScaleShortToQuantum(*p++);
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"BGRP") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->blue=ScaleShortToQuantum(*p++);
              q->green=ScaleShortToQuantum(*p++);
              q->red=ScaleShortToQuantum(*p++);
              p++;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"I") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->red=ScaleShortToQuantum(*p++);
              q->green=q->red;
              q->blue=q->red;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"RGB") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->red=ScaleShortToQuantum(*p++);
              q->green=ScaleShortToQuantum(*p++);
              q->blue=ScaleShortToQuantum(*p++);
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"RGBA") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->red=ScaleShortToQuantum(*p++);
              q->green=ScaleShortToQuantum(*p++);
              q->blue=ScaleShortToQuantum(*p++);
              q->opacity=MaxRGB-ScaleShortToQuantum(*p++);
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"RGBP") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->red=ScaleShortToQuantum(*p++);
              q->green=ScaleShortToQuantum(*p++);
              q->blue=ScaleShortToQuantum(*p++);
              p++;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      for (y=0; y < (long) rows; y++)
      {
        q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
        if (q == (PixelPacket *) NULL)
          break;
        indexes=GetIndexes(image);
        for (x=0; x < (long) columns; x++)
        {
          for (i=0; i < (long) length; i++)
          {
            switch (quantum_map[i])
            {
              case RedQuantum:
              case CyanQuantum:
              {
                q->red=ScaleShortToQuantum(*p);
                break;
              }
              case GreenQuantum:
              case MagentaQuantum:
              {
                q->green=ScaleShortToQuantum(*p);
                break;
              }
              case BlueQuantum:
              case YellowQuantum:
              {
                q->blue=ScaleShortToQuantum(*p);
                break;
              }
              case AlphaQuantum:
              {
                q->opacity=MaxRGB-ScaleShortToQuantum(*p);
                break;
              }
              case OpacityQuantum:
              {
                q->opacity=ScaleShortToQuantum(*p);
                break;
              }
              case BlackQuantum:
              {
                indexes[x]=ScaleShortToQuantum(*p);
                break;
              }
              case IndexQuantum:
              {
                q->red=ScaleShortToQuantum(*p);
                q->green=q->red;
                q->blue=q->red;
                break;
              }
              default:
                break;
            }
            p++;
          }
          q++;
        }
        if (SyncImagePixels(image) == MagickFalse)
          break;
      }
      break;
    }
    case IntegerPixel:
    {
      register unsigned int
        *p;

      p=(unsigned int *) pixels;
      if (LocaleCompare(map,"BGR") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->blue=ScaleLongToQuantum(*p++);
              q->green=ScaleLongToQuantum(*p++);
              q->red=ScaleLongToQuantum(*p++);
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"BGRA") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->blue=ScaleLongToQuantum(*p++);
              q->green=ScaleLongToQuantum(*p++);
              q->red=ScaleLongToQuantum(*p++);
              q->opacity=MaxRGB-ScaleLongToQuantum(*p++);
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"BGRP") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->blue=ScaleLongToQuantum(*p++);
              q->green=ScaleLongToQuantum(*p++);
              q->red=ScaleLongToQuantum(*p++);
              p++;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"I") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->red=ScaleLongToQuantum(*p++);
              q->green=q->red;
              q->blue=q->red;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"RGB") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->red=ScaleLongToQuantum(*p++);
              q->green=ScaleLongToQuantum(*p++);
              q->blue=ScaleLongToQuantum(*p++);
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"RGBA") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->red=ScaleLongToQuantum(*p++);
              q->green=ScaleLongToQuantum(*p++);
              q->blue=ScaleLongToQuantum(*p++);
              q->opacity=MaxRGB-ScaleLongToQuantum(*p++);
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"RGBP") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->red=ScaleLongToQuantum(*p++);
              q->green=ScaleLongToQuantum(*p++);
              q->blue=ScaleLongToQuantum(*p++);
              p++;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      for (y=0; y < (long) rows; y++)
      {
        q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
        if (q == (PixelPacket *) NULL)
          break;
        indexes=GetIndexes(image);
        for (x=0; x < (long) columns; x++)
        {
          for (i=0; i < (long) length; i++)
          {
            switch (quantum_map[i])
            {
              case RedQuantum:
              case CyanQuantum:
              {
                q->red=ScaleLongToQuantum(*p);
                break;
              }
              case GreenQuantum:
              case MagentaQuantum:
              {
                q->green=ScaleLongToQuantum(*p);
                break;
              }
              case BlueQuantum:
              case YellowQuantum:
              {
                q->blue=ScaleLongToQuantum(*p);
                break;
              }
              case AlphaQuantum:
              {
                q->opacity=MaxRGB-ScaleLongToQuantum(*p);
                break;
              }
              case OpacityQuantum:
              {
                q->opacity=ScaleLongToQuantum(*p);
                break;
              }
              case BlackQuantum:
              {
                indexes[x]=ScaleLongToQuantum(*p);
                break;
              }
              case IndexQuantum:
              {
                q->red=ScaleLongToQuantum(*p);
                q->green=q->red;
                q->blue=q->red;
                break;
              }
              default:
                break;
            }
            p++;
          }
          q++;
        }
        if (SyncImagePixels(image) == MagickFalse)
          break;
      }
      break;
    }
    case LongPixel:
    {
      register unsigned long
        *p;

      p=(unsigned long *) pixels;
      if (LocaleCompare(map,"BGR") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->blue=ScaleLongToQuantum(*p++);
              q->green=ScaleLongToQuantum(*p++);
              q->red=ScaleLongToQuantum(*p++);
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"BGRA") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->blue=ScaleLongToQuantum(*p++);
              q->green=ScaleLongToQuantum(*p++);
              q->red=ScaleLongToQuantum(*p++);
              q->opacity=MaxRGB-ScaleLongToQuantum(*p++);
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"BGRP") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->blue=ScaleLongToQuantum(*p++);
              q->green=ScaleLongToQuantum(*p++);
              q->red=ScaleLongToQuantum(*p++);
              p++;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"I") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->red=ScaleLongToQuantum(*p++);
              q->green=q->red;
              q->blue=q->red;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"RGB") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->red=ScaleLongToQuantum(*p++);
              q->green=ScaleLongToQuantum(*p++);
              q->blue=ScaleLongToQuantum(*p++);
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"RGBA") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->red=ScaleLongToQuantum(*p++);
              q->green=ScaleLongToQuantum(*p++);
              q->blue=ScaleLongToQuantum(*p++);
              q->opacity=MaxRGB-ScaleLongToQuantum(*p++);
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"RGBP") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->red=ScaleLongToQuantum(*p++);
              q->green=ScaleLongToQuantum(*p++);
              q->blue=ScaleLongToQuantum(*p++);
              p++;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      for (y=0; y < (long) rows; y++)
      {
        q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
        if (q == (PixelPacket *) NULL)
          break;
        indexes=GetIndexes(image);
        for (x=0; x < (long) columns; x++)
        {
          for (i=0; i < (long) length; i++)
          {
            switch (quantum_map[i])
            {
              case RedQuantum:
              case CyanQuantum:
              {
                q->red=ScaleLongToQuantum(*p);
                break;
              }
              case GreenQuantum:
              case MagentaQuantum:
              {
                q->green=ScaleLongToQuantum(*p);
                break;
              }
              case BlueQuantum:
              case YellowQuantum:
              {
                q->blue=ScaleLongToQuantum(*p);
                break;
              }
              case AlphaQuantum:
              {
                q->opacity=MaxRGB-ScaleLongToQuantum(*p);
                break;
              }
              case OpacityQuantum:
              {
                q->opacity=ScaleLongToQuantum(*p);
                break;
              }
              case BlackQuantum:
              {
                indexes[x]=ScaleLongToQuantum(*p);
                break;
              }
              case IndexQuantum:
              {
                q->red=ScaleLongToQuantum(*p);
                q->green=q->red;
                q->blue=q->red;
                break;
              }
              default:
                break;
            }
            p++;
          }
          q++;
        }
        if (SyncImagePixels(image) == MagickFalse)
          break;
      }
      break;
    }
    case FloatPixel:
    {
      register float
        *p;

      p=(float *) pixels;
      if (LocaleCompare(map,"BGR") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->blue=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->green=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->red=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"BGRA") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->blue=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->green=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->red=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->opacity=MaxRGB-RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"BGRP") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->blue=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->green=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->red=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"I") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->red=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              q->green=q->red;
              q->blue=q->red;
              p++;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"RGB") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->red=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->green=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->blue=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"RGBA") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->red=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->green=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->blue=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->opacity=MaxRGB-RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"RGBP") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->red=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->green=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->blue=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      for (y=0; y < (long) rows; y++)
      {
        q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
        if (q == (PixelPacket *) NULL)
          break;
        indexes=GetIndexes(image);
        for (x=0; x < (long) columns; x++)
        {
          for (i=0; i < (long) length; i++)
          {
            switch (quantum_map[i])
            {
              case RedQuantum:
              case CyanQuantum:
              {
                q->red=RoundToQuantum((MagickRealType) MaxRGB*(*p));
                break;
              }
              case GreenQuantum:
              case MagentaQuantum:
              {
                q->green=RoundToQuantum((MagickRealType) MaxRGB*(*p));
                break;
              }
              case BlueQuantum:
              case YellowQuantum:
              {
                q->blue=RoundToQuantum((MagickRealType) MaxRGB*(*p));
                break;
              }
              case AlphaQuantum:
              {
                q->opacity=MaxRGB-RoundToQuantum((MagickRealType) MaxRGB*(*p));
                break;
              }
              case OpacityQuantum:
              {
                q->opacity=RoundToQuantum((MagickRealType) MaxRGB*(*p));
                break;
              }
              case BlackQuantum:
              {
                indexes[x]=RoundToQuantum((MagickRealType) MaxRGB*(*p));
                break;
              }
              case IndexQuantum:
              {
                q->red=RoundToQuantum((MagickRealType) MaxRGB*(*p));
                q->green=q->red;
                q->blue=q->red;
                break;
              }
              default:
                break;
            }
            p++;
          }
          q++;
        }
        if (SyncImagePixels(image) == MagickFalse)
          break;
      }
      break;
    }
    case DoublePixel:
    {
      register double
        *p;

      p=(double *) pixels;
      if (LocaleCompare(map,"BGR") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->blue=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->green=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->red=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"BGRA") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->blue=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->green=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->red=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->opacity=MaxRGB-RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"BGRP") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->blue=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->green=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->red=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"I") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->red=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              q->green=q->red;
              q->blue=q->red;
              p++;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"RGB") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->red=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->green=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->blue=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"RGBA") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->red=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->green=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->blue=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->opacity=MaxRGB-RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      if (LocaleCompare(map,"RGBP") == 0)
        {
          for (y=0; y < (long) rows; y++)
          {
            q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) columns; x++)
            {
              q->red=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->green=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q->blue=RoundToQuantum((MagickRealType) MaxRGB*(*p));
              p++;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
          break;
        }
      for (y=0; y < (long) rows; y++)
      {
        q=GetImagePixels(image,x_offset,y_offset+y,columns,1);
        if (q == (PixelPacket *) NULL)
          break;
        indexes=GetIndexes(image);
        for (x=0; x < (long) columns; x++)
        {
          for (i=0; i < (long) length; i++)
          {
            switch (quantum_map[i])
            {
              case RedQuantum:
              case CyanQuantum:
              {
                q->red=RoundToQuantum((MagickRealType) MaxRGB*(*p));
                break;
              }
              case GreenQuantum:
              case MagentaQuantum:
              {
                q->green=RoundToQuantum((MagickRealType) MaxRGB*(*p));
                break;
              }
              case BlueQuantum:
              case YellowQuantum:
              {
                q->blue=RoundToQuantum((MagickRealType) MaxRGB*(*p));
                break;
              }
              case AlphaQuantum:
              {
                q->opacity=MaxRGB-RoundToQuantum((MagickRealType) MaxRGB*(*p));
                break;
              }
              case OpacityQuantum:
              {
                q->opacity=RoundToQuantum((MagickRealType) MaxRGB*(*p));
                break;
              }
              case BlackQuantum:
              {
                indexes[x]=RoundToQuantum((MagickRealType) MaxRGB*(*p));
                break;
              }
              case IndexQuantum:
              {
                q->red=RoundToQuantum((MagickRealType) MaxRGB*(*p));
                q->green=q->red;
                q->blue=q->red;
                break;
              }
              default:
                break;
            }
            p++;
          }
          q++;
        }
        if (SyncImagePixels(image) == MagickFalse)
          break;
      }
      break;
    }
    default:
    {
      quantum_map=(QuantumType *) RelinquishMagickMemory(quantum_map);
      (void) ThrowMagickException(&image->exception,GetMagickModule(),
        OptionError,"UnrecognizedPixelMap",map);
      break;
    }
  }
  quantum_map=(QuantumType *) RelinquishMagickMemory(quantum_map);
  return(MagickTrue);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   P i n g I m a g e                                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  PingImage() returns all the attributes of an image or image sequence
%  except for the pixels.  It is much faster and consumes far less memory
%  than ReadImage().  On failure, a NULL image is returned and exception
%  describes the reason for the failure.
%
%  The format of the PingImage method is:
%
%      Image *PingImage(const ImageInfo *image_info,ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o image_info: Ping the image defined by the file or filename members of
%      this structure.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/

#if defined(__cplusplus) || defined(c_plusplus)
extern "C" {
#endif

static MagickBooleanType PingStream(const Image *image,const void *pixels,
  const size_t columns)
{
  return(MagickTrue);
}

#if defined(__cplusplus) || defined(c_plusplus)
}
#endif

MagickExport Image *PingImage(const ImageInfo *image_info,
  ExceptionInfo *exception)
{
  Image
    *image;

  ImageInfo
    *ping_info;

  assert(image_info != (ImageInfo *) NULL);
  assert(image_info->signature == MagickSignature);
  if (image_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),image_info->filename);
  assert(exception != (ExceptionInfo *) NULL);
  (void) SetExceptionInfo(exception,UndefinedException);
  ping_info=CloneImageInfo(image_info);
  ping_info->ping=MagickTrue;
  image=ReadStream(ping_info,&PingStream,exception);
  DestroyImageInfo(ping_info);
  if ((image != (Image *) NULL) && (image_info->verbose != MagickFalse))
    (void) DescribeImage(image,stdout,MagickFalse);
  return(image);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   P o p I m a g e P i x e l s                                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  PopImagePixels() transfers one or more pixel components from the image pixel
%  cache to a user supplied buffer.  The pixels are returned in network byte
%  order.  True is returned if the pixels are successfully transferred,
%  otherwise MagickFalse.
%
%  The format of the PopImagePixels method is:
%
%      MagickBooleanType PopImagePixels(Image *,const QuantumType quantum,
%        unsigned char *destination)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o quantum: Declare which pixel components to transfer (RGB, RGBA, etc).
%
%    o destination:  The components are transferred to this buffer.
%
%
*/
MagickExport MagickBooleanType PopImagePixels(Image *image,
  const QuantumType quantum,unsigned char *destination)
{
#define PopCharPixel(q,pixel) \
{ \
  *(q)++=(unsigned char) (pixel); \
}
#define PopLongPixel(q,pixel) \
{ \
  *(q)++=(unsigned char) ((pixel) >> 24); \
  *(q)++=(unsigned char) ((pixel) >> 16); \
  *(q)++=(unsigned char) ((pixel) >> 8); \
  *(q)++=(unsigned char) (pixel); \
}
#define PopShortPixel(q,pixel) \
{ \
  *(q)++=(unsigned char) ((pixel) >> 8); \
  *(q)++=(unsigned char) (pixel); \
}

  long
    bit;

  MagickSizeType
    number_pixels;

  register IndexPacket
    *indexes;

  register long
    i,
    x;

  register PixelPacket
    *p;

  register unsigned char
    *q;

  unsigned long
    depth,
    pixel;

  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (image->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),image->filename);
  assert(destination != (unsigned char *) NULL);
  number_pixels=GetPixelCacheArea(image);
  depth=GetImageQuantumDepth(image);
  x=0;
  p=GetPixels(image);
  indexes=GetIndexes(image);
  q=destination;
  switch (quantum)
  {
    case IndexQuark:
    {
      if (image->storage_class != PseudoClass)
        ThrowBinaryException(ImageError,"ColormappedImageRequired",
          image->filename);
      for (depth=1; (1UL << depth) < image->colors; depth*=2);
      switch (depth)
      {
        case 1:
        {
          for (x=((long) image->columns-7); x > 0; x-=8)
          {
            pixel=(unsigned long) *indexes++;
            *q=(unsigned char) ((pixel & 0x01) << 7);
            pixel=(unsigned long) *indexes++;
            *q|=(unsigned char) ((pixel & 0x01) << 6);
            pixel=(unsigned long) *indexes++;
            *q|=(unsigned char) ((pixel & 0x01) << 5);
            pixel=(unsigned long) *indexes++;
            *q|=(unsigned char) ((pixel & 0x01) << 4);
            pixel=(unsigned long) *indexes++;
            *q|=(unsigned char) ((pixel & 0x01) << 3);
            pixel=(unsigned long) *indexes++;
            *q|=(unsigned char) ((pixel & 0x01) << 2);
            pixel=(unsigned long) *indexes++;
            *q|=(unsigned char) ((pixel & 0x01) << 1);
            pixel=(unsigned long) *indexes++;
            *q|=(unsigned char) ((pixel & 0x01) << 0);
            q++;
          }
          if ((image->columns % 8) != 0)
            {
              *q='\0';
              for (bit=7; bit >= (long) (8-(image->columns % 8)); bit--)
              {
                pixel=(unsigned long) *indexes++;
                *q|=(unsigned char) ((pixel & 0x01) << (unsigned long) bit);
              }
              q++;
            }
          break;
        }
        case 2:
        {
          for (x=0; x < ((long) image->columns-3); x+=4)
          {
            pixel=(unsigned long) *indexes++;
            *q=(unsigned char) ((pixel & 0x03) << 6);
            pixel=(unsigned long) *indexes++;
            *q|=(unsigned char) ((pixel & 0x03) << 4);
            pixel=(unsigned long) *indexes++;
            *q|=(unsigned char) ((pixel & 0x03) << 2);
            pixel=(unsigned long) *indexes++;
            *q|=(unsigned char) ((pixel & 0x03) << 0);
            q++;
          }
          if ((image->columns % 4) != 0)
            {
              *q='\0';
              for (i=3; i >= (4-((long) image->columns % 4)); i--)
              {
                pixel=(unsigned long) *indexes++;
                *q|=(unsigned char) ((pixel & 0x03) << ((unsigned long) i*2));
              }
              q++;
            }
          break;
        }
        case 4:
        {
          for (x=0; x < (long) (image->columns-1) ; x+=2)
          {
            pixel=(unsigned long) *indexes++;
            *q=(unsigned char) ((pixel & 0xf) << 4);
            pixel=(unsigned long) *indexes++;
            *q|=(unsigned char) ((pixel & 0xf) << 0);
            q++;
          }
          if ((image->columns % 2) != 0)
            {
              pixel=(unsigned long) *indexes++;
              *q=(unsigned char) ((pixel & 0xf) << 4);
              q++;
            }
          break;
        }
        default:
          break;
      }
      if (x != 0)
        break;
    }
    case IndexQuantum:
    {
      if (image->storage_class != PseudoClass)
        ThrowBinaryException(ImageError,"ColormappedImageRequired",
          image->filename);
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
            PopCharPixel(q,indexes[x]);
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
            PopShortPixel(q,indexes[x]);
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
            PopLongPixel(q,indexes[x]);
          break;
        }
      }
      break;
    }
    case IndexAlphaQuark:
    {
      if (image->storage_class != PseudoClass)
        ThrowBinaryException(ImageError,"ColormappedImageRequired",
          image->filename);
      for (depth=1; (1UL << depth) < image->colors; depth*=2);
      switch (depth)
      {
        case 1:
        {
          for (x=((long) image->columns-3); x > 0; x-=4)
          {
            pixel=(unsigned long) *indexes++;
            *q=(unsigned char) ((pixel & 0x01) << 7);
            pixel=(unsigned long) (p->opacity == TransparentOpacity);
            *q|=(unsigned char) ((pixel & 0x01) << 6);
            p++;
            pixel=(unsigned long) *indexes++;
            *q|=(unsigned char) ((pixel & 0x01) << 5);
            pixel=(unsigned long) (p->opacity == TransparentOpacity);
            *q|=(unsigned char) ((pixel & 0x01) << 4);
            p++;
            pixel=(unsigned long) *indexes++;
            *q|=(unsigned char) ((pixel & 0x01) << 3);
            pixel=(unsigned long) (p->opacity == TransparentOpacity);
            *q|=(unsigned char) ((pixel & 0x01) << 2);
            p++;
            pixel=(unsigned long) *indexes++;
            *q|=(unsigned char) ((pixel & 0x01) << 1);
            pixel=(unsigned long) (p->opacity == TransparentOpacity);
            *q|=(unsigned char) ((pixel & 0x01) << 0);
            p++;
            q++;
          }
          if ((image->columns % 4) != 0)
            {
              *q='\0';
              for (bit=3; bit >= (long) (4-(image->columns % 4)); bit-=2)
              {
                pixel=(unsigned long) *indexes++;
                *q|=(unsigned char) ((pixel & 0x01) << (unsigned long) bit);
                pixel=(unsigned long) (p->opacity == TransparentOpacity);
                *q|=(unsigned char) ((pixel & 0x01) << (unsigned long) (bit-1));
                p++;
              }
              q++;
            }
          break;
        }
        case 2:
        {
          for (x=0; x < (long) image->columns; x++)
          {
            pixel=(unsigned long) *indexes++;
            *q=(unsigned char) ((pixel & 0x03) << 6);
            pixel=(unsigned long) (4*QuantumScale*p->opacity+0.5);
            *q|=(unsigned char) ((pixel & 0x03) << 4);
            p++;
            pixel=(unsigned long) *indexes++;
            *q|=(unsigned char) ((pixel & 0x03) << 2);
            pixel=(unsigned long) (4*QuantumScale*p->opacity+0.5);
            *q|=(unsigned char) ((pixel & 0x03) << 0);
            p++;
            q++;
          }
          break;
        }
        case 4:
        {
          for (x=0; x < (long) image->columns ; x++)
          {
            pixel=(unsigned long) *indexes++;
            *q=(unsigned char) ((pixel & 0xf) << 4);
            pixel=(unsigned long) (16*QuantumScale*(MaxRGB-p->opacity)+0.5);
            *q|=(unsigned char) ((pixel & 0xf) << 0);
            p++;
            q++;
          }
          break;
        }
        default:
          break;
      }
      if (x != 0)
        break;
    }
    case IndexAlphaQuantum:
    {
      if (image->storage_class != PseudoClass)
        ThrowBinaryException(ImageError,"ColormappedImageRequired",
          image->filename);
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PopCharPixel(q,indexes[x]);
            pixel=(unsigned long) ScaleQuantumToChar(MaxRGB-p->opacity);
            PopCharPixel(q,pixel);
            p++;
          }
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PopShortPixel(q,indexes[x]);
            pixel=(unsigned long) ScaleQuantumToShort(MaxRGB-p->opacity);
            PopShortPixel(q,pixel);
            p++;
          }
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PopLongPixel(q,indexes[x]);
            pixel=(unsigned long) ScaleQuantumToLong(MaxRGB-p->opacity);
            PopLongPixel(q,pixel);
            p++;
          }
          break;
        }
      }
      break;
    }
    case GrayQuark:
    {
      for (depth=1; depth < image->depth; depth*=2);
      switch (depth)
      {
        case 1:
        {
          for (x=((long) image->columns-7); x > 0; x-=8)
          {
            *q='\0';
            pixel=(unsigned long) PixelIntensityToQuantum(p);
            *q|=(unsigned char) ((int) (int) pixel != 0 ? 0x00 : 0x01) << 7;
            p++;
            pixel=(unsigned long) PixelIntensityToQuantum(p);
            *q|=(unsigned char) ((int) pixel != 0 ? 0x00 : 0x01) << 6;
            p++;
            pixel=(unsigned long) PixelIntensityToQuantum(p);
            *q|=(unsigned char) ((int) pixel != 0 ? 0x00 : 0x01) << 5;
            p++;
            pixel=(unsigned long) PixelIntensityToQuantum(p);
            *q|=(unsigned char) ((int) pixel != 0 ? 0x00 : 0x01) << 4;
            p++;
            pixel=(unsigned long) PixelIntensityToQuantum(p);
            *q|=(unsigned char) ((int) pixel != 0 ? 0x00 : 0x01) << 3;
            p++;
            pixel=(unsigned long) PixelIntensityToQuantum(p);
            *q|=(unsigned char) ((int) pixel != 0 ? 0x00 : 0x01) << 2;
            p++;
            pixel=(unsigned long) PixelIntensityToQuantum(p);
            *q|=(unsigned char) ((int) pixel != 0 ? 0x00 : 0x01) << 1;
            p++;
            pixel=(unsigned long) PixelIntensityToQuantum(p);
            *q|=(unsigned char) ((int) pixel != 0 ? 0x00 : 0x01) << 0;
            p++;
            q++;
          }
          if ((image->columns % 8) != 0)
            {
              *q='\0';
              for (bit=7; bit >= (long) (8-(image->columns % 8)); bit--)
              {
                pixel=(unsigned long) PixelIntensityToQuantum(p);
                *q|=(unsigned char) ((int) pixel != 0 ? 0x00 : 0x01) << bit;
                p++;
              }
              q++;
            }
          break;
        }
        case 2:
        {
          for (x=0; x < ((long) image->columns-3); x+=4)
          {
            *q='\0';
            pixel=(unsigned long) PixelIntensityToQuantum(p);
            *q|=(unsigned char) ((pixel & 0x03) << 6);
            p++;
            pixel=(unsigned long) PixelIntensityToQuantum(p);
            *q|=(unsigned char) ((pixel & 0x03) << 4);
            p++;
            pixel=(unsigned long) PixelIntensityToQuantum(p);
            *q|=(unsigned char) ((pixel & 0x03) << 2);
            p++;
            pixel=(unsigned long) PixelIntensityToQuantum(p);
            *q|=(unsigned char) ((pixel & 0x03));
            p++;
            q++;
          }
          if ((image->columns % 4) != 0)
            {
              *q='\0';
              for (i=3; i >= (4-((long) image->columns % 4)); i--)
              {
                pixel=(unsigned long) PixelIntensityToQuantum(p);
                *q|=(unsigned char) (pixel << ((unsigned long) i*2));
                p++;
              }
              q++;
            }
          break;
        }
        case 4:
        {
          for (x=0; x < (long) (image->columns-1) ; x+=2)
          {
            pixel=(unsigned long) PixelIntensityToQuantum(p);
            *q=(unsigned char) ((pixel & 0xf) << 4);
            p++;
            pixel=(unsigned long) PixelIntensityToQuantum(p);
            *q|=(unsigned char) ((pixel & 0xf) << 0);
            p++;
            q++;
          }
          if ((image->columns % 2) != 0)
            {
              pixel=(unsigned long) PixelIntensityToQuantum(p);
              *q=(unsigned char) ((pixel & 0xf) << 4);
              p++;
              q++;
            }
          break;
        }
        default:
          break;
      }
      if (x != 0)
        break;
    }
    case GrayQuantum:
    {
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long)
              ScaleQuantumToChar(PixelIntensityToQuantum(p));
            PopCharPixel(q,pixel);
            p++;
          }
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long)
              ScaleQuantumToShort(PixelIntensityToQuantum(p));
            PopShortPixel(q,pixel);
            p++;
          }
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long)
              ScaleQuantumToLong(PixelIntensityToQuantum(p));
            PopLongPixel(q,pixel);
            p++;
          }
          break;
        }
      }
      break;
    }
    case GrayAlphaQuark:
    {
      for (depth=1; depth < image->depth; depth*=2);
      switch (depth)
      {
        case 1:
        {
          for (x=((long) image->columns-3); x > 0; x-=4)
          {
            pixel=(unsigned long) PixelIntensityToQuantum(p);
            *q=(unsigned char) ((pixel & 0x01) << 7);
            pixel=(unsigned long) (p->opacity != TransparentOpacity);
            *q|=(unsigned char) ((pixel & 0x01) << 6);
            p++;
            pixel=(unsigned long) PixelIntensityToQuantum(p);
            *q|=(unsigned char) ((pixel & 0x01) << 5);
            pixel=(unsigned long) (p->opacity != TransparentOpacity);
            *q|=(unsigned char) ((pixel & 0x01) << 4);
            p++;
            pixel=(unsigned long) PixelIntensityToQuantum(p);
            *q|=(unsigned char) ((pixel & 0x01) << 3);
            pixel=(unsigned long) (p->opacity != TransparentOpacity);
            *q|=(unsigned char) ((pixel & 0x01) << 2);
            p++;
            pixel=(unsigned long) PixelIntensityToQuantum(p);
            *q|=(unsigned char) ((pixel & 0x01) << 1);
            pixel=(unsigned long) (p->opacity != TransparentOpacity);
            *q|=(unsigned char) ((pixel & 0x01) << 0);
            p++;
            q++;
          }
          if ((image->columns % 4) != 0)
            {
              *q='\0';
              for (bit=3; bit >= (long) (4-(image->columns % 4)); bit-=2)
              {
                pixel=(unsigned long) PixelIntensityToQuantum(p);
                *q|=(unsigned char) ((pixel & 0x01) << (unsigned long) bit);
                pixel=(unsigned long) (p->opacity != TransparentOpacity);
                *q|=(unsigned char) ((pixel & 0x01) << (unsigned long) (bit-1));
                p++;
              }
              q++;
            }
          break;
        }
        case 2:
        {
          for (x=0; x < (long) image->columns; x++)
          {
            pixel=(unsigned long) PixelIntensityToQuantum(p);
            *q=(unsigned char) ((pixel & 0x03) << 6);
            pixel=(unsigned long) (4*QuantumScale*p->opacity+0.5);
            *q|=(unsigned char) ((pixel & 0x03) << 4);
            p++;
            pixel=(unsigned long) PixelIntensityToQuantum(p);
            *q|=(unsigned char) ((pixel & 0x03) << 2);
            pixel=(unsigned long) (4*QuantumScale*p->opacity+0.5);
            *q|=(unsigned char) ((pixel & 0x03) << 0);
            p++;
            q++;
          }
          break;
        }
        case 4:
        {
          for (x=0; x < (long) image->columns ; x++)
          {
            pixel=(unsigned long) PixelIntensityToQuantum(p);
            *q=(unsigned char) ((pixel & 0xf) << 4);
            pixel=(unsigned long) (16*QuantumScale*(MaxRGB-p->opacity)+0.5);
            *q|=(unsigned char) ((pixel & 0xf) << 0);
            p++;
            q++;
          }
          break;
        }
        default:
          break;
      }
      if (x != 0)
        break;
    }
    case GrayAlphaQuantum:
    {
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long)
              ScaleQuantumToChar(PixelIntensityToQuantum(p));
            PopCharPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToChar(MaxRGB-p->opacity);
            PopCharPixel(q,pixel);
            p++;
          }
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long)
              ScaleQuantumToShort(PixelIntensityToQuantum(p));
            PopShortPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToShort(MaxRGB-p->opacity);
            PopShortPixel(q,pixel);
            p++;
          }
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long)
              ScaleQuantumToLong(PixelIntensityToQuantum(p));
            PopLongPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToLong(MaxRGB-p->opacity);
            PopLongPixel(q,pixel);
            p++;
          }
          break;
        }
      }
      break;
    }
    case RedQuantum:
    case CyanQuantum:
    {
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToChar(p->red);
            PopCharPixel(q,pixel);
            p++;
          }
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToShort(p->red);
            PopShortPixel(q,pixel);
            p++;
          }
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToLong(p->red);
            PopLongPixel(q,pixel);
            p++;
          }
          break;
        }
      }
      break;
    }
    case GreenQuantum:
    case MagentaQuantum:
    {
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToChar(p->green);
            PopCharPixel(q,pixel);
            p++;
          }
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToShort(p->green);
            PopShortPixel(q,pixel);
            p++;
          }
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToLong(p->green);
            PopLongPixel(q,pixel);
            p++;
          }
          break;
        }
      }
      break;
    }
    case BlueQuantum:
    case YellowQuantum:
    {
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToChar(p->blue);
            PopCharPixel(q,pixel);
            p++;
          }
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToShort(p->blue);
            PopShortPixel(q,pixel);
            p++;
          }
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToLong(p->blue);
            PopLongPixel(q,pixel);
            p++;
          }
          break;
        }
      }
      break;
    }
    case AlphaQuantum:
    {
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToChar(MaxRGB-p->opacity);
            PopCharPixel(q,pixel);
            p++;
          }
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToShort(MaxRGB-p->opacity);
            PopShortPixel(q,pixel);
            p++;
          }
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToLong(MaxRGB-p->opacity);
            PopLongPixel(q,pixel);
            p++;
          }
          break;
        }
      }
      break;
    }
    case BlackQuantum:
    {
      if (image->colorspace != CMYKColorspace)
        ThrowBinaryException(ImageError,"ColorSeparatedImageRequired",
          image->filename);
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToChar(indexes[x]);
            PopCharPixel(q,pixel);
            p++;
          }
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToShort(indexes[x]);
            PopShortPixel(q,pixel);
            p++;
          }
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToLong(indexes[x]);
            PopLongPixel(q,pixel);
            p++;
          }
          break;
        }
      }
      break;
    }
    case RGBQuantum:
    default:
    {
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToChar(p->red);
            PopCharPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToChar(p->green);
            PopCharPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToChar(p->blue);
            PopCharPixel(q,pixel);
            p++;
          }
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToShort(p->red);
            PopShortPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToShort(p->green);
            PopShortPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToShort(p->blue);
            PopShortPixel(q,pixel);
            p++;
          }
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToLong(p->red);
            PopLongPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToLong(p->green);
            PopLongPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToLong(p->blue);
            PopLongPixel(q,pixel);
            p++;
          }
          break;
        }
      }
      break;
    }
    case RGBAQuantum:
    {
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToChar(p->red);
            PopCharPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToChar(p->green);
            PopCharPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToChar(p->blue);
            PopCharPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToChar(MaxRGB-p->opacity);
            PopCharPixel(q,pixel);
            p++;
          }
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToShort(p->red);
            PopShortPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToShort(p->green);
            PopShortPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToShort(p->blue);
            PopShortPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToShort(MaxRGB-p->opacity);
            PopShortPixel(q,pixel);
            p++;
          }
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToLong(p->red);
            PopLongPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToLong(p->green);
            PopLongPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToLong(p->blue);
            PopLongPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToLong(MaxRGB-p->opacity);
            PopLongPixel(q,pixel);
            p++;
          }
          break;
        }
      }
      break;
    }
    case CMYKQuantum:
    {
      if (image->colorspace != CMYKColorspace)
        ThrowBinaryException(ImageError,"ColorSeparatedImageRequired",
          image->filename);
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToChar(p->red);
            PopCharPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToChar(p->green);
            PopCharPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToChar(p->blue);
            PopCharPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToChar(indexes[x]);
            PopCharPixel(q,pixel);
            p++;
          }
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToShort(p->red);
            PopShortPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToShort(p->green);
            PopShortPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToShort(p->blue);
            PopShortPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToShort(indexes[x]);
            PopShortPixel(q,pixel);
            p++;
          }
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToLong(p->red);
            PopLongPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToLong(p->green);
            PopLongPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToLong(p->blue);
            PopLongPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToLong(indexes[x]);
            PopLongPixel(q,pixel);
            p++;
          }
          break;
        }
      }
      break;
    }
    case CMYKAQuantum:
    {
      if (image->colorspace != CMYKColorspace)
        ThrowBinaryException(ImageError,"ColorSeparatedImageRequired",
          image->filename);
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToChar(p->red);
            PopCharPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToChar(p->green);
            PopCharPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToChar(p->blue);
            PopCharPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToChar(indexes[x]);
            PopCharPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToChar(MaxRGB-p->opacity);
            PopCharPixel(q,pixel);
            p++;
          }
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToShort(p->red);
            PopShortPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToShort(p->green);
            PopShortPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToShort(p->blue);
            PopShortPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToShort(indexes[x]);
            PopShortPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToShort(MaxRGB-p->opacity);
            PopShortPixel(q,pixel);
            p++;
          }
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            pixel=(unsigned long) ScaleQuantumToLong(p->red);
            PopLongPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToLong(p->green);
            PopLongPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToLong(p->blue);
            PopLongPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToLong(indexes[x]);
            PopLongPixel(q,pixel);
            pixel=(unsigned long) ScaleQuantumToLong(MaxRGB-p->opacity);
            PopLongPixel(q,pixel);
            p++;
          }
          break;
        }
      }
      break;
    }
  }
  if (image->endian == LSBEndian)
    switch (depth)
    {
      case 16:
      {
        MSBOrderShort(destination,(size_t) (q-destination));
        break;
      }
      case 32:
      {
        MSBOrderLong(destination,(size_t) (q-destination));
        break;
      }
      default:
        break;
    }
  return(MagickTrue);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   P u s h I m a g e P i x e l s                                             %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  PushImagePixels() transfers one or more pixel components from a user
%  supplied buffer into the image pixel cache of an image.  The pixels are
%  expected in network byte order.  It returns MagickTrue if the pixels are
%  successfully transferred, otherwise MagickFalse.
%
%  The format of the PushImagePixels method is:
%
%      MagickBooleanType PushImagePixels(Image *image,const QuantumType quantum,
%        const unsigned char *source)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o quantum: Declare which pixel components to transfer (red, green, blue,
%      opacity, RGB, or RGBA).
%
%    o source:  The pixel components are transferred from this buffer.
%
%
*/

static inline IndexPacket PushColormapIndex(Image *image,
  const unsigned long index)
{
  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (index < image->colors)
    return((IndexPacket) index);
  (void) ThrowMagickException(&image->exception,GetMagickModule(),
    CorruptImageError,"InvalidColormapIndex",image->filename);
  return(0);
}

MagickExport MagickBooleanType PushImagePixels(Image *image,
  const QuantumType quantum,const unsigned char *source)
{
#define PushCharPixel(pixel,p) \
{ \
  pixel=(unsigned long) (*(p)); \
  (p)++; \
}
#define PushLongPixel(pixel,p) \
{ \
  pixel=(unsigned long) \
    ((*(p) << 24) | (*((p)+1) << 16) | (*((p)+2) << 8) | *((p)+3)); \
  (p)+=4; \
}
#define PushShortPixel(pixel,p) \
{ \
  pixel=(unsigned long) ((*(p) << 8) | *((p)+1)); \
  (p)+=2; \
}

  long
    bit;

  MagickSizeType
    number_pixels;

  register const unsigned char
    *p;

  register IndexPacket
    *indexes;

  register long
    x;

  register PixelPacket
    *q;

  unsigned long
    depth,
    pixel;

  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (image->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),image->filename);
  assert(source != (const unsigned char *) NULL);
  number_pixels=GetPixelCacheArea(image);
  depth=GetImageQuantumDepth(image);
  if (image->endian == LSBEndian)
    {
      size_t
        packet_size;

      switch (quantum)
      {
        default: packet_size=1;  break;
        case GrayAlphaQuantum:
        case GrayAlphaQuark:
        case IndexAlphaQuantum:
        case IndexAlphaQuark: packet_size=2; break;
        case RGBQuantum: packet_size=3; break;
        case CMYKQuantum:
        case RGBAQuantum: packet_size=4; break;
        case CMYKAQuantum: packet_size=5; break;
      }
      switch (depth)
      {
        case 16:
        {
          MSBOrderShort((unsigned char *) source,
            (size_t) (2*packet_size*number_pixels));
          break;
        }
        case 32:
        {
          MSBOrderLong((unsigned char *) source,
            (size_t) (3*packet_size*number_pixels));
          break;
        }
        default:
          break;
      }
    }
  x=0;
  p=source;
  q=GetPixels(image);
  indexes=GetIndexes(image);
  switch (quantum)
  {
    case IndexQuark:
    {
      if (image->storage_class != PseudoClass)
        ThrowBinaryException(ImageError,"ColormappedImageRequired",
          image->filename);
      switch (image->depth)
      {
        case 1:
        {
          for (x=0; x < ((long) image->columns-7); x+=8)
          {
            for (bit=0; bit < 8; bit++)
            {
              pixel=(unsigned long)
                (((*p) & (1UL << (unsigned long) (7-bit))) != 0 ? 0x01 : 0x00);
              indexes[x+bit]=PushColormapIndex(image,pixel);
              *q=image->colormap[indexes[x+bit]];
              q++;
            }
            p++;
          }
          for (bit=0; bit < (long) (image->columns % 8); bit++)
          {
            pixel=(unsigned long)
              (((*p) & (1UL << (unsigned long) (7-bit))) != 0 ? 0x01 : 0x00);
            indexes[x+bit]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x+bit]];
            q++;
          }
          break;
        }
        case 2:
        {
          for (x=0; x < ((long) image->columns-3); x+=4)
          {
            pixel=(unsigned long) ((*p >> 6) & 0x03);
            indexes[x]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x]];
            q++;
            pixel=(unsigned long) ((*p >> 4) & 0x03);
            indexes[x+1]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x+1]];
            q++;
            pixel=(unsigned long) ((*p >> 2) & 0x03);
            indexes[x+2]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x+2]];
            q++;
            pixel=(unsigned long) ((*p) & 0x03);
            indexes[x+3]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x+3]];
            p++;
            q++;
          }
          for (bit=0; bit < (long) (image->columns % 4); bit++)
          {
            pixel=(unsigned long) ((*p >> (unsigned long) (2*(3-bit))) & 0x03);
            indexes[x+bit]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x+bit]];
            q++;
          }
          break;
        }
        case 4:
        {
          for (x=0; x < ((long) image->columns-1); x+=2)
          {
            pixel=(unsigned long) ((*p >> 4) & 0xf);
            indexes[x]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x]];
            q++;
            pixel=(unsigned long) ((*p) & 0xf);
            indexes[x+1]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x+1]];
            p++;
            q++;
          }
          for (bit=0; bit < (long) (image->columns % 2); bit++)
          {
            pixel=(unsigned long) ((*p++ >> 4) & 0xf);
            indexes[x+bit]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x+bit]];
            q++;
          }
          break;
        }
        case 12:
        {
          for (x=0; x < (long) (image->columns-1); x+=2)
          {
            pixel=(unsigned long) (((*(p+1) >> 4) & 0xf) | (*p));
            indexes[x]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x]];
            q++;
            pixel=(unsigned long) (((*(p+1) & 0xf) << 8) | (*(p+2)));
            indexes[x+1]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x+1]];
            p+=3;
            q++;
          }
          for (bit=0; bit < (long) (image->columns % 2); bit++)
          {
            pixel=(unsigned long) (((*(p+1) >> 4) & 0xf) | (*p));
            indexes[x+bit]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x+bit]];
            q++;
          }
          break;
        }
        default:
          break;
      }
      if (x != 0)
        break;
    }
    case IndexQuantum:
    {
      if (image->storage_class != PseudoClass)
        ThrowBinaryException(ImageError,"ColormappedImageRequired",
          image->filename);
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushCharPixel(pixel,p);
            indexes[x]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x]];
            q++;
          }
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushShortPixel(pixel,p);
            indexes[x]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x]];
            q++;
          }
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushLongPixel(pixel,p);
            indexes[x]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x]];
            q++;
          }
          break;
        }
      }
      break;
    }
    case IndexAlphaQuark:
    {
      if (image->storage_class != PseudoClass)
        ThrowBinaryException(ImageError,"ColormappedImageRequired",
          image->filename);
      switch (image->depth)
      {
        case 1:
        {
          for (x=0; x < ((long) image->columns-3); x+=4)
          {
            for (bit=0; bit < 8; bit+=2)
            {
              pixel=(unsigned long)
                (((*p) & (1UL << (unsigned long) (7-bit))) != 0 ? 0x01 : 0x00);
              indexes[x+bit]=PushColormapIndex(image,pixel);
              *q=image->colormap[indexes[x+bit]];
              pixel=(unsigned long)
                (((*p) & (1UL << (unsigned long) (8-bit))) != 0 ? 0x01 : 0x00);
              q->opacity=pixel == 0x00 ? TransparentOpacity : OpaqueOpacity;
              q++;
            }
            p++;
          }
          for (bit=0; bit < (long) (image->columns % 4); bit+=2)
          {
            pixel=(unsigned long)
              (((*p) & (1UL << (unsigned long) (7-bit))) != 0 ? 0x01 : 0x00);
            indexes[x+bit]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x+bit]];
            pixel=(unsigned long)
              (((*p) & (1UL << (unsigned long) (8-bit))) != 0 ? 0x01 : 0x00);
            q->opacity=pixel == 0x00 ? TransparentOpacity : OpaqueOpacity;
            q++;
          }
          break;
        }
        case 2:
        {
          for (x=0; x < ((long) image->columns-1); x+=2)
          {
            pixel=(unsigned long) ((*p >> 6) & 0x03);
            indexes[x]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x]];
            pixel=(unsigned long) (MaxRGB*((*p >> 4) & 0x03)/4);
            q->opacity=(Quantum) pixel;
            q++;
            pixel=(unsigned long) ((*p >> 2) & 0x03);
            indexes[x+2]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x+2]];
            pixel=(unsigned long) (MaxRGB*((*p) & 0x03)/4);
            q->opacity=(Quantum) pixel;
            p++;
            q++;
          }
          break;
        }
        case 4:
        {
          for (x=0; x < (long) image->columns; x++)
          {
            pixel=(unsigned long) ((*p >> 4) & 0xf);
            indexes[x]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x]];
            pixel=(unsigned long) (MaxRGB*((*p) & 0xf)/15);
            q->opacity=(Quantum) (MaxRGB-pixel);
            p++;
            q++;
          }
          break;
        }
        case 12:
        {
          for (x=0; x < (long) image->columns; x++)
          {
            pixel=(unsigned long) (((*(p+1) >> 4) & 0xf) | (*p));
            indexes[x]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x]];
            pixel=(unsigned long) (((*(p+1) & 0xf) << 8) | (*(p+2)));
            q->opacity=(Quantum) (MaxRGB*pixel/1024);
            p+=3;
            q++;
          }
          break;
        }
        default:
          break;
      }
      if (x != 0)
        break;
    }
    case IndexAlphaQuantum:
    {
      if (image->storage_class != PseudoClass)
        ThrowBinaryException(ImageError,"ColormappedImageRequired",
          image->filename);
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushCharPixel(pixel,p);
            indexes[x]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x]];
            PushCharPixel(pixel,p);
            q->opacity=(Quantum) (MaxRGB-ScaleCharToQuantum(pixel));
            q++;
          }
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushShortPixel(pixel,p);
            indexes[x]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x]];
            PushShortPixel(pixel,p);
            q->opacity=MaxRGB-ScaleShortToQuantum(pixel);
            q++;
          }
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushLongPixel(pixel,p);
            indexes[x]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x]];
            PushLongPixel(pixel,p);
            q->opacity=MaxRGB-ScaleLongToQuantum(pixel);
            q++;
          }
          break;
        }
      }
      break;
    }
    case GrayQuark:
    {
      if (image->storage_class != PseudoClass)
        ThrowBinaryException(ImageError,"ColormappedImageRequired",
          image->filename);
      switch (image->depth)
      {
        case 1:
        {
          for (x=0; x < ((long) image->columns-7); x+=8)
          {
            for (bit=0; bit < 8; bit++)
            {
              pixel=(unsigned long)
                (((*p) & (1UL << (unsigned long) (7-bit))) != 0 ? 0x01 : 0x00);
              indexes[x+bit]=PushColormapIndex(image,pixel);
              *q=image->colormap[indexes[x+bit]];
              q++;
            }
            p++;
          }
          for (bit=0; bit < (long) (image->columns % 8); bit++)
          {
            pixel=(unsigned long)
              (((*p) & (1UL << (unsigned long) (7-bit))) != 0 ? 0x01 : 0x00);
            indexes[x+bit]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x+bit]];
            q++;
          }
          break;
        }
        case 2:
        {
          for (x=0; x < ((long) image->columns-3); x+=4)
          {
            pixel=(unsigned long) ((*p >> 6) & 0x03);
            indexes[x]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x]];
            q++;
            pixel=(unsigned long) ((*p >> 4) & 0x03);
            indexes[x+1]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x+1]];
            q++;
            pixel=(unsigned long) ((*p >> 2) & 0x03);
            indexes[x+2]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x+2]];
            q++;
            pixel=(unsigned long) ((*p) & 0x03);
            indexes[x+3]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x+3]];
            p++;
            q++;
          }
          for (bit=0; bit < (long) (image->columns % 4); bit++)
          {
            pixel=(unsigned long) ((*p >> (unsigned long) (2*(3-bit))) & 0x03);
            indexes[x+bit]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x+bit]];
            q++;
          }
          break;
        }
        case 4:
        {
          for (x=0; x < ((long) image->columns-1); x+=2)
          {
            pixel=(unsigned long) ((*p >> 4) & 0xf);
            indexes[x]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x]];
            q++;
            pixel=(unsigned long) ((*p) & 0xf);
            indexes[x+1]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x+1]];
            p++;
            q++;
          }
          for (bit=0; bit < (long) (image->columns % 2); bit++)
          {
            pixel=(unsigned long) ((*p++ >> 4) & 0xf);
            indexes[x+bit]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x+bit]];
            q++;
          }
          break;
        }
        case 12:
        {
          for (x=0; x < (long) (image->columns-1); x+=2)
          {
            pixel=(unsigned long) (((*(p+1) >> 4) & 0xf) | (*p));
            indexes[x]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x]];
            q++;
            pixel=(unsigned long) (((*(p+1) & 0xf) << 8) | (*(p+2)));
            indexes[x+1]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x+1]];
            p+=3;
            q++;
          }
          for (bit=0; bit < (long) (image->columns % 2); bit++)
          {
            pixel=(unsigned long) (((*(p+1) >> 4) & 0xf) | (*p));
            indexes[x+bit]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x+bit]];
            q++;
          }
          break;
        }
        default:
          break;
      }
      if (x != 0)
        break;
    }
    case GrayQuantum:
    {
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushCharPixel(pixel,p);
            indexes[x]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x]];
            q++;
          }
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushShortPixel(pixel,p);
            indexes[x]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x]];
            q++;
          }
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushLongPixel(pixel,p);
            indexes[x]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x]];
            q++;
          }
          break;
        }
      }
      break;
    }
    case GrayAlphaQuark:
    {
      if (image->storage_class != PseudoClass)
        ThrowBinaryException(ImageError,"ColormappedImageRequired",
          image->filename);
      switch (image->depth)
      {
        case 1:
        {
          for (x=0; x < ((long) image->columns-3); x+=4)
          {
            for (bit=0; bit < 8; bit+=2)
            {
              pixel=(unsigned long)
                (((*p) & (1UL << (unsigned long) (7-bit))) != 0 ? 0x01 : 0x00);
              indexes[x+bit/2]=PushColormapIndex(image,pixel);
              *q=image->colormap[indexes[x+bit/2]];
              pixel=(unsigned long)
                (((*p) & (1UL << (unsigned long) (6-bit))) != 0 ? 0x01 : 0x00);
              q->opacity=pixel == 0x00 ? TransparentOpacity : OpaqueOpacity;
              q++;
            }
            p++;
          }
          for (bit=0; bit < (long) (image->columns % 4); bit+=2)
          {
            pixel=(unsigned long)
              (((*p) & (1UL << (unsigned long) (7-bit))) != 0 ? 0x01 : 0x00);
            indexes[x+bit/2]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x+bit/2]];
            pixel=(unsigned long)
              (((*p) & (1UL << (unsigned long) (6-bit))) != 0 ? 0x01 : 0x00);
            q->opacity=pixel == 0x00 ? TransparentOpacity : OpaqueOpacity;
            q++;
          }
          break;
        }
        case 2:
        {
          for (x=0; x < ((long) image->columns-1); x+=2)
          {
            pixel=(unsigned long) ((*p >> 6) & 0x03);
            indexes[x]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x]];
            pixel=(unsigned long) (MaxRGB*((*p >> 4) & 0x03)/4);
            q->opacity=(Quantum) pixel;
            q++;
            pixel=(unsigned long) ((*p >> 2) & 0x03);
            indexes[x+2]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x+2]];
            pixel=(unsigned long) (MaxRGB*((*p) & 0x03)/4);
            q->opacity=(Quantum) pixel;
            p++;
            q++;
          }
          break;
        }
        case 4:
        {
          for (x=0; x < (long) image->columns; x++)
          {
            pixel=(unsigned long) ((*p >> 4) & 0xf);
            indexes[x]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x]];
            pixel=(unsigned long) (MaxRGB*((*p) & 0xf)/15);
            q->opacity=(Quantum) (MaxRGB-pixel);
            p++;
            q++;
          }
          break;
        }
        case 12:
        {
          for (x=0; x < (long) image->columns; x++)
          {
            pixel=(unsigned long) (((*(p+1) >> 4) & 0xf) | (*p));
            indexes[x]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x]];
            pixel=(unsigned long) (((*(p+1) & 0xf) << 8) | (*(p+2)));
            q->opacity=(Quantum) (MaxRGB*pixel/1024);
            p+=3;
            q++;
          }
          break;
        }
        default:
          break;
      }
      if (x != 0)
        break;
    }
    case GrayAlphaQuantum:
    {
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushCharPixel(pixel,p);
            indexes[x]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x]];
            PushCharPixel(pixel,p);
            q->opacity=(Quantum) (MaxRGB-pixel);
            q++;
          }
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushShortPixel(pixel,p);
            indexes[x]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x]];
            PushShortPixel(pixel,p);
            q->opacity=MaxRGB-ScaleShortToQuantum(pixel);
            q++;
          }
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushLongPixel(pixel,p);
            indexes[x]=PushColormapIndex(image,pixel);
            *q=image->colormap[indexes[x]];
            PushLongPixel(pixel,p);
            q->opacity=MaxRGB-ScaleLongToQuantum(pixel);
            q++;
          }
          break;
        }
      }
      break;
    }
    case RedQuantum:
    case CyanQuantum:
    {
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushCharPixel(pixel,p);
            q->red=ScaleCharToQuantum(pixel);
            q++;
          }
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushShortPixel(pixel,p);
            q->red=ScaleShortToQuantum(pixel);
            q++;
          }
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushLongPixel(pixel,p);
            q->red=ScaleLongToQuantum(pixel);
            q++;
          }
          break;
        }
      }
      break;
    }
    case GreenQuantum:
    case MagentaQuantum:
    {
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushCharPixel(pixel,p);
            q->green=ScaleCharToQuantum(pixel);
            q++;
          }
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushShortPixel(pixel,p);
            q->green=ScaleShortToQuantum(pixel);
            q++;
          }
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushLongPixel(pixel,p);
            q->green=ScaleLongToQuantum(pixel);
            q++;
          }
          break;
        }
      }
      break;
    }
    case BlueQuantum:
    case YellowQuantum:
    {
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushCharPixel(pixel,p);
            q->blue=ScaleCharToQuantum(pixel);
            q++;
          }
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushShortPixel(pixel,p);
            q->blue=ScaleShortToQuantum(pixel);
            q++;
          }
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushLongPixel(pixel,p);
            q->blue=ScaleLongToQuantum(pixel);
            q++;
          }
          break;
        }
      }
      break;
    }
    case AlphaQuantum:
    {
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushCharPixel(pixel,p);
            q->opacity=MaxRGB-ScaleCharToQuantum(pixel);
            q++;
          }
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushShortPixel(pixel,p);
            q->opacity=MaxRGB-ScaleShortToQuantum(pixel);
            q++;
          }
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushLongPixel(pixel,p);
            q->opacity=MaxRGB-ScaleLongToQuantum(pixel);
            q++;
          }
          break;
        }
      }
      break;
    }
    case BlackQuantum:
    {
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushCharPixel(pixel,p);
            indexes[x]=ScaleCharToQuantum(pixel);
          }
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushShortPixel(pixel,p);
            indexes[x]=ScaleShortToQuantum(pixel);
          }
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushLongPixel(pixel,p);
            indexes[x]=ScaleLongToQuantum(pixel);
          }
          break;
        }
      }
      break;
    }
    case RGBQuantum:
    default:
    {
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushCharPixel(pixel,p);
            q->red=ScaleCharToQuantum(pixel);
            PushCharPixel(pixel,p);
            q->green=ScaleCharToQuantum(pixel);
            PushCharPixel(pixel,p);
            q->blue=ScaleCharToQuantum(pixel);
            q++;
          }
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushShortPixel(pixel,p);
            q->red=ScaleShortToQuantum(pixel);
            PushShortPixel(pixel,p);
            q->green=ScaleShortToQuantum(pixel);
            PushShortPixel(pixel,p);
            q->blue=ScaleShortToQuantum(pixel);
            q++;
          }
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushLongPixel(pixel,p);
            q->red=ScaleLongToQuantum(pixel);
            PushLongPixel(pixel,p);
            q->green=ScaleLongToQuantum(pixel);
            PushLongPixel(pixel,p);
            q->blue=ScaleLongToQuantum(pixel);
            q++;
          }
          break;
        }
      }
      break;
    }
    case RGBAQuantum:
    {
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushCharPixel(pixel,p);
            q->red=ScaleCharToQuantum(pixel);
            PushCharPixel(pixel,p);
            q->green=ScaleCharToQuantum(pixel);
            PushCharPixel(pixel,p);
            q->blue=ScaleCharToQuantum(pixel);
            PushCharPixel(pixel,p);
            q->opacity=MaxRGB-ScaleCharToQuantum(pixel);
            q++;
          }
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushShortPixel(pixel,p);
            q->red=ScaleShortToQuantum(pixel);
            PushShortPixel(pixel,p);
            q->green=ScaleShortToQuantum(pixel);
            PushShortPixel(pixel,p);
            q->blue=ScaleShortToQuantum(pixel);
            PushShortPixel(pixel,p);
            q->opacity=MaxRGB-ScaleShortToQuantum(pixel);
            q++;
          }
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushLongPixel(pixel,p);
            q->red=ScaleLongToQuantum(pixel);
            PushLongPixel(pixel,p);
            q->green=ScaleLongToQuantum(pixel);
            PushLongPixel(pixel,p);
            q->blue=ScaleLongToQuantum(pixel);
            PushLongPixel(pixel,p);
            q->opacity=MaxRGB-ScaleLongToQuantum(pixel);
            q++;
          }
          break;
        }
      }
      break;
    }
    case CMYKQuantum:
    {
      if (image->colorspace != CMYKColorspace)
        ThrowBinaryException(ImageError,"ColorSeparatedImageRequired",
          image->filename);
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushCharPixel(pixel,p);
            q->red=ScaleCharToQuantum(pixel);
            PushCharPixel(pixel,p);
            q->green=ScaleCharToQuantum(pixel);
            PushCharPixel(pixel,p);
            q->blue=ScaleCharToQuantum(pixel);
            PushCharPixel(pixel,p);
            indexes[x]=ScaleCharToQuantum(pixel);
            q++;
          }
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushShortPixel(pixel,p);
            q->red=ScaleShortToQuantum(pixel);
            PushShortPixel(pixel,p);
            q->green=ScaleShortToQuantum(pixel);
            PushShortPixel(pixel,p);
            q->blue=ScaleShortToQuantum(pixel);
            PushShortPixel(pixel,p);
            indexes[x]=ScaleShortToQuantum(pixel);
            q++;
          }
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushLongPixel(pixel,p);
            q->red=ScaleLongToQuantum(pixel);
            PushLongPixel(pixel,p);
            q->green=ScaleLongToQuantum(pixel);
            PushLongPixel(pixel,p);
            q->blue=ScaleLongToQuantum(pixel);
            PushLongPixel(pixel,p);
            indexes[x]=ScaleLongToQuantum(pixel);
            q++;
          }
          break;
        }
      }
      break;
    }
    case CMYKAQuantum:
    {
      if (image->colorspace != CMYKColorspace)
        ThrowBinaryException(ImageError,"ColorSeparatedImageRequired",
          image->filename);
      switch (depth)
      {
        case 8:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushCharPixel(pixel,p);
            q->red=ScaleCharToQuantum(pixel);
            PushCharPixel(pixel,p);
            q->green=ScaleCharToQuantum(pixel);
            PushCharPixel(pixel,p);
            q->blue=ScaleCharToQuantum(pixel);
            PushCharPixel(pixel,p);
            indexes[x]=ScaleCharToQuantum(pixel);
            PushCharPixel(pixel,p);
            q->opacity=MaxRGB-ScaleCharToQuantum(pixel);
            q++;
          }
          break;
        }
        case 16:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushShortPixel(pixel,p);
            q->red=ScaleShortToQuantum(pixel);
            PushShortPixel(pixel,p);
            q->green=ScaleShortToQuantum(pixel);
            PushShortPixel(pixel,p);
            q->blue=ScaleShortToQuantum(pixel);
            PushShortPixel(pixel,p);
            indexes[x]=ScaleShortToQuantum(pixel);
            PushShortPixel(pixel,p);
            q->opacity=MaxRGB-ScaleShortToQuantum(pixel);
            q++;
          }
          break;
        }
        case 32:
        {
          for (x=0; x < (long) number_pixels; x++)
          {
            PushLongPixel(pixel,p);
            q->red=ScaleLongToQuantum(pixel);
            PushLongPixel(pixel,p);
            q->green=ScaleLongToQuantum(pixel);
            PushLongPixel(pixel,p);
            q->blue=ScaleLongToQuantum(pixel);
            PushLongPixel(pixel,p);
            indexes[x]=ScaleLongToQuantum(pixel);
            PushLongPixel(pixel,p);
            q->opacity=MaxRGB-ScaleLongToQuantum(pixel);
            q++;
          }
          break;
        }
      }
      break;
    }
  }
  if (image->endian == LSBEndian)
    switch (depth)
    {
      case 16:
      {
        MSBOrderShort((unsigned char *) source,(size_t) (p-source));
        break;
      }
      case 32:
      {
        MSBOrderLong((unsigned char *) source,(size_t) (p-source));
        break;
      }
      default:
        break;
    }
  return(MagickTrue);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   R e a d I m a g e                                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  ReadImage() reads an image or image sequence from a file or file handle.
%  The method returns a NULL if there is a memory shortage or if the image
%  cannot be read.  On failure, a NULL image is returned and exception
%  describes the reason for the failure.
%
%  The format of the ReadImage method is:
%
%      Image *ReadImage(const ImageInfo *image_info,ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o image_info: Read the image defined by the file or filename members of
%      this structure.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/
MagickExport Image *ReadImage(const ImageInfo *image_info,
  ExceptionInfo *exception)
{
  char
    filename[MaxTextExtent],
    magick[MaxTextExtent];

  const char
    *comment,
    *delay,
    *dispose,
    *label,
    *page;

  const DelegateInfo
    *delegate_info;

  const ImageAttribute
    *attribute;

  const MagickInfo
    *magick_info;

  GeometryInfo
    geometry_info;

  Image
    *image,
    *next;

  ImageInfo
    *read_info;

  /*
    Determine image type from filename prefix or suffix (e.g. image.jpg).
  */
  assert(image_info != (ImageInfo *) NULL);
  assert(image_info->signature == MagickSignature);
  assert(image_info->filename != (char *) NULL);
  if (image_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),image_info->filename);
  assert(exception != (ExceptionInfo *) NULL);
  (void) SetExceptionInfo(exception,UndefinedException);
  if (*image_info->filename == '@')
    return(ReadImages(image_info,exception));
  read_info=CloneImageInfo(image_info);
  (void) SetImageInfo(read_info,MagickFalse,exception);
  (void) CopyMagickString(filename,read_info->filename,MaxTextExtent);
  (void) CopyMagickString(magick,read_info->magick,MaxTextExtent);
  /*
    Call appropriate image reader based on image type.
  */
  magick_info=GetMagickInfo(read_info->magick,exception);
  if (exception->severity > UndefinedException)
    {
      delegate_info=GetDelegateInfo(read_info->magick,(char *) NULL,exception);
      if (delegate_info == (const DelegateInfo *) NULL)
        {
          DestroyImageInfo(read_info);
          return(MagickFalse);
        }
    }
  if ((magick_info != (const MagickInfo *) NULL) &&
      (GetMagickSeekableStream(magick_info) != MagickFalse))
    {
      MagickBooleanType
        status;

      image=AllocateImage(read_info);
      if (image == (Image *) NULL)
        return(MagickFalse);
      (void) CopyMagickString(image->filename,read_info->filename,
        MaxTextExtent);
      status=OpenBlob(image_info,image,ReadBinaryBlobMode,exception);
      if (status == MagickFalse)
        {
          DestroyImageInfo(read_info);
          DestroyImage(image);
          return(MagickFalse);
        }
      if (IsBlobSeekable(image) == MagickFalse)
        {
          /*
            Coder requires a seekable stream.
          */
          *read_info->filename='\0';
          status=ImageToFile(image,read_info->filename,exception);
          if (status == MagickFalse)
            {
              CloseBlob(image);
              DestroyImageInfo(read_info);
              DestroyImage(image);
              return(MagickFalse);
            }
          read_info->temporary=MagickTrue;
        }
      CloseBlob(image);
      DestroyImage(image);
    }
  image=(Image *) NULL;
  if ((magick_info != (const MagickInfo *) NULL) &&
      (GetMagickDecoder(magick_info) != (DecoderHandler *) NULL))
    {
      if (GetMagickThreadSupport(magick_info) == MagickFalse)
        AcquireSemaphoreInfo(&constitute_semaphore);
      image=GetMagickDecoder(magick_info)(read_info,exception);
      if (GetMagickThreadSupport(magick_info) == MagickFalse)
        RelinquishSemaphoreInfo(&constitute_semaphore);
    }
  else
    {
      delegate_info=GetDelegateInfo(read_info->magick,(char *) NULL,exception);
      if (delegate_info == (const DelegateInfo *) NULL)
        {
          if (IsAccessible(read_info->filename) == MagickFalse)
            (void) ThrowMagickException(exception,GetMagickModule(),
              FileOpenError,"UnableToOpenFile",read_info->filename,
              strerror(errno));
          else
            (void) ThrowMagickException(exception,GetMagickModule(),
              MissingDelegateError,"NoDecodeDelegateForThisImageFormat",
              read_info->filename);
          if (read_info->temporary != MagickFalse)
            (void) RelinquishUniqueFileResource(read_info->filename);
          DestroyImageInfo(read_info);
          return((Image *) NULL);
        }
      /*
        Let our decoding delegate process the image.
      */
      image=AllocateImage(read_info);
      if (image == (Image *) NULL)
        {
          DestroyImageInfo(read_info);
          return((Image *) NULL);
        }
      (void) SetExceptionInfo(exception,UndefinedException);
      (void) CopyMagickString(image->filename,read_info->filename,
        MaxTextExtent);
      *read_info->filename='\0';
      (void) InvokeDelegate(read_info,image,read_info->magick,(char *) NULL,
        exception);
      DestroyImageList(image);
      image=(Image *) NULL;
      read_info->temporary=MagickTrue;
      (void) SetImageInfo(read_info,MagickFalse,exception);
      magick_info=GetMagickInfo(read_info->magick,exception);
      if ((magick_info == (const MagickInfo *) NULL) ||
          (GetMagickDecoder(magick_info) == (DecoderHandler *) NULL))
        {
          if (IsAccessible(read_info->filename) == MagickFalse)
            (void) ThrowMagickException(exception,GetMagickModule(),
              FileOpenError,"UnableToOpenFile",read_info->filename,
              strerror(errno));
          else
            (void) ThrowMagickException(exception,GetMagickModule(),
              MissingDelegateError,"NoDecodeDelegateForThisImageFormat",
              read_info->filename);
          DestroyImageInfo(read_info);
          return((Image *) NULL);
        }
      if (GetMagickThreadSupport(magick_info) == MagickFalse)
        AcquireSemaphoreInfo(&constitute_semaphore);
      image=(Image *) (GetMagickDecoder(magick_info))(read_info,exception);
      if (GetMagickThreadSupport(magick_info) == MagickFalse)
        RelinquishSemaphoreInfo(&constitute_semaphore);
    }
  if (read_info->temporary != MagickFalse)
    {
      (void) RelinquishUniqueFileResource(read_info->filename);
      read_info->temporary=MagickFalse;
      if (image != (Image *) NULL)
        (void) CopyMagickString(image->filename,filename,MaxTextExtent);
    }
  if (image == (Image *) NULL)
    {
      DestroyImageInfo(read_info);
      return(image);
    }
  if (IsBlobTemporary(image) != MagickFalse)
    (void) RelinquishUniqueFileResource(read_info->filename);
  if ((image->next != (Image *) NULL) &&
      (IsSceneGeometry(read_info->scenes,MagickFalse) != MagickFalse))
    {
      char
        *q;

      Image
        *read_image,
        *scenes;

      long
        quantum;

      register char
        *p;

      register long
        i;

      unsigned long
        first,
        last;

      /*
        User specified scenes (e.g. image.miff[1,3-5,7-6,2]).
      */
      scenes=NewImageList();
      p=read_info->scenes;
      for (q=p; *q != '\0'; p++)
      {
        while ((isspace((int) ((unsigned char) *p)) != 0) || (*p == ','))
          p++;
        first=strtoul(p,&q,10);
        last=first;
        while (isspace((int) ((unsigned char) *q)) != 0)
          q++;
        if (*q == '-')
          last=strtoul(q+1,&q,10);
        quantum=first > last ? -1 : 1;
        for (p=q; first != (last+quantum); first+=quantum)
        {
          i=0;
          for (next=image; next != (Image *) NULL; next=next->next)
          {
            if (next->scene != 0)
              i=(long) next->scene;
            if (i == (long) first)
              {
                read_image=CloneImage(next,0,0,MagickTrue,exception);
                if (read_image == (Image *) NULL)
                  break;
                AppendImageToList(&scenes,read_image);
              }
            i++;
          }
        }
      }
      if (scenes != (Image *) NULL)
        image=GetFirstImageInList(scenes);
      else
        (void) ThrowMagickException(exception,GetMagickModule(),OptionError,
          "SubimageSpecificationReturnsNoImages",read_info->filename);
    }
  if (GetBlobError(image) != MagickFalse)
    {
      (void) ThrowMagickException(exception,GetMagickModule(),CorruptImageError,
        "AnErrorHasOccurredReadingFromFile",read_info->filename,
        strerror(errno));
      DestroyImageList(image);
      DestroyImageInfo(read_info);
      return((Image *) NULL);
    }
  for (next=image; next; next=next->next)
  {
    next->taint=MagickFalse;
    (void) CopyMagickString(next->magick,magick,MaxTextExtent);
    (void) CopyMagickString(next->magick_filename,filename,MaxTextExtent);
    if (IsBlobTemporary(image) != MagickFalse)
      (void) CopyMagickString(next->filename,filename,MaxTextExtent);
    attribute=GetImageAttribute(next,"EXIF:XResolution");
    if ((attribute != (const ImageAttribute *) NULL) &&
        (attribute->value != (char *) NULL))
      {
        geometry_info.rho=next->x_resolution;
        geometry_info.sigma=1.0;
        (void) ParseGeometry(attribute->value,&geometry_info);
        if (geometry_info.sigma != 0)
          next->x_resolution=geometry_info.rho/geometry_info.sigma;
        (void) SetImageAttribute(next,"EXIF:XResolution",(char *) NULL);
      }
    attribute=GetImageAttribute(next,"EXIF:YResolution");
    if ((attribute != (const ImageAttribute *) NULL) &&
        (attribute->value != (char *) NULL))
      {
        geometry_info.rho=next->y_resolution;
        geometry_info.sigma=1.0;
        (void) ParseGeometry(attribute->value,&geometry_info);
        if (geometry_info.sigma != 0)
          next->y_resolution=geometry_info.rho/geometry_info.sigma;
        (void) SetImageAttribute(next,"EXIF:YResolution",(char *) NULL);
      }
    page=GetImageOption(image_info,"page");
    if (page != (char *) NULL)
      {
        char
          *geometry;

        geometry=GetPageGeometry(page);
        (void) GetGeometry(geometry,&next->page.x,&next->page.y,
          &next->page.width,&next->page.height);
        geometry=(char *) RelinquishMagickMemory(geometry);
      }
    delay=GetImageOption(image_info,"delay");
    if (delay != (char *) NULL)
      {
        double
          maximum_delay,
          minimum_delay;

        int
          count;

        count=sscanf(delay,"%lf-%lf",&minimum_delay,&maximum_delay);
        if (count == 1)
          next->delay=(unsigned long) minimum_delay;
        else
          {
            if ((double) next->delay < minimum_delay)
              next->delay=(unsigned long) (minimum_delay+0.5);
            if ((double) next->delay > maximum_delay)
              next->delay=(unsigned long) (maximum_delay+0.5);
          }
      }
    dispose=GetImageOption(image_info,"dispose");
    if (dispose != (char *) NULL)
      next->dispose=(DisposeType)
        ParseMagickOption(MagickDisposeOptions,MagickFalse,dispose);
    comment=GetImageOption(image_info,"comment");
    if (comment != (char *) NULL)
      {
        (void) SetImageAttribute(next,"comment",(char *) NULL);
        (void) SetImageAttribute(next,"comment",comment);
      }
    label=GetImageOption(image_info,"label");
    if (label != (char *) NULL)
      {
        (void) SetImageAttribute(next,"label",(char *) NULL);
        (void) SetImageAttribute(next,"label",label);
      }
    if (next->magick_columns == 0)
      next->magick_columns=next->columns;
    if (next->magick_rows == 0)
      next->magick_rows=next->rows;
    if (image_info->verbose != MagickFalse)
      (void) DescribeImage(next,stdout,MagickFalse);
  }
  DestroyImageInfo(read_info);
  return(image);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
+   R e a d I m a g e s                                                       %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  ReadImages() reads a list of image names from a file and then returns the
%  images as a linked list.
%
%  The format of the ReadImage method is:
%
%      Image *ReadImages(const ImageInfo *image_info,ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o image: Method ReadImage returns a pointer to the image after
%      reading.  A null image is returned if there is a memory shortage or
%      if the image cannot be read.
%
%    o image_info: The list of filenames are defined in the filename member of
%      this structure.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/
static Image *ReadImages(const ImageInfo *image_info,ExceptionInfo *exception)
{
  char
    *command,
    **images;

  Image
    *image;

  ImageInfo
    *read_info;

  int
    number_images;

  register Image
    *next;

  register long
    i;

  size_t
    length;

  /*
    Read image list from a file.
  */
  assert(image_info != (ImageInfo *) NULL);
  assert(image_info->signature == MagickSignature);
  if (image_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),image_info->filename);
  assert(exception != (ExceptionInfo *) NULL);
  command=(char *) FileToBlob(image_info->filename+1,&length,exception);
  if (command == (char *) NULL)
    return((Image *) NULL);
  StripString(command);
  images=StringToArgv(command,&number_images);
  command=(char *) RelinquishMagickMemory(command);
  /*
    Read the images into a linked list.
  */
  image=(Image *) NULL;
  read_info=CloneImageInfo(image_info);
  for (i=1; i < number_images; i++)
  {
    (void) CopyMagickString(read_info->filename,images[i],MaxTextExtent);
    next=ReadImage(read_info,exception);
    if (next == (Image *) NULL)
      continue;
    if (image == (Image *) NULL)
      image=next;
    else
      {
        register Image
          *q;

        /*
          Link image into image list.
        */
        for (q=image; q->next != (Image *) NULL; q=q->next);
        next->previous=q;
        q->next=next;
      }
  }
  DestroyImageInfo(read_info);
  for (i=1; i < number_images; i++)
    images[i]=(char *) RelinquishMagickMemory(images[i]);
  images=(char **) RelinquishMagickMemory(images);
  return(image);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   R e a d I n l i n e I m a g e                                             %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  ReadInlineImage() reads a Base64-encoded inline image or image sequence.
%  The method returns a NULL if there is a memory shortage or if the image
%  cannot be read.  On failure, a NULL image is returned and exception
%  describes the reason for the failure.
%
%  The format of the ReadInlineImage method is:
%
%      Image *ReadInlineImage(const ImageInfo *image_info,const char *content,
%        ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o image_info: The image info.
%
%    o content: The image encoded in Base64.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/
MagickExport Image *ReadInlineImage(const ImageInfo *image_info,
  const char *content,ExceptionInfo *exception)
{
  Image
    *image;

  MonitorHandler
    handler;

  unsigned char
    *blob;

  size_t
    length;

  register const char
    *p;

  (void) SetExceptionInfo(exception,UndefinedException);
  image=(Image *) NULL;
  for (p=content; (*p != ',') && (*p != '\0'); p++);
  if (*p == '\0')
    ThrowReaderException(CorruptImageWarning,"CorruptImage");
  p++;
  length=0;
  blob=Base64Decode(p,&length);
  if (length == 0)
    ThrowReaderException(CorruptImageWarning,"CorruptImage");
  handler=SetMonitorHandler((MonitorHandler) NULL);
  image=BlobToImage(image_info,blob,length,exception);
  (void) SetMonitorHandler(handler);
  blob=(unsigned char *) RelinquishMagickMemory(blob);
  return(image);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   W r i t e I m a g e                                                       %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Use Write() to write an image or an image sequence to a file or
%  filehandle.  If writing to a file on disk, the name is defined by the
%  filename member of the image structure.  Write() returns 0 is there is a
%  memory shortage or if the image cannot be written.  Check the exception
%  member of image to determine the cause for any failure.
%
%  The format of the WriteImage method is:
%
%      MagickBooleanType WriteImage(const ImageInfo *image_info,Image *image)
%
%  A description of each parameter follows:
%
%    o image_info: The image info.
%
%    o image: The image.
%
%
*/
MagickExport MagickBooleanType WriteImage(const ImageInfo *image_info,
  Image *image)
{
  char
    filename[MaxTextExtent];

  const DelegateInfo
    *delegate_info;

  const MagickInfo
    *magick_info;

  ImageInfo
    *write_info;

  MagickBooleanType
    status;

  /*
    Determine image type from filename prefix or suffix (e.g. image.jpg).
  */
  assert(image_info != (ImageInfo *) NULL);
  assert(image_info->signature == MagickSignature);
  assert(image_info->filename != (char *) NULL);
  if (image->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),image_info->filename);
  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  write_info=CloneImageInfo(image_info);
  (void) CopyMagickString(write_info->filename,image->filename,MaxTextExtent);
  (void) CopyMagickString(write_info->magick,image->magick,MaxTextExtent);
  (void) SetImageInfo(write_info,MagickTrue,&image->exception);
  (void) CopyMagickString(filename,image->filename,MaxTextExtent);
  (void) CopyMagickString(image->filename,write_info->filename,MaxTextExtent);
  (void) SetExceptionInfo(&image->exception,UndefinedException);
  if ((image->next == (Image *) NULL) && (image->previous == (Image *) NULL) &&
      (write_info->page == (char *) NULL) &&
      (IsTaintImage(image) == MagickFalse))
    {
      delegate_info=GetDelegateInfo(image->magick,write_info->magick,
        &image->exception);
      if ((delegate_info != (const DelegateInfo *) NULL) &&
          (GetDelegateMode(delegate_info) == 0) &&
          (IsAccessible(image->magick_filename) != MagickFalse))
        {
          /*
            Process image with bi-modal delegate.
          */
          (void) CopyMagickString(image->filename,image->magick_filename,
            MaxTextExtent);
          status=InvokeDelegate(write_info,image,image->magick,
            write_info->magick,&image->exception);
          DestroyImageInfo(write_info);
          (void) CopyMagickString(image->filename,filename,MaxTextExtent);
          return((MagickBooleanType) (status == MagickFalse));
        }
    }
  status=MagickFalse;
  magick_info=GetMagickInfo(write_info->magick,&image->exception);
  if ((magick_info != (const MagickInfo *) NULL) &&
      (GetMagickEncoder(magick_info) != (EncoderHandler *) NULL))
    {
      /*
        Call appropriate image writer based on image type.
      */
      if (GetMagickThreadSupport(magick_info) == MagickFalse)
        AcquireSemaphoreInfo(&constitute_semaphore);
      status=GetMagickEncoder(magick_info)(write_info,image);
      if (GetMagickThreadSupport(magick_info) == MagickFalse)
        RelinquishSemaphoreInfo(&constitute_semaphore);
    }
  else
    {
      delegate_info=GetDelegateInfo((char *) NULL,write_info->magick,
        &image->exception);
      if (delegate_info != (DelegateInfo *) NULL)
        {
          /*
            Process the image with delegate.
          */
          *write_info->filename='\0';
          status=InvokeDelegate(write_info,image,(char *) NULL,
            write_info->magick,&image->exception);
          (void) CopyMagickString(image->filename,filename,MaxTextExtent);
        }
      else
        {
          magick_info=GetMagickInfo(write_info->magick,&image->exception);
          if ((write_info->affirm == MagickFalse) &&
              (magick_info == (const MagickInfo *) NULL))
            magick_info=(MagickInfo *) GetMagickInfo(image->magick,
              &image->exception);
          if ((magick_info == (MagickInfo *) NULL) ||
              (GetMagickEncoder(magick_info) == (EncoderHandler *) NULL))
            (void) ThrowMagickException(&image->exception,GetMagickModule(),
              MissingDelegateError,"NoEncodeDelegateForThisImageFormat",
              image->filename);
          else
            {
              /*
                Call appropriate image writer based on image type.
              */
              if (GetMagickThreadSupport(magick_info) == MagickFalse)
                AcquireSemaphoreInfo(&constitute_semaphore);
              status=GetMagickEncoder(magick_info)(write_info,image);
              if (GetMagickThreadSupport(magick_info) == MagickFalse)
                RelinquishSemaphoreInfo(&constitute_semaphore);
            }
        }
    }
  if (GetBlobError(image) != MagickFalse)
    (void) ThrowMagickException(&image->exception,GetMagickModule(),
      CorruptImageError,"AnErrorHasOccurredWritingToFile",image->filename,
      strerror(errno));
  DestroyImageInfo(write_info);
  return(status);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   W r i t e I m a g e s                                                     %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  WriteImages() writes an image sequence.
%
%  The format of the WriteImages method is:
%
%      MagickBooleanType WriteImages(const ImageInfo *image_info,Image *image,
%        const char *filename,ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o image_info: The image info.
%
%    o images: The image list.
%
%    o filename: The image filename.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/
MagickExport MagickBooleanType WriteImages(const ImageInfo *image_info,
  Image *image,const char *filename,ExceptionInfo *exception)
{
  ImageInfo
    *write_info;

  MagickBooleanType
    status;

  register Image
    *p;

  /*
    Write converted images.
  */
  assert(image_info != (const ImageInfo *) NULL);
  assert(image_info->signature == MagickSignature);
  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (image->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),image->filename);
  assert(exception != (ExceptionInfo *) NULL);
  write_info=CloneImageInfo(image_info);
  if (filename != (const char *) NULL)
    {
      (void) CopyMagickString(write_info->filename,filename,MaxTextExtent);
      for (p=image; p != (Image *) NULL; p=p->next)
        (void) CopyMagickString(p->filename,filename,MaxTextExtent);
    }
  (void) SetImageInfo(write_info,MagickTrue,exception);
  (void) SetExceptionInfo(exception,UndefinedException);
  status=MagickTrue;
  for (p=image; p != (Image *) NULL; p=p->next)
  {
    status&=WriteImage(write_info,p);
    if (p->exception.severity != UndefinedException)
      InheritException(exception,&p->exception);
    GetImageException(p,exception);
    if (write_info->verbose != MagickFalse)
      (void) DescribeImage(p,stdout,MagickFalse);
    if (write_info->adjoin != MagickFalse)
      break;
  }
  DestroyImageInfo(write_info);
  return(status);
}
