/*

GObject wrapper for the gwibber service

Copyright 2010 Canonical Ltd.

Authors:
    David Barth <david.barth@canonical.com>

This program is free software: you can redistribute it and/or modify it 
under the terms of the GNU General Public License version 3, as published 
by the Free Software Foundation.

This program is distributed in the hope that it will be useful, but 
WITHOUT ANY WARRANTY; without even the implied warranties of 
MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along 
with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "me-service-gwibber.h"

#include <glib/gi18n.h>

#include <glib.h>
#include <glib/gprintf.h>

#include <gwibber.h>

/* gobject infrastructure */

typedef struct _MeGwibberServicePrivate MeGwibberServicePrivate;
struct _MeGwibberServicePrivate {
	GwibberService * me_gwibber_service;
	GwibberAccounts * me_gwibber_accounts;
	int status;
	gboolean has_configured_accounts;
	gchar * accounts_string;
};

/* Signals */
enum {
	STATUS_CHANGED,
	LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0 };

/* Prototypes */
static void me_gwibber_service_class_init (MeGwibberServiceClass *klass);
static void me_gwibber_service_init       (MeGwibberService *self);
static void me_gwibber_service_dispose    (GObject *object);
static void me_gwibber_service_finalize   (GObject *object);

static void me_gwibber_service_exists_cb (GwibberService * me_gwibber_service, gboolean exists, gpointer userdata);
static void me_gwibber_accounts_exists_cb (GwibberAccounts * me_gwibber_accounts, gboolean exists, gpointer userdata);
static void setup_service_proxies (MeGwibberService *self);
static void query_account_manager (MeGwibberService *self);

G_DEFINE_TYPE (MeGwibberService, me_gwibber_service, G_TYPE_OBJECT);

#define ME_GWIBBER_SERVICE_GET_PRIVATE(o) \
(G_TYPE_INSTANCE_GET_PRIVATE ((o), ME_GWIBBER_SERVICE_TYPE, MeGwibberServicePrivate))

static void
me_gwibber_service_class_init (MeGwibberServiceClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	g_type_class_add_private (klass, sizeof (MeGwibberServicePrivate));

	object_class->dispose = me_gwibber_service_dispose;
	object_class->finalize = me_gwibber_service_finalize;

	signals[STATUS_CHANGED]  = g_signal_new(ME_GWIBBER_SERVICE_SIGNAL_STATUS_CHANGED,
											G_TYPE_FROM_CLASS(klass),
											G_SIGNAL_RUN_LAST,
											G_STRUCT_OFFSET(MeGwibberServiceClass, status_changed),
											NULL, NULL,
											g_cclosure_marshal_VOID__UINT,
											G_TYPE_NONE, 1, G_TYPE_UINT);
	return;
}

static void
me_gwibber_service_init (MeGwibberService *self)
{
	MeGwibberServicePrivate * priv = ME_GWIBBER_SERVICE_GET_PRIVATE(self);

	priv->me_gwibber_service = NULL;
	priv->me_gwibber_accounts = NULL;
	priv->status = ME_GWIBBER_SERVICE_STATUS_NOT_RUNNING;
	priv->has_configured_accounts = FALSE;
	priv->accounts_string = NULL;

	setup_service_proxies(self);
	return;
}

static void
me_gwibber_service_dispose (GObject *object)
{
	MeGwibberServicePrivate * priv = ME_GWIBBER_SERVICE_GET_PRIVATE(object);

	if (priv->me_gwibber_service != NULL) {
		g_object_unref(priv->me_gwibber_service);
		priv->me_gwibber_service = NULL;
	}

	if (priv->me_gwibber_accounts != NULL) {
		g_object_unref(priv->me_gwibber_accounts);
		priv->me_gwibber_accounts = NULL;
	}

	if (priv->accounts_string != NULL) {
		g_free(priv->accounts_string);
		priv->accounts_string = NULL;
	}

	G_OBJECT_CLASS (me_gwibber_service_parent_class)->dispose (object);
	return;
}

static void
me_gwibber_service_finalize (GObject *object)
{
	G_OBJECT_CLASS (me_gwibber_service_parent_class)->finalize (object);
	return;
}

static void
query_account_manager (MeGwibberService *self)
{
	g_return_if_fail (IS_ME_GWIBBER_SERVICE (self));

	GList* accounts_list;
	GwibberAccount* account;

	MeGwibberServicePrivate * priv = ME_GWIBBER_SERVICE_GET_PRIVATE(self);

	if (priv->me_gwibber_accounts == NULL) {
		g_warning ("no accounts, can't query for accounts");
		goto reset_accounts_string;
	}
	accounts_list = gwibber_accounts_list (priv->me_gwibber_accounts);
	
        if (accounts_list == NULL) {
                g_warning ("failed to get accounts list");
				goto reset_accounts_string;
        }

	priv->has_configured_accounts = FALSE;
	GList *list = NULL;

	GList* account_it;
	for (account_it = accounts_list; account_it != NULL; account_it = account_it->next) {
		account = g_object_ref ((GwibberAccount*) account_it->data);
		if (g_strcmp0(gwibber_account_get_send_enabled (account), "1") == 0) {
			priv->has_configured_accounts = TRUE;
			const gchar * service_name = gwibber_account_get_service (account);
			if (service_name != NULL) {
				list = g_list_append (list, g_strdup (service_name));
			}
		}
	}

	if (priv->accounts_string != NULL) {
		g_free(priv->accounts_string);
		priv->accounts_string = NULL;
	}	

	guint len = g_list_length (list);
	if (len > 0) {
		GList *item0 = g_list_nth (list, 0);
		GList *item1 = g_list_nth (list, 1);
		if (len == 1)
			priv->accounts_string = g_strconcat (_("Post to: "),
												 item0->data,
												 "...", NULL);
		else if (len < 3)
			priv->accounts_string = g_strconcat (_("Post to: "),
												 item0->data,
												 ", ",
												 item1->data,
												 "...", NULL);
		else
			priv->accounts_string = g_strdup (_("Post to: multiple networks..."));
		g_debug ("accounts_string: %s", priv->accounts_string);
	}

	g_list_foreach (list, (GFunc) g_free, NULL);
	g_list_free (list);

	g_signal_emit (G_OBJECT (self),
				   ME_GWIBBER_SERVICE_SIGNAL_STATUS_CHANGED_ID,
				   0, priv->status, TRUE);

	return;

reset_accounts_string:

	if (priv->accounts_string != NULL) {
		g_free(priv->accounts_string);
		priv->accounts_string = NULL;
	}	
}

static void
accounts_changed (GwibberAccounts *me_gwibber_accounts, GHashTable *account_table, MeGwibberService *self)
{
	g_return_if_fail (IS_ME_GWIBBER_SERVICE (self));

	query_account_manager (self);

	return;
}


static void
setup_service_proxies (MeGwibberService *self)
{
	g_return_if_fail (IS_ME_GWIBBER_SERVICE (self));

	MeGwibberServicePrivate * priv = ME_GWIBBER_SERVICE_GET_PRIVATE(self);

	if (priv->me_gwibber_service == NULL) {
		priv->me_gwibber_service = gwibber_service_new ();
		if (priv->me_gwibber_service == NULL) {
			g_warning ("can't setup me_gwibber_service");
			return;
		}
	}

	g_signal_connect (priv->me_gwibber_service, "is-available", 
				G_CALLBACK(me_gwibber_service_exists_cb), ME_GWIBBER_SERVICE (self));

	if (priv->me_gwibber_accounts == NULL) {
		priv->me_gwibber_accounts = gwibber_accounts_new ();;
		if (priv->me_gwibber_accounts == NULL) {
			g_warning ("can't setup me_gwibber_accounts");
			return;
		}
	}

	g_signal_connect (priv->me_gwibber_accounts, "is-available", 
				G_CALLBACK(me_gwibber_accounts_exists_cb), ME_GWIBBER_SERVICE (self));
}

static void
me_gwibber_service_exists_cb (GwibberService * me_gwibber_service, gboolean exists, gpointer userdata)
{
	MeGwibberServicePrivate * priv = ME_GWIBBER_SERVICE_GET_PRIVATE(ME_GWIBBER_SERVICE (userdata));

	if (exists) {
		priv->status = ME_GWIBBER_SERVICE_STATUS_RUNNING;
	}


	if (!exists) {
		priv->status = ME_GWIBBER_SERVICE_STATUS_NOT_RUNNING;
		g_signal_emit (G_OBJECT (ME_GWIBBER_SERVICE (userdata)), ME_GWIBBER_SERVICE_SIGNAL_STATUS_CHANGED_ID, 0, priv->status, TRUE);
	}
	return;
}

static void
me_gwibber_accounts_exists_cb (GwibberAccounts * me_gwibber_accounts, gboolean exists, gpointer userdata)
{
	MeGwibberServicePrivate * priv = ME_GWIBBER_SERVICE_GET_PRIVATE(ME_GWIBBER_SERVICE (userdata));
	if (exists) {
		if (priv->me_gwibber_accounts == NULL) {
			priv->me_gwibber_accounts = gwibber_accounts_new ();;
			if (priv->me_gwibber_accounts == NULL) {
				g_warning ("can't setup me_gwibber_accounts");
				return;
			}
		}

		query_account_manager (ME_GWIBBER_SERVICE (userdata));
		g_signal_emit (G_OBJECT (ME_GWIBBER_SERVICE (userdata)), ME_GWIBBER_SERVICE_SIGNAL_STATUS_CHANGED_ID, 0, priv->status, TRUE);
		g_signal_connect (priv->me_gwibber_accounts, "account-updated",
				G_CALLBACK(accounts_changed), ME_GWIBBER_SERVICE (userdata));
	}
	return;
}
		

MeGwibberService * me_gwibber_service_new (void){
	return ME_GWIBBER_SERVICE (g_object_new (ME_GWIBBER_SERVICE_TYPE, NULL));
}

void
me_gwibber_service_send (MeGwibberService *self, gchar * msg)
{
	g_return_if_fail (IS_ME_GWIBBER_SERVICE (self));

	MeGwibberServicePrivate * priv = ME_GWIBBER_SERVICE_GET_PRIVATE(self);

	if (priv->me_gwibber_service == NULL) {
		setup_service_proxies (self);
		if (priv->me_gwibber_service == NULL) {
			g_warning ("can't setup me_gwibber_service");
			return;
		}
	}

	gwibber_service_send_message (priv->me_gwibber_service, msg);

	return;
}

MeGwibberService *
me_gwibber_service_get (void)
{
  static MeGwibberService *singleton = NULL;

  if (! singleton) {
	  singleton = me_gwibber_service_new ();
  }	

  return singleton;
}

gboolean
me_gwibber_service_has_configured_accounts (MeGwibberService *self)
{
	g_return_val_if_fail (IS_ME_GWIBBER_SERVICE (self), FALSE);

	MeGwibberServicePrivate * priv = ME_GWIBBER_SERVICE_GET_PRIVATE(self);

	return priv->has_configured_accounts;
}

const gchar *
me_gwibber_service_get_accounts_string (MeGwibberService *self)
{
	g_return_val_if_fail (IS_ME_GWIBBER_SERVICE (self), FALSE);

	MeGwibberServicePrivate * priv = ME_GWIBBER_SERVICE_GET_PRIVATE(self);

	return priv->accounts_string;
}
