
/****************************************************************************
 *
 * All portions copyright their respective authors.  All rights reserved.
 *
 * This file is part of IVMan (ivm).
 *
 * This file may be distributed under the terms of the Q Public License
 * as defined by Troll Tech AS of Norway and appearing in the file
 * LICENSE.QPL included in the packaging of this file.
 * 
 * See http://www.troll.no/qpl for QPL licensing information.
 *
 * $Id: IvmConfigCommon.c,v 1.4 2005/05/12 01:42:34 ro_han Exp $
 *****************************************************************************/

// Common functions which more than one IvmConfig* needs to call.

#include "IvmConfigCommon.h"

#include <libhal.h>
#include <libxml/xmlstring.h>
#include <dbus/dbus.h>
#include <stdlib.h>

LibHalContext *hal_ctx;

gboolean
hal_xml_property_matches(xmlChar const *const property,
                         xmlChar const *const value, char const *udi)
{
    if (udi == NULL)
        return FALSE;

    DBusError dbus_error;
    dbus_error_init(&dbus_error);

    int length = xmlStrlen(property) - 4;
    xmlChar *devProperty = xmlStrsub(property, 4, length);

    if (libhal_device_property_exists(hal_ctx, udi, devProperty, &dbus_error))
    {
        int type = libhal_device_get_property_type(hal_ctx, udi, devProperty,
                                                   &dbus_error);

        if (type == DBUS_TYPE_STRING)
        {
            char *propVal =
                (char *) libhal_device_get_property_string(hal_ctx, udi,
                                                           devProperty,
                                                           &dbus_error);

            if (!xmlStrcmp(value, (const xmlChar *) propVal))
            {
                libhal_free_string(propVal);
                return TRUE;
            } else
            {
                libhal_free_string(propVal);
                return FALSE;
            }
        }                       // string


        else if (type == DBUS_TYPE_BOOLEAN)
        {
            // have to cast to boolean...
            dbus_bool_t value_b;

            if (!xmlStrcmp(value, (const xmlChar *) "true"))
                value_b = TRUE;
            else
                value_b = FALSE;

            return (gboolean) (libhal_device_get_property_bool
                               (hal_ctx, udi, devProperty,
                                &dbus_error) == value_b);
        }                       // boolean


        else if (type == DBUS_TYPE_UINT32 || type == DBUS_TYPE_UINT64)
        {
            dbus_uint64_t value_i = atoll((const char *) value);

            return (gboolean)
                (libhal_device_get_property_uint64
                 (hal_ctx, udi, devProperty, &dbus_error) == value_i);
        }                       // unsigned int


        else if (type == DBUS_TYPE_INT32 || type == DBUS_TYPE_INT64)
        {
            dbus_int64_t value_i = atoll((const char *) value);

            return (gboolean) (libhal_device_get_property_int(hal_ctx, udi, devProperty, &dbus_error) == value_i);
        }  // signed int

        // we don't handle other types yet
        DEBUG("Warning: unhandled HAL type encountered!");
    } // if property exists

    ivm_check_dbus_error(&dbus_error);

    return FALSE;
}

gboolean ivm_device_is_mountable(char const *udi)
{
    if (udi == NULL)
        return FALSE;

    DBusError dbus_error;
    dbus_error_init(&dbus_error);

    /* Necessary conditions for a volume to be mountable: (all of these
       must be true)
       1) device must be block device
       2) device must be volume
       3) volume.policy.should_mount == TRUE OR
          storage.policy.should_mount == TRUE on block.storage_device
     */

    if (!libhal_device_property_exists
        (hal_ctx, udi, "block.device", &dbus_error))
        return FALSE;

    char *device = (char *) libhal_device_get_property_string(hal_ctx, udi,
                                                              "block.device",
                                                              &dbus_error);

    DEBUG("%s is %s", udi, device, &dbus_error);
    // if we get here, condition one is satisfied


    if (!libhal_device_property_exists
        (hal_ctx, udi, "block.is_volume", &dbus_error))
    {
        DEBUG
            ("%s won't be mounted because it does not specify whether or not it is a volume",
             device);
        libhal_free_string(device);
        return FALSE;
    } else if (FALSE ==
               libhal_device_get_property_bool(hal_ctx, udi,
                                               "block.is_volume",
                                               &dbus_error))
    {
        DEBUG("%s can't be mounted because it is not a volume", device);
        libhal_free_string(device);
        return FALSE;
    }
    // if we get here, condition two is satisfied


    if (!libhal_device_property_exists
        (hal_ctx, udi, "volume.policy.should_mount", &dbus_error))
    {
        if (!libhal_device_property_exists
            (hal_ctx, udi, "block.storage_device", &dbus_error))
        {
            DEBUG
                ("%s won't be mounted because no mount policy or parent device is specified",
                 device);
            libhal_free_string(device);
            return FALSE;
        }

        char *new_udi = (char *) libhal_device_get_property_string(hal_ctx,
                                                                   udi,
                                                                   "block.storage_device",
                                                                   &dbus_error);

        if (!libhal_device_property_exists
            (hal_ctx, new_udi, "storage.policy.should_mount", &dbus_error))
        {
            DEBUG
                ("%s won't be mounted because no mount policy was specified on either the volume or parent device",
                 device);
            libhal_free_string(device);
            libhal_free_string(new_udi);
            return FALSE;
        } else if (FALSE ==
                   libhal_device_get_property_bool(hal_ctx,
                                                   new_udi,
                                                   "storage.policy.should_mount",
                                                   &dbus_error))
        {
            DEBUG
                ("%s won't be mounted because no mount policy was specified on volume and mount policy on parent device says not to mount",
                 device);
            libhal_free_string(device);
            libhal_free_string(new_udi);
            return FALSE;
        }
        libhal_free_string(new_udi);
    } else if (FALSE ==
               libhal_device_property_exists(hal_ctx, udi,
                                             "volume.policy.should_mount",
                                             &dbus_error))
    {
        DEBUG
            ("%s won't be mounted because mount policy on volume says not to mount",
             device);
        libhal_free_string(device);
        return FALSE;
    }
    // if we get here, condition three is satisfied



    /* Sufficient conditions for a volume to _not_ be mountable: (just
       one of these being true is sufficient)
       1) volume is a disc with no data */
    if (libhal_device_property_exists
        (hal_ctx, udi, "volume.disc.has_data", &dbus_error)
        && (FALSE ==
            libhal_device_get_property_bool(hal_ctx, udi,
                                            "volume.disc.has_data",
                                            &dbus_error)))
    {
        DEBUG
            ("%s can't be mounted because it is a disc with no data tracks",
             device);
        libhal_free_string(device);
        return FALSE;
    }
    // if we get here, condition one is not satisfied


    // yay, we made it!!!
    DEBUG
        ("%s appears to be mountable",
         device);
    libhal_free_string(device);

    ivm_check_dbus_error(&dbus_error);

    return TRUE;
}
