/****************************************************************************
 *
 * All portions copyright their respective authors.  All rights reserved.
 *
 * This file is part of IVMan (ivm).
 *
 * This file may be distributed under the terms of the Q Public License
 * as defined by Troll Tech AS of Norway and appearing in the file
 * LICENSE.QPL included in the packaging of this file.
 * 
 * See http://www.troll.no/qpl for QPL licensing information.
 *
 * $Id: hal_psi.c,v 1.4 2005/05/12 01:42:34 ro_han Exp $
 *****************************************************************************/

// The following are a few functions which make HAL property set iterators
// easier to work with (i.e., it duplicates all of the libhal calls that Ivman
// likes for PSIs.)

#include "hal_psi.h"
#include <assert.h>
#include <stdlib.h>
#include <string.h>

LibHalPropertySetIterator *psi_get_psi_at_key(LibHalPropertySet * set,
                                              char const *const key)
{
    LibHalPropertySetIterator *psi =
        (LibHalPropertySetIterator *)
        calloc(sizeof(LibHalPropertySetIterator),
               1);

    libhal_psi_init(psi, set);
    char *currentKey = NULL;

    currentKey = libhal_psi_get_key(psi);
    if (strcmp(currentKey, key) == 0)
        return psi;

    while (libhal_psi_has_more(psi))
    {
        libhal_psi_next(psi);
        currentKey = libhal_psi_get_key(psi);
        if (strcmp(currentKey, key) == 0)
            return psi;
    }

    return NULL;
}

LibHalPropertyType
psi_get_property_type(LibHalPropertySet * set, char const *const key)
{
    LibHalPropertySetIterator *psi = psi_get_psi_at_key(set, key);

    if (!psi)
        return LIBHAL_PROPERTY_TYPE_INVALID;

    LibHalPropertyType type = libhal_psi_get_type(psi);

    free(psi);
    return type;
}

dbus_bool_t
psi_property_exists(LibHalPropertySet * set, char const *const key)
{
    LibHalPropertySetIterator *psi = psi_get_psi_at_key(set, key);

    if (!psi)
        return FALSE;

    free(psi);
    return TRUE;
}

char *psi_get_property_string(LibHalPropertySet * set,
                              char const *const key)
{
    LibHalPropertySetIterator *psi = psi_get_psi_at_key(set, key);

    assert(psi != NULL);

    char *string = (char *)strdup(libhal_psi_get_string(psi));

    free(psi);
    return string;
}

dbus_bool_t
psi_get_property_bool(LibHalPropertySet * set, char const *const key)
{
    LibHalPropertySetIterator *psi = psi_get_psi_at_key(set, key);

    assert(psi != NULL);

    dbus_bool_t prop = libhal_psi_get_bool(psi);

    free(psi);
    return prop;
}

dbus_int32_t
psi_get_property_int(LibHalPropertySet * set, char const *const key)
{
    LibHalPropertySetIterator *psi = psi_get_psi_at_key(set, key);

    assert(psi != NULL);

    dbus_int32_t prop = libhal_psi_get_int(psi);

    free(psi);
    return prop;
}

dbus_uint64_t
psi_get_property_uint64(LibHalPropertySet * set, char const *const key)
{
    LibHalPropertySetIterator *psi = psi_get_psi_at_key(set, key);

    assert(psi != NULL);

    dbus_uint64_t prop = libhal_psi_get_uint64(psi);

    free(psi);
    return prop;
}
