#!/usr/bin/python
# -*- coding: UTF-8 -*-
"""
Copyright 2006 Sebastian Kügler, Canonical Ltd

Authors: 
    Sebastian Kügler <sebas@kde.org>
    Jonathan Riddell <jriddell@ubuntu.com>

This program is free software; you can redistribute it and/or modify 
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

"""

"""
A frontend to HAL's power features for KDE.
Supports screen brightness, battery level, plugged/unplugged notices, laptop lid closing actions
Specification at https://wiki.kubuntu.org/KubuntuPowerManagement

Issues:
    - We have to keep polling HAL rather than listening for signals because the Python DBUS bindings
      don't have Qt mainloop integration
    - Written in Python so will be slow to load up, will probably port to C++ Qt 4.2 in future
    - Should also handle UPS and bluetooth batteries
    - systray applet should be hidden if no battery, but then how do you suspend if no battery?  
      (ksmserver integration please)
    - Needs lots more testing
    - Use KUniqueApplication again as soon as dcop problem is sorted out:
      or check via dcop for an existing instance and exit if there is one.
    - dcop calls need patch to dcopexport.py, already submitted upstream
"""

import sys
import subprocess
import dbus

from qt import QLabel
from kdecore import *
from kdeui import *

from dcopext import DCOPClient, DCOPApp # used to lock the screen
from dcopexport import DCOPExObj

from guidance_power_manager_ui import *
from notify import *
from tooltip import *

from powermanage import *

POLL_INTERVAL = 5000 # in milliseconds

class Notify(NotifyWidget):
    """ Pop up a passive notication windows. """

    def __init__(self,parent,msg,icon,caption=None):
        NotifyWidget.__init__(self,parent,"notify")
        self.setIcon(icon)
        self.setText(msg)
        if caption:
            self.Caption(caption)

    def setIcon(self,pixmap):
        """ Set an icon to be displayed in the notification. """
        self.Icon.setPixmap(pixmap)

    def setCaption(self,caption):
        """ Text to show in bold letters. """
        self.Caption.setText(QString("<b>")+caption+QString("</b>"))

    def setText(self,msg):
        """" Set actual notification message. """
        self.Text.setText(msg)


class PowerManager(PowerManagerUI):
    """ Our configuration dialog. """

    def __init__ (self, parent, name):
        PowerManagerUI.__init__(self, parent, name)        
        KGlobal.iconLoader().addAppDir("guidance")
        #self.icon_loader = KIconLoader()

        # The systray icon should show and hide the KDialogBase, not only this widget,
        # therefore, it gets our parent as parent.
        self.systray = KSystemTray(parent)
        self.icon = "battery-charging-100"
        self.systray.setPixmap(QPixmap(UserIcon(self.icon)))
        self.connect(self.systray, SIGNAL("quitSelected()"), self.quit)

        # Configuration filename
        self.config = KConfig("power-managerrc")

        self.powermanager = PowerManage()

        self.bus = dbus.SystemBus()

    def prepare(self):
        self._initBrightness()
        self._initLid()
        self._initUI(self.parent())
        self._initBattery()

        self.configToUi()

        # Polling: evil.  can't receive signals in python-dbus unless we have a glib mainloop, 
        # so we need to poll
        self.pollTimer = QTimer(self)
        self.connect(self.pollTimer, SIGNAL("timeout()"), self.poll)
        self.pollTimer.start(POLL_INTERVAL) # 5 second poll, maybe make this configurable
        self.poll()
        self.systray.show()

    def _initBrightness(self):
        """ Check for brightness support and disable widgets if it's not there. """
        if not self.powermanager.hasBrightness:
            self.MainsPoweredBox.hide()
            self.BrightnessLabel.hide()
            self.BatteryBrightnessSlider.hide()

    def _initLid(self):
        """ Check for lid support and disable widgets if it's not there. """
        if not self.powermanager.hasLid:
            self.LaptopLidRadios.setEnabled(False)
            if self.powermanager.onBattery():
                self.powerHasBeenPlugged()
            else:
                self.powerHasBeenUnplugged()

    def _initUI(self, parent):
        """ Build dynamic parts of the UI: context menu and tooltip. """
        menu = self.systray.contextMenu()
        #menu.setIcon(QPixmap(UserIcon("power-information"))) # FIXME: doesn't work :/
        #menu.setIcon(self.icon_loader.loadIcon("power-information", KIcon.NoGroup, 16)) # FIXME: doesn't work :/
        if self.powermanager.canSuspend:
            action = KAction( i18n("Suspend"), KShortcut(), self.suspend, 
                                                    self.systray.actionCollection(), "suspend");
            action.setIcon("suspend")
            action.plug(menu)
        if self.powermanager.canHibernate:
            action = KAction( i18n("Hibernate"), KShortcut(), self.hibernate, 
                                                    self.systray.actionCollection(), "hibernate");
            action.setIcon("hibernate")
            action.plug(menu)


        # Connect some signals.  Updates in the dialogue apply instantly
        self.connect(self.poweredBrightnessSlider, SIGNAL("valueChanged(int)"), self.changePoweredBrightness)
        self.connect(self.BatteryBrightnessSlider, SIGNAL("valueChanged(int)"), self.changeBatteryBrightness)

        #Add a blank tooltip, the tooltipgroup signals are then used for our KPassivePopup
        toolTipGroup = QToolTipGroup(self.systray)
        QToolTip.add(self.systray, "", toolTipGroup, "blah")
        self.connect(toolTipGroup, SIGNAL("showTip(const QString&)"), self.showTip)
        self.connect(toolTipGroup, SIGNAL("removeTip()"), self.hideTip)

        # Popup tooltip showing battery level
        self.popup = KPassivePopup(self.systray)
        self.tooltip = ToolTip(self.popup)

        self._addBatteryWidgets()

        self._addCpuWidgets()
        self.popup.setView(self.tooltip)
        
        if not self.powermanager.canSuspend:
            self.batteryCriticalSuspend.hide()
            self.laptopClosedSuspend.hide()
        
        if not self.powermanager.canHibernate:
            self.batteryCriticalHibernate.hide()
            self.laptopClosedHibernate.hide()
            

        # KGlobalAccel crashes the application in pykde
        # see http://mats.gmd.de/pipermail/pykde/2006-May/013224.html
        #self.globalActions = KGlobalAccel(self)
        #self.suspendShortcut = KShortcut("XF86Sleep")
        #self.hibernateShortcut = KShortcut("XF86Standby")
        #self.hshutdownShortcut = KShortcut("XF86PowerOff")
        #self.globalActions.insert("suspend", i18n("Suspend"), i18n("what's this?"), self.suspendShortcut, #self.suspendShortcut, self.suspend)
        #self.globalActions.updateConnections()

    def _initBattery(self):
        """ Remove non-battery-related widgets if there's no battery bay. """
        if not self.powermanager.hasBattery:
            # Disable the Batterybox in the config dialogue,
            self.BatteryBox.setEnabled(False)
            # And change the icon in the systray, remove the restore option
            # This way, we're basically becoming a systray applet, you can 
            # hibernate and suspend from
            self.systray.setPixmap(QPixmap(UserIcon(self.icon)))
            #menu.removeItem(2)  #FIXME: has no effect
        if self.powermanager.hasAC:
            self.wasOnBattery = self.powermanager.onBattery()

    def configToUi(self):
        """ Setup the the values from the config file in the UI."""
        # brightness.
        #if self.powermanager.hasBattery:
        #    self.CriticalLabel.setText(i18n("When remaining capacity drops below %i%%" % BATTERY_CRITICAL_PERCENT))
        if self.powermanager.hasBrightness:
            brightness_high = self.powermanager.brightness_levels
            self.BatteryBrightnessSlider.setMaxValue(self.powermanager.brightness_levels-1)
            self.poweredBrightnessSlider.setMaxValue(self.powermanager.brightness_levels-1)
            self.BatteryBrightnessSlider.setValue(self.config.readNumEntry("batteryBrightness", int(brightness_high/2))) #default middle
            self.poweredBrightnessSlider.setValue(self.config.readNumEntry("poweredBrightness", brightness_high)) #default highest                #self.BatteryBrightnessSlider.setValue(self.powermanager.getBrightness())                
            tt_text = "Every step increases or decreases the brightness by %i%%" % int(100/brightness_high)
            QToolTip.add(self.BatteryBrightnessSlider, tt_text)
            QToolTip.add(self.poweredBrightnessSlider, tt_text)
        # battery critical and lid actions.
        self.batteryCriticalRadios.setButton(self.config.readNumEntry("batteryCriticalAction", 2)) #default Suspend
        self.LaptopLidRadios.setButton(self.config.readNumEntry("laptopLidAction", 0)) #default None
        self.CriticalRemainTime.setValue(self.config.readNumEntry("criticalRemainTime", BATTERY_CRITICAL_MINUTES))
        self.criticalLevel = self.CriticalRemainTime.value()

    def uiToConfig(self):
        """ Read all values from the UI and write them to the config file. """
        self.config.writeEntry("poweredBrightness", self.poweredBrightnessSlider.value())
        self.config.writeEntry("batteryBrightness", self.BatteryBrightnessSlider.value())
        self.config.writeEntry("batteryCriticalAction", self.batteryCriticalRadios.selectedId())
        self.config.writeEntry("laptopLidAction", self.LaptopLidRadios.selectedId())
        self.config.writeEntry("criticalRemainTime", self.CriticalRemainTime.value())
        self.criticalLevel = self.CriticalRemainTime.value()

        self.config.sync()

    def quit(self):
        kapp.quit()

    def showTip(self, text=""):
        """ Pop up the tooltip showing battery data and CPU frequencies. """
        self.popup.show()

    def hideTip(self):
        """ Hide the tooltip."""
        self.popup.hide()

    def lockScreen(self):
        """ locks the screen using kdesktop """
        # create a new DCOP-Client:
        client = DCOPClient()
        # connect the client to the local DCOP-server:
        client.attach()
        # create a DCOP-Application-Object to talk to kdesktop:
        kdesktop = DCOPApp('kdesktop', client)
        # call a DCOP-function:
        ok, foo = kdesktop.KScreensaverIface.lock()

    def suspend(self):
        """ Lock the screen and initiate a suspend to RAM (S3). """
        self.lockScreen()
        return self.powermanager.suspend()

    def hibernate(self):
        """ Lock the screen and initiate a suspend to disk (S4). """
        self.lockScreen()
        return self.powermanager.hibernate()

    def shutdown(self):
        self.powermanager.shutdown()

    def blankScreen(self):
        """ Lock and blank screen. """
        self.lockScreen()
        self.powermanager.blankScreen()

    def _getIcon(self):
        """ Set systray icon depending on battery status/level. """
        if self.powermanager.hasBattery:
            if self.batt_state == "charged":
                self.icon = "battery-charging-100"
            elif self.batt_state == "discharging":
                if self.batt_level >= 95:
                    self.icon = "battery-discharging-100"
                elif self.batt_level < 95 and self.batt_level >= 70:
                    self.icon = "battery-discharging-080"
                elif self.batt_level < 70 and self.batt_level >= 40:
                    self.icon = "battery-discharging-060"
                elif self.batt_level < 40 and self.batt_level >= 15:
                    self.icon = "battery-discharging-040"
                elif self.batt_level < 15 and self.batt_level >= 5:
                    self.icon = "battery-discharging-020"
                else:
                    self.icon = "battery-discharging-000"
            elif self.batt_state == "charging":
                if self.batt_level >= 95:
                    self.icon = "battery-charging-100"
                elif self.batt_level < 95 and self.batt_level >= 70:
                    self.icon = "battery-charging-080"
                elif self.batt_level < 70 and self.batt_level >= 40:
                    self.icon = "battery-charging-060"
                elif self.batt_level < 40 and self.batt_level >= 15:
                    self.icon = "battery-charging-040"
                elif self.batt_level < 15 and self.batt_level >= 5:
                    self.icon = "battery-charging-020"
                else:
                    self.icon = "battery-charging-000"
        else:
            self.icon = "power-information"
        return self.icon

    def setIcon(self):
        """ Change the systray/tooltip icon."""
        oldIcon = self.icon
        icon = self._getIcon()
        if self.icon != oldIcon:
            self.systray.setPixmap(QPixmap(UserIcon(self.icon)))
            self.BattPixmap.setPixmap(QPixmap(UserIcon(self.icon)))

    def notify(self, msg, icon="power-information"):
        """ Send a notification popup. """
        icon = QPixmap(UserIcon(icon))
        try: 
            del self.warningPopup
        except:
            pass
        self.warningPopup = KPassivePopup(self.systray)
        label = Notify(self.warningPopup, msg, icon)
        self.warningPopup.setView(label)
        position = QPoint(5,5)
        self.warningPopup.show(position)

    def poll(self):
        """ Check for changes in plugged in status, battery status and laptop lid closed status. """
        debug( "------------ POLL ---------------")

        # Battery stuff:
        # check for last state, and run plugged / unplugged message if the state changed.
        if self.powermanager.hasBattery:
            plugged_num = 0
            self.batt_state = "not present" # unknown yet
            self.batt_level = self.batt_remain = 0
            self.batt_rate = self.batt_charge = self.batt_full = 0
            for batt in self.powermanager.batteries:
                state, level, remain, rate, current, full = self.powermanager.getBatteryState(batt)
                self._updateBatteryWidget(batt, state, level, remain)

                ## notify plugged/unplugged batteries
                if state == "not present":
                    if self.powermanager.batteryIsPresent[batt]:
                        self.notify(i18n("The battery has been removed."))
                        self.powermanager.batteryIsPresent[batt] = False
                else: # battery present
                    if not self.powermanager.batteryIsPresent[batt]:
                        self.notify(i18n("The battery has been inserted."), self._getIcon())
                        self.powermanager.batteryIsPresent[batt] = True

                    ## get cumulative charge levels/rate 
                    self.batt_rate += rate
                    self.batt_charge += current
                    self.batt_full += full

                    ## calculate overall level (average of present batteries)
                    self.batt_remain += remain
                    self.batt_level += level
                    plugged_num += 1

                    ## calculate overall state (charging/discharging/charged)
                    if state in ("charging","discharging"):
                        self.batt_state = state
                    elif not self.batt_state in ("charging, discharging"):
                        self.batt_state = state

            # if we know charge and full -> recalculate overall level
            if self.batt_full > 0 and self.batt_charge > 0:
               self.batt_level = 100 * self.batt_charge / self.batt_full
            else:
                # if more than one battery present, we need to calculate average level
                if plugged_num > 1:
                    self.batt_level /= plugged_num;

            # if rate is reported, calculate remaining time on our own
            if self.batt_rate > 0:
                if self.batt_state == "charging":
                    self.batt_remain = 3600 * (float(self.batt_full - self.batt_charge) / self.batt_rate)
                if self.batt_state == "discharging":
                    self.batt_remain = 3600 * (float(self.batt_charge) / self.batt_rate)

            remain_h = self.batt_remain/3600
            remain_m = (self.batt_remain/60)%60 

            blabel = i18n("<b>Battery:</b>")
            if self.batt_state == "charged":
               blabel += i18n(" Fully charged")
            elif self.batt_state == "charging":
               blabel += i18n(" %i:%02ih to charge" % (remain_h,remain_m))
            elif self.batt_state == "discharging":
               blabel += i18n(" %i:%02ih remaining" % (remain_h,remain_m))
            self.BattMainLabel.setText(blabel)

            # update tray icon if needed
            self.setIcon()

            # check battery state
            self.checkBatteryCritical()

        # CPU stuff
        self._updateCpuWidgets()

        if self.powermanager.hasBattery:
            on_battery = self.powermanager.onBattery()
            if self.powermanager.wasOnBattery is not on_battery:
                debug("poll: states differ")
                if not on_battery: 
                    debug("poll: Now on AC")
                    self.powerHasBeenPlugged()
                else:
                    debug("poll: Now on battery")
                    self.powerHasBeenUnplugged()
            else:
                debug("poll: state is the same")
            self.powermanager.wasOnBattery = on_battery

        # Lid stuff
        if self.powermanager.hasLid:
            properties = self.powermanager.lidObject.GetAllProperties(dbus_interface="org.freedesktop.Hal.Device")
            if properties["button.state.value"]:
                if not self.powermanager.lidClosedState:
                    self.powermanager.lidClosedState = True
                    self.warningPopup = KPassivePopup(self.systray)

                    action = self.LaptopLidRadios.selectedId()
                    if action == 0:  #None
                        actionText = i18n("doing nothing")
                    elif action == 1: #Blank
                        actionText = i18n("blanking screen")
                        QTimer.singleShot(2000, self.blankScreen)
                    elif action == 2: #Suspend
                        actionText = i18n("suspending")
                        QTimer.singleShot(2000, self.suspend)
                    elif action == 3: #hibernate
                        self.hibernate()
                        QTimer.singleShot(2000, self.hibernate)
                        actionText = i18n("hibernating")
                    elif action == 4: #shutdown
                        QTimer.singleShot(2000, self.shutdown)
                        actionText = i18n("shutting down")
                    else: #error
                        print "Error: unknown battery critical action"
                    label = QLabel(i18n("Laptop lid is closed, %1 now").arg(actionText), self.warningPopup)
                    self.warningPopup.setView(label)
                    self.warningPopup.show()
            else:
                self.powermanager.lidClosedState = False

    def _addBatteryWidgets(self):
        """ Adds progressbars to show battery status to the tooltip."""
        BattLayout = QHBoxLayout(None,0,6,"BattLayout")

        self.BattPixmap = QLabel(self.tooltip,"BattLabLayout")
        self.BattPixmap.setSizePolicy(QSizePolicy(QSizePolicy.Fixed,QSizePolicy.Fixed,0,0,self.BattPixmap.sizePolicy().hasHeightForWidth()))
        self.BattPixmap.setPixmap(QPixmap(UserIcon(self.icon)))
        self.BattPixmap.setScaledContents(1)
        BattLayout.addWidget(self.BattPixmap)
        self.BattMainLabel = QLabel(self.tooltip,"BattMainLabel")
        self.BattMainLabel.setText(i18n("<b>Battery:</b>"))
        BattLayout.addWidget(self.BattMainLabel)

        # Add to tooltip
        self.tooltip.layout().addLayout(BattLayout)

        # Create a progressbar and a label for every battery found, and add it to tooltip
        self.BattLabel = {}
        self.BattLayout = {}
        self.BattProgress = {}
        i = 1
        for batt in self.powermanager.batteries:
            self.BattLayout[batt] = QHBoxLayout(None,0,6,"BattBarLayout")
            self.BattLabel[batt] = QLabel(self.tooltip,"BattLabel")
            if len(self.powermanager.batteries) > 1:
                self.BattLabel[batt].setText(i18n("Battery %i" % i))
            self.BattLayout[batt].addWidget(self.BattLabel[batt])
            self.BattProgress[batt] = KProgress(self.tooltip,"BattProgress")
            self.BattProgress[batt].setMinimumSize(QSize(200,0))
            self.BattLayout[batt].addWidget(self.BattProgress[batt])
            self.tooltip.layout().addLayout(self.BattLayout[batt])
            i += 1


    def _updateBatteryWidget(self, batt, state, level, remain):
        """ Retrieve battery information and update the related widgets accordingly. """
        self.BattProgress[batt].setEnabled(True)
        self.BattProgress[batt].setTotalSteps(100)
        self.BattProgress[batt].setProgress(level)
        if state == "not present":
            self.BattProgress[batt].setFormat(i18n("not present"))
        elif state == "charging":
            self.BattProgress[batt].setFormat(i18n("Charging (%p%)"))
        elif state == "discharging":
            self.BattProgress[batt].setFormat("Discharging (%p%)")
        else:
            self.BattProgress[batt].setFormat("%p%")

    def _addCpuWidgets(self):
        """ Adds progressbars to show CPU frequencies to the tooltip."""
        if not self.powermanager.USE_CPUFREQ:
            return
        LabelLayout = QHBoxLayout(None,0,6,"layout5")

        self.CpuPixmap = QLabel(self.tooltip,"CpuPixmap")
        self.CpuPixmap.setSizePolicy(QSizePolicy(QSizePolicy.Fixed,QSizePolicy.Fixed,0,0,self.CpuPixmap.sizePolicy().hasHeightForWidth()))
        self.CpuPixmap.setPixmap(QPixmap(UserIcon("processor")))
        self.CpuPixmap.setScaledContents(1)
        LabelLayout.addWidget(self.CpuPixmap)
        self.CpuMainLabel = QLabel(self.tooltip,"CpuMainLabel")
        self.CpuMainLabel.setText(i18n("<b>CPU Frequency:</b>"))
        LabelLayout.addWidget(self.CpuMainLabel)

        # Add to tooltip
        self.tooltip.layout().addLayout(LabelLayout)

        # Create a progressbar and a label for every CPU found, and add it to tooltip
        self.CpuLabel = {}
        self.CpuLayout = {}
        self.CpuProgress = {}
        i = 1
        for cpu in self.powermanager.cpus:
            self.CpuLayout[cpu] = QHBoxLayout(None,0,6,"layout2")
            self.CpuLabel[cpu] = QLabel(self.tooltip,"CpuLabel")
            if len(self.powermanager.cpus) > 1:
                self.CpuLabel[cpu].setText(i18n("Processor %i" % i))
            self.CpuLayout[cpu].addWidget(self.CpuLabel[cpu])    
            self.CpuProgress[cpu] = KProgress(self.tooltip,"CpuProgress")
            self.CpuProgress[cpu].setFormat("%v MHz")
            self.CpuLayout[cpu].addWidget(self.CpuProgress[cpu])
            self.tooltip.layout().addLayout(self.CpuLayout[cpu])
            i += 1

    def _updateCpuWidgets(self):
        """ Retrieve CPU freq information and update the related widgets accordingly. """
        if not self.powermanager.USE_CPUFREQ:
            return
        for cpu in self.powermanager.cpus:
            cpustate = self.powermanager.getCpuState(cpu)
            if not cpustate['online']:
                self.CpuProgress[cpu].setEnabled(False)
                continue
            else:
                self.CpuProgress[cpu].setEnabled(True)
                self.CpuProgress[cpu].setTotalSteps(cpustate['max'])
                self.CpuProgress[cpu].setProgress(cpustate['cur'])

    def changePoweredBrightness(self, level=None):
        """ Mains-powered brigthness slider has been moved. """
        # Check if the state applies and adjust brightness immediately.
        if not self.powermanager.onBattery() and self.powermanager.hasBrightness:
            if not level:
                level = self.poweredBrightnessSlider.value()
            self.powermanager.adjustBrightness(level)

    def changeBatteryBrightness(self, level=None):
        """ Battery-powered brigthness slider has been moved. """
        # Check if the state applies and adjust brightness immediately.
        if self.powermanager.onBattery() and self.powermanager.hasBrightness:
            if not level:
                level = self.BatteryBrightnessSlider.value()
            self.powermanager.adjustBrightness(level)

    def powerHasBeenUnplugged(self):
        """ Actions to perform when the plug has been pulled."""
        if self.powermanager.hasAC:
            self.notify(i18n("The AC adapter has been unplugged, switching to battery mode."), self._getIcon())
        if self.powermanager.hasBrightness:
            self.powermanager.brightnessObject.SetBrightness(self.BatteryBrightnessSlider.value(), 
                                            dbus_interface="org.freedesktop.Hal.Device.LaptopPanel")
        # No SetPowerSave in Ubuntu's HAL
        try:
            self.computerObject.SetPowerSave(True, 
                                dbus_interface="org.freedesktop.Hal.Device.SystemPowerManagement")
        except:
            pass
        self.checkBatteryCritical()
        self.changeBatteryBrightness()
        self.powermanager.setScreensaverBlankOnly(True)

    def powerHasBeenPlugged(self):
        """ Actions to perform when AC adapter has been plugged in. """
        self.notify(i18n("The AC adapter has been plugged in, switching to AC mode."), self._getIcon())
        if self.powermanager.hasBrightness:
            self.powermanager.brightnessObject.SetBrightness(self.poweredBrightnessSlider.value(), 
                dbus_interface="org.freedesktop.Hal.Device.LaptopPanel")
        # No SetPowerSave in Ubuntu's HAL
        try:
            self.computerObject.SetPowerSave(True, 
                                dbus_interface="org.freedesktop.Hal.Device.SystemPowerManagement")
        except:
            pass
        self.changePoweredBrightness()
        self.powermanager.setScreensaverBlankOnly(False)

    def checkBatteryCritical(self):
        """ Check for warning and critical battery label and notify-warn or 
            initiate the configured action. """

        if not self.powermanager.hasBattery:
            return

        if self.batt_state == "discharging": 
            currentLevel = int(self.batt_remain/60)

            warningLevel = self.criticalLevel + 5 # warn five minutes before critical
            criticalLevel = self.criticalLevel

            debug("CurrentBat: %i, WarningBat: %i, CriticalBat: %i" % (currentLevel, warningLevel, criticalLevel))
            # We only want to suspend if the chargelevel is above a certain threshold, 
            # it sometimes takes some time for HAL to report remaining time correctly
            if currentLevel <= criticalLevel and self.batt_level < CHARGE_LEVEL_THRESHOLD:
                if not self.powermanager.criticalBatteryState and self.powermanager.onBattery():
                    self.powermanager.criticalBatteryState = True

                    action = self.config.readNumEntry("batteryCriticalAction", 2)
                    if action == 0:  #None
                        actionText = i18n("doing nothing")
                    elif action == 1: #Suspend
                        actionText = i18n("suspending")
                        QTimer.singleShot(2000, self.suspend)
                    elif action == 2: #hibernate
                        actionText = i18n("hibernating")
                        QTimer.singleShot(2000, self.hibernate)
                    elif action == 3: #shutdown
                        actionText = i18n("shutting down")
                        QTimer.singleShot(2000, self.shutdown)
                    else: #error
                        print "Error: unknown battery critical action"
                        actionText = "unknown"
                    self.notify(i18n("You are about to run out of battery power, %1 now.").arg(actionText), 
                                        self._getIcon())
            else: 
                self.powermanager.criticalBatteryState = False
                if currentLevel <= warningLevel:
                    if not self.powermanager.warningBatteryState:
                        self.powermanager.warningBatteryState = True
                        self.notify(i18n("You are low on battery power."), self._getIcon())
                else:
                    self.powermanager.warningBatteryState = False

        
def doDcop(kapp):
    """ Register kvandale in dcop, so it can be controlled from outside. """
    my_dcop = kapp.dcopClient()
    #my_dcop.attach()
    my_dcop.registerAs("power-manager")

    
class DcopIface (DCOPExObj):
    """ Add some interface so we can use powermanager from the outside. """
    def __init__ (self, app, id='power-manager'):
        DCOPExObj.__init__ (self, id)
        #     addMethod (<signature>, <Python method>)
        #self.addMethod ('QString getQuery()', gvd.getZoekbegrip)

        # PM related.
        self.addMethod ('void suspend ()', app.suspend)
        self.addMethod ('void hibernate ()', app.hibernate)
        self.addMethod ('void shutdown ()', app.shutdown)

        # UI related.
        self.addMethod ('void showTip ()', app.showTip)
        #self.addMethod ('void show ()', app.parent().show)
        #self.addMethod ('void hide ()', app.parent().hide)

        #self.addMethod ('void plugged ()', app.powerHasBeenPlugged)
        #self.addMethod ('void unplugged ()', app.powerHasBeenUnplugged)
        self.addMethod ('bool onBattery ()', app.powermanager.onBattery)

        #self.addMethod ('QString getCurrentResult()', gvd.getRawResult)


class PowermanagerApp(KDialogBase):
    """ The KDialog providing the OK, Apply and Cancel buttons."""

    def __init__(self,parent=None,name=None):
        """ Initialise dialog and set mainwidget. """
        KGlobal.locale().insertCatalogue("guidance")
        KGlobal.iconLoader().addAppDir("guidance")
        
        # We would like to use a KUniqueApplication, but that breaks dcop due to some
        # strange bug. The following line is the revenge code for this bug, it is
        # intentionally ugly.
        if len(os.popen("dcop |grep power-manager").readlines()) > 1:
            print "There is already an instance of power manager running. Exiting."
            sys.exit(0)

        # Which buttons do we want?
        KDialogBase.__init__(self,KJanusWidget.Swallow,i18n("Power Manager"),
            KDialogBase.Ok|KDialogBase.Apply|KDialogBase.Cancel|KDialogBase.User1, KDialogBase.Close)
        self.pmwidget = PowerManager(self,name)
        self.setButtonText(KDialogBase.User1, i18n("About"))
        
        if not self.pmwidget.powermanager.isLaptop():
            print "This is not a laptop, quitting ... "
            sys.exit(1)
            
        self.pmwidget.prepare()
            
        self.setMainWidget(self.pmwidget)
        self.aboutus = KAboutApplication(self)
        
    def slotOk(self):
        """ The OK button has been pressed, save configuration and pass on do whatever 
            needs to be done by KDialog. """
        self.pmwidget.uiToConfig()
        KDialogBase.slotOk(self)

    def slotApply(self):
        """ The Apply button has been pressed, save configuration and pass on do whatever 
            needs to be done by KDialog. """
        self.pmwidget.uiToConfig()
        KDialogBase.slotApply(self)

    def slotCancel(self):
        """ The Cancel button has been pressed, reset some values and hide dialogue. """
        # In case brightness has changed, we reset it to the configured value.
        if self.pmwidget.powermanager.hasBrightness:
            brightness_high = self.pmwidget.powermanager.brightness_levels
            if not self.pmwidget.powermanager.onBattery():
                level = self.pmwidget.config.readNumEntry("poweredBrightness", brightness_high)
            else:
                level = self.pmwidget.config.readNumEntry("batteryBrightness", int(brightness_high/2))
            self.pmwidget.powermanager.adjustBrightness(level)
        self.pmwidget.configToUi()
        KDialogBase.slotCancel(self)
        
    def slotUser1(self):
        self.aboutus.show()


if __name__ == "__main__":
    aboutdata = KAboutData("power-manager", "Power Manager", "0.42", 
        "Handles battery, display and suspend modes for your computer.", KAboutData.License_GPL, 
        "(C) 2006 Sebastian Kügler, (C) 2006 Canonical Ltd", None, None, "jriddell@ubuntu.com")
    aboutdata.addAuthor("Sebastian Kügler", "Developer", "sebas@kde.org","http://vizZzion.org")
    aboutdata.addAuthor("Jonathan Riddell", "Developer", "jriddell@ubuntu.com")
    aboutdata.addAuthor("Luka Renko", "Developer", "lure@kubuntu.org")
    aboutdata.setProgramLogo(QImage("power-manager.png"))
    KCmdLineArgs.init(sys.argv, aboutdata)
    kapp = KUniqueApplication(True, True, False)
    #kapp = KApplication()
    mainWindow = PowermanagerApp(None, "main window")
    doDcop(kapp)
    dcop_iface = DcopIface(mainWindow.pmwidget)

    kapp.exec_loop()
