
/**
 *
 *  This file is part of the kdewebdev package
 *  Copyright (c) 2002 Keith Isdale <keith@kdewebdev.org>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 **/

#include "kxsldbgadaptor.h"
#include "kxsldbg_part.h"
#include "libxsldbg/files.h"
#include "libxsldbg/utils.h"
#include "libxsldbg/xsldbg.h"
#include "libxsldbg/debugXSL.h"

#include <kcomponentdata.h>
#include <kaction.h>
#include <kstandardaction.h>
#include <kfiledialog.h>
#include <kparts/genericfactory.h>
#include <ktexteditor/markinterface.h>
#include <ktexteditor/editor.h>

#include <QFile>
#include <QTextStream>

#include "../kxsldbg.h"
#include <kcmdlineargs.h>
#include <kiconloader.h>
#include <QMessageBox>
#include <klocale.h>
#include <kdeversion.h>
#include <kinputdialog.h>

// Qxsldbg specific includes
#include "qxsldbgdoc.h"
#include <QVariant>
#include <QStatusBar>
#include <QSplitter>

#include <QLayout>
#include <QToolTip>
#include <QAction>
#include <QMenuBar>
#include <QToolBar>
#include <QImage>
#include <QPixmap>
#include <QTextBrowser>
#include <QLineEdit>
#include <QPushButton>
#include <QInputDialog>
#include <QObject>
#include <QStackedWidget>
#include "xsldbgoutputview.h"
#include "xsldbgconfigimpl.h"
#include <kdebug.h>
#include <kvbox.h>
#include <kactioncollection.h>
#include <QApplication>
#include "configdialogprivate.h"

typedef KParts::GenericFactory<KXsldbgPart> KXsldbgPartFactory;
K_EXPORT_COMPONENT_FACTORY( libkxsldbgpart, KXsldbgPartFactory )

KXsldbgPart::KXsldbgPart( QWidget *parentWidget,
                                  QObject *parent,
                                  const QStringList & /*args*/ )
  : KParts::ReadOnlyPart(parent)
{
    new KXsldbgPartAdaptor( this );
    QDBusConnection::sessionBus().registerObject("/KXsldbgPart", this, QDBusConnection::ExportAdaptors);
    currentLineNo = 0;
    currentColumnNo = 0;
    inspector = 0L;
    debugger = 0L;
    dlg = 0L;
    configWidget = 0L;
    currentDoc = 0L;

    // we need an instance
    setComponentData( KXsldbgPartFactory::componentData() );
    KVBox *frame = new KVBox(parentWidget);
    KHBox *h = new KHBox(frame);
    newXPath = new QLineEdit(h);
    xPathBtn = new QPushButton(i18n("Goto XPath"), h);
/* Disable searching as searching documentation is not ready
    h = new QHBox(frame);
    newSearch = new QLineEdit(h);
    searchBtn = new QPushButton(i18n("Search"), h);
*/
    h = new KHBox(frame);
    newEvaluate = new QLineEdit(h);
    evaluateBtn = new QPushButton(i18n("Evaluate"), h);

    QSplitter *splitter = new QSplitter(Qt::Vertical, frame);
    mainView = new QStackedWidget(splitter);
    mainView->setMinimumHeight(400); //## TODO don't use a magic number  
    outputview = new XsldbgOutputView(splitter);
    setWidget(frame);

    // create our actions
    KStandardAction::open(this, SLOT(fileOpen()), actionCollection());

    // set our XML-UI resource file
   setXMLFile("kxsldbg_part.rc");
   KAction *action ;
   action = actionCollection()->addAction( "configureEditorCmd" );
   action->setText( i18n("Configure Editor...") );
   action->setIcon( KIcon("configure") );
   connect(action, SIGNAL(triggered(bool)), SLOT(configureEditorCmd_activated()));
   action = actionCollection()->addAction( "configureCmd" );
   action->setText( i18n("Configure...") );
   action->setIcon( KIcon("configure") );
   connect(action, SIGNAL(triggered(bool)), SLOT(configureCmd_activated()));
   action->setShortcut(QKeySequence(Qt::Key_C));

   action = actionCollection()->addAction( "inspectCmd" );
   action->setText( i18n("Inspect...") );
   action->setIcon( KIcon("edit-find") );
    connect(action, SIGNAL(triggered(bool)), SLOT(inspectorCmd_activated()));
    action->setShortcut(QKeySequence(Qt::Key_I));


    // Motions commands
    action = actionCollection()->addAction( "runCmd" );
    action->setText( i18n("Run") );
    action->setIcon( KIcon("system-run") );

    connect(action, SIGNAL(triggered(bool)), SLOT(runCmd_activated()));
    action->setShortcut(QKeySequence(Qt::Key_F5));

    action = actionCollection()->addAction( "continueCmd" );
    action->setText( i18n("Continue") );
    action->setIcon( KIcon("media-playback-start") );

    connect(action, SIGNAL(triggered(bool)), SLOT(continueCmd_activated()));
    action->setShortcut(QKeySequence(Qt::Key_F4));

    action = actionCollection()->addAction( "stepCmd" );
    action->setText( i18n("Step") );
    action->setIcon( KIcon("step") );

    connect(action, SIGNAL(triggered(bool)), SLOT(stepCmd_activated()));
    action->setShortcut(QKeySequence(Qt::Key_F8));

    action  = new KAction(KIcon("go-down-search"), i18n("Next"), this);
    actionCollection()->addAction("nextCmd", action );
    connect(action, SIGNAL(triggered(bool)), SLOT(nextCmd_activated()));
    action->setShortcut(QKeySequence(Qt::Key_F10));

    action  = new KAction(KIcon("xsldbg_stepup"), i18n("Step Up"), this);
    actionCollection()->addAction("stepupCmd", action );
    connect(action, SIGNAL(triggered(bool)), SLOT(stepupCmd_activated()));
    action->setShortcut(QKeySequence(Qt::Key_F6));

    action  = new KAction(KIcon("xsldbg_stepdown"), i18n("Step Down"), this);
    actionCollection()->addAction("stepdownCmd", action );
    connect(action, SIGNAL(triggered(bool)), SLOT(stepdownCmd_activated()));
    action->setShortcut(QKeySequence(Qt::Key_F7));

    // Breakpoint commands
    action  = new KAction(KIcon("xsldbg_break"), i18n("Break"), this);
    actionCollection()->addAction("breakCmd", action );
    connect(action, SIGNAL(triggered(bool)), SLOT(breakCmd_activated()));
    action->setShortcut(QKeySequence(Qt::Key_F2));

    action  = new KAction(KIcon("xsldbg_enable"), i18n("Enable/Disable"), this);
    actionCollection()->addAction("enableCmd", action );
    connect(action, SIGNAL(triggered(bool)), SLOT(enableCmd_activated()));
    action->setShortcut(QKeySequence(Qt::Key_F3));

    action  = new KAction(KIcon("xsldbg_delete"), i18n("Delete"), this);
    actionCollection()->addAction("deleteCmd", action );
    connect(action, SIGNAL(triggered(bool)), SLOT(deleteCmd_activated()));
    action->setShortcut(QKeySequence(Qt::Key_Delete));

    action  = new KAction(KIcon("xsldbg_source"), i18n("&Source"), this);
    actionCollection()->addAction("sourceCmd", action );
    connect(action, SIGNAL(triggered(bool)), SLOT(sourceCmd_activated()));
    action->setShortcut(QKeySequence(Qt::Key_S));

    action  = new KAction(KIcon("xsldbg_data"), i18n("&Data"), this);
    actionCollection()->addAction("dataCmd", action );
    connect(action, SIGNAL(triggered(bool)), SLOT(dataCmd_activated()));
    action->setShortcut(QKeySequence(Qt::Key_D));

    action  = new KAction(KIcon("xsldbg_output"), i18n("&Output"), this);
    actionCollection()->addAction("outputCmd", action );
    connect(action, SIGNAL(triggered(bool)), SLOT(outputCmd_activated()));
    action->setShortcut(QKeySequence(Qt::Key_O));

    action  = new KAction(KIcon("xsldbg_refresh"), i18n("Reload"), this);
    actionCollection()->addAction("refreshCmd", action );
    connect(action, SIGNAL(triggered(bool)), SLOT(refreshCmd_activated()));
    action->setShortcut(QKeySequence(Qt::CTRL + Qt::Key_F5));

    /* tracing and walking */
    action  = new KAction(i18n("Walk Through Stylesheet..."), this);
    actionCollection()->addAction("walkCmd", action );
    connect(action, SIGNAL(triggered(bool)), SLOT(walkCmd_activated()));
    action->setShortcut(QKeySequence(Qt::Key_W));
    action  = new KAction(i18n("Stop Wal&king Through Stylesheet"), this);
    actionCollection()->addAction("walkStopCmd", action );
    connect(action, SIGNAL(triggered(bool)), SLOT(walkStopCmd_activated()));
    action->setShortcut(QKeySequence(Qt::Key_K));
    action  = new KAction(i18n("Tr&ace Execution of Stylesheet"), this);
    actionCollection()->addAction("traceCmd", action );
    connect(action, SIGNAL(triggered(bool)), SLOT(traceCmd_activated()));
    action->setShortcut(QKeySequence(Qt::Key_A));
    action  = new KAction(i18n("Stop Tracing of Stylesheet"), this);
    actionCollection()->addAction("traceStopCmd", action );
    connect(action, SIGNAL(triggered(bool)), SLOT(traceStopCmd_activated()));
    action->setShortcut(QKeySequence(Qt::Key_K));

    action  = new KAction(i18n("&Evaluate Expression..."), this);
    actionCollection()->addAction("evaluateCmd", action );
    connect(action, SIGNAL(triggered(bool)), SLOT(evaluateCmd_activated()));
    action->setShortcut(QKeySequence(Qt::Key_E));

    action  = new KAction(i18n("Goto &XPath..."), this);
    actionCollection()->addAction("gotoXPathCmd", action );
    connect(action, SIGNAL(triggered(bool)), SLOT(gotoXPathCmd_activated()));
    action->setShortcut(QKeySequence(Qt::Key_X));

    action  = new KAction(i18n("Lookup SystemID..."), this);
    actionCollection()->addAction("lookupSystemID", action );
    connect(action, SIGNAL(triggered(bool)), SLOT(slotLookupSystemID()));

    action  = new KAction(i18n("Lookup PublicID..."), this);
    actionCollection()->addAction("lookupPublicID", action );
    connect(action, SIGNAL(triggered(bool)), SLOT(slotLookupPublicID()));

    action  = new KAction(i18n("Quit"), this);
    actionCollection()->addAction("file_quit", action );
    connect(action, SIGNAL(triggered(bool)), SLOT(quit()));
    action->setShortcut(QKeySequence(Qt::CTRL + Qt::Key_Q));

    /*
    action = new KAction( i18n("Exit KXsldbg"),
                        "xsldbg_output", actionCollection(), "exitCmd" );
    connect(action, SIGNAL(triggered(bool)), SLOT(exitCmd_activated()));
    action->setShortcut(QKeySequence(CTRL + Key_Q));
    */
    connect( xPathBtn, SIGNAL( clicked() ),
             this, SLOT( slotGotoXPath() ) );
    connect( evaluateBtn, SIGNAL( clicked() ),
             this, SLOT( slotEvaluate() ) );
/*

    connect( searchBtn, SIGNAL( clicked() ),
             this, SLOT( slotSearch() ) );
*/
/* We must have a valid debugger and inspector */
    createInspector();
    if (checkDebugger()){
        dlg = new  ConfigDialogPrivate(0);
        Q_CHECK_PTR(dlg);
        configWidget = dlg->configWidget;
    }
}

KXsldbgPart::~KXsldbgPart()
{
    if (debugger){
        disconnect(debugger, SIGNAL( debuggerReady()),
                this, SLOT(debuggerStarted()));
        disconnect(debugger, SIGNAL( debuggerStopped()),
                this, SLOT(debuggerStopped()));
        debugger->stop();
    }

    closeUrl();

    delete dlg;
    dlg = 0;
    delete debugger;
    debugger = 0;
    delete inspector;
    inspector = 0;
}

void KXsldbgPart::quit()
{
   qWarning("quit");
   closeUrl();
}


KAboutData *KXsldbgPart::createAboutData()
{
    // the non-i18n name here must be the same as the directory in
    // which the part's rc file is installed ('partrcdir' in the
    // Makefile)
    KAboutData *aboutData = new KAboutData("kxsldbgpart", 0, ki18n("KXsldbgPart"), "0.1");
    aboutData->addAuthor(ki18n("Keith Isdale"), ki18n(0L), "keith@kdewebdev.org");
    return aboutData;
}


bool  KXsldbgPart::openUrl(const KUrl &url)
{
    bool result = fetchURL(url);
    if (result){
        QString docID = url.url();
	QXsldbgDoc *docPtr = docDictionary[docID];
	if (docPtr && docPtr->kateView()){
	    if (docPtr != currentDoc){
		currentDoc = docPtr;
		currentFileName = docID;
		mainView->addWidget(currentDoc->kateView());
		mainView->setCurrentWidget(currentDoc->kateView());
		emit setWindowCaption(currentDoc->url().prettyUrl());
	    }
	} else{
	    result = false;
	}
    }

    return result;
}


/* Don't show the content of URL just loaded it into our data structures */
bool  KXsldbgPart::fetchURL(const KUrl &url)
{
    bool result = false;
    QString docID = url.url();
    QXsldbgDoc *docPtr = docDictionary[docID];
    if (!docPtr){
	docPtr = new QXsldbgDoc(mainView, url);
	docDictionary.insert(docID, docPtr);
	QHash<QString, QXsldbgDoc*>::const_iterator it;
	if (docPtr->kateView()){
		mainView->addWidget(docPtr->kateView());
		KTextEditor::View *v = docPtr->kateView();
		connect(v, SIGNAL(cursorPositionChanged(KTextEditor::View *, const KTextEditor::Cursor &))
			, this, SLOT(cursorPositionChanged()));
	}
        docPtr = docDictionary[docID];
        result = docPtr != 0;
    }else{
        result = true;
    }

    if (!result)
       qWarning("Fetch of URL %s failed", docID.toUtf8().constData());

    return result;
}

bool KXsldbgPart::openFile()
{
    qWarning("bool KXsldbgPart::openFile() called");
    return false;
}

bool KXsldbgPart::closeUrl()
{
    QWidget * w;
    for (int index = 0; index < mainView->count(); index++) {
        w = mainView->widget(index);
        mainView->removeWidget(w);
    }


    if (currentDoc && currentDoc->kateView() && currentDoc->kateDoc()->views().count() == 1){
        delete currentDoc->kateView()->document();
        currentDoc = 0L;
    }

    return true;
}

void KXsldbgPart::fileOpen()
{
    // this slot is called whenever the File->Open menu is selected,
    // the Open shortcut is pressed (usually CTRL+O) or the Open toolbar
    // button is clicked
    QString file_name = KFileDialog::getOpenFileName();

    if (file_name.isEmpty() == false)
        openUrl(KUrl( file_name ));
}

void KXsldbgPart::configureEditorCmd_activated()
{
    if (currentDoc){
      currentDoc->kateDoc()->editor()->configDialog(mainView);
    }
}

bool KXsldbgPart::checkDebugger()
{
    bool result = debugger && debugger->getInitialized();
    if (!result){
        QMessageBox::information(0L, i18n("Debugger Not Ready"),
                i18n("Configure and start the debugger first."),
                QMessageBox::Ok);
    }

    return result;
}


void KXsldbgPart::lookupSystemID( QString systemID)
{
   bool ok = false;
   if (!checkDebugger())
     return;

   if (systemID.isEmpty()){
     systemID = KInputDialog::getText(
			      i18n( "Lookup SystemID" ),
			      i18n( "Please enter SystemID to find:" ),
			      QString(), &ok,
			      mainView);
   }else{
     ok = true;
   }
  if ( ok && !systemID.isEmpty() ){
      // user entered something and pressed ok
	QString msg(QString("system %1").arg(systemID));  // noTr
	debugger->fakeInput(msg, true);
  }

}


void KXsldbgPart::lookupPublicID(QString publicID)
{
   bool ok = false;
   if (!checkDebugger())
     return;

   if (publicID.isEmpty()){
     publicID = KInputDialog::getText(
			      i18n( "Lookup PublicID" ),
			      i18n( "Please enter PublicID to find:" ),
			      QString(), &ok, mainView );
   }else{
     ok = true;
   }
   if ( ok && !publicID.isEmpty()){
     // user entered something and pressed ok
	QString msg(QString("public %1").arg(publicID));  // noTr
	debugger->fakeInput(msg, true);
  }
}


void KXsldbgPart::slotLookupSystemID()
{
  lookupSystemID("");
}

void KXsldbgPart::slotLookupPublicID()
{
  lookupPublicID("");
}

void KXsldbgPart::configureCmd_activated()
{
  if (!checkDebugger() || !dlg || !configWidget)
    return;

  configWidget->setModel(optionDataModel());

  dlg->show();
}

void
KXsldbgPart::runCmd_activated()
{
    if ( checkDebugger() )
        debugger->slotRunCmd();
}

void KXsldbgPart::inspectorCmd_activated()
{
  if (inspector == 0L)
    createInspector();

  if (checkDebugger() && (inspector != 0L)){
    inspector->show();
  }
}


void KXsldbgPart::createInspector()
{
    if ( inspector == 0L ) {
      debugger = new XsldbgDebugger();
      Q_CHECK_PTR( debugger );
      if ( debugger != 0L ) {
	       connect(debugger, SIGNAL( debuggerReady()),
		     this, SLOT(debuggerStarted()));
	       connect(debugger, SIGNAL( debuggerStopped()),
		     this, SLOT(debuggerStopped()));
	       if (outputview){
		 connect(debugger,
			 SIGNAL( showMessage(QString /* msg*/)),
			 outputview,
		  SLOT(slotProcShowMessage(QString /* msg*/)));
	       }
	      inspector = new XsldbgInspector( debugger );
	      Q_CHECK_PTR( inspector );
	      debugger->setInspector( inspector );
	      if (inspector != 0L){
		/*process line number and/or file name changed */
		connect(debugger,
			SIGNAL(lineNoChanged
				(QString /* fileName */ ,
				 int /* lineNumber */ ,
				 bool /* breakpoint */ ) ),
			this,
			SLOT(lineNoChanged
			     ( QString /* fileName */ ,
			       int /* lineNumber */ ,
			       bool /* breakpoint */ ) ) );
		connect(debugger,
			SIGNAL(breakpointItem(QString /* fileName*/,
					      int /* lineNumber */,
					      QString /*templateName*/,
					      QString /* modeName */,
					      bool /* enabled */,
					      int /* id */)),
			this,
			SLOT( breakpointItem(QString /* fileName*/,
					     int /* lineNumber */,
					     QString /*templateName*/,
					     QString /* modeName */,
					     bool /* enabled */,
					     int /* id */)));
		connect(debugger, SIGNAL(resolveItem(QString /*URI*/)),
			this, SLOT(slotProcResolveItem(QString /*URI*/)));
	      }
        }
        debugger->start();
    }
}

void KXsldbgPart::emitOpenFile(QString file, int line, int row)
{
    Q_UNUSED(file)
    Q_UNUSED(line)
    Q_UNUSED(row)
    qWarning("KXsldbgPart::emitOpenFile called but not supported");
}

void KXsldbgPart::continueCmd_activated()
{
    if ( checkDebugger() )
        debugger->slotContinueCmd();

}

void KXsldbgPart::stepCmd_activated()
{
    if ( checkDebugger() )
        debugger->slotStepCmd();
}

void KXsldbgPart::nextCmd_activated()
{
    if ( checkDebugger() )
        debugger->fakeInput("next", true);  // noTr
}


void KXsldbgPart::stepupCmd_activated()
{
    if ( checkDebugger() )
      debugger->fakeInput("stepup", true);  // noTr
}


void KXsldbgPart::stepdownCmd_activated()
{
    if ( checkDebugger() )
      debugger->fakeInput("stepdown", true);  // noTr
}


void KXsldbgPart::dataCmd_activated()
{
    if ( checkDebugger() )
        debugger->slotDataCmd();
}

void
KXsldbgPart::sourceCmd_activated()
{
    if ( checkDebugger() )
        debugger->slotSourceCmd();
}

void
KXsldbgPart::outputCmd_activated()
{
    XsldbgSettingData item;
    if ( ( inspector != 0L ) &&  checkDebugger() && ( configWidget != 0L ) && optionDataModel()
            && optionDataModel()->findSetting(OPTIONS_OUTPUT_FILE_NAME, item)){
        debugger->setOutputFileActive(true);
        lineNoChanged( item.m_value.toString(), 1, false );
        QXsldbgDoc *docPtr = docDictionary[item.m_value.toString()];
        if(docPtr)
            docPtr->refresh();
    }
}

void KXsldbgPart::refreshCmd_activated()
{

    if ( !currentFileName.isEmpty() ){
        QXsldbgDoc *docPtr;
        QHash<QString, QXsldbgDoc*>::const_iterator it;
        for (it = docDictionary.constBegin(); it != docDictionary.constEnd(); ++it){
            docPtr = it.value();
            if(docPtr)
                docPtr->refresh();
        }
        if ( checkDebugger() ){
            debugger->fakeInput("showbreak", true);  // noTr
        }
    }
}

void KXsldbgPart::enableCmd_activated()
{
  if ( checkDebugger() ){
    debugger->slotEnableCmd( currentFileName, currentLineNo);
  }
}

void KXsldbgPart::deleteCmd_activated()
{
  if ( checkDebugger() ){
        debugger->slotDeleteCmd( currentFileName, currentLineNo);
  }
}

void KXsldbgPart::breakCmd_activated()
{
  if ( checkDebugger() ){
    debugger->slotBreakCmd( currentFileName, currentLineNo);
  }
}

void KXsldbgPart::evaluateCmd_activated()
{
  QString expression = KInputDialog::getText(i18n("Evaluate Expression"), i18n("XPath:"));
  if (checkDebugger()  && (expression.length() > 0)){
    debugger->slotCatCmd( expression);
  }
}

void KXsldbgPart::gotoXPathCmd_activated()
{
  QString xpath = KInputDialog::getText(i18n("Goto XPath"), i18n("XPath:"));
  if (checkDebugger() && xpath.length() > 0){
    debugger->slotCdCmd( xpath );
  }
}

void
KXsldbgPart::lineNoChanged(QString fileName, int lineNumber, bool breakpoint)
{
  if ( fileName.isEmpty() ) {
    return;
  }

  openUrl(fileName);

 /* Did we stop at a breakpoint if so move the marker */
  if (breakpoint && currentDoc) {
      currentDoc->selectBreakPoint(lineNumber);
      emit newDebuggerPosition(currentFileName, lineNumber);
  }

  /* Move cursor and update status bar */
  if (currentDoc && currentDoc->kateView()){
    currentDoc->kateView()->setCursorPosition(KTextEditor::Cursor(lineNumber - 1, 0));
	  currentLineNo = lineNumber;
  }
}

void KXsldbgPart::cursorPositionChanged()
{
	if (currentDoc && currentDoc->kateView()){
		KTextEditor::Cursor c = currentDoc->kateView()->cursorPosition();
		currentLineNo = c.line();
		currentColumnNo = c.column();
		currentLineNo++;
		currentColumnNo++;
		emit newCursorPosition(currentFileName, currentLineNo,0);
	}
}

void  KXsldbgPart::docChanged()
{
    if (!currentDoc || currentDoc->kateDoc() || currentDoc->kateView())
	return;
}

void  KXsldbgPart::debuggerStarted()
{
    KCmdLineArgs *args = KCmdLineArgs::parsedArgs();
    if (configWidget != 0L && args){
	if (args->count()){
            QVariant value;
	    int i=0, result=1, noFilesFound = 0;
	    QString expandedName;      /* contains file name with path expansion if any */

	    for (i = 0; i < args->count(); i++) {
		if (!result)
		    break;

		if (args->arg(i)[0] != '-') {
		    expandedName = filesExpandName(args->arg(i));
		    if (expandedName.isEmpty()) {
			result = 0;
			break;
		    }
		    switch (noFilesFound) {
			case 0:
                            value = expandedName;
                            optionDataModel()->updateSetting(OPTIONS_SOURCE_FILE_NAME, value);
			    noFilesFound++;
			    break;
			case 1:
                            value = expandedName;
                            optionDataModel()->updateSetting(OPTIONS_DATA_FILE_NAME, value);
			    noFilesFound++;
			    break;
			case 2:
                            value = expandedName;
                            optionDataModel()->updateSetting(OPTIONS_OUTPUT_FILE_NAME, value);
			    noFilesFound++;
			    break;

			default:
			    xsldbgGenericErrorFunc(i18n("Error: Too many file names supplied via command line.\n"));
			    result = 0;
		    }
		    continue;
		}
	    }
	}else {
            xmlChar *profile=0;
            xsldbgReadConfig(profile);
        }
    }
}

void  KXsldbgPart::debuggerStopped()
{
    delete inspector;
    inspector = 0;
    delete debugger;
    debugger = 0;
}

void  KXsldbgPart::addBreakPoint(int lineNumber)
{
  if ( checkDebugger() ){
    debugger->slotBreakCmd( currentFileName, lineNumber);
  }
}

void  KXsldbgPart::enableBreakPoint(int lineNumber)
{
  if ( checkDebugger() ){
    debugger->slotEnableCmd( currentFileName, lineNumber);
  }
}


void  KXsldbgPart::deleteBreakPoint(int lineNumber)
{
  if ( checkDebugger() ){
        debugger->slotDeleteCmd( currentFileName, lineNumber);
  }
}



void KXsldbgPart::slotSearch()
{
  if ((newSearch != 0L)  && checkDebugger() ) {
    QString msg(QString("search \"%1\"").arg(newSearch->text()));  // noTr
 		debugger->fakeInput(msg, false);
  }
}


void KXsldbgPart::slotEvaluate()
{
  if ((newEvaluate != 0L) && checkDebugger() ){
    debugger->slotCatCmd( newEvaluate->text() );
  }
}

void KXsldbgPart::slotGotoXPath()
{
  if ((newXPath != 0L) && checkDebugger() ){
    debugger->slotCdCmd( newXPath->text() );
  }
}



void KXsldbgPart::slotProcResolveItem(QString URI)
{
  if (!URI.isEmpty()){
    QMessageBox::information(mainView, i18n("SystemID or PublicID Resolution Result"),
        i18n("SystemID or PublicID has been resolved to\n.%1", URI),
	QMessageBox::Ok);
  }
}

void  KXsldbgPart::breakpointItem(QString fileName, int lineNumber ,
			      QString /*templateName*/, QString /* modeName */,
			      bool  enabled , int /* id */)
{

    if (fileName == 0L){
        /* Go through all documents and remove all breakpoints */
        QXsldbgDoc *docPtr=0;
        KTextEditor::MarkInterface *markIf = 0L;
        QHash<QString, QXsldbgDoc*>::iterator it;
        for (it = docDictionary.begin(); it != docDictionary.end(); ++it){
            docPtr = it.value();
            if (docPtr){
                markIf = qobject_cast<KTextEditor::MarkInterface*>( docPtr->kateDoc() );
                if (markIf)
                    markIf->clearMarks();
            }
        }
    }else{
        KUrl url(fileName);
        fetchURL(url);
        QXsldbgDoc *docPtr = docDictionary[url.url()] ;
        if (docPtr)
            docPtr->enableBreakPoint(--lineNumber, enabled);
    }
}



void KXsldbgPart::walkCmd_activated()
{
  if (checkDebugger()){
    debugger->slotWalkCmd();
  }
}

void KXsldbgPart::walkStopCmd_activated()
{
  if (checkDebugger()){
    debugger->slotWalkStopCmd();
  }
}

void KXsldbgPart::traceCmd_activated()
{
  if (checkDebugger()){
    debugger->slotTraceCmd();
  }
}

void KXsldbgPart::traceStopCmd_activated()
{
  walkStopCmd_activated();
}


#include "kxsldbg_part.moc"
