
/**
 *
 *  This file is part of the kdewebdev package
 *  Copyright (c) 2002 Keith Isdale <keith@kdewebdev.org>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 **/

#include <klocale.h>
#include <QTableWidget>
#include <QLineEdit>
#include <QMessageBox>
#include "xsldbgdebugger.h"
#include "xsldbgbreakpointsimpl.h"
#include <kdebug.h> 


XsldbgBreakpointsImpl::XsldbgBreakpointsImpl(XsldbgDebugger *debugger,
	QWidget *parent /*=0*/)
	: QWidget(parent)
{
	setupUi(this);
	this->debugger = debugger;
	connect(debugger, SIGNAL(breakpointItem(QString /* file*/,
                         int /*line number */, QString /*templateName*/,
                         QString /* modeName*/,
	                 bool /* enabled */, int /* id */)),
		this, SLOT(slotProcBreakpointItem(QString /* file*/,
			 int /*line number */, QString /*templateName*/,
			 QString /* modeName */,
			 bool /* enabled */, int /* id */)));
	connect( breakpointView,  SIGNAL(itemSelectionChanged()),
						this, SLOT(itemSelectionChanged()));

	connect(&resizeTimer, SIGNAL(timeout()), this, SLOT(resizeContents()));
	resizeContents();

	show();
	refresh();
}

XsldbgBreakpointsImpl::~XsldbgBreakpointsImpl()
{
    debugger = 0L;
}

int XsldbgBreakpointsImpl::getLineNumber()
{
	bool isOk = false;
	int lineNo = lineNumberEdit->text().toInt(&isOk);
	if (isOk == false)
	    lineNo = -1;

	return lineNo;
}

int XsldbgBreakpointsImpl::getId()
{
	bool isOk = false;
	int id = idEdit->text().toInt(&isOk);
	if (isOk == false){
		id = -1;
		kDebug() << "Invalid line number";
	}

	return id;
}

void XsldbgBreakpointsImpl::slotAddBreakpoint()
{
	int lineNo = getLineNumber();
	if (lineNo != -1) {
	  if (!sourceFileEdit->text().isEmpty()){
	        debugger->slotBreakCmd(sourceFileEdit->text(), lineNo);
	  }else {
	      QMessageBox::information(this, i18n("Operation Failed"),
		  i18n("A line number was provided without a file name."),
  	           QMessageBox::Ok);
	  }
	}else if (!templateNameEdit->text().isEmpty() ||
		  !modeNameEdit->text().isEmpty()){
		debugger->slotBreakCmd(templateNameEdit->text(),
				       modeNameEdit->text());
	}else{
	    QMessageBox::information(this, i18n("Operation Failed"),
		 i18n("No details provided or an invalid line number was supplied."),
  	          QMessageBox::Ok);
	}
}

void  XsldbgBreakpointsImpl::slotAddAllTemplateBreakpoints()
{
  if (debugger != 0L){
    debugger->fakeInput("break *", true);
    debugger->fakeInput("show", true);
  }
}

void XsldbgBreakpointsImpl::slotDeleteBreakpoint()
{
	int lineNo = getLineNumber(), id = getId();
	if (id != -1){
		debugger->slotDeleteCmd(id);
	}else if (lineNo != -1) {
	  if (!sourceFileEdit->text().isEmpty()){
		debugger->slotDeleteCmd(sourceFileEdit->text(), lineNo);
	  }else {
	    QMessageBox::information(this, i18n("Operation Failed"),
	         i18n("A line number was provided without a file name."),
  	         QMessageBox::Ok);
	  }
	}else {
	    QMessageBox::information(this, i18n("Operation Failed"),
		 i18n("No details provided or an invalid line or ID was supplied."),
  	          QMessageBox::Ok);
	}
}

void  XsldbgBreakpointsImpl::slotDeleteAllBreakpoints()
{
  if (debugger != 0L){
    debugger->fakeInput("delete *", true);
    debugger->fakeInput("show", true);
  }
}

void XsldbgBreakpointsImpl::slotEnableBreakpoint()
{
	int lineNo = getLineNumber(), id = getId();
	if (id != -1){
		debugger->slotEnableCmd(id);
	}else if (lineNo != -1){
	  if (!sourceFileEdit->text().isEmpty()){
		debugger->slotEnableCmd(sourceFileEdit->text(), lineNo);
	  }else {
	       QMessageBox::information(this, i18n("Operation Failed"),
	           i18n("A line number was provided without a file name."),
  	           QMessageBox::Ok);
	  }
	}else {
	    QMessageBox::information(this, i18n("Operation Failed"),
		 i18n("No details provided."),
  	          QMessageBox::Ok);
	}
}

void XsldbgBreakpointsImpl::itemSelectionChanged()
{
    QList<QTableWidgetItem *> items = breakpointView->selectedItems();
    if (!items.isEmpty()){
        QString fileName;
        int lineNumber = 0;
        bool ok=false;
        QTableWidgetItem *cellItem;
        for (int i = 0; i < items.size(); ++i) {
            cellItem = items[i];	
            switch(cellItem->column()){
                case IDColumn:
                    idEdit->setText(cellItem->text());
                    break;
                case NameColumn:
                    templateNameEdit->setText(cellItem->text());
                    break;
                case ModeColumn:
                    modeNameEdit->setText(cellItem->text());
                    break;
                case FileNameColumn:
                    sourceFileEdit->setText(cellItem->text());
                    fileName = cellItem->text();
                    break;
                case LineNumberColumn:
                    lineNumber = cellItem->text().toInt(&ok);
                    if (!ok){
                        lineNumber = 0;
                    }else{
                        if (lineNumber != -1) 
                            lineNumberEdit->setText(cellItem->text());
                    }
                    break;
            }
        }
        if (!fileName.isEmpty())
            debugger->gotoLine(fileName, lineNumber);			 
    }
}


void XsldbgBreakpointsImpl::resizeContents()
{
    breakpointView->resizeColumnsToContents();
}

	
void XsldbgBreakpointsImpl::refresh()
{
	/* get xsldbg to tell what breakpoints are set,
	   we'll get the notification back via slotProcBreakpointItem */
	debugger->fakeInput("showbreak", true);
}


void XsldbgBreakpointsImpl::slotClear()
{
	idEdit->setText("");
	templateNameEdit->setText("");
	modeNameEdit->setText("");
 	sourceFileEdit->setText("");
 	lineNumberEdit->setText("");
}

void XsldbgBreakpointsImpl::slotProcBreakpointItem(QString  fileName,
						   int lineNumber ,
						   QString templateName,
						   QString modeName,
						   bool enabled, int id )
{
    static int insertPosition = 0;
    if (fileName.isNull()){
        breakpointView->clearContents();
        insertPosition=0;
    }else{
        resizeTimer.stop();
        if (insertPosition >= breakpointView->rowCount())
            breakpointView->insertRow(insertPosition);
        QTableWidgetItem *cellItem;
        for (int column=IDColumn; column<=EnabledColumn; column++){
            cellItem = breakpointView->item(insertPosition, column);
            if (!cellItem)
                breakpointView->setItem(insertPosition, column, new QTableWidgetItem);
            cellItem = breakpointView->item(insertPosition, column);
            if (cellItem){
                switch(column){
                    case IDColumn:
                        cellItem->setText(QString::number(id));
                        break;
                    case NameColumn:
                        cellItem->setText(templateName);
                        break;
                    case ModeColumn:
                        cellItem->setText(modeName);
                        break;
                    case FileNameColumn:
                        cellItem->setText(fileName);
                        break;
                    case LineNumberColumn:
                        if (lineNumber != -1)
                            cellItem->setText(QString::number(lineNumber));
                        else
                            cellItem->setText("");
                        break;
                    case EnabledColumn:
                        if (enabled) 
                            cellItem->setText(i18n("Enabled"));
                        else
                            cellItem->setText(i18n("Disabled"));
                        break;
                }
            }
        }
        resizeTimer.start();
        insertPosition++;
    }
}




#include "xsldbgbreakpointsimpl.moc"
