/*
* page_export_1394.cc Notebook Firewire/AVI/Still Frame Export Page Object
* Copyright (C) 2001 Dan Dennedy <dan@dennedy.org>
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
using std::cerr;
using std::endl;

#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include "page_export_1394.h"
#include "preferences.h"
#include "ieee1394io.h"
#include "message.h"
#include "frame.h"

/** Constructor for page.

\param exportPage the export page object to bind to
\param common	common object to which this page belongs
*/

Export1394::Export1394( PageExport *_exportPage, KinoCommon *_common ) :
		Export( _exportPage, _common ), avc( NULL ), exportWriter( NULL ),
		system_loaded( false )
{
	cerr << "> Creating Export1394 Page" << endl;

	entry_cip_n = GTK_ENTRY( lookup_widget( common->getWidget(), "entry_export_1394_cip_n" ) );
	entry_cip_d = GTK_ENTRY( lookup_widget( common->getWidget(), "entry_export_1394_cip_d" ) );
	entry_syt_offset = GTK_ENTRY( lookup_widget( common->getWidget(), "entry_export_1394_syt_offset" ) );
	entry_preroll = GTK_ENTRY( lookup_widget( common->getWidget(), "spinbutton_export_1394_preroll" ) );
	feedbackLabel = GTK_LABEL( lookup_widget( common->getWidget(), "label_export_1394_load_messages" ) );
}

/** Destructor for page.
 */

Export1394::~Export1394()
{
	cerr << "> Destroying Export1394 Page" << endl;
}

/** Start of page.
 */

void Export1394::start()
{
	static raw1394handle_t handle;
	GtkLabel * load_label
	= GTK_LABEL( lookup_widget ( exportPage->getWidget(),
	                             "label_export_1394_load_messages" ) );

	gchar s[ 512 ];

	cerr << ">> Starting Export1394" << endl;

	sprintf( s, "%d", Preferences::getInstance().cip_n );
	gtk_entry_set_text( entry_cip_n, s );
	sprintf( s, "%d", Preferences::getInstance().cip_d );
	gtk_entry_set_text( entry_cip_d, s );
	sprintf( s, "%d", Preferences::getInstance().syt_offset );
	gtk_entry_set_text( entry_syt_offset, s );
	sprintf( s, "%d", Preferences::getInstance().dvExportPrerollSec );
	gtk_entry_set_text( entry_preroll, s );

#ifdef LIBRAW1394_OLD

	if ( ( handle = raw1394_get_handle() ) )
	{
#else
	if ( ( handle = raw1394_new_handle() ) )
	{
#endif
		if ( raw1394_set_port( handle, 0 ) >= 0 )
		{
			raw1394_destroy_handle( handle );
			handle = NULL;
			avc = new AVC( Preferences::getInstance().interface );
			system_loaded = true;
			gtk_label_set_text( load_label, "" );
		}
		else
		{
			system_loaded = false;
			gtk_label_set_text( load_label,
			                    _( "Error setting the IEEE 1394 port (host adapater); all IEEE 1394 options are disabled." ) );
		}
	}
	else
	{
		system_loaded = false;
		gtk_label_set_text( load_label,
		                    _( "The IEEE 1394 subsystem is not loaded; all IEEE 1394 options are disabled." ) );
	}
	/* Change the sensitivity of the hosting widget according to
	   wheter the system loaded or not */
	GtkVBox * tmp
	= GTK_VBOX( lookup_widget ( exportPage->getWidget(),
	                            "vbox_export_1394" ) );
	if ( system_loaded )
	{
		gtk_widget_set_sensitive ( GTK_WIDGET ( tmp ), TRUE );
	}
	else
	{
		gtk_widget_set_sensitive ( GTK_WIDGET ( tmp ), FALSE );
	}

}

/** Define active widgets.
    This page does not support pausing, and if the system is
    not loaded, it does not support anything at all.
 */

gulong Export1394::onActivate()
{
	if ( system_loaded )
	{
		return EXPORT_PREVIEW | EXPORT_EXPORT | EXPORT_SCENE_LIST;
	}
	else
	{
		return 0;
	}
}

/** Leaving the page
 */

void Export1394::clean()
{
	cerr << ">> Leaving Export1394" << endl;
	if ( avc )
	{
		delete avc;
		avc = NULL;
	}
	Preferences::getInstance().Save();
}

/** The actual export function */

enum export_result
Export1394::doExport( PlayList * playlist, int begin, int end, int every,
                      bool preview )
{
	int i = -1;
	Preferences::getInstance().cip_n = atoi( gtk_entry_get_text( entry_cip_n ) );
	Preferences::getInstance().cip_d = atoi( gtk_entry_get_text( entry_cip_d ) );
	Preferences::getInstance().syt_offset = atoi( gtk_entry_get_text( entry_syt_offset ) );
	Preferences::getInstance().dvExportPrerollSec = atoi( gtk_entry_get_text( entry_preroll ) );
	enum export_result result = EXPORT_RESULT_FAILURE;
	string filename = "";
	string currentFilename = "";

	exportWriter = new dv1394Writer(
					   Preferences::getInstance().dvExportDevice,
					   Preferences::getInstance().channel,
					   Preferences::getInstance().dvExportBuffers,
					   Preferences::getInstance().cip_n,
					   Preferences::getInstance().cip_d,
					   Preferences::getInstance().syt_offset
				   );

	// Only for export, NOT for preview.
	if ( !preview )
	{
		Preferences::getInstance().phyID = avc->getNodeId( Preferences::getInstance().avcGUID );
		avc->Record( Preferences::getInstance().phyID );
	}

	assert( exportWriter );
	if ( exportWriter->isValid() )
	{
		Frame& frame = *GetFramePool()->GetFrame();

		bool resample = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(
		                    lookup_widget( common->getWidget(), "checkbutton_export_1394_resample" ) ) );

		int16_t *audio_buffers[ 4 ];
		AudioInfo info;
		AudioResample<int16_ne_t,int16_ne_t> *resampler = NULL;

		for ( i = 0; i < 4; i++ )
			audio_buffers[ i ] = ( int16_t * ) calloc( 2 * DV_AUDIO_MAX_SAMPLES, sizeof( int16_t ) );

		/* pre-roll */
		playlist->GetFrame( begin, frame );
		for ( i = begin; i <= end && i <= begin +
		        Preferences::getInstance().dvExportPrerollSec * ( frame.IsPAL() ? 25 : 30 )
		        && exportPage->isExporting; i++ )
		{

			/* re-encode the audio to ensure consistent 16bit bitstream */
			frame.GetAudioInfo( info );
			info.channels = 2;
			frame.EncodeAudio( info, audio_buffers );
			exportWriter->SendFrame( frame );
		}

		/* Iterate over all frames in selection */
		for ( i = begin; i <= end && exportPage->isExporting; i += every )
		{
			/* Call innerLoopUpdate */
			innerLoopUpdate( i, begin, end, every );

			/* display filename */
			currentFilename = playlist->GetFileNameOfFrame( i );
			if ( currentFilename != filename )
			{
				filename = currentFilename;
				gtk_label_set_text( feedbackLabel, filename.c_str() );
			}

			/* Get the frame, write it */
			playlist->GetFrame( i, frame );

			/* resample audio */
			if ( resample )
			{
				if ( resampler == NULL )
				{
					frame.GetAudioInfo( info );
					resampler = AudioResampleFactory<int16_ne_t,int16_ne_t>::createAudioResample(
					                AUDIO_RESAMPLE_SRC_SINC_MEDIUM_QUALITY, info.frequency );
				}
				resampler->Resample( frame );
				info.samples = resampler->size / info.channels / 2;
				int16_t *p = resampler->output;
				for ( int s = 0; s < info.samples; s++ )
					for ( int c = 0; c < info.channels; c++ )
						audio_buffers[ c ][ s ] = *p++;
				frame.EncodeAudio( info, audio_buffers );
			}

			exportWriter->SendFrame( frame );
		}

		if ( !exportPage->isExporting )
			result = EXPORT_RESULT_ABORT;
		else if ( i > end )
			result = EXPORT_RESULT_SUCCESS;
			
		for ( i = 0; i < 4; i++ )
			free( audio_buffers[ i ] );
		if ( resampler != NULL )
			delete resampler;
			
		GetFramePool()->DoneWithFrame( &frame );
	}

	gtk_label_set_text( feedbackLabel, "" );
	if ( !preview )
	{
		avc->Stop( Preferences::getInstance().phyID );
	}

	delete exportWriter;
	exportWriter = NULL;

	return result;
}
