/***************************************************************************
                          knetworkconfigparser.cpp  -  description
                             -------------------
    begin                : Mon Jan 13 2003
    copyright            : (C) 2003 by Juan Luis Baptiste
    email                : jbaptiste@merlinux.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "knetworkconfigparser.h"
#include "knetworkconfigparser.moc"

KNetworkConfigParser::KNetworkConfigParser(){
  networkInfo = new KNetworkInfo();
  
  QString platform;
  bool askAgain = readAskAgain(platform);
    
  if (!askAgain || platform.length() > 0)
      runDetectionScript(platform);
  else 
    runDetectionScript(QString::null);
}
/** Runs the gst backend to get network values. You can pass to the script a specific platform to load using the platform parameter.  */
void KNetworkConfigParser::runDetectionScript(QString platform){
  KDetectDistroDlg* dialog = new KDetectDistroDlg(0, 0);
  dialog->show();
  procDetect = new QProcess(this);
  QString pathToProgram =  locate("data",BACKEND_PATH);
  if (pathToProgram == "")
  {
    KMessageBox::error(0,
                        i18n("Couldn't find the network detection scripts. Something is wrong with your installation.\n Please check that  \n{KDE_PATH}/%1 \nfile is present.").arg(BACKEND_PATH),
                        i18n("Couldn't find network backend script"));
      dialog->close();
      exit(5);
      //kapp->quit();
  }
  else
  {
    procDetect->addArgument( pathToProgram );
    if (platform != QString::null)
    {
      procDetect->addArgument( "--platform" );
      procDetect->addArgument( platform );
    }
    procDetect->addArgument( "--get" );
    connect( this, SIGNAL(readyLoadingNetworkInfo()), dialog, SLOT(close()) );
    connect( this, SIGNAL(errorDetectingPlatform()), dialog, SLOT(close()) );
    connect( procDetect, SIGNAL(processExited()), this, SLOT(readNetworkInfo()) );
    connect( procDetect, SIGNAL(readyReadStdout()),this, SLOT(concatXMLOutputSlot()));
    connect( procDetect, SIGNAL(readyReadStderr()),this, SLOT(readXMLErrSlot()));

    if ( !procDetect->start() )
    {
    // error handling
      KMessageBox::error(0,
                        i18n("Couldn't execute network detection scripts. Something is wrong with your installation."),
                        i18n("Couldn't launch network backend script"));
      dialog->close();
      exit(5);
    }
  }
}
/** runs gst to find out the state of network devices.It runs the command:
$knetworkconf_home/backends/networkconf [--platform platform] -d list_ifaces. */
void KNetworkConfigParser::listIfaces(QString platform){
  procDetect = new QProcess(this);
  procDetect->addArgument( locate("data",BACKEND_PATH) );
  if (platform != QString::null)
  {
    procDetect->addArgument( "--platform" );
    procDetect->addArgument( platform );
  }
  //procDetect->addArgument( "--get" );
  procDetect->addArgument( "-d" );
  procDetect->addArgument( "list_ifaces" );

  connect( procDetect, SIGNAL(processExited()), this, SLOT(readListIfacesSlot()) );
  connect( procDetect, SIGNAL(readyReadStdout()),this, SLOT(concatXMLOutputSlot()));
  connect( procDetect, SIGNAL(readyReadStderr()),this, SLOT(readXMLErrSlot()));

  xmlOuput = "";
  xmlErr = "";
 if ( !procDetect->start() )
  {
// error handling
    KMessageBox::error(0,
                        i18n("Couldn't execute network detection scripts. Something is wrong with your installation."),
                        i18n("Couldn't launch network backend script"));
    exit(5);
  }
}

void KNetworkConfigParser::readListIfacesSlot(){
  QPtrList<KNetworkInterface> tempDeviceList;

  //The gst backend puts a \n at the beginning of the xml output, so
  //we have to erase it first before we parse it.
  xmlOuput = xmlOuput.section('\n',1);
  qDebug("XML -d list_ifaces: %s",xmlOuput.latin1());
    QString err;
    int x,y;
    QDomDocument doc( "network-ifaces");
    if ( !doc.setContent( xmlOuput.utf8(),false,&err,&x,&y  ) )
    {
      KMessageBox::error(0,
                      i18n("Couldn't parse the XML file."),
                      i18n("Error while listing network interfaces"));
        qDebug("error: %s %d,%d",err.latin1(),x,y);
    }
    QDomElement root = doc.documentElement();
    QDomNode node = root.firstChild();
    QString s;
    while( !node.isNull() )
    {
      if ( node.isElement() && node.nodeName() == "interface" )
      {
        QDomElement interface = node.toElement();
        KNetworkInterface *tempDevice = new KNetworkInterface();
        tempDevice = getInterfaceInfo(interface,QString::null);

        if (tempDevice->getDeviceName().lower() != "lo")
	      {
	        KNetworkInterface *originalDevice = getDeviceInfo(tempDevice->getDeviceName());
      	  if (originalDevice == NULL)
      	  {
		        node = node.nextSibling();
		        continue;
	        }
      	  originalDevice->setActive(tempDevice->isActive());
       	  if (tempDevice->getBroadcast() != "")
	          originalDevice->setBroadcast(tempDevice->getBroadcast());
	        if (tempDevice->getDescription() != "")
     	      originalDevice->setDescription(tempDevice->getDescription());
	        if (tempDevice->getIpAddress() != "")
	          originalDevice->setIpAddress(tempDevice->getIpAddress());
	        if (tempDevice->getMacAddress() != "")
	          originalDevice->setMacAddress(tempDevice->getMacAddress());
          if (tempDevice->getNetmask() != "")
	          originalDevice->setNetmask(tempDevice->getNetmask());
	        if (tempDevice->getNetwork() != "")
            originalDevice->setNetwork(tempDevice->getNetwork());
        }
      }
      node = node.nextSibling();
    }
    //networkInfo->setDeviceList(deviceList);
    //Tell to interested parties when the network info is ready.
   emit readyLoadingNetworkInfo();
}

KNetworkConfigParser::~KNetworkConfigParser(){
}

/** return tyhe number of configured devices. */
unsigned KNetworkConfigParser::numDevices(){
  return _numDevices;
}

void KNetworkConfigParser::setProgramVersion(QString ver)
{
  KNetworkConfigParser::programVersion = ver;
}


void KNetworkConfigParser::readIfconfigOutput(){
  QString s = proc->readStdout();
  ifconfigOutput = s;
}

/** 
  Reads /proc/net/route looking for the default gateway. 
  
  NOTE:We should use the gateway reported by gst, but if there's a
  gw in a config file and one of the network interfaces is 
  configured to use dhcp, gst returns the value of the config 
  file instead of the gw configured by dhcp.
*/
void KNetworkConfigParser::loadRoutingInfo(){
//  routingInfo = NULL;
  QFile f( "/proc/net/route");
  if ( !f.open(IO_ReadOnly) )
  {
//    return routingInfo;
    KMessageBox::error(0,
                        i18n("Could not open file"),
                        i18n("Couldn't open file /proc/net/route!"));
  }
  else
  {
//    routingInfo = new KRoutingInfo();
    QTextStream t( &f );        // use a text stream
    QString s;
    while (!t.eof())
    {
      s = t.readLine();       // line of text excluding '\n'
      QString interface = s.section('\t',0,0);      
      QString destination = s.section('\t',1,1);
      QString gw = s.section('\t',2,2);
        
      if (destination == "00000000")
      {
        routingInfo->setGateway(hexIPv4ToDecIPv4(gw));
//        routingInfo->setGateway(gw);
        routingInfo->setGatewayDevice(interface);
      }  
    }
  }
  f.close();
}
QString
KNetworkConfigParser::hexIPv4ToDecIPv4(QString hex)
{
  bool ok;
  QString dec = "";
  QString dec2 = "";
  QString temp = "";
  QString temp2 = "";
  temp = hex.mid(6,2);
  temp2 = temp2.setNum(temp.toInt(&ok,16));
  dec.append(temp2);
  dec.append('.');
  temp = hex.mid(4,2);
  temp2 = temp2.setNum(temp.toInt(&ok,16));  
  dec.append(temp2);
  dec.append('.');
  temp = hex.mid(2,2);
  temp2 = temp2.setNum(temp.toInt(&ok,16));  
  dec.append(temp2);
  dec.append('.');
  temp = hex.mid(0,2);
  temp2 = temp2.setNum(temp.toInt(&ok,16));
  dec.append(temp2);

  dec2 = dec;
  if (ok)
    return dec2;
  else
    return "";
}

void KNetworkConfigParser::saveNetworkInfo(KNetworkInfo *networkInfo)
{
  QPtrList<KNetworkInterface> devList = networkInfo->getDeviceList();
  dnsInfo = networkInfo->getDNSInfo();
  QStringList nameServerList = dnsInfo->getNameServers();
  QPtrList<KKnownHostInfo> knownHostsList = dnsInfo->getKnownHostsList();
  QPtrListIterator<KKnownHostInfo> knownHostsIt(knownHostsList);
  QPtrListIterator<KNetworkInterface> devIt(devList);

  KNetworkInterface *device;
  KKnownHostInfo *host;
  routingInfo = networkInfo->getRoutingInfo();
  
  //Start xml file
  QDomDocument doc( "network []" );
  QDomProcessingInstruction instr = doc.createProcessingInstruction("xml","version=\"1.0\" ");
  doc.appendChild(instr);
  QDomElement root = doc.createElement( "network" );
  doc.appendChild( root );

  //Add routing info
  QDomElement tag = doc.createElement( "gateway" );
  root.appendChild( tag );
  QDomText t = doc.createTextNode( routingInfo->getGateway() );
  tag.appendChild( t );

  tag = doc.createElement( "gatewaydev" );
  root.appendChild( tag );
  t = doc.createTextNode( routingInfo->getGatewayDevice() );
  tag.appendChild( t );

  tag = doc.createElement( "hostname" );
  root.appendChild( tag );
  t = doc.createTextNode( dnsInfo->getMachineName() );
  tag.appendChild( t );
  tag = doc.createElement( "domain" );
  root.appendChild( tag );
  t = doc.createTextNode( dnsInfo->getDomainName() );
  tag.appendChild( t );

  //Add the list of name servers
  for ( QStringList::Iterator it = nameServerList.begin(); it != nameServerList.end(); ++it ) {
    tag = doc.createElement( "nameserver" );
    root.appendChild( tag );
    t = doc.createTextNode( *it );
    tag.appendChild( t );
  }
  
  //Add the list of static hosts
  while ( (host = knownHostsIt.current()) != 0 )
  {
    ++knownHostsIt;
    tag = doc.createElement( "statichost" );
    root.appendChild( tag );
    QDomElement innerTag;
    if (host->getIpAddress() != "")
    {
      innerTag = doc.createElement( "ip" );
      tag.appendChild( innerTag );
      t = doc.createTextNode( host->getIpAddress() );
      innerTag.appendChild( t );
    } 
/*    if (host->getHostName() != "")
    {
      innerTag = doc.createElement( "alias" );
      tag.appendChild( innerTag );
      t = doc.createTextNode( host->getHostName() );
      innerTag.appendChild( t );
    }
*/     
    QStringList aliases = host->getAliases(); 
      
    for ( QStringList::Iterator it = aliases.begin(); it != aliases.end(); ++it ) 
    {
      innerTag = doc.createElement( "alias" );        
      tag.appendChild( innerTag );
      t = doc.createTextNode( *it );
      innerTag.appendChild( t );
    }            
    
  }
  
  //Save in the configuration file the description of the interfaces as
  //the backend no longer handles this
  KSimpleConfig cfg("knetworkconfrc");
  cfg.setGroup("Interfaces");
  
  //Add the network interfaces list
  while ( (device = devIt.current()) != 0 )
  {
    ++devIt;
    tag = doc.createElement( "interface" );
    tag.setAttribute("type",device->getType());
    root.appendChild( tag );
    QDomElement configurationTag;
    configurationTag = doc.createElement( "configuration" );
    tag.appendChild( configurationTag );
    
    QDomElement innerTag;
    if ((device->getBootProto().lower() != "dhcp") && (device->getBootProto().lower() != "bootp"))
    {
      if (device->getIpAddress() != "")
      {
        innerTag = doc.createElement( "address" );
        configurationTag.appendChild( innerTag );
        t = doc.createTextNode( device->getIpAddress() );
        innerTag.appendChild( t );
      }
      if (device->getBroadcast() != "")
      {
        innerTag = doc.createElement( "broadcast" );
        configurationTag.appendChild( innerTag );
        t = doc.createTextNode( device->getBroadcast() );
        innerTag.appendChild( t );
      }
      if (device->getNetmask() != "")
      {
        innerTag = doc.createElement( "netmask" );
        configurationTag.appendChild( innerTag );
        t = doc.createTextNode( device->getNetmask() );
        innerTag.appendChild( t );
      }
      if (device->getNetwork() != "")
      {
        innerTag = doc.createElement( "network" );
        configurationTag.appendChild( innerTag );
        t = doc.createTextNode( device->getNetwork() );
        innerTag.appendChild( t );
      }
    }
    innerTag = doc.createElement( "auto" );
    configurationTag.appendChild( innerTag );

    if (device->getOnBoot().lower() == "yes")
      t = doc.createTextNode( "1" );
    else
      t = doc.createTextNode( "0" );

    innerTag.appendChild( t );

    innerTag = doc.createElement( "bootproto" );
    configurationTag.appendChild( innerTag );
    if (device->getBootProto().lower() == "manual")
      t = doc.createTextNode( "none" );
    else
      t = doc.createTextNode( device->getBootProto().lower() );
    innerTag.appendChild( t );

    innerTag = doc.createElement( "file" );
    configurationTag.appendChild( innerTag );
    t = doc.createTextNode( device->getDeviceName() );
    innerTag.appendChild( t );
    
    innerTag = doc.createElement( "dev" );
    tag.appendChild( innerTag );
    t = doc.createTextNode( device->getDeviceName() );
    innerTag.appendChild( t );

    innerTag = doc.createElement( "enabled" );
    tag.appendChild( innerTag );
    if (device->isActive())
      t = doc.createTextNode( "1" );
    else  
      t = doc.createTextNode( "0" );
    innerTag.appendChild( t );

    innerTag = doc.createElement( "hwaddr" );
    tag.appendChild( innerTag );
    t = doc.createTextNode( device->getMacAddress() );
    innerTag.appendChild( t );        

    if (device->getDescription() != "" && device->getDeviceName()!= "lo")
        cfg.writeEntry(device->getDeviceName(),device->getDescription());
    
    cfg.sync();
  }

  //If we don't add this comment to the end of the xml file, the gst process never exits!
  QDomComment endComment = doc.createComment(" GST: end of request ");
  doc.appendChild( endComment );

  QString xml = doc.toString();
  qDebug("--set XML:\n%s",xml.latin1());
  procSaveNetworkInfo = new QProcess(this);
  procSaveNetworkInfo->addArgument( locate("data",BACKEND_PATH) );

  if (!networkInfo->getPlatformName().isEmpty())
  {
    procSaveNetworkInfo->addArgument( "--platform" );
    procSaveNetworkInfo->addArgument( networkInfo->getPlatformName() );
  }
  procSaveNetworkInfo->addArgument( "--set" );

//  KDetectDistroDlg* dialog = new KDetectDistroDlg(0, 0, true,QDialog::WStyle_Customize|QDialog::WStyle_NormalBorder|QDialog::WStyle_Title|QDialog::WStyle_SysMenu); //made it semi-modal
  KDetectDistroDlg* dialog = new KDetectDistroDlg(0, 0);
  dialog->setCaption(i18n("Reloading the network"));
  dialog->text->setText(i18n("%1Please wait while reloading the Network...%2").arg("<center>").arg("</center>"));
  dialog->show();

  xmlOuput = "";

  connect( this, SIGNAL(readyLoadingNetworkInfo()), dialog, SLOT(close()) );
  connect(procSaveNetworkInfo,SIGNAL(readyReadStdout()),this,SLOT(readFromStdoutSaveNetworkInfo()));
  connect(procSaveNetworkInfo,SIGNAL(wroteToStdin()),this,SLOT(sendNetworkInfoSavedSignalSlot()));
  connect(procSaveNetworkInfo,SIGNAL(processExited()),this,SLOT(listIfacesSlot()));
   

  if ( !procSaveNetworkInfo->start() )
  {
    KMessageBox::error(0,
                        i18n("Couldn't execute network saving scripts. Something is wrong with your installation."),
                        i18n("Couldn't launch network backend script"));
  }

  procSaveNetworkInfo->writeToStdin(xml);

}

/** Loads the network info from a xml file generated by the gnome system tools
network backends that are included with this app. */
KNetworkInfo * KNetworkConfigParser::getNetworkInfo(){

    return networkInfo;
}
/** Parses all of the <interface>...</interface> entries in the xml configuration file. */
KNetworkInterface * KNetworkConfigParser::getInterfaceInfo(QDomElement interface, QString type){
  QDomNode node = interface.firstChild();
  KNetworkInterface *tempDevice = new KNetworkInterface();
  //tempDevice->setDescription(i18n("Ethernet Network Device"));

  while ( !node.isNull() )
  {
    if ( node.isElement() )
    {
      QString nodeName =node.nodeName();
      
      //Parsing --get interfaces configuration
      if ( node.isElement() && node.nodeName() == "configuration" )
      {
        QDomNode configNode = node.firstChild();
        while ( !configNode.isNull() )
        {
          if ( configNode.isElement() )
          {
            QString configNodeName =configNode.nodeName();
            
            if ( configNodeName == "auto" )
            {
              QDomElement e = configNode.toElement();
              if (e.text() == "1")
                tempDevice->setOnBoot("yes");
              else
                tempDevice->setOnBoot("no");
            }
            else if ( configNodeName == "bootproto" )
            {
              QDomElement e = configNode.toElement();
              tempDevice->setBootProto(e.text());
            }
            if ( configNodeName == "address" || configNodeName == "addr")
            {
              QDomElement e = configNode.toElement();
              if (e.text() != "")
	              tempDevice->setIpAddress(e.text());
            }
            else if ( configNodeName == "netmask" || configNodeName == "mask")
            {
              QDomElement e = configNode.toElement();
              if (e.text() != "")
            	  tempDevice->setNetmask(e.text());
            }
            else if ( configNodeName == "network" )
            {
              QDomElement e = configNode.toElement();
              if (e.text() != "")
                tempDevice->setNetwork(e.text());
            }
            else if ( configNodeName == "broadcast" || configNodeName == "bdcast")
            {
              QDomElement e = configNode.toElement();
              if (e.text() != "")
	              tempDevice->setBroadcast(e.text());
            }
            configNode = configNode.nextSibling();
          }
        }  
      }
      
      //Parse -d list_ifaces interfaces configuration
      if ( nodeName == "addr")
      {
        QDomElement e = node.toElement();
        if (e.text() != "")
	       tempDevice->setIpAddress(e.text());
      }
      else if ( nodeName == "mask")
      {
        QDomElement e = node.toElement();
        if (e.text() != "")
          tempDevice->setNetmask(e.text());
      }
      else if ( nodeName == "bdcast")
      {
        QDomElement e = node.toElement();
        if (e.text() != "")
	        tempDevice->setBroadcast(e.text());
      }
     
       
      //These ones are common for both --get and -d list_ifaces
      else if ( nodeName == "dev" )
      {
        QDomElement e = node.toElement();
        tempDevice->setDeviceName(e.text());
      }
      //we had to add the OR because the xml sintax when listing the interfaces
      //is different than when loading the network info. ie.: enabled->active
      //address->addr, etc...
      else if ( nodeName == "enabled"  || nodeName == "active")
      {
        QDomElement e = node.toElement();
        if (e.text() == "1")        
          tempDevice->setActive(true);
        else
          tempDevice->setActive(false);  
      }
      else if ( nodeName == "hwaddr" )
      {
        QDomElement e = node.toElement();
        if (e.text() != "")
          tempDevice->setMacAddress(e.text());
      }
    }
    node = node.nextSibling();
  }
  if (type != QString::null)
    tempDevice->setType(type);
  
  QString description;
  KSimpleConfig cfg("knetworkconfrc");
  cfg.setGroup("Interfaces");
  description = cfg.readEntry(tempDevice->getDeviceName());
  if (description != "")
    tempDevice->setDescription(description);
  else
  {
    if (tempDevice->getType() == "ethernet")
      tempDevice->setDescription(i18n("Ethernet Network Device"));
    else if (tempDevice->getType() == "wireless")
      tempDevice->setDescription(i18n("Wireless Network Device"));
  }

  //If the backend reports that the interface is up e(ven if it isn't true),
  //and the interface doesn't have an IP address assgined it means that effectively
  //the interface is down and should be marked as inactive.
  if (tempDevice->isActive() && tempDevice->getIpAddress() == "")
    tempDevice->setActive(false);
    
  //Clear IP address settings if boot protocol is dhcp or bootp, in case that they are
  //setted in the config files.
  if ((tempDevice->getBootProto().lower() == "dhcp") || (tempDevice->getBootProto().lower() == "bootp"))
  {
    tempDevice->setIpAddress("");
    tempDevice->setNetmask("");
    tempDevice->setNetwork("");
    tempDevice->setBroadcast("");
  }

  return tempDevice;
}

/** Parses all of the <statichost>...</statichost> entries in the xml configuration file. */
KKnownHostInfo * KNetworkConfigParser::getStaticHostInfo(QDomElement host)
{
  QDomNode node = host.firstChild();
  KKnownHostInfo *tempHost = new KKnownHostInfo();

  while ( !node.isNull() )
  {
    if ( node.isElement() )
    {
      QString nodeName =node.nodeName();
      
      if ( nodeName == "ip")
      {
        QDomElement e = node.toElement();
        if (e.text() != "")
	       tempHost->setIpAddress(e.text());
      }
      else if ( nodeName == "alias")
      {
        QDomElement e = node.toElement();
        if (e.text() != "")
          tempHost->addAlias(e.text());
      }
    }  
    node = node.nextSibling();
  }

  return tempHost;
}

/** Reads the xml with the network info. */
void KNetworkConfigParser::readNetworkInfo()
{
  dnsInfo = NULL;
  QStringList serverList;
  
  routingInfo = NULL;
  QPtrList<KNetworkInterface> tempDeviceList;
  routingInfo = new KRoutingInfo();
  dnsInfo = new KDNSInfo();
  QPtrList<KKnownHostInfo> knownHostsList;

  deviceList.clear();
  //The gst backend puts a \n at the beginning of the xml output, so
  //we have to erase it first before we can parse it.
  xmlOuput = xmlOuput.section('\n',1);
  qDebug("--get XML:\n%s",xmlOuput.latin1());

  //If the platform where knetworkconf is running isn't supported, show the
  //user a dialog with all the supported platforms to choose.
  if (xmlErr.contains("platform_unsup::"))
  {    
      connect( this, SIGNAL(readyLoadingSupportedPlatforms()), this, SLOT(showSupportedPlatformsDialogSlot()) );
      loadSupportedPlatforms();
      emit errorDetectingPlatform();  
  }
  else  //parse the XML file
  {
    QString err;
    int x,y;
    QDomDocument doc( "network");
    if ( !doc.setContent( xmlOuput.utf8(),false,&err,&x,&y  ) )
    {
      KMessageBox::error(0,
                      i18n("Couldn't parse the XML file."),
                      i18n("Error"));
        qDebug("error: %s %d,%d",err.latin1(),x,y);
    }
    //dnsInfo = loadDNSInfo();
    dnsInfo = new KDNSInfo();  
      
    QDomElement root = doc.documentElement();
    QDomNode node = root.firstChild();
    QString s;
    
    while( !node.isNull() )
    {
      if ( node.isElement())
      {
        QString nodeName = node.nodeName();
        
        //Get the gatway
        if ( nodeName == "gateway" )
        {
          QDomElement gateway = node.toElement();
          routingInfo->setGateway(gateway.text());
        }        
        if ( nodeName == "gatewaydev" )
        {
          QDomElement gatewaydev = node.toElement();
          routingInfo->setGatewayDevice(gatewaydev.text());
        }
        
        //Get the network interfaces
        else if ( nodeName == "interface" )
        {
          QDomElement interface = node.toElement();
          KNetworkInterface *tempDevice = new KNetworkInterface();
        
          QString attr = interface.attribute("type").lower();  
          if ( attr != "modem" ){          
            tempDevice = getInterfaceInfo(interface,attr);
            tempDevice->setType(attr);
            deviceList.append(tempDevice);
          }
          /*
          else if (interface.attribute("type","").lower() == "wireless"){
            KNetworkInterface *tempDevice = new KNetworkInterface();
            tempDevice = getInterfaceInfo(interface);
          }  */
          //if ((/*attr != "loopback" && */attr != "modem"))
           
        }
        
        //Get domain and host names
        else if ( nodeName == "hostname" )
        {
          QDomElement hostname = node.toElement();
          dnsInfo->setMachineName(hostname.text());
        }
        else if ( nodeName == "domain" )
        {
          QDomElement domainname = node.toElement();
          dnsInfo->setDomainName(domainname.text());
        }        
        
        //Get the nameServers
        else if ( nodeName == "nameserver" )
        {
          QDomElement nameserver = node.toElement();
          serverList.append(nameserver.text());
        }        
        
        //Get the static hosts
        else if ( nodeName == "statichost" )
        {
          QDomElement host = node.toElement();       
          KKnownHostInfo *tempHost = getStaticHostInfo(host);
          knownHostsList.append(tempHost); 
        }      
      }
      node = node.nextSibling();
    }
    dnsInfo->setNameServers(serverList);
    dnsInfo->setKnownHostsList(knownHostsList);
    
    listIfaces(networkInfo->getPlatformName());
    networkInfo->setDeviceList(deviceList);
    loadRoutingInfo();
    networkInfo->setRoutingInfo(routingInfo);
    networkInfo->setDNSInfo(dnsInfo);
    //Tell to anyone interested when the network info is loaded.
    //emit readyLoadingNetworkInfo();
    }
}
void KNetworkConfigParser::readFromStdoutReloadScript(){
  QString s = procReloadNetwork->readStdout();
  reloadScriptOutput.append(s);
}
/** emits a signal when the network changes have benn saved. */
void KNetworkConfigParser::sendNetworkInfoSavedSignalSlot(){
//  loadDhcpDevices();
  procSaveNetworkInfo->closeStdin();
 // emit readySavingNetworkInfo();
}
/** Concatenates into a QString the xml output of the network backend. */
void KNetworkConfigParser::concatXMLOutputSlot(){
  xmlOuput.append(procDetect->readStdout());
}

void KNetworkConfigParser::readXMLErrSlot(){
  xmlErr.append(procDetect->readStderr());
}
/** lists all platforms supported by GST. */
void KNetworkConfigParser::loadSupportedPlatforms(){
	procDetect = new QProcess(this);
  procDetect->addArgument( locate("data",BACKEND_PATH) );
  procDetect->addArgument( "-d" );
  procDetect->addArgument( "platforms" );
  connect( procDetect, SIGNAL(processExited()), this, SLOT(readSupportedPlatformsSlot()) );
  xmlOuput = "";
  connect( procDetect, SIGNAL(readyReadStdout()),this, SLOT(concatXMLOutputSlot()));
//  connect( procDetect, SIGNAL(readyReadStderr()),this, SLOT(readXMLErrSlot()));

 	if ( !procDetect->start() )
  {
// error handling
    KMessageBox::error(0,
                        i18n("Couldn't list supported platforms. Something is wrong with your installation."),
                        i18n("Couldn't launch network backend script"));
    exit(5);
	}

}
/** Parses the xml ouput generated by GST that has all the supported platforms. */
void KNetworkConfigParser::readSupportedPlatformsSlot(){
  //The gst backend puts a \n at the beginning of the xml output, so
  //we have to erase it first before we parse it.
  xmlOuput = xmlOuput.section('\n',1);
  QDomDocument doc( "platforms" );
  if ( !doc.setContent( xmlOuput.utf8() ) )
  {
    KMessageBox::error(0,
                         i18n("Couldn't parse the XML file of supported platforms."),
                         i18n("Error"));
  }
  QDomElement root = doc.documentElement();
  QDomNode node = root.firstChild();
  QString s;
  while( !node.isNull() )
  {
    if ( node.isElement() && node.nodeName() == "platform" )
    {
      QDomElement platform = node.toElement();
      s = getPlatformInfo(platform);
    }
    supportedPlatformsList << s;
    node = node.nextSibling();
  }
  emit readyLoadingSupportedPlatforms();
}
/** Returns the info of a platform in the form of 'key:value' . */
QString KNetworkConfigParser::getPlatformInfo(QDomElement platform){
  QDomNode node = platform.firstChild();
  QString s;
  while ( !node.isNull() )
  {
    if ( node.isElement() )
    {
      if ( node.nodeName() == "key" )
      {
        QDomElement e = node.toElement();
        s += e.text();
        s += ":";
      }
      else if ( node.nodeName() == "name" )
      {
        QDomElement e = node.toElement();
        s += e.text();
      }

    }
    node = node.nextSibling();
  }

  return s;
}
/** Shows the dialog with all the supported platforms by GST. */
void KNetworkConfigParser::showSupportedPlatformsDialogSlot(){
  KSelectDistroDlg* dialog = new KSelectDistroDlg(0, 0);

  for ( QStringList::Iterator it = supportedPlatformsList.begin(); it != supportedPlatformsList.end(); ++it )
  {
    QString key = (*it);
    key = key.section(":",0,0);
    QString name = (*it);
    name = name.section(":",1,1);
    if (key.contains("debian"))
    {
      QPixmap distroImg(locate("data","knetworkconf/pixmaps/debian.png"));
      dialog->klbDistroList->insertItem(distroImg, name);
    }
    else if (key.contains("mandrake"))
    {
      QPixmap distroImg(locate("data","knetworkconf/pixmaps/mandrake.png"));
      dialog->klbDistroList->insertItem(distroImg, name);
    }
    else if (key.contains("conectiva"))
    {
      QPixmap distroImg(locate("data","knetworkconf/pixmaps/conectiva.png"));
      dialog->klbDistroList->insertItem(distroImg, name);
    }    
    else if (key.contains("pld"))
    {
      QPixmap distroImg(locate("data","knetworkconf/pixmaps/pld.png"));
      dialog->klbDistroList->insertItem(distroImg, name);
    }
    else if (key.contains("redhat"))
    {
      QPixmap distroImg(locate("data","knetworkconf/pixmaps/redhat.png"));
      dialog->klbDistroList->insertItem(distroImg, name);
    }
    else if (key.contains("suse"))
    {
      QPixmap distroImg(locate("data","knetworkconf/pixmaps/suse.png"));
      dialog->klbDistroList->insertItem(distroImg, name);
    }
    else if (key.contains("turbolinux"))
    {
      QPixmap distroImg(locate("data","knetworkconf/pixmaps/turbolinux.png"));
      dialog->klbDistroList->insertItem(distroImg, name);
    }
    else if (key.contains("fedora"))
    {
      QPixmap distroImg(locate("data","knetworkconf/pixmaps/fedora.png"));
      dialog->klbDistroList->insertItem(distroImg, name);
    }
    else if (key.contains("openna"))
    {
      QPixmap distroImg(locate("data","knetworkconf/pixmaps/openna.png"));
      dialog->klbDistroList->insertItem(distroImg, name);
    }
    else if (key.contains("slackware"))
    {
      QPixmap distroImg(locate("data","knetworkconf/pixmaps/slackware.png"));
      dialog->klbDistroList->insertItem(distroImg, name);
    }
    else if (key.contains("freebsd"))
    {
      QPixmap distroImg(locate("data","knetworkconf/pixmaps/freebsd.png"));
      dialog->klbDistroList->insertItem(distroImg, name);
    }
    else if (key.contains("gentoo"))
    {
      QPixmap distroImg(locate("data","knetworkconf/pixmaps/gentoo.png"));
      dialog->klbDistroList->insertItem(distroImg, name);
    }
    else if (key.contains("blackpanther"))
    {
      QPixmap distroImg(locate("data","knetworkconf/pixmaps/blackpanther.png"));
      dialog->klbDistroList->insertItem(distroImg, name);
    }
    else if (key.contains("specifix"))
    {
      QPixmap distroImg(locate("data","knetworkconf/pixmaps/specifix.png"));
      dialog->klbDistroList->insertItem(distroImg, name);
    }    
    else if (key.contains("vine"))
    {
      QPixmap distroImg(locate("data","knetworkconf/pixmaps/vine.png"));
      dialog->klbDistroList->insertItem(distroImg, name);
    }    
  } 
  
  if (!dialog->exec())
    //kapp->quit();
    emit setReadOnly(true);
  else
  {
    int i = 0;
    QStringList::Iterator it = supportedPlatformsList.begin();
    while (i < dialog->klbDistroList->currentItem())
    {
      i++;
      ++it;
    }
    QString key = (*it);
    key = key.section(":",0,0); 
    //clean variables and run again the detection script but now don't
    //auto-detect.
    xmlErr = "";
    xmlOuput = "";
    networkInfo->setPlatformName(key);
    runDetectionScript(key);
    if (dialog->cbAskAgain->isChecked())
      saveAskAgain(key,!dialog->cbAskAgain->isChecked());
  }  
}

void KNetworkConfigParser::saveAskAgain(QString platform, bool askAgain)
{
  KSimpleConfig cfg("knetworkconfrc");
  cfg.setGroup("General");
  cfg.writeEntry("detectedPlatform",platform);
  cfg.writeEntry("askAgainPlatform",askAgain);
  cfg.sync();
}

bool KNetworkConfigParser::readAskAgain(QString &platform)
{
  KSimpleConfig cfg("knetworkconfrc");
  cfg.setGroup("General");
  platform = cfg.readEntry("detectedPlatform");
  return cfg.readEntry("askAgainPlatform");
}

/** No descriptions */
void KNetworkConfigParser::readFromStdoutSaveNetworkInfo(){
  xmlOuput.append(procSaveNetworkInfo->readStdout());
}
/** Calls runDetectionScript(). */
void KNetworkConfigParser::listIfacesSlot(){
  listIfaces(networkInfo->getPlatformName());
}
/**Returns the info of the network device 'device or NULL if not found.'*/
KNetworkInterface * KNetworkConfigParser::getDeviceInfo(QString device){
  QPtrListIterator<KNetworkInterface>  i(deviceList);
  KNetworkInterface *temp;
  while ((temp = i.current()) != 0)
  {
    if (temp->getDeviceName() == device)
    {
      return temp;
    }
    ++i;
  }
  return NULL;
}
