/* This file is part of the KDE project
 * Copyright (C) 2009 Fredy Yanardi <fyanardi@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "SearchSuggestion.h"

#include <KDebug>
#include <KUrl>
#include <kio/scheduler.h>

#include "engines/GoogleSuggestionEngine.h"

SearchSuggestion::SearchSuggestion(QObject *parent)
    : QObject(parent)
{
    // Register all available Suggestion Engines here
    GoogleSuggestionEngine *googleEng = new GoogleSuggestionEngine(this);
    m_enginesMap.insert(googleEng->engineName(), googleEng);
}

bool SearchSuggestion::isEngineAvailable(const QString &searchProvider)
{
    return m_enginesMap.contains(searchProvider);
}

void SearchSuggestion::requestSuggestion(const QString &searchProvider, const QString &searchText)
{
    if (!m_enginesMap.contains(searchProvider)) {
        // There is no such engine registered
        return;
    }
    m_activeEngine = m_enginesMap.value(searchProvider);

    const QString requestURL = m_activeEngine->requestURL();

    const QString encodedText = QUrl::toPercentEncoding(searchText);
    QString subsReqURL = requestURL;
    subsReqURL.replace("\\{@}", encodedText);

    KUrl url(subsReqURL);
    m_jobData.clear();
    KIO::TransferJob *job = KIO::get(url, KIO::NoReload, KIO::HideProgressInfo);
    connect(job, SIGNAL(data(KIO::Job *, const QByteArray &)),
            this, SLOT(dataReceived(KIO::Job *, const QByteArray &)));
    connect(job, SIGNAL(result(KJob *)), SLOT(jobFinished(KJob *)));
}

void SearchSuggestion::jobFinished(KJob *job)
{
    if (job->error()) {
        return; // just silently return
    }

    const QStringList suggestionsList = m_activeEngine->parseSuggestion(m_jobData);
    kDebug(1202) << "Received suggestion from " << m_activeEngine->engineName() << ": " << suggestionsList;

    emit suggestionReceived(suggestionsList);
}

void SearchSuggestion::dataReceived(KIO::Job *job, const QByteArray &data)
{
    Q_UNUSED(job);
    m_jobData.append(data);
}

#include "SearchSuggestion.moc"

