/*
 * Copyright (c) 2001-2003 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 *
 * $HEADER$
 *
 * $Id: ssi_boot_open.c,v 1.19.2.1 2004/01/02 00:10:06 jsquyres Exp $
 *
 */

#include <lam_config.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <syslog.h>

#include <typical.h>
#include <net.h>
#include <etc_misc.h>
#include <lamdebug.h>
#include <lam-ssi-boot.h>
#include <all_opt.h>


/*
 * The following file was created by configure.  It contains extern
 * statements and the definition of an array of pointers to each
 * module's public lam_ssi_rpi_t struct.
 */

#include "lam-ssi-boot-modules.h"


/*
 * Global variables
 */

lam_ssi_boot_actions_t lam_ssi_boot;
lam_ssi_boot_t lam_ssi_boot_base_module;
int lam_ssi_boot_verbose = -1;
int lam_ssi_boot_did = -1;
OPT *lam_ssi_boot_optd = NULL;


/*
 * local variables
 */
static lam_debug_stream_info_t lds = {
  0, -1, -1, NULL, "boot:", -1, -1, -1, -1, "ssi-boot.txt"
};


/*
 * Local functions
 */
static int check_specific_module(OPT *aod, 
				 lam_ssi_boot_location_t where, char *name);
static int check_all_modules(OPT *aod, lam_ssi_boot_location_t where);


/*
 * Figure out which boot SSI module to use.  Also determine verbosity
 * level.
 *
 *  Verbosity level:
 *   < 0 - only print things if you really, really have to
 *   = 0 - normal, errors and warnings only, behavior
 *   > 0 - increased verbosity.  Higher the number, the more to print
 */
int
lam_ssi_boot_open(OPT *aod, lam_ssi_boot_location_t where)
{
  int i;
  char *env;

  lam_ssi_boot_optd = aod;

  /* Call to base SSI open */

  lam_ssi_base_open(aod);

  /* Setup verbosity for this kind */

  lam_ssi_base_set_verbose("LAM_MPI_SSI_boot_verbose", &lds, aod,
			   &lam_ssi_boot_verbose, &lam_ssi_boot_did);
  if (lam_ssi_boot_verbose >= 1)
    lam_debug(lam_ssi_boot_did, " Opening");
  
  /* See if a specific boot module was requested */
  
  if ((env = getenv("LAM_MPI_SSI_boot")) != NULL)
    i = check_specific_module(aod, where, env);
  
  /* Nope -- a specific one was not selected.  Go choose one. */
  
  else
    i = check_all_modules(aod, where);
  
  if (i == LAMERROR)
    return LAMERROR;
  
  if (lam_ssi_boot_verbose > 0)
    lam_debug(lam_ssi_boot_did, " Selected boot module %s", 
	      lam_ssi_boot_base_module.lsb_meta_info.ssi_module_name);
  
  return 0;
}


/*
 * A specific module was selected on the command line.  If a module by
 * that name is found, call its open (if it exists) and init
 * functions.  If they both return happiness, that module is selected.
 * Otherwise, call its finalize and close functions and return an
 * error (i.e., don't try to find any other available modules).
 */
static int
check_specific_module(OPT *aod, lam_ssi_boot_location_t where, char *name)
{
  int i;
  int priority;
  const lam_ssi_t *ls;
  const lam_ssi_boot_actions_t *lba = NULL;
  
  if (lam_ssi_boot_verbose > 10)
    lam_debug(lam_ssi_boot_did, " looking for module named %s", name);
  
  for (i = 0; lam_ssi_boot_modules[i] != NULL; ++i) {
    ls = &(lam_ssi_boot_modules[i]->lsb_meta_info);
    if (strcmp(name, ls->ssi_module_name) == 0) {
      if (lam_ssi_boot_verbose > 10)
	lam_debug(lam_ssi_boot_did, " opening module %s", ls->ssi_module_name);
      if (ls->ssi_open_module == NULL ||
	  (ls->ssi_open_module != NULL &&
	   ls->ssi_open_module(aod) == 1)) {
	
	/* If it's available, run its init function and see if it
	   wants to run.  And if so, ignore the priority but get its
	   function list. */
	
	if (lam_ssi_boot_verbose > 10)
	  lam_debug(lam_ssi_boot_did, " initializing module %s", 
		    ls->ssi_module_name);
	lba = lam_ssi_boot_modules[i]->lsb_init(where, &priority);
	
	/* If we got NULL back, then this module does not want to run.
	   Give up in despair. */
	
	if (lba == NULL) {
	  if (lam_ssi_boot_modules[i]->lsb_finalize != NULL)
	    lam_ssi_boot_modules[i]->lsb_finalize();
	  
	  if (ls->ssi_close_module != NULL)
	    ls->ssi_close_module();
	  
	  show_help("boot-ssi", "selected-module-unavailable", name);
	  return LAMERROR;
	}
	
	/* Otherwise, we're found the module, and it wants to run.
	   We're done! */
	
	break;
      }

      /* Otherwise, this module's init function did not want to run.
	 Abort. */

      else {
	show_help("boot-ssi", "selected-module-unavailable", name);
	return LAMERROR;
      }
    }
  }

  /* Did we find one? */

  if (lam_ssi_boot_modules[i] == NULL) {
    show_help("ssi", "module-not-found", "boot", name, NULL);
    return LAMERROR;
  }

  /* We found one.  Load up the global variables for the boot
     module */

  lam_ssi_boot = *lba;
  lam_ssi_boot_base_module = *lam_ssi_boot_modules[i];

  return 0;
}


/*
 * Call open on all the available modules (if it exists).  If open
 * returns happiness (or doesn't exist), call the init function to see
 * if that module wants to run.  If it does, do the priority matchup
 * and remember the one with the best priority.  Fill the global
 * structs with the data from the winner.  Call finalize and close on
 * all the losers that we invoked initialize on.
 *
 * Return LAMERROR if there are no modules available.
 */
static int
check_all_modules(OPT *aod, lam_ssi_boot_location_t where)
{
  int i, cur_priority, best_priority, best_index;
  int *ran_init;
  const lam_ssi_boot_actions_t *cur_lba, *best_lba = NULL;
  const lam_ssi_t *ls;

  /* Setup space to save who we ran the init function on */

  for (i = 0; lam_ssi_boot_modules[i] != NULL; ++i)
    continue;
  ran_init = malloc(sizeof(int) * i);
  if (ran_init == NULL)
    return LAMERROR;
  memset(ran_init, 0, sizeof(int) * i);

  /* Call the open function in every module and see if they want to
     run */

  best_index = -1;
  best_priority = -1;
  best_lba = NULL;
  for (i = 0; lam_ssi_boot_modules[i] != NULL; ++i) {
    ls = &(lam_ssi_boot_modules[i]->lsb_meta_info);
    if (lam_ssi_boot_verbose > 0)
      lam_debug(lam_ssi_boot_did, " opening module %s", ls->ssi_module_name);
    if (ls->ssi_open_module == NULL ||
	(ls->ssi_open_module != NULL &&
	 ls->ssi_open_module(aod) == 1)) {
      
      /* If it's available, run its init function and see if it wants
	 to run.  And if so, what's its priority and function list. */

      if (lam_ssi_boot_verbose > 0)
	lam_debug(lam_ssi_boot_did, " initializing module %s", 
		  ls->ssi_module_name);
      cur_lba = lam_ssi_boot_modules[i]->lsb_init(where, &cur_priority);
      ran_init[i] = 1;
      if (cur_lba != NULL) {
	if (lam_ssi_boot_verbose >= 10)
	  lam_debug(lam_ssi_boot_did, 
		    " module available: %s, priority: %d", 
		    ls->ssi_module_name, cur_priority);
	
	cur_priority = LAM_min(cur_priority, 100);
	if (cur_priority > best_priority) {
	  best_index = i;
	  best_priority = cur_priority;
	  best_lba = cur_lba;
	}
      } else {
	if (lam_ssi_boot_verbose >= 10)
	  lam_debug(lam_ssi_boot_did, " module not available: %s",
		    ls->ssi_module_name);
      }
    }
  }

  /* If there were no modules available, it's an error. */

  if (best_index == -1) {
    if (lam_ssi_boot_verbose > 1)
      lam_debug(lam_ssi_boot_did, " No boot moduless available!");
    show_help("ssi-boot", "none-available", NULL);
    free(ran_init);
    return LAMERROR;
  }

  /* Otherwise, load up the global variables for the boot module */

  lam_ssi_boot = *best_lba;
  lam_ssi_boot_base_module = *lam_ssi_boot_modules[best_index];

  /* Since we have a winner, call finalize and close on all the rest
     to tell them that they will not be run.  */

  for (i = 0; lam_ssi_boot_modules[i] != NULL; ++i) {
    if (i == best_index || ran_init[i] != 1)
      continue;

    ls = &(lam_ssi_boot_modules[i]->lsb_meta_info);

    /* Only run finalize on the modules that we actually called init
       on */

    if (ran_init[i] == 1) {
      if (lam_ssi_boot_verbose > 0)
	lam_debug(lam_ssi_boot_did, " finalizing module %s", 
		  ls->ssi_module_name);
      if (lam_ssi_boot_modules[i]->lsb_finalize != NULL)
	lam_ssi_boot_modules[i]->lsb_finalize();
    }

    /* Run everyone's module_close */

    if (lam_ssi_boot_verbose > 0)
      lam_debug(lam_ssi_boot_did, " closing module %s", ls->ssi_module_name);
    if (ls->ssi_close_module != NULL)
      ls->ssi_close_module();
  }
  free(ran_init);

  return 0;
}
