/*
 * Copyright (c) 2001-2003 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 * $HEADER$
 *
 * $Id: ssi_crmpi_open.c,v 1.7 2003/04/17 03:48:30 ssankara Exp $
 *
 */

#include <stdio.h>
#include <string.h>

#include <lam_config.h>
#include <etc_misc.h>
#include <lamdebug.h>

#include <lam-ssi.h>
#include <lam-ssi-rpi.h>
#include <lam-ssi-cr.h>


/*
 * The following file was created by configure.  It contains extern
 * statements and the definition of an array of pointers to each
 * module's public lam_ssi_crmpi_t struct.
 */

#include "lam-ssi-crmpi-modules.h"


/*
 * Global variables
 */
lam_ssi_crmpi_t lam_ssi_crmpi_base_module;
lam_ssi_crmpi_actions_t lam_ssi_crmpi;
LIST *lam_ssi_crmpi_base_available = NULL;
volatile lam_ssi_crmpi_base_handler_state_t
    lam_ssi_crmpi_base_handler_state = LAM_SSI_CRMPI_BASE_HANDLER_STATE_IDLE;


/*
 * Local variables
 */
#if LAM_HAVE_THREADS
static lam_debug_stream_info_t lds = {
  0, 0, 0, NULL, "crmpi:", 0, 1, 1, 0, "ssi-crmpi.txt"
};
#endif


/*
 * Local functions
 */
#if LAM_HAVE_THREADS
static int check_all_modules(OPT *aod);
static int check_specific_module(OPT *aod, char *);
#endif


int
lam_ssi_crmpi_base_open(OPT *aod)
{
  char *env;

#if LAM_HAVE_THREADS
  /* The CR modules must have thread support in order to run */

  int i;

  /* Call to base SSI open */

  lam_ssi_base_open(aod);

  /* Setup verbosity for this kind */

  lam_ssi_base_set_verbose("LAM_MPI_SSI_cr_verbose", &lds, aod,
			   &lam_ssi_cr_verbose, &lam_ssi_cr_did);
  if (lam_ssi_cr_verbose >= 10)
    lam_debug(lam_ssi_cr_did, " Opening");

  /* Initialize the list */

  lam_ssi_crmpi_base_available = 
    al_init(sizeof(lam_ssi_module_t), lam_ssi_base_module_compare);
  if (lam_ssi_crmpi_base_available == NULL) {
    show_help(NULL, "system-call-failed", "malloc", NULL);
    return LAMERROR;
  }

  /* check env overrides -- was a specific module requested? */

  if ((env = getenv("LAM_MPI_SSI_cr")) != NULL)
    i = check_specific_module(aod, env);

  /* else select one, based on priority */

  else
    i = check_all_modules(aod);

  if (LAMERROR == i) {
    al_free(lam_ssi_crmpi_base_available);
    lam_ssi_crmpi_base_available = NULL;
    return LAMERROR;
  }
#else

  /* If we do not have thread support, then simply set the available
     list to NULL */

  lam_ssi_crmpi_base_available = NULL;

  /* If the user requested a specific module, it's an error (because
     we cannot possibly use that module, since there is no thread
     support). */

  if (((env = getenv("LAM_MPI_SSI_cr")) != NULL) 
      && (!strcmp(env, "none"))) {
    show_help("cr-ssi", "specific-module-no-threads", env, NULL);
    return LAMERROR;
  }
#endif

  /* All done */

  return 0;
}


#if LAM_HAVE_THREADS
/*
 * A specific module was selected on the command line.  If a module by
 * that name is found, call its open (if it exists) and query
 * functions.  If they both return happiness, that module is eligible
 * for secondary selection.  Otherwise, call its close function and
 * return an error (i.e., don't try to find any other available
 * modules).
 */
static int 
check_specific_module(OPT *aod, char *name)
{
  int i;
  int priority, thread_min, thread_max;
  const lam_ssi_t *ls;
  lam_ssi_module_t entry;

  /* In 64 bit mode, this struct can have empty padding */

  LAM_ZERO_ME(entry);

  if (!strcmp(name, "none")) {
    if (lam_ssi_cr_verbose >= 0)
      lam_debug(lam_ssi_cr_did, "module \"none\" explicitly requested. "
                "Disabling cr support");
    return 0;
  }

  if (lam_ssi_cr_verbose >= 10)
    lam_debug(lam_ssi_cr_did, " looking for module named %s", name);

  for (i = 0; lam_ssi_crmpi_modules[i] != NULL; ++i) {
    ls = &(lam_ssi_crmpi_modules[i]->lscrm_meta_info);
    if (0 == strcmp(name, ls->ssi_module_name)) {
      if (lam_ssi_cr_verbose > 10)
        lam_debug(lam_ssi_cr_did, " opening module %s", ls->ssi_module_name);
      if (ls->ssi_open_module == NULL ||
          (ls->ssi_open_module != NULL &&
           1 == ls->ssi_open_module(aod))) {

        /* If it's available, run its query function and see if it
           wants to run.  And if so, ignore the priority but get its
           function list. */

        if (lam_ssi_cr_verbose > 10)
          lam_debug(lam_ssi_cr_did, " query module %s",
                    ls->ssi_module_name);
        if (lam_ssi_crmpi_modules[i]->lscrm_query(&priority, &thread_min, 
                                                  &thread_max) != 0) {

          /* If this module does not want to run, give up in
             despair. */

          if (ls->ssi_close_module != NULL)
            ls->ssi_close_module();

          show_help("cr-ssi", "selected-module-unavailable", name);
          return LAMERROR;
        }

        /* Otherwise, we're found the module and it wants to
           run. We're done */

	entry.lsm_priority = priority;
        entry.lsm_thread_min = thread_min;
        entry.lsm_thread_max = thread_max;
        entry.lsm_module = (lam_ssi_t*) lam_ssi_crmpi_modules[i];
        break;
      }

      /* Otherwise this module's init function didn't want to
         run. Abort. */

      else {
        show_help("cr-ssi", "selected-module-unavailable", name);
        return LAMERROR;
      }
    }
  }

  /* Did we find one ? */

  if (NULL == lam_ssi_crmpi_modules[i]) {
    show_help("cr-ssi", "module-not-found", name, NULL);
    return LAMERROR;
  }

  /* We found the desired module, and it wanted to be considered.  Add
     it to the list.  It'll be the only entry, but that's what the
     user asked for.  :-) */

  al_insert(lam_ssi_crmpi_base_available, &entry);

  return 0;
}


/*
 * Call open on all the available modules (if any exist).  If open
 * returns happiness (or doesn't exist), call the query function to
 * see if that module wants to run.  If it does, add it to the
 * priority list for secondary selection.
 *
 * It is not an error if there are no CR modules available.
 */
static int
check_all_modules(OPT *aod)
{
  int i;
  const lam_ssi_t *ls;
  int priority, thread_min, thread_max;
  lam_ssi_module_t entry;

  /* In 64 bit mode, this struct can have empty padding */

  LAM_ZERO_ME(entry);

  for (i = 0; lam_ssi_crmpi_modules[i] != NULL; ++i) {
    ls = &(lam_ssi_crmpi_modules[i]->lscrm_meta_info);
    if (lam_ssi_cr_verbose > 0)
      lam_debug(lam_ssi_cr_did, " opening module %s", ls->ssi_module_name);
    if (ls->ssi_open_module == NULL ||
        (ls->ssi_open_module != NULL &&
         1 == ls->ssi_open_module(aod))) {

      /* If it's available, run its query function and see if it wants
         to run.  And if so, what's its priority and function
         list. */

      if (lam_ssi_cr_verbose > 0)
        lam_debug(lam_ssi_cr_did, " querying module %s",
                  ls->ssi_module_name);
      if (lam_ssi_crmpi_modules[i]->lscrm_query(&priority, &thread_min,
                                                &thread_max) == 0) {
        if (lam_ssi_cr_verbose >= 10)
          lam_debug(lam_ssi_cr_did,
                    "module available: %s, priority %d, thread level (%d,%d)",
                    lam_ssi_crmpi_modules[i]->lscrm_meta_info.ssi_module_name,
                    priority, thread_min, thread_max);

        /* This module wants to be considered */

	entry.lsm_priority = priority;
        entry.lsm_thread_min = thread_min;
        entry.lsm_thread_max = thread_max;
        entry.lsm_module = (lam_ssi_t*) lam_ssi_crmpi_modules[i];
        al_insert(lam_ssi_crmpi_base_available, &entry);
      } else {
        if (lam_ssi_cr_verbose >= 10)
          lam_debug(lam_ssi_cr_did, " module not available: %s",
                    ls->ssi_module_name);
      }
    }
  }

  /* It doesn't matter if there were no modules found to be available.
     We always return happiness. */

  return 0;
}
#endif
