/*
 * Copyright (c) 2001-2003 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 * $HEADER$
 *
 * $Id: ssi_rpi_gm_unexpected.c,v 1.9.2.3 2004/03/12 01:19:12 vsahay Exp $
 *
 *	Function:	- handle gm unexpected messages
 */

#include <lam_config.h>

#include <stdlib.h>
#include <string.h>

#include <rpisys.h>
#include <lamdebug.h>

#include <rpi_gm.h>
#include <rpi_gm_unexpected.h>
#include <rpi_gm_dreg.h>
#include <rpi_gm_long.h>
#include <rpi_gm_short.h>
#include <rpi_gm_ack.h>
#include <rpi_gm_util.h>


/*
 * Advance a request based on a buffered (i.e., previously unexpected)
 * message.
 */
int 
lam_ssi_rpi_gm_unexpected_advance(MPI_Request req, 
				  struct lam_ssi_rpi_cbuf_msg *msg)
{
  struct lam_ssi_rpi_gm_envl gmenv;
  struct lam_ssi_rpi_envl *env = &(gmenv.ge_env);
  
  gmenv.ge_env = msg->cm_env;

  /* If the matched request is a probe, then set the status and leave
     the envelope buffered */

  if (req->rq_type == LAM_RQIPROBE) {
    lam_debug_cond((lam_ssi_rpi_gm_did,
                    "unexpected_advance: matched a probe"));
    lam_ssi_rpi_fill_mpi_status(req, env->ce_rank, env->ce_tag, env->ce_len);
    req->rq_state = LAM_RQSDONE;
    --lam_rq_nactv;

    return 0;
  }

  /* Special case of a synchronous send with the sender being the same
     as the receiver.  Copy the message data directly from the
     sender's buffer and both requests are done.  */

  else if (msg->cm_req != MPI_REQUEST_NULL) {
    lam_debug_cond((lam_ssi_rpi_gm_did, 
                    "unexpected_advance: matched sync send to self"));
    if (env->ce_len > req->rq_packsize) {
      req->rq_flags |= LAM_RQFTRUNC;
      env->ce_len = req->rq_packsize;
    }
    
    if (env->ce_len > 0)
      lam_memcpy(req->rq_packbuf, msg->cm_buf, env->ce_len);

    lam_ssi_rpi_fill_mpi_status(req, env->ce_rank, env->ce_tag, env->ce_len);
    req->rq_state = LAM_RQSDONE;
    msg->cm_req->rq_state = LAM_RQSDONE;
    lam_rq_nactv -= 2;
  }

  /* If the matches request is a long message, jump into the long
     message protocol (i.e., send an ACK) */

  else if (env->ce_flags & C2CLONG) {
    lam_debug_cond((lam_ssi_rpi_gm_did, 
                    "unexpected_advance: matched long"));

    /* First pin the buffer so that we can receive into it */

    if (lam_ssi_rpi_gm_setup_request_dma(req) != 0)
      return LAMERROR;

    /* Now fet a second envelope that we'll need for the long protocol */

    req->rq_rpi->cq_envbuf2 = lam_ssi_rpi_gm_dma_env_malloc();
    if (req->rq_rpi->cq_envbuf2 == NULL)
      return LAMERROR;

    /* Now proceed as normal with the long protocol */

    if (lam_ssi_rpi_gm_long_receive_env_first(&gmenv, req) != 0)
      return LAMERROR;
  }

  /* Otherwise, it was a short or tiny message */
  
  else {
    lam_debug_cond((lam_ssi_rpi_gm_did,
                    "unexpected_advance: matched short/tiny"));

    /* If we've got the whole message, then the cm_proc pointer will
       be set to the sentinel value of NULL.  We can copy the message
       over to the destination and advance the request (potentially
       sending an ACK).  Don't worry about freeing the cm_buf; that
       will be handled in lam_ssi_rpi_cbuf_delete. */

    if (msg->cm_proc == NULL) {
      lam_debug_cond((lam_ssi_rpi_gm_did,
                      "unexpected_advance: have entire message"));
      if (env->ce_len > req->rq_packsize) {
	req->rq_flags |= LAM_RQFTRUNC;
	env->ce_len = req->rq_packsize;
      }
      
      if (env->ce_len > 0)
	lam_memcpy(req->rq_packbuf, msg->cm_buf, env->ce_len);
      lam_ssi_rpi_fill_mpi_status(req, env->ce_rank, env->ce_tag, env->ce_len);
      
      /* If this was not a synchronous send, it is done.  If it was a
         syncronous send, then we need to send an ACK back */

      if (env->ce_flags & C2CSSEND) {
	req->rq_state = LAM_RQSACTIVE;
	req->rq_rpi->is_advancing = 1;
	if (lam_ssi_rpi_gm_send_ack_done(req, &gmenv) != 0)
	  return LAMERROR;
	lam_debug_cond((lam_ssi_rpi_gm_did,
                        "unexepcted_advance: sync ACK queued up"));
      } else {
	lam_debug_cond((lam_ssi_rpi_gm_did,
                        "unexpected_advance: all done!"));
	req->rq_state = LAM_RQSDONE;
	--lam_rq_nactv;
      }
    }

    /* Otherwise, it was a short message and we have only received the
       envelope so far.  So we need to setup to receive the body. */

    else {
      lam_debug_cond((lam_ssi_rpi_gm_did, 
                      "unexpected_advance: prepare to receive short body"));
      if (lam_ssi_rpi_gm_short_prepare_unexpected_body(&gmenv, req) != 0)
	return LAMERROR;
    }
  }

  /* Discard the buffered message */

  lam_ssi_rpi_cbuf_delete(msg);

  return 0;
}


/*
 * We have received an envelope for which we did not already have a
 * matching receive posted.  Look at the envelope and decide how to
 * buffer it.
 */
int 
lam_ssi_rpi_gm_unexpected_receive_env(struct lam_ssi_rpi_proc *src_proc, 
				      struct lam_ssi_rpi_gm_envl *env)
{
  struct lam_ssi_rpi_cbuf_msg cm;

  /* In 64 bit mode, this struct may have extra padding */

  LAM_ZERO_ME(cm);
  cm.cm_env = env->ge_env;
  cm.cm_req = MPI_REQUEST_NULL;
  cm.cm_dont_delete = 0;

  /* Note that we add the C2CBUFFERED flag to the ce_flags so that we
     know later that this envelope buffer is not a pinned buffer. */
  
  cm.cm_env.ce_flags |= C2CBUFFERED;
  
  /* If it's a tiny message, then this is easy -- the entire message
     has already arrived.  Buffer the envelope and the message. */

  if (env->ge_env.ce_len <= lam_ssi_rpi_gm_tinymsglen) {
    lam_debug_cond((lam_ssi_rpi_gm_did, "unexpected_recv: got tiny"));

    /* Set the cm_proc to be NULL as a sentinel that the entire body
       of the message has already been read in. */

    cm.cm_proc = NULL;

    /* Allocate a buffer just large enough and memcpy the body over */

    cm.cm_buf = malloc(env->ge_env.ce_len);
    if (cm.cm_buf == NULL)
      return LAMERROR;
    lam_memcpy(cm.cm_buf, env + 1, env->ge_env.ce_len);

    /* Now add this message to the unexpected list */

    if (lam_ssi_rpi_cbuf_append(&cm) == NULL) {
      free(cm.cm_buf);
      return LAMERROR;
    }
  }

  /* If it's a short message, save the proc.  Put a copy of this
     cm_buf on the proc so that it knows to receive the body into this
     unexpected message buffer.  Receiving the body will be the next
     thing that we do with this proc.  Note, however, that if -- by
     some race condition -- we actually manage to post a matching
     receive *before* the body is received, the buffered_advance
     function will do the Right Thing. */

  else if (env->ge_env.ce_len <= lam_ssi_rpi_gm_shortmsglen) {
    lam_debug_cond((lam_ssi_rpi_gm_did, "unexpected_recv: got short"));
    cm.cm_proc = src_proc->cp_proc;

    /* Allocate a buffer just large enough and memcpy the body over */

    cm.cm_buf = malloc(env->ge_env.ce_len);
    if (cm.cm_buf == NULL)
      return LAMERROR;

    /* Now put this unexpected buffer on the proc, and add it to the
       unexpected list */

    if ((src_proc->cp_bmsg = lam_ssi_rpi_cbuf_append(&cm)) == NULL) {
      free(cm.cm_buf);
      return LAMERROR;
    }
  }

  /* Otherwise, it's a long message */
  
  else {
    lam_debug_cond((lam_ssi_rpi_gm_did, "unexpected_recv: got long"));

    /* Just buffer up the envelope and don't send an ACK back -- we'll
       send it when a matching receive is posted. */

    cm.cm_buf = NULL;
    cm.cm_proc = NULL;

    /* Now add this message to the unexpected list */

    if (lam_ssi_rpi_cbuf_append(&cm) == NULL)
      return LAMERROR;
  }

  return 0;
}


/*
 * Receive the body of a short message into an unexpected buffer that
 * has already been allocated.
 */
int 
lam_ssi_rpi_gm_unexpected_receive_short(struct lam_ssi_rpi_proc *src_proc, 
					char *body)
{
  lam_debug_cond((lam_ssi_rpi_gm_did, 
		  "unexpected_recv_short: received body into buffer"));

  /* Copy the body over to the buffer */

  lam_memcpy(src_proc->cp_bmsg->cm_buf, body, 
	     src_proc->cp_bmsg->cm_env.ce_len);

  /* Set the sentinel value in the lam_ssi_rpi_cbuf_msg that says that
     we received the entire message */

  src_proc->cp_bmsg->cm_proc = NULL;

  /* Reset the proc so that it doesn't think that the next thing
     received is also an unexpected message :-) */

  src_proc->cp_bmsg = 0;

  /* Return the buffer to the pool */

  lam_ssi_rpi_gm_dma_short_free(body);

  return 0;
}
