import re
from linda import libchecks, checks
from sets import Set

class BinaryCheck(libchecks.LindaChecker):
    'Checks binary objects contained within a package.'
    def init(self):
        self.arch = 'none'
        self.reloc = re.compile(r'(statically linked|[[LM]SB relocatable)')
        self.debug = re.compile(r'[-_](db|debu)?g$')
        
    def check_binary_2(self):
        self.arch_checks()
        self.binary_in_wierd_places()
        self.depends()
        for file in self.information['collector']('files', 'elf'):
            self.sonames(file)
            self.note_comment(file)
            self.file_checks(file)
            self.objdump_checks(file)
            self.debug_checks(file)
    
    def arch_checks(self):
        if self.information['control']['self'][0].has_key('architecture'):
            self.arch = self.information['control']['self'][0]['architecture']
        if not DataValidator('arch', self.arch):
            self.signal_error('unknown-arch', [self.arch])
        if self.information['collector']('files', 'elf') and \
            self.arch == 'all':
            self.signal_error('pkg-should-be-arch-any')
    def binary_in_wierd_places(self):
        for file in self.information['collector']('files', 'elf'):
            if file.startswith('/etc'):
                self.signal_error('binary-in-etc', [file])
            elif file.startswith('/usr/doc') or \
                file.startswith('/usr/share/doc'):
                self.signal_error('binary-in-doc-dir', [file])
            elif file.find('/usr/share') != -1:
                self.signal_error('arch-dep-in-usr-share', [file])
    def depends(self):
        keys = Set(('depends', 'pre-depends'))
        reloc_found = 0
        for file in self.information['collector']('files', 'elf'):
            if re.search(self.reloc, self.information['collector']('output', \
                'file')[file]):
                reloc_found = 1
        if self.information['collector']('files', 'elf') and \
            len(keys.intersection(self.information['control']['self'][0].keys())) == 0 and not reloc_found:
            self.signal_error('no-depends-line')
    def sonames(self, file):
        sonames = {}
        if self.information['collector']('output', 'ldd')[file].find('statically linked') != -1:
            return
        for line in self.information['collector']('output', 'ldd')[file].split('\n'):
            if line.find('.so.') == -1:
                continue
            tmp_array = line[1:].split(' => ')[0].split('.so.')
            if tmp_array[0] == 'libc':
                continue
            if tmp_array[0].find('-') != -1 and \
                tmp_array[0].find('libstdc++') != -1:
                tmp_array[0] = tmp_array[0][:tmp_array[0].find('-')]
            if sonames.has_key(tmp_array[0]):
                if sonames[tmp_array[0]] == tmp_array[1]:
                    # Same sover, drop it.
                    continue
                dprint(_("Sonames during double: %s") % sonames, 2)
                self.signal_error('double-shlib', [file, \
                    sonames[tmp_array[0]], tmp_array[1], tmp_array[0]])
            else:
                sonames[tmp_array[0]] = tmp_array[1]
    def note_comment(self, file):
        if not self.considered_debug(file) and not re.search(self.reloc, \
            self.information['collector']('output', 'file')[file]):
            sects = re.findall(r'\s+\.(note|comment)\s+', \
                self.information['collector']('output', 'objdump')[file])
            if len(sects) == 2:
                self.signal_error('not-quite-stripped-binary', [file])
            elif len(sects) == 1:
                self.signal_error('binary-note-or-comment', [file, sects[0]])
    def file_checks(self, file):
        if self.information['collector']('output', 'file')[file].find('statically') != -1 \
            and not file.startswith('/boot') and file.find('.static') == -1 \
            and not self.considered_debug(file):
            self.signal_error('static-binary', [file])
        if self.information['collector']('output', 'file')[file].find(', stripped') != -1 and self.considered_debug(file):
            self.signal_error('stripped-binary-not', [file])
    def objdump_checks(self, file):
        if self.information['collector']('output', 'objdump')[file].\
            find('format not recognized') != -1:
            return
        if self.information['collector']('output', 'objdump')[file].\
            find('RPATH') != -1:
            rpath_str = self.information['collector']('output', 'objdump')[file]\
                [self.information['collector']('output', 'objdump')[file].find('RPATH'):\
                self.information['collector']('output', 'objdump')[file].find('\n', \
                self.information['collector']('output', 'objdump')[file].find('RPATH'))]
            rpath_lst = re.split('\s+', rpath_str)
            if rpath_lst[-1]:
                self.signal_error('binary-rpath', [file, rpath_lst[-1]])
        if self.information['collector']('output', 'objdump')[file].find('libc.so.5') != -1 and \
            file.find('/libc5-compat') == -1:
            self.signal_error('binary-libc5', [file])
        if self.information['collector']('output', 'objdump')[file].find('libc.so.6') != -1 and \
            file.find('/libc5-compat') != -1:
            self.signal_error('libc6-binary-in-libc5-dir', [file])
        if self.information['collector']('output', 'objdump')[file].find('Packed with UPX') != -1:
            self.signal_error('binary-upx', [file])
        if re.search(r'\.text.*__gmon_start__', \
            self.information['collector']('output', 'objdump')[file]) and self.arch != 'hppa':
            self.signal_error('binary-profiling', [file])
    def debug_checks(self, file):
        if not self.considered_debug(file) and not \
            re.search(self.reloc, self.information['collector']('output', \
            'file')[file]):
            if self.information['collector']('output', 'objdump')[file].find('NEEDED') == -1:
                self.signal_error('shared-binary', [file])
            if self.information['collector']('output', 'file')[file].find('not stripped') != -1:
                self.signal_error('unstripped-binary', [file])
            if self.information['collector']('output', 'objdump')[file].find('libc.so') == -1:
                self.signal_error('binary-libc6', [file])

    def considered_debug(self, file):
        if re.search(self.debug, file) or re.search(self.debug, self.pkg_name):
            return 1
        else:
            return 0
        
checks.register(BinaryCheck)

