import unittest, os, linda
from po_count import POFileParser
from linda.debug import dprint
from linda.parser.datavalidator import DataValidator
from linda.libchecks import Checks
from sets import Set

linda.checks = Checks()

class ControlCheckBinaryTest(unittest.TestCase):
    def setUp(self):
        execfile('checks/control.py', {'_': __builtins__['_'], 'dprint': \
            dprint, 'DataValidator': DataValidator, 'os': os})
        for klass in linda.checks.registry['binary'][1]:
            self.instance = klass('/tmp', '/tmp/test_0.0.1-1_all.deb', {})
        self.po_keys = POFileParser('po/en.po')
        
    def test_package(self):
        "Check package() method of ControlCheck"
        method = getattr(self.instance, 'package')
        self.instance.information['control'] = {'self': [{'package': 'test'}]}
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['control']['self'][0]['package'] = 'Test'
        method()
        self.assert_('upper-in-package' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'upper-in-package'), True)

    def test_perl_section(self):
        "Check Section: perl part of package() method"
        method = getattr(self.instance, 'package')
        self.instance.information['control'] = {'self': [{'package': \
            'test-perl', 'section': ['perl']}]}
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['control']['self'][0]['section'] = \
            ['interpreters']
        method()
        self.assert_('perl-pkg-should-be-sct-perl' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'perl-pkg-should-be-sct-perl'), True)
        
    def test_python_section(self):
        "Check Section: python part of package() method"
        method = getattr(self.instance, 'package')
        self.instance.information['control'] = {'self': [{'package': \
            'python-test', 'section': ['python']}]}
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['control']['self'][0]['section'] = \
            ['interpreters']
        method()
        self.assert_('python-pkg-should-be-sct-py' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'python-pkg-should-be-sct-py'), True)

    def test_priority(self):
        "Check priority() method of ControlCheck"
        method = getattr(self.instance, 'priority')
        for priority in ('required', 'important', 'standard', 'optional', \
            'extra'):
            self.instance.information['control'] = {'self': [{'priority': \
                priority}]}
            method()
            self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['control']['self'][0]['priority'] = \
            'something'
        method()
        self.assert_('unknown-priority' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'unknown-priority'), True)

    def test_section(self):
        "Check section() method of ControlCheck"
        method = getattr(self.instance, 'section')
        self.instance.information['control'] = {'self': [{'section': ['doc']}]}
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['control']['self'][0]['section'] = \
            ['non-US', 'doc']
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['control']['self'][0]['section'] = \
            ['something']
        method()
        self.assert_('unknown-section' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'unknown-section'), True)
        self.instance.information['control']['self'][0]['section'] = \
            ['non-us', 'doc']
        method()
        self.assert_('spelling-non-us' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'spelling-non-us'), True)
        self.instance.information['control']['self'][0]['section'] = \
            ['something', 'doc']
        method()
        self.assert_('unknown-component' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'unknown-component'), True)

    def test_maintainer(self):
        "Check maintainer() method of ControlCheck"
        method = getattr(self.instance, 'maintainer')
        self.instance.information['control'] = {'self': [{}]}
        method()
        self.assert_('no-maintainer' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'no-maintainer'), True)
        self.instance.information['control']['self'][0]['maintainer'] = ''
        method()
        self.assert_('empty-maintainer' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'empty-maintainer'), True)
        self.instance.information['control']['self'][0]['maintainer'] = \
            'Some One'
        method()
        self.assert_('no-maintainer-email' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'no-maintainer-email'), True)
        self.instance.information['control']['self'][0]['maintainer'] = \
            'S <k@>'
        method()
        self.assert_('malformed-maint-mail-addr' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'malformed-maint-mail-addr'), True)
        self.instance.err_dict.clear()
        self.instance.information['control']['self'][0]['maintainer'] = \
            'Some One <k_@foo.com>'
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['control']['self'][0]['maintainer'] = \
            'Some One <someone@something.com>'
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])

    def test_essential(self):
        "Check essential() method of ControlCheck"
        method = getattr(self.instance, 'essential')
        self.instance.pkg_name = 'xringd'
        self.instance.information['control'] = {'self': [{'essential': 'no'}]}
        method()
        self.assert_('essential-no-redundant' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'essential-no-redundant'), True)
        self.instance.information['control']['self'][0]['essential'] = 'maybe'
        method()
        self.assert_('essential-not-yes-or-no' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'essential-not-yes-or-no'), True)
        self.instance.information['control']['self'][0]['essential'] = 'yes'
        method()
        self.assert_('new-essential-package' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'new-essential-package'), True)

    def test_description(self):
        "Check description() method of ControlCheck"
        method = getattr(self.instance, 'description')
        self.instance.pkg_name = 'xringd'
        self.instance.information['control'] = {'self': [{}]}
        method()
        self.assert_('no-description' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'no-description'), True)
        self.instance.information['control']['self'][0]['description'] = \
            ['xringd something', ' xringd']
        method()
        self.assert_('desc-starts-with-package' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'desc-starts-with-package'), True)
        self.instance.information['control']['self'][0]['description'][0] = \
            'xringd  something'
        method()
        self.assert_('multiple-spaces-in-desc' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'multiple-spaces-in-desc'), True)
        self.instance.information['control']['self'][0]['description'][0] = \
            'somengobnwognlsnvlsnvlsnvlsnvsnvslknvlksnvsngjqoighwoghwognowenvnlwenvlwnvlvnlenvmnw'
        method()
        self.assert_('description-too-long' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'description-too-long'), True)
        self.instance.information['control']['self'][0]['description'][0] = \
            'something foo	and something else'
        method()
        self.assert_('tab-in-desc' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'tab-in-desc'), True)
        self.instance.information['control']['self'][0]['description'] = \
            ['This is a description', 'Something This is a description of foo']
        method()
        self.assert_('ex-desc-contains-desc' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'ex-desc-contains-desc'), True)
        self.instance.err_dict.clear()
        self.instance.information['control']['self'][0]['description'][0] = \
            'This is a normal description'
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])

    def test_depends(self):
        "Check depends() method of ControlCheck"
        method = getattr(self.instance, 'depends')
        self.instance.pkg_name = 'xringd'
        self.instance.information['control'] = {'self': [{'depends': \
            {'debianutils': {'arch': '', 'relation': ''}}}]}
        method()
        self.assert_('no-version-dep-on-essential' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'no-version-dep-on-essential'), True)
        self.instance.information['control']['self'][0]['depends'] = \
            {'xringd': {'arch': '', 'relation': []}}
        method()
        self.assert_('pkg-depends-on-self' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'pkg-depends-on-self'), True)
        self.instance.err_dict.clear()
        self.instance.information['control']['self'][0]['depends'] = \
            {'libc6': {'arch': '', 'relation': ['>=', '2.3.2-5']}}            
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])

    def test_provides(self):
        "Check provides() method of ControlCheck"
        method = getattr(self.instance, 'provides')
        self.instance.information['control'] = {'self': [{'version': '1.0-1', \
            'package': 'test', 'priority': 'extra', 'provides': {'xringer': \
            {'arch': [None], 'relation': [[None, None]]}}}]}
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['control'] = {'self': [{'provides': \
            {'xringer': {'arch': [None], 'relation': [['>=', '1.30']]}}}]}
        method()
        self.assert_('versioned-provides' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'versioned-provides'), True)
        self.instance.err_dict.clear()
        self.instance.information['control']['self'][0] = {}
        method()
        lines = Set(['no-package-line', 'no-priority-line', 'no-version-line'])
        self.assertEqual(lines.intersection(self.instance.err_dict.keys()), \
            lines)
        for key in lines:
            self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
                key), True)
        self.instance.information['control']['self'][0]['version'] = '1.2-0'
        method()
        self.assert_('deb-revision-zero' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'deb-revision-zero'), True)

    def test_obselete(self):
        "Check obselete() method of ControlCheck"
        method = getattr(self.instance, 'obselete')
        self.instance.information['control'] = {'self': [{}]}
        for x in ('revision', 'package-revision', 'package_revision', \
            'recommended', 'optional', 'class'):
            self.instance.err_dict.clear()
            self.instance.information['control']['self'][0][x] = ''
            method()
            self.assert_('obsolete-field-in-control' in \
                self.instance.err_dict.keys())
            self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
                'obsolete-field-in-control'), True)

    def test_recommends_suggests(self):
        "Check recommends_suggests() method of ControlCheck"
        method = getattr(self.instance, 'recommends_suggests')
        self.instance.information['control'] = {'self': [{'suggests': 'foo'}]}
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['control'] = {'self': [{'depends': \
            'foobar'}]}
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['control']['self'][0]['suggests'] = \
            'foobar'
        method()
        self.assert_('recommends-suggests-dependant-pkg' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'recommends-suggests-dependant-pkg'), True)

