import os, unittest, shutil, tempfile
from linda.funcs import run_external_cmd
from linda.parser.control import DebianControlParser, DCPException
from sets import Set

class DebianControlParserTest(unittest.TestCase):
    def setUp(self):
        self.dir = tempfile.mkdtemp()
        run_external_cmd('tar zxCf %s tests/parser_control.tar.gz' % self.dir)

    def tearDown(self):
        shutil.rmtree(self.dir)

    def test_binary(self):
        "Check that a binary control file parses correctly"
        dcp = DebianControlParser(os.path.join(self.dir, 'binary'))
        self.assertEqual(dcp.stanzas(), 1)
        keys = Set(('recommends', 'maintainer', 'description', 'package', \
            'section', 'installed-size', 'priority', 'suggests', 'depends', \
            'version', 'architecture', 'provides', 'conflicts'))
        self.assertEqual(keys.intersection(dcp[0].keys()), keys)
        self.assertEqual(dcp[0]['maintainer'], \
            'Steve Kowalik <stevenk@debian.org>')
        self.assertEqual(dcp[0]['package'], 'linda')
        self.assertEqual(dcp[0]['section'], ['devel'])
        self.assertEqual(dcp[0]['installed-size'], '716')
        self.assertEqual(dcp[0]['priority'], 'optional')
        self.assertEqual(dcp[0]['version'], '0.3.0')
        self.assertEqual(dcp[0]['architecture'], 'all')
        self.assertEqual(dcp[0]['description'], ['Debian package checker, not unlike lintian', ' Linda is a Debian package checker, much like lintian, that\n runs some rudimentary checks over source and binary packages\n to see if they comply to Policy.'])
        self.assertEqual(dcp[0]['suggests'].keys(), ['binutils-multiarch'])
        verrel_keys = Set(('arch', 'relation'))
        self.assertEqual(dcp[0]['provides'].keys(), ['lintian-python'])
        self.assertEqual(verrel_keys.intersection(dcp[0]['suggests']['binutils-multiarch'].keys()), verrel_keys)
        self.assertEqual(verrel_keys.intersection(dcp[0]['provides']['lintian-python'].keys()), verrel_keys)
        self.assertEqual(dcp[0]['suggests']['binutils-multiarch']['relation'], [[None, None]])
        self.assertEqual(dcp[0]['provides']['lintian-python']['relation'], [[None, None]])
        self.assertEqual(dcp[0]['suggests']['binutils-multiarch']['arch'], \
            [None])
        self.assertEqual(dcp[0]['provides']['lintian-python']['arch'], [None])
        self.assertEqual(dcp[0]['conflicts'].keys(), ['file'])
        self.assertEqual(verrel_keys.intersection(dcp[0]['conflicts']['file'].keys()), verrel_keys)
        self.assertEqual(dcp[0]['conflicts']['file']['arch'], [None])
        self.assertEqual(dcp[0]['conflicts']['file']['relation'], \
            [['=', '4.04']])
        depends = Set(('python', 'dash', 'groff-base', 'file', 'binutils'))
        self.assertEqual(depends.intersection(dcp[0]['depends']), depends)
        for pkg in depends:
            self.assertEqual(verrel_keys.intersection(dcp[0]['depends'][pkg].keys()), verrel_keys)
        self.assertEqual(dcp[0]['depends']['python']['arch'], [None, None])
        self.assertEqual(dcp[0]['depends']['python']['relation'], \
            [['>=', '2.3.1'], ['<<', '2.4']])
        for pkg in ('dash', 'groff-base', 'file', 'binutils'):
            self.assertEqual(dcp[0]['depends'][pkg]['arch'], [None])
            self.assertEqual(dcp[0]['depends'][pkg]['relation'], \
                [[None, None]])
        
    def test_source_single(self):
        "Check that a single-pkg source control file parses correctly"
        dcp = DebianControlParser(os.path.join(self.dir, 'source-single'))
        self.assertEqual(dcp.stanzas(), 2)
        keys = Set(('maintainer', 'build-depends-indep', 'section', \
            'priority', 'source', 'standards-version'))
        self.assertEqual(keys.intersection(dcp[0].keys()), keys)
        keys = Set(('recommends', 'description', 'package', 'suggests', \
            'depends', 'architecture', 'provides', 'conflicts'))
        self.assertEqual(keys.intersection(dcp[1].keys()), keys)
        self.assertEqual(dcp[0]['maintainer'], \
            'Steve Kowalik <stevenk@debian.org>')
        self.assertEqual(dcp[0]['section'], ['devel'])
        self.assertEqual(dcp[0]['priority'], 'optional')
        self.assertEqual(dcp[0]['source'], 'linda')
        self.assertEqual(dcp[0]['standards-version'], '3.6.1')
        bdi = Set(('python', 'debhelper'))
        verrel_keys = Set(('arch', 'relation'))
        self.assertEqual(bdi.intersection(dcp[0]['build-depends-indep'].keys()), bdi)
        for pkg in bdi:
            self.assertEqual(verrel_keys.intersection(dcp[0]['build-depends-indep'][pkg].keys()), verrel_keys)
        self.assertEqual(dcp[0]['build-depends-indep']['python']['arch'], \
            [None, None])
        self.assertEqual(dcp[0]['build-depends-indep']['debhelper']['arch'], \
            [None])
        self.assertEqual(dcp[0]['build-depends-indep']['python']['relation'], \
            [['>=', '2.3.1'], ['<<', '2.4']])
        self.assertEqual(dcp[0]['build-depends-indep']['debhelper']['relation'], [['>=', '4.0.0']])
        self.assertEqual(dcp[1]['package'], 'linda')
        self.assertEqual(dcp[1]['architecture'], 'all')
        self.assertEqual(dcp[1]['description'], ['Debian package checker, not unlike lintian', ' Linda is a Debian package checker, much like lintian, that\n runs some rudimentary checks over source and binary packages\n to see if they comply to Policy.'])
        self.assertEqual(dcp[1]['suggests'].keys(), ['binutils-multiarch'])
        verrel_keys = Set(('arch', 'relation'))
        self.assertEqual(dcp[1]['provides'].keys(), ['lintian-python'])
        self.assertEqual(verrel_keys.intersection(dcp[1]['suggests']['binutils-multiarch'].keys()), verrel_keys)
        self.assertEqual(verrel_keys.intersection(dcp[1]['provides']['lintian-python'].keys()), verrel_keys)
        self.assertEqual(dcp[1]['suggests']['binutils-multiarch']['relation'], [[None, None]])
        self.assertEqual(dcp[1]['provides']['lintian-python']['relation'], [[None, None]])
        self.assertEqual(dcp[1]['suggests']['binutils-multiarch']['arch'], \
            [None])
        self.assertEqual(dcp[1]['provides']['lintian-python']['arch'], [None])
        self.assertEqual(dcp[1]['conflicts'].keys(), ['file'])
        self.assertEqual(verrel_keys.intersection(dcp[1]['conflicts']['file'].keys()), verrel_keys)
        self.assertEqual(dcp[1]['conflicts']['file']['arch'], [None])
        self.assertEqual(dcp[1]['conflicts']['file']['relation'], \
            [['=', '4.04']])
        depends = Set(('python', 'dash', 'groff-base', 'file', 'binutils'))
        self.assertEqual(depends.intersection(dcp[1]['depends']), depends)
        for pkg in depends:
            self.assertEqual(verrel_keys.intersection(dcp[1]['depends'][pkg].keys()), verrel_keys)
        self.assertEqual(dcp[1]['depends']['python']['arch'], [None, None])
        self.assertEqual(dcp[1]['depends']['python']['relation'], \
            [['>=', '2.3.1'], ['<<', '2.4']])
        for pkg in ('dash', 'groff-base', 'file', 'binutils'):
            self.assertEqual(dcp[1]['depends'][pkg]['arch'], [None])
            self.assertEqual(dcp[1]['depends'][pkg]['relation'], \
                [[None, None]])
        
    def test_source_multiple(self):
        "Check that a multi-pkg source control file parses correctly"
        dcp = DebianControlParser(os.path.join(self.dir, 'source-multiple'))
        self.assertEqual(dcp.stanzas(), 4)
        keys = Set(('uploaders', 'maintainer', 'build-depends-indep', \
            'section', 'priority', 'source', 'standards-version', \
            'build-depends'))
        self.assertEqual(keys.intersection(dcp[0].keys()), keys)
        keys = Set(('replaces', 'description', 'package', 'suggests', \
            'depends', 'architecture', 'provides', 'conflicts'))
        self.assertEqual(keys.intersection(dcp[1].keys()), keys)
        keys = Set(('recommends', 'description', 'package', 'suggests', \
            'depends', 'architecture'))
        self.assertEqual(keys.intersection(dcp[2].keys()), keys)
        keys = Set(('replaces', 'description', 'package', 'architecture', \
            'conflicts'))
        self.assertEqual(keys.intersection(dcp[3].keys()), keys)
        self.assertEqual(dcp[0]['section'], ['sound'])
        self.assertEqual(dcp[0]['priority'], 'optional')
        self.assertEqual(dcp[0]['source'], 'alsa-driver')
        self.assertEqual(dcp[0]['standards-version'], '3.6.1')
        self.assertEqual(dcp[0]['uploaders'], 'Jordi Mallach <jordi@debian.org>, Steve Kowalik <stevenk@debian.org>, David B. Harris <david@eelf.ddts.net>')
        self.assertEqual(dcp[0]['maintainer'], 'Debian-Alsa Psychos <pkg-alsa-devel@lists.alioth.debian.org>')
        bdi = Set(('dpatch', 'bzip2', 'debhelper', 'cpio', 'po-debconf'))
        verrel_keys = Set(('arch', 'relation'))
        self.assertEqual(bdi.intersection(dcp[0]['build-depends-indep'].keys()), bdi)
        for pkg in bdi:
            self.assertEqual(verrel_keys.intersection(dcp[0]['build-depends-indep'][pkg].keys()), verrel_keys)
        for pkg in ('bzip2', 'cpio', 'po-debconf'):
            self.assertEqual(dcp[0]['build-depends-indep'][pkg]['arch'], \
                [None])
            self.assertEqual(dcp[0]['build-depends-indep'][pkg]['relation'], \
                [[None, None]])
        self.assertEqual(dcp[0]['build-depends-indep']['dpatch']['arch'], \
            [None])
        self.assertEqual(dcp[0]['build-depends-indep']['debhelper']['arch'], \
            [None])
        self.assertEqual(dcp[0]['build-depends-indep']['dpatch']['relation'], \
            [['>=', '1.11']])
        self.assertEqual(dcp[0]['build-depends-indep']['debhelper']['relation'], \
            [['>=', '4.1.17']])
        self.assertEqual(dcp[0]['build-depends'].keys(), ['foobar'])
        self.assertEqual(dcp[0]['build-depends']['foobar']['relation'], \
            [[None, None]])
        arches = Set(('!arm', '!m68k'))
        self.assertEqual(arches.intersection(dcp[0]['build-depends']['foobar']['arch']), arches)
        self.assertEqual(dcp[1]['package'], 'alsa-base')
        self.assertEqual(dcp[1]['architecture'], 'all')
        self.assertEqual(dcp[1]['description'], ['ALSA sound driver common files', ' ALSA is an effort to create a modules sound system for Linux,\n while maintaining full compatibility with OSS/Lite.\n .\n These are the common files to be used on ALSA such as the scripts used\n to load and unload the modules at boot time. In addition, this package\n is for mutex control between several ALSA versions.'])
        repl = Set(('alsa-modules', 'alsa-base-0.4'))
        self.assertEqual(repl.intersection(dcp[1]['replaces'].keys()), repl)
        for pkg in repl:
            self.assertEqual(verrel_keys.intersection(dcp[1]['replaces'][pkg].keys()), verrel_keys)
            self.assertEqual(dcp[1]['replaces'][pkg]['arch'], [None])
            self.assertEqual(dcp[1]['replaces'][pkg]['relation'], \
                [[None, None]])
        self.assertEqual(dcp[1]['provides'].keys(), ['alsa'])
        self.assertEqual(verrel_keys.intersection(dcp[1]['provides']['alsa'].keys()), verrel_keys)
        self.assertEqual(dcp[1]['provides']['alsa']['arch'], [None])
        self.assertEqual(dcp[1]['provides']['alsa']['relation'], \
            [[None, None]])
        self.assertEqual(dcp[1]['suggests'].keys(), ['apmd'])
        self.assertEqual(verrel_keys.intersection(dcp[1]['suggests']['apmd'].keys()), verrel_keys)
        self.assertEqual(dcp[1]['suggests']['apmd']['arch'], [None])
        self.assertEqual(dcp[1]['suggests']['apmd']['relation'], \
            [['>=', '3.0.2-1']])
        confl = Set(('modutils', 'alsaconf', 'alsa-modules', 'lsof-2.2', \
            'alsa-base-0.4'))
        self.assertEqual(confl.intersection(dcp[1]['conflicts'].keys()), confl)
        for pkg in confl:
            self.assertEqual(verrel_keys.intersection(dcp[1]['conflicts'][pkg].keys()), verrel_keys)
            self.assertEqual(dcp[1]['conflicts'][pkg]['arch'], [None])
        self.assertEqual(dcp[1]['conflicts']['alsa-modules']['relation'], \
            [[None, None]])
        self.assertEqual(dcp[1]['conflicts']['alsa-base-0.4']['relation'], \
            [[None, None]])
        self.assertEqual(dcp[1]['conflicts']['modutils']['relation'], \
            [['=', '2.3.20-1']])
        self.assertEqual(dcp[1]['conflicts']['alsaconf']['relation'], \
            [['<<', '0.4.3b-4']])
        self.assertEqual(dcp[1]['conflicts']['lsof-2.2']['relation'], \
            [['<<', '4.64-1']])
        deps = Set(('debconf', 'alsa-utils', 'modutils', 'psmisc', \
            'module-init-tools', 'debianutils', 'lsof', 'procps'))
        self.assertEqual(deps.intersection(dcp[1]['depends'].keys()), deps)
        for pkg in deps:
            self.assertEqual(verrel_keys.intersection(dcp[1]['depends'][pkg].keys()), verrel_keys)
            self.assertEqual(dcp[1]['depends'][pkg]['arch'], [None])
        self.assertEqual(dcp[1]['depends']['debconf']['relation'], \
            [[None, None]])
        self.assertEqual(dcp[1]['depends']['alsa-utils']['relation'], \
            [[None, None]])
        self.assertEqual(dcp[1]['depends']['psmisc']['relation'], \
            [[None, None]])
        self.assertEqual(dcp[1]['depends']['module-init-tools']['relation'], \
            [[None, None]])
        self.assertEqual(dcp[1]['depends']['procps']['relation'], \
            [[None, None]])
        self.assertEqual(dcp[1]['depends']['modutils']['relation'], \
            [['>=', '2.3.5-1']])
        self.assertEqual(dcp[1]['depends']['debianutils']['relation'], \
            [['>=', '1.6']])
        self.assertEqual(dcp[1]['depends']['lsof']['relation'], \
            [['>=', '4.64-1']])
        keys = Set(('recommends', 'description', 'package', 'suggests', \
            'depends', 'architecture'))
        self.assertEqual(keys.intersection(dcp[2].keys()), keys)
        self.assertEqual(dcp[2]['package'], 'alsa-source')
        self.assertEqual(dcp[2]['architecture'], 'all')
        self.assertEqual(dcp[2]['description'], ['ALSA sound driver source', ' ALSA is an effort to create a modules sound system for Linux,\n while maintaining full compatibility with OSS/Lite. \n .\n This is the source code for the ALSA drivers.  Please note that the\n kernel sources or headers must be installed to compile these modules.'])
        self.assertEqual(dcp[2]['suggests'].keys(), ['devscripts'])
        self.assertEqual(verrel_keys.intersection(dcp[2]['suggests']['devscripts']), verrel_keys)
        self.assertEqual(dcp[2]['suggests']['devscripts']['arch'], [None])
        self.assertEqual(dcp[2]['suggests']['devscripts']['relation'], \
            [[None, None]])
        recs = Set(('dpkg-dev', 'kernel-package'))
        self.assertEqual(recs.intersection(dcp[2]['recommends'].keys()), recs)
        for pkg in recs:
            self.assertEqual(verrel_keys.intersection(dcp[2]['recommends'][pkg].keys()), verrel_keys)
            self.assertEqual(dcp[2]['recommends'][pkg]['arch'], [None])
            self.assertEqual(dcp[2]['recommends'][pkg]['relation'], \
                [[None, None]])
        deps = Set(('debconf', 'gcc', 'make', 'debhelper', 'bzip2', \
            'c-compiler', 'debconf-utils'))
        self.assertEqual(deps.intersection(dcp[2]['depends'].keys()), deps)
        for pkg in deps:
            self.assertEqual(verrel_keys.intersection(dcp[2]['depends'][pkg].keys()), verrel_keys)
            self.assertEqual(dcp[2]['depends'][pkg]['arch'], [None])
            if not pkg.startswith('deb'):
                self.assertEqual(dcp[2]['depends'][pkg]['relation'], \
                    [[None, None]])
        self.assertEqual(dcp[2]['depends']['debconf']['relation'], \
            [['>=', '0.2.26']])
        self.assertEqual(dcp[2]['depends']['debhelper']['relation'], \
            [['>=', '4.0.0']])
        self.assertEqual(dcp[3]['package'], 'alsa-headers')
        self.assertEqual(dcp[3]['architecture'], 'all')
        self.assertEqual(dcp[3]['description'], ['ALSA sound driver header files', ' ALSA is an effort to create a modules sound system for Linux,\n while maintaining full compatibility with OSS/Lite.\n .\n These are the necessary header files to develop interface libraries\n between the kernel and user space.'])
        for field in ('replaces', 'conflicts'):
            keys = Set(('alsa-headers-0.5', 'alsa-headers-0.4', \
                'alsa-headers-unstable'))
            self.assertEqual(keys.intersection(dcp[3][field].keys()), keys)
        for pkg in keys:
            self.assertEqual(verrel_keys.intersection(dcp[3][field][pkg].keys()), verrel_keys)
            self.assertEqual(dcp[3][field][pkg]['arch'], [None])
            self.assertEqual(dcp[3][field][pkg]['relation'], [[None, \
                None]])

    def test_whitespace(self):
        "Test that whitespace doesn't affect parsing"
        dcp = DebianControlParser(os.path.join(self.dir, 'version-spaces'))
        self.assertEqual(dcp.stanzas(), 1)
        self.assertEqual(dcp[0]['package'], 'linda')
        self.assertEqual(dcp[0]['version'], '0.3.0')
        self.assertEqual(dcp[0]['maintainer'], 'Steve Kowalik <stevenk@debian.org>')
        self.assertEqual(dcp[0]['description'][0], 'Debian package checker, not unlike lintian')
        verrel_keys = Set(('arch', 'relation'))
        depends = Set(('python', 'dash', 'groff-base', 'file', 'binutils'))
        self.assertEqual(depends.intersection(dcp[0]['depends']), depends)
        for pkg in depends:
            self.assertEqual(verrel_keys.intersection(dcp[0]['depends'][pkg].keys()), verrel_keys)
        self.assertEqual(dcp[0]['depends']['python']['arch'], [None, None])
        self.assertEqual(dcp[0]['depends']['python']['relation'], \
            [['>=', '2.3.1'], ['<<', '2.4']])
        for pkg in ('dash', 'groff-base', 'file', 'binutils'):
            self.assertEqual(dcp[0]['depends'][pkg]['arch'], [None])
            self.assertEqual(dcp[0]['depends'][pkg]['relation'], \
                [[None, None]])
        
    def test_section_parsing(self):
        "Test parsing of the Section: header"
        dcp = DebianControlParser(os.path.join(self.dir, 'section-sound'))
        self.assertEqual(dcp[0]['section'], ['sound'])
        dcp = DebianControlParser(os.path.join(self.dir, 'section-nonus'))
        self.assertEqual(dcp[0]['section'], ['non-US', 'sound'])

    def test_broken_control(self):
        "Test parsing of a broken control file"
        dcp = DebianControlParser(os.path.join(self.dir, 'broken-control'))
        keys = Set(('${shlibs:Depends}', '${misc:Depends}', 'perl', \
            'lockfile-progs', 'gawk'))
        self.assertEqual(keys.intersection(dcp[1]['depends'].keys()), keys)
        keys = Set(('catdoc', 'pstotext'))
        self.assertEqual(keys.intersection(dcp[1]['suggests'].keys()), keys)

    def test_multiline_depends(self):
        "Test parsing of a control file that uses a multiple line field"
        dcp = DebianControlParser(os.path.join(self.dir, 'multiline-depends'))
        depends = Set(('baz', 'bar', 'quux'))
        self.assertEqual(depends.intersection(dcp[1]['depends'].keys()), \
            depends)

