import os, unittest, tempfile, shutil
from linda.collector import Collector
from linda.funcs import run_external_cmd, iterate_dir
from sets import Set

class CollectorBaseTest(unittest.TestCase):
    def setUp(self):
        self.cur_dir = os.getcwd()

    def tearDown(self):
        os.chdir(self.cur_dir)
    
    def test_binary_simple(self):
        "Check a simple binary instance of Collector"
        dir = tempfile.mkdtemp()
        o = run_external_cmd('tar zxvvCf %s tests/base_collectorbinsim.tar.gz' % dir)
        collect = Collector('bin', dir, o)
        files = Set(('/ch-embed.html', '/file with spaces in it', '/ls', \
            '/README.gz', '/libao.so.2', '/file with: bad chars', \
            '/file$foo.so'))
        self.assertEqual(files.intersection(collect('files', 'files').keys()),\
            files)
        self.assertEqual(collect('files', 'files').has_key('/'), False)
        self.assertEqual(collect('files', 'dirs'), {})
        self.assertEqual(collect('output', 'file')['/file with: bad chars'], \
            'ASCII text')
        self.assertEqual(collect('output', 'file')['/file$foo.so'], 'empty')
        elf = Set(('/ls', '/libao.so.2'))
        self.assertEqual(elf.intersection(collect('files', 'elf')), elf)
        shutil.rmtree(dir)

    def test_binary_complex(self):
        "Check a complex binary instance of Collector"
        dir = tempfile.mkdtemp()
        o = run_external_cmd('tar zxvvCf %s tests/base_collectorbincom.tar.gz' % dir)
        collect = Collector('bin', dir, o)
        files = Set(('/usr/share/doc/xringd/changelog.Debian.gz', \
            '/lib/libBrokenLocale.so.1', '/bin/foo1', '/bin/foo2'))
        self.assertEqual(files.intersection(collect('files', 'files').keys()),\
            files)
        dirs = Set(('/bin', '/lib', '/opt', '/usr', '/usr/share', \
            '/usr/share/doc', '/usr/share/doc/xringd'))
        self.assertEqual(dirs.intersection(collect('files', 'dirs').keys()), \
            dirs)
        self.assertEqual(collect('files', 'elf'), \
            ['/lib/libBrokenLocale.so.1'])
        shutil.rmtree(dir)

    def test_source(self):
        "Check a source instance of Collector"
        dir = tempfile.mkdtemp()
        run_external_cmd('tar zxCf %s tests/base_collectorsrc.tar.gz' % dir)
        os.chdir(dir)
        collect = Collector('src', dir, iterate_dir('.'))
        files = Set(('xringd.c', 'debian/a'))
        self.assertEqual(files.intersection(collect('files', 'files').keys()),\
            files)
        os.chdir(self.cur_dir)
        shutil.rmtree(dir)

