import unittest, gettext, os, linda
from linda.collector import Collector
from linda.debug import dprint
from linda.parser.datavalidator import DataValidator
from linda.libchecks import Checks

linda.checks = Checks()

class BinaryCheckBinaryTest(unittest.TestCase):
    def setUp(self):
        execfile('checks/binary.py', {'_': gettext.gettext, 'dprint': \
            dprint, 'DataValidator': DataValidator, 'os': os})
        for klass in linda.checks.registry['binary'][2]:
            self.instance = klass('/tmp', '/tmp/test_0.0.1-1_all.deb', {})

    def clear_errdict(self):
        for key in self.instance.err_dict.keys():
            del self.instance.err_dict[key]

    def test_arch_checks(self):
        "Check arch_checks() method of BinaryCheck"
        method = getattr(self.instance, 'arch_checks')
        self.instance.information = {'control': {'self': [{}]}}
        self.instance.information['collector'] = Collector('bin', '/tmp', '')
        method()
        self.assert_('unknown-arch' in self.instance.err_dict.keys())
        self.clear_errdict()
        self.instance.information['control']['self'][0]['architecture'] = \
            'i386'
        self.instance.information['collector'].set('files', 'elf', \
            ['/usr/bin/lala'])
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['control']['self'][0]['architecture'] = \
            'all'
        method()
        self.assert_('pkg-should-be-arch-any' in self.instance.err_dict.keys())
        
    def test_binary_wierd(self):
        "Check binary_in_wierd_places() method of BinaryCheck"
        method = getattr(self.instance, 'binary_in_wierd_places')
        self.instance.information['collector'] = Collector('bin', '/tmp', '')
        self.instance.information['collector'].set('files', 'elf', \
            ['/etc/lala'])
        method()
        self.assert_('binary-in-etc' in self.instance.err_dict.keys())
        self.instance.information['collector'].set('files', 'elf', \
            ['/usr/share/doc/lala'])
        method()
        self.assert_('binary-in-doc-dir' in self.instance.err_dict.keys())
        self.instance.information['collector'].set('files', 'elf', \
            ['/usr/share/lala'])
        method()
        self.assert_('arch-dep-in-usr-share' in self.instance.err_dict.keys())
        
    def test_depends(self):
        "Check depends() method of BinaryCheck"
        method = getattr(self.instance, 'depends')
        self.instance.information = {'control': {'self': [{'depends': \
            {'libc6': []}}]}}
        self.instance.information['collector'] = Collector('bin', '/tmp', '')
        self.instance.information['collector'].set('files', 'elf', \
            ['/usr/bin/lala'])
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['control']['self'][0] = {}
        method()
        self.assert_('no-depends-line' in self.instance.err_dict.keys())

    def test_sonames(self):
        "Check sonames() method of BinaryCheck"
        method = getattr(self.instance, 'sonames')
        self.instance.information['collector'] = Collector('bin', '/tmp', '')
        self.instance.information['collector'].set('output', 'ldd', \
            {'/usr/bin/foo': '\tlibbar.so => /tmp/unpacked/usr/lib/libbar.so (0x40016000)\n\tlibbaz.so => /tmp/unpacked/usr/lib/libbaz.so (0x40018000)\n\tlibstdc++.so.5 => /usr/lib/libstdc++.so.5 (0x40023000)\n\tlibm.so.6 => /lib/libm.so.6 (0x400da000)\n\tlibgcc_s.so.1 => /lib/libgcc_s.so.1 (0x400fd000)\n\tlibc.so.6 => /lib/libc.so.6 (0x40105000)\n\tlibstdc++-libc6.2-2.so.3 => /usr/lib/libstdc++-libc6.2-2.so.3 (0x40233000)\n\t/lib/ld-linux.so.2 => /lib/ld-linux.so.2 (0x40000000)'})
        method('/usr/bin/foo')
        self.assert_('double-shlib' in self.instance.err_dict.keys())

    def test_note_comment(self):
        "Check note_comment() method of BinaryCheck"
        method = getattr(self.instance, 'note_comment')
        method('/usr/bin/foo_g')
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['collector'] = Collector('bin', '/tmp', '')
        self.instance.information['collector'].set('output', 'objdump', \
            {'/usr/bin/foo': ''})
        method('/usr/bin/foo')
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['collector'].set('output', 'objdump', \
            {'/usr/bin/foo': ' 21 .comment      0000014d  00000000  00000000  00003220  2**0'})
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/bin/foo': ''})
        method('/usr/bin/foo')
        self.assert_('binary-note-or-comment' in self.instance.err_dict.keys())
        self.clear_errdict()
        self.instance.information['collector'].set('output', 'objdump', \
            {'/lib/modules/1.0/test.o': ' 21 .comment      0000014d  00000000  00000000  00003220  2**0', '/usr/lib/lala.o': ' 21 .comment      0000014d  00000000  00000000  00003220  2**0'})
        self.instance.information['collector'].set('output', 'file', \
            {'/lib/modules/1.0/test.o': 'ELF 32-bit LSB relocatable, Intel 80386, version 1 (SYSV), not stripped', '/usr/lib/lala.o': 'ELF 64-bit LSB relocatable, IA-64 (Intel 64 bit architecture) version 1 (SYSV), not stripped'})
        method('/lib/modules/1.0/test.o')
        self.assertEqual(self.instance.err_dict.keys(), [])
        method('/usr/lib/lala.o')
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['collector'].set('output', 'objdump', \
            {'/usr/bin/foo': '22 .note         00000050  00000000  00000000  0000336d  2**0'})
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/bin/foo': ''})
        method('/usr/bin/foo')
        self.assert_('binary-note-or-comment' in self.instance.err_dict.keys())
        self.instance.information['collector'].set('output', 'objdump', \
            {'/usr/bin/foo': ' 21 .comment      0000014d  00000000  00000000  00003220  2**0\n 22 .note         00000050  00000000  00000000  0000336d  2**0'})
        method('/usr/bin/foo')
        self.assert_('not-quite-stripped-binary' in \
            self.instance.err_dict.keys())
        
    def test_file_checks(self):
        "Check file_checks() method of BinaryCheck"
        method = getattr(self.instance, 'file_checks')
        self.instance.information['collector'] = Collector('bin', '/tmp', '')
        self.instance.information['collector'].set('output', 'file', \
            {'/boot/foo': 'ELF 32-bit LSB executable, Intel 80386, version 1 (SYSV), statically linked'})
        method('/boot/foo')
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['collector'].set('output', 'file', \
            {'/bin/foo.static': 'ELF 32-bit LSB executable, Intel 80386, version 1 (SYSV), statically linked'})
        method('/bin/foo.static')
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/bin/foo-dbg': 'ELF 32-bit LSB executable, Intel 80386, version 1 (SYSV), statically linked'})
        method('/usr/bin/foo-dbg')
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/bin/foo': 'ELF 32-bit LSB executable, Intel 80386, version 1 (SYSV), statically linked'})
        method('/usr/bin/foo')
        self.assert_('static-binary' in self.instance.err_dict.keys())
        self.clear_errdict()
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/bin/foobar': 'ELF 64-bit LSB executable, Alpha, version 1 (SYSV), dynamically linked (uses shared libs), stripped'})
        method('/usr/bin/foobar')
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['collector'].set('output', 'file', \
            {'/lib/modules/0.0.1/module.o': 'ELF 32-bit LSB relocatable, Intel 80386, version 1 (SYSV), not stripped'})
        method('/lib/modules/0.0.1/module.o')
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['collector'].set('output', 'file', \
            {'/lib/lala.o': 'ELF 32-bit LSB relocatable, Intel 80386, version 1 (SYSV), not stripped'})
        method('/lib/lala.o')
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/bin/barfoo': 'ELF 64-bit LSB executable, Alpha, version 1 (SYSV), dynamically linked (uses shared libs), not stripped'})
        method('/usr/bin/barfoo')
        self.assert_('unstripped-binary' in self.instance.err_dict.keys())
        self.clear_errdict()
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/bin/lala': 'ELF 32-bit MSB executable, SPARC, version 1 (SYSV), dynamically linked (uses shared libs), stripped'})
        method('/usr/bin/lala')
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/bin/foo-dbg': 'ELF 32-bit MSB executable, SPARC, version 1 (SYSV), dynamically linked (uses shared libs), stripped'})
        method('/usr/bin/foo-dbg')
        self.assert_('stripped-binary-not' in self.instance.err_dict.keys())

    def test_objdump_checks(self):
        "Check objdump_checks() method of BinaryCheck"
        method = getattr(self.instance, 'objdump_checks')
        self.instance.information['collector'] = Collector('bin', '/tmp', '')
        self.instance.information['collector'].set('output', 'objdump', \
            {'/usr/bin/lala': '  RPATH       /usr/lib'})
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/bin/lala': ''})
        method('/usr/bin/lala')
        self.assert_('binary-rpath' in self.instance.err_dict.keys())
        self.clear_errdict()
        self.instance.information['collector'].set('output', 'objdump', \
            {'/usr/bin/libc5-compat/foo': '  NEEDED      libc.so.5'})
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/bin/libc5-compat/foo': ''})
        method('/usr/bin/libc5-compat/foo')
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['collector'].set('output', 'objdump', \
            {'/usr/bin/foo': '  NEEDED      libc.so.5'})
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/bin/foo': ''})
        method('/usr/bin/foo')
        self.assert_('binary-libc5' in self.instance.err_dict.keys())
        self.instance.information['collector'].set('output', 'objdump', \
            {'/usr/bin/libc5-compat/foo': '  NEEDED      libc.so.6'})
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/bin/libc5-compat/foo': ''})
        method('/usr/bin/libc5-compat/foo')
        self.assert_('libc6-binary-in-libc5-dir' in \
            self.instance.err_dict.keys())
        self.clear_errdict()
        self.instance.information['collector'].set('output', 'objdump', \
            {'/usr/bin/foobar-debug': ''})
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/bin/foobar-debug': ''})
        method('/usr/bin/foobar-debug')
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['collector'].set('output', 'objdump', \
            {'/usr/bin/foobar': ''})
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/bin/foobar': ''})
        method('/usr/bin/foobar')
        self.assert_('binary-libc6' in self.instance.err_dict.keys())
        self.instance.information['collector'].set('output', 'objdump', \
            {'/usr/bin/foobar': 'Packed with UPX'})
        method('/usr/bin/foobar')
        self.assert_('binary-upx' in self.instance.err_dict.keys())
        self.clear_errdict()
        self.instance.information['collector'].set('output', 'objdump', \
            {'/usr/bin/lala': 'NEEDED libc.so.6\n000132d4  w   DF .text  00000004  Base        __gmon_start__'})
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/bin/lala': ''})
        self.instance.arch = 'hppa'
        method('/usr/bin/lala')
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.arch = 'sparc'
        method('/usr/bin/lala')
        self.assert_('binary-profiling' in self.instance.err_dict.keys())
        self.clear_errdict()
        self.instance.information['collector'].set('output', 'objdump', \
            {'/usr/bin/foo-debug': ''})
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/bin/foo-debug': ''})
        method('/usr/bin/foo-debug')
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['collector'].set('output', 'objdump', \
            {'/usr/X11R6/lib/modules/input/foo.o': ''})
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/X11R6/lib/modules/input/foo.o': 'ELF 32-bit LSB relocatable, Intel 80386, version 1 (SYSV), not stripped'})
        method('/usr/X11R6/lib/modules/input/foo.o')
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['collector'].set('output', 'objdump', \
            {'/usr/bin/foo': ''})
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/bin/foo': ''})
        method('/usr/bin/foo')
        self.assert_('shared-binary' in self.instance.err_dict.keys())

