/*
*******************************************************************************
**        O.S   : Linux
**   FILE NAME  : arcmsr_hba.c
**        BY    : Erich Chen
**   Description: SCSI RAID Device Driver for
**                ARECA RAID Host adapter
*******************************************************************************
** Copyright (C) 2002 - 2005, Areca Technology Corporation All rights reserved
**
**     Web site: www.areca.com.tw
**       E-mail: erich@areca.com.tw
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License version 2 as
** published by the Free Software Foundation.
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
*******************************************************************************
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 3. The name of the author may not be used to endorse or promote products
**    derived from this software without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
** IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
** OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
** IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
** INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES(INCLUDING,BUT
** NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION)HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE)ARISING IN ANY WAY OUT OF THE USE OF
** THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************
** For history of changes, see Documentation/scsi/ChangeLog.arcmsr
**     Firmware Specification, see Documentation/scsi/arcmsr_spec.txt
*******************************************************************************
*/
#include <linux/module.h>
#include <linux/reboot.h>
#include <linux/spinlock.h>
#include <linux/pci_ids.h>
#include <linux/interrupt.h>
#include <linux/moduleparam.h>
#include <linux/errno.h>
#include <linux/types.h>
#include <linux/delay.h>
#include <linux/dma-mapping.h>
#include <linux/timer.h>
#include <linux/pci.h>
#include <asm/dma.h>
#include <asm/io.h>
#include <asm/system.h>
#include <asm/uaccess.h>
#include <scsi/scsi_host.h>
#include <scsi/scsi.h>
#include <scsi/scsi_cmnd.h>
#include <scsi/scsi_tcq.h>
#include <scsi/scsi_device.h>
#include <scsi/scsi_transport.h>
#include <scsi/scsicam.h>
#include "arcmsr.h"

MODULE_AUTHOR("Erich Chen <erich@areca.com.tw>");
MODULE_DESCRIPTION("ARECA (ARC11xx/12xx) SATA RAID HOST Adapter");
MODULE_LICENSE("Dual BSD/GPL");
MODULE_VERSION(ARCMSR_DRIVER_VERSION);

static int arcmsr_initialize(struct AdapterControlBlock *pACB, struct pci_dev *pdev);
static int arcmsr_iop_message_xfer(struct AdapterControlBlock *pACB, struct scsi_cmnd *cmd);
static int arcmsr_cmd_abort(struct scsi_cmnd *);
static int arcmsr_bus_reset(struct scsi_cmnd *);
static int arcmsr_bios_param(struct scsi_device *sdev,
				struct block_device *bdev, sector_t capacity, int *info);
static int arcmsr_queue_command(struct scsi_cmnd * cmd,
				void (*done) (struct scsi_cmnd *));
static int arcmsr_device_probe(struct pci_dev *pdev,
				const struct pci_device_id *id);
static void arcmsr_device_remove(struct pci_dev *pdev);
static void arcmsr_device_shutdown(struct pci_dev *pdev);
static void arcmsr_pcidev_disattach(struct AdapterControlBlock *pACB);
static void arcmsr_iop_init(struct AdapterControlBlock *pACB);
static void arcmsr_free_ccb_pool(struct AdapterControlBlock *pACB);
static void arcmsr_stop_adapter_bgrb(struct AdapterControlBlock *pACB);
static void arcmsr_flush_adapter_cache(struct AdapterControlBlock *pACB);
static uint8_t arcmsr_wait_msgint_ready(struct AdapterControlBlock *pACB);
static const char *arcmsr_info(struct Scsi_Host *);

static int arcmsr_adjust_disk_queue_depth(struct scsi_device *sdev, int queue_depth)
{
	if (queue_depth > ARCMSR_MAX_CMD_PERLUN)
		queue_depth = ARCMSR_MAX_CMD_PERLUN;
	scsi_adjust_queue_depth(sdev, MSG_ORDERED_TAG, queue_depth);
	return queue_depth;
}

static struct scsi_host_template arcmsr_scsi_host_template = {
	.module			= THIS_MODULE,
	.name			= "ARCMSR ARECA SATA RAID HOST Adapter" ARCMSR_DRIVER_VERSION,
	.info			= arcmsr_info,
	.queuecommand		= arcmsr_queue_command,
	.eh_abort_handler	= arcmsr_cmd_abort,
	.eh_bus_reset_handler	= arcmsr_bus_reset,
	.bios_param		= arcmsr_bios_param,
	.change_queue_depth	= arcmsr_adjust_disk_queue_depth,
	.can_queue		= ARCMSR_MAX_OUTSTANDING_CMD,
	.this_id		= ARCMSR_SCSI_INITIATOR_ID,
	.sg_tablesize		= ARCMSR_MAX_SG_ENTRIES,
	.max_sectors    	= ARCMSR_MAX_XFER_SECTORS,
	.cmd_per_lun		= ARCMSR_MAX_CMD_PERLUN,
	.use_clustering		= ENABLE_CLUSTERING,
	.shost_attrs		= arcmsr_host_attrs,
};

static struct pci_device_id arcmsr_device_id_table[] = {
	{PCI_DEVICE(PCI_VENDOR_ID_ARECA, PCI_DEVICE_ID_ARECA_1110)},
	{PCI_DEVICE(PCI_VENDOR_ID_ARECA, PCI_DEVICE_ID_ARECA_1120)},
	{PCI_DEVICE(PCI_VENDOR_ID_ARECA, PCI_DEVICE_ID_ARECA_1130)},
	{PCI_DEVICE(PCI_VENDOR_ID_ARECA, PCI_DEVICE_ID_ARECA_1160)},
	{PCI_DEVICE(PCI_VENDOR_ID_ARECA, PCI_DEVICE_ID_ARECA_1170)},
	{PCI_DEVICE(PCI_VENDOR_ID_ARECA, PCI_DEVICE_ID_ARECA_1210)},
	{PCI_DEVICE(PCI_VENDOR_ID_ARECA, PCI_DEVICE_ID_ARECA_1220)},
	{PCI_DEVICE(PCI_VENDOR_ID_ARECA, PCI_DEVICE_ID_ARECA_1230)},
	{PCI_DEVICE(PCI_VENDOR_ID_ARECA, PCI_DEVICE_ID_ARECA_1260)},
	{PCI_DEVICE(PCI_VENDOR_ID_ARECA, PCI_DEVICE_ID_ARECA_1270)},
	{PCI_DEVICE(PCI_VENDOR_ID_ARECA, PCI_DEVICE_ID_ARECA_1280)},
	{0, 0}, /* Terminating entry */
};
MODULE_DEVICE_TABLE(pci, arcmsr_device_id_table);
static struct pci_driver arcmsr_pci_driver = {
	.name			= "arcmsr",
	.id_table		= arcmsr_device_id_table,
	.probe			= arcmsr_device_probe,
	.remove			= arcmsr_device_remove,
	.shutdown		= arcmsr_device_shutdown
};

static irqreturn_t arcmsr_do_interrupt(int irq, void *dev_id,
	struct pt_regs *regs)
{
	irqreturn_t handle_state;
	struct AdapterControlBlock *pACB;
	unsigned long flags;

	pACB = (struct AdapterControlBlock *)dev_id;

	spin_lock_irqsave(pACB->host->host_lock, flags);
	handle_state = arcmsr_interrupt(pACB);
	spin_unlock_irqrestore(pACB->host->host_lock, flags);
	return handle_state;
}

static int arcmsr_bios_param(struct scsi_device *sdev,
	struct block_device *bdev, sector_t capacity, int *geom)
{
	int ret, heads, sectors, cylinders, total_capacity;
	unsigned char *buffer;/* return copy of block device's partition table */

	buffer = scsi_bios_ptable(bdev);
	if (buffer) {
		ret = scsi_partsize(buffer, capacity, &geom[2], &geom[0], &geom[1]);
		kfree(buffer);
		if (ret != -1)
			return ret;
	}
	total_capacity = capacity;
	heads = 64;
	sectors = 32;
	cylinders = total_capacity / (heads * sectors);
	if (cylinders > 1024) {
		heads = 255;
		sectors = 63;
		cylinders = total_capacity / (heads * sectors);
	}
	geom[0] = heads;
	geom[1] = sectors;
	geom[2] = cylinders;
	return 0;
}

static int arcmsr_device_probe(struct pci_dev *pdev,
	const struct pci_device_id *id)
{
	struct Scsi_Host *host;
	struct AdapterControlBlock *pACB;
	uint8_t bus, dev_fun;

	if (pci_enable_device(pdev)) {
		printk(KERN_NOTICE
		"arcmsr: adapter probe pci_enable_device error \n");
		return -ENODEV;
	}
	host = scsi_host_alloc(&arcmsr_scsi_host_template, sizeof (struct AdapterControlBlock));
	if (!host) {
		printk(KERN_NOTICE
		"arcmsr: adapter probe scsi_host_alloc error \n");
		pci_disable_device(pdev);
		return -ENODEV;
	}
	if (!pci_set_dma_mask(pdev, DMA_64BIT_MASK))
		printk(KERN_INFO
		"ARECA RAID ADAPTER%d: 64BITS PCI BUS DMA ADDRESSING SUPPORTED\n"
		, host->host_no);
	else if (!pci_set_dma_mask(pdev, DMA_32BIT_MASK))
		printk(KERN_INFO
		"ARECA RAID ADAPTER%d: 32BITS PCI BUS DMA ADDRESSING SUPPORTED\n"
		, host->host_no);
	else {
		printk(KERN_NOTICE
		"ARECA RAID ADAPTER%d: No suitable DMA available.\n"
		, host->host_no);
		pci_disable_device(pdev);
		scsi_host_put(host);
		return -ENOMEM;
	}
	if (pci_set_consistent_dma_mask(pdev, DMA_32BIT_MASK)) {
		printk(KERN_NOTICE
		"ARECA RAID ADAPTER%d:"
		" No 32BIT coherent DMA adressing available\n"
		, host->host_no);
		pci_disable_device(pdev);
		scsi_host_put(host);
		return -ENOMEM;
	}
	bus = pdev->bus->number;
	dev_fun = pdev->devfn;
	pACB = (struct AdapterControlBlock *) host->hostdata;
	memset(pACB, 0, sizeof (struct AdapterControlBlock));
	pACB->host = host;
	pACB->pdev = pdev;
	host->max_sectors = ARCMSR_MAX_XFER_SECTORS;
	host->max_lun = ARCMSR_MAX_TARGETLUN;
	host->max_id = ARCMSR_MAX_TARGETID;/*16:8*/
	host->max_cmd_len = 16;    /*this is issue of 64bit LBA, over 2T byte*/
	host->sg_tablesize = ARCMSR_MAX_SG_ENTRIES;
	host->can_queue = ARCMSR_MAX_FREECCB_NUM; /* max simultaneous cmds */
	host->cmd_per_lun = ARCMSR_MAX_CMD_PERLUN;
	host->this_id = ARCMSR_SCSI_INITIATOR_ID;
	host->unique_id = (bus << 8) | dev_fun;
	host->io_port = 0;
	host->n_io_port = 0;
	host->irq = pdev->irq;
	pci_set_master(pdev);
	if (arcmsr_initialize(pACB, pdev)) {
		printk(KERN_NOTICE
		"arcmsr%d: initialize got error \n"
		, host->host_no);
		pci_disable_device(pdev);
		scsi_host_put(host);
		return -ENODEV;
	}
	if (pci_request_regions(pdev, "arcmsr")) {
		printk(KERN_NOTICE
		"arcmsr%d: adapter probe: pci_request_regions failed \n"
		, host->host_no);
		arcmsr_pcidev_disattach(pACB);
		return -ENODEV;
	}
	if (request_irq(pdev->irq, arcmsr_do_interrupt, SA_INTERRUPT | SA_SHIRQ,
		"arcmsr", pACB)) {
		printk(KERN_NOTICE
		"arcmsr%d: request IRQ=%d failed !\n"
		, host->host_no, pdev->irq);
		arcmsr_pcidev_disattach(pACB);
		return -ENODEV;
	}
	arcmsr_iop_init(pACB);
	if (scsi_add_host(host, &pdev->dev)) {
		printk(KERN_NOTICE
		"arcmsr%d: scsi_add_host got error \n"
		, host->host_no);
		arcmsr_pcidev_disattach(pACB);
		return -ENODEV;
	}
	arcmsr_alloc_sysfs_attr(pACB);
	pci_set_drvdata(pdev, host);
	scsi_scan_host(host);
	return 0;
}

static void arcmsr_device_remove(struct pci_dev *pdev)
{
	struct Scsi_Host * host = pci_get_drvdata(pdev);
	struct AdapterControlBlock *pACB = (struct AdapterControlBlock *) host->hostdata;

	arcmsr_pcidev_disattach(pACB);
}

static void arcmsr_device_shutdown(struct pci_dev *pdev)
{
	struct Scsi_Host *host = pci_get_drvdata(pdev);
	struct AdapterControlBlock *pACB = (struct AdapterControlBlock *) host->hostdata;

	arcmsr_stop_adapter_bgrb(pACB);
	arcmsr_flush_adapter_cache(pACB);
}

static int arcmsr_module_init(void)
{
	int error = 0;

	error = pci_register_driver(&arcmsr_pci_driver);
	return error;
}

static void arcmsr_module_exit(void)
{
	pci_unregister_driver(&arcmsr_pci_driver);
}
module_init(arcmsr_module_init);
module_exit(arcmsr_module_exit);

static void arcmsr_pci_unmap_dma(struct CommandControlBlock *pCCB)
{
	struct AdapterControlBlock *pACB = pCCB->pACB;
	struct scsi_cmnd *pcmd = pCCB->pcmd;

	if (pcmd->use_sg != 0) {
		struct scatterlist *sl;

		sl = (struct scatterlist *)pcmd->request_buffer;
		pci_unmap_sg(pACB->pdev, sl, pcmd->use_sg, pcmd->sc_data_direction);
	}
	else if (pcmd->request_bufflen != 0)
		pci_unmap_single(pACB->pdev,
			(dma_addr_t)(unsigned long)pcmd->SCp.ptr,
			pcmd->request_bufflen, pcmd->sc_data_direction);
}

static void arcmsr_flush_adapter_cache(struct AdapterControlBlock *pACB)
{
	struct MessageUnit __iomem *reg=pACB->pmu;

	writel(ARCMSR_INBOUND_MESG0_FLUSH_CACHE, &reg->inbound_msgaddr0);
	if (arcmsr_wait_msgint_ready(pACB))
		printk(KERN_NOTICE
			"arcmsr%d: wait 'flush adapter cache' timeout \n"
			, pACB->host->host_no);
}

static void arcmsr_ccb_complete(struct CommandControlBlock *pCCB, int stand_flag)
{
	struct AdapterControlBlock *pACB = pCCB->pACB;
	struct scsi_cmnd *pcmd = pCCB->pcmd;

	arcmsr_pci_unmap_dma(pCCB);
	if (stand_flag == 1)
		atomic_dec(&pACB->ccboutstandingcount);
	pCCB->startdone = ARCMSR_CCB_DONE;
	pCCB->ccb_flags = 0;
	list_add_tail(&pCCB->list, &pACB->ccb_free_list);
	pcmd->scsi_done(pcmd);
}

static void arcmsr_report_sense_info(struct CommandControlBlock *pCCB)
{
	struct scsi_cmnd *pcmd = pCCB->pcmd;
	struct SENSE_DATA *psenseBuffer = (struct SENSE_DATA *)pcmd->sense_buffer;

	pcmd->result = DID_OK << 16;
	if (psenseBuffer) {
		int sense_data_length =
			sizeof (struct SENSE_DATA) < sizeof (pcmd->sense_buffer)
			? sizeof (struct SENSE_DATA) : sizeof (pcmd->sense_buffer);
		memset(psenseBuffer, 0, sizeof (pcmd->sense_buffer));
		memcpy(psenseBuffer, pCCB->arcmsr_cdb.SenseData, sense_data_length);
		psenseBuffer->ErrorCode = SCSI_SENSE_CURRENT_ERRORS;
		psenseBuffer->Valid = 1;
	}
}

static void arcmsr_abort_allcmd(struct AdapterControlBlock *pACB)
{
	struct MessageUnit __iomem *reg = pACB->pmu;

	writel(ARCMSR_INBOUND_MESG0_ABORT_CMD, &reg->inbound_msgaddr0);
	if (arcmsr_wait_msgint_ready(pACB))
		printk(KERN_NOTICE
			"arcmsr%d: wait 'abort all outstanding command' timeout \n"
			, pACB->host->host_no);
}

static uint8_t arcmsr_wait_msgint_ready(struct AdapterControlBlock *pACB)
{
	struct MessageUnit __iomem *reg = pACB->pmu;
	uint32_t Index;
	uint8_t Retries = 0x00;

	do {
		for(Index = 0; Index < 100; Index++) {
			if (readl(&reg->outbound_intstatus)
				& ARCMSR_MU_OUTBOUND_MESSAGE0_INT) {
				writel(ARCMSR_MU_OUTBOUND_MESSAGE0_INT
					, &reg->outbound_intstatus);
				return 0x00;
			}
			msleep_interruptible(10);
		}/*max 1 seconds*/
	} while (Retries++ < 20);/*max 20 sec*/
	return 0xff;
}

void arcmsr_iop_reset(struct AdapterControlBlock *pACB)
{
	struct MessageUnit __iomem *reg = pACB->pmu;
	struct CommandControlBlock *pCCB;
	uint32_t intmask_org, mask;
	int i = 0;

	if (atomic_read(&pACB->ccboutstandingcount) != 0) {
		/* talk to iop 331 outstanding command aborted */
		arcmsr_abort_allcmd(pACB);
		/* wait for 3 sec for all command aborted*/
		msleep_interruptible(3000);
		/* disable all outbound interrupt */
		intmask_org = readl(&reg->outbound_intmask);
		writel(intmask_org | ARCMSR_MU_OUTBOUND_ALL_INTMASKENABLE,
			&reg->outbound_intmask);
		/* clear all outbound posted Q */
		for(i = 0; i < ARCMSR_MAX_OUTSTANDING_CMD; i++)
			readl(&reg->outbound_queueport);
		for(i = 0; i < ARCMSR_MAX_FREECCB_NUM; i++) {
			pCCB = pACB->pccb_pool[i];
			if (pCCB->startdone == ARCMSR_CCB_START) {
				pCCB->startdone = ARCMSR_CCB_ABORTED;
				pCCB->pcmd->result = DID_ABORT << 16;
				arcmsr_ccb_complete(pCCB, 1);
			}
		}
		/* enable all outbound interrupt */
		mask =~ (ARCMSR_MU_OUTBOUND_POSTQUEUE_INTMASKENABLE
			| ARCMSR_MU_OUTBOUND_DOORBELL_INTMASKENABLE);
		writel(intmask_org & mask, &reg->outbound_intmask);
	}
	atomic_set(&pACB->ccboutstandingcount, 0);
}

static void arcmsr_build_ccb(struct AdapterControlBlock *pACB, struct CommandControlBlock *pCCB,
	struct scsi_cmnd *pcmd)
{
	struct ARCMSR_CDB *pARCMSR_CDB = (struct ARCMSR_CDB *)&pCCB->arcmsr_cdb;
	int8_t *psge = (int8_t *)&pARCMSR_CDB->u;
	uint32_t address_lo, address_hi;
	int arccdbsize = 0x30;

	pCCB->pcmd = pcmd;
	memset(pARCMSR_CDB, 0, sizeof (struct ARCMSR_CDB));
	pARCMSR_CDB->Bus = 0;
	pARCMSR_CDB->TargetID = pcmd->device->id;
	pARCMSR_CDB->LUN = pcmd->device->lun;
	pARCMSR_CDB->Function = 1;
	pARCMSR_CDB->CdbLength = (uint8_t)pcmd->cmd_len;
	pARCMSR_CDB->Context = (unsigned long)pARCMSR_CDB;
	memcpy(pARCMSR_CDB->Cdb, pcmd->cmnd, pcmd->cmd_len);
	if (pcmd->use_sg) {
		int length, sgcount, i, cdb_sgcount = 0;
		struct scatterlist *sl;

		/* Get Scatter Gather List from scsiport. */
		sl = (struct scatterlist *) pcmd->request_buffer;
		sgcount = pci_map_sg(pACB->pdev, sl, pcmd->use_sg,
				pcmd->sc_data_direction);
		/* map stor port SG list to our iop SG List. */
		for(i = 0; i < sgcount; i++) {
			/* Get the physical address of the current data pointer */
			length = cpu_to_le32(sg_dma_len(sl));
			address_lo = cpu_to_le32(dma_addr_lo32(sg_dma_address(sl)));
			address_hi = cpu_to_le32(dma_addr_hi32(sg_dma_address(sl)));
			if (address_hi == 0) {
				struct SG32ENTRY *pdma_sg = (struct SG32ENTRY *)psge;

				pdma_sg->address = address_lo;
				pdma_sg->length = length;
				psge += sizeof (struct SG32ENTRY);
				arccdbsize += sizeof (struct SG32ENTRY);
			} else {
				struct SG64ENTRY *pdma_sg = (struct SG64ENTRY *)psge;

				pdma_sg->addresshigh = address_hi;
				pdma_sg->address = address_lo;
				pdma_sg->length = length|IS_SG64_ADDR;
				psge += sizeof (struct SG64ENTRY);
				arccdbsize += sizeof (struct SG64ENTRY);
			}
			sl++;
			cdb_sgcount++;
		}
		pARCMSR_CDB->sgcount = (uint8_t)cdb_sgcount;
		pARCMSR_CDB->DataLength = pcmd->request_bufflen;
		if ( arccdbsize > 256)
			pARCMSR_CDB->Flags |= ARCMSR_CDB_FLAG_SGL_BSIZE;
	} else if (pcmd->request_bufflen) {
		dma_addr_t dma_addr;
		dma_addr = pci_map_single(pACB->pdev, pcmd->request_buffer,
				pcmd->request_bufflen, pcmd->sc_data_direction);
		pcmd->SCp.ptr = (char *)(unsigned long) dma_addr;
		address_lo = cpu_to_le32(dma_addr_lo32(dma_addr));
		address_hi = cpu_to_le32(dma_addr_hi32(dma_addr));
		if (address_hi == 0) {
			struct  SG32ENTRY *pdma_sg = (struct SG32ENTRY *)psge;
			pdma_sg->address = address_lo;
			pdma_sg->length = pcmd->request_bufflen;
		} else {
			struct SG64ENTRY *pdma_sg = (struct SG64ENTRY *)psge;
			pdma_sg->addresshigh = address_hi;
			pdma_sg->address = address_lo;
			pdma_sg->length = pcmd->request_bufflen|IS_SG64_ADDR;
		}
		pARCMSR_CDB->sgcount = 1;
		pARCMSR_CDB->DataLength = pcmd->request_bufflen;
	}
	if (pcmd->sc_data_direction == DMA_TO_DEVICE ) {
		pARCMSR_CDB->Flags |= ARCMSR_CDB_FLAG_WRITE;
		pCCB->ccb_flags |= CCB_FLAG_WRITE;
	}
}

static void arcmsr_post_ccb(struct AdapterControlBlock *pACB, struct CommandControlBlock *pCCB)
{
	struct MessageUnit __iomem *reg = pACB->pmu;
	uint32_t cdb_shifted_phyaddr = pCCB->cdb_shifted_phyaddr;
	struct ARCMSR_CDB *pARCMSR_CDB = (struct ARCMSR_CDB *)&pCCB->arcmsr_cdb;

	atomic_inc(&pACB->ccboutstandingcount);
	pCCB->startdone = ARCMSR_CCB_START;
	if (pARCMSR_CDB->Flags & ARCMSR_CDB_FLAG_SGL_BSIZE)
		writel(cdb_shifted_phyaddr | ARCMSR_CCBPOST_FLAG_SGL_BSIZE,
			&reg->inbound_queueport);
	else
		writel(cdb_shifted_phyaddr, &reg->inbound_queueport);
}

void arcmsr_post_Qbuffer(struct AdapterControlBlock *pACB)
{
	struct MessageUnit __iomem *reg = pACB->pmu;
	struct QBUFFER __iomem *pwbuffer = (struct QBUFFER __iomem *) &reg->message_wbuffer;
	uint8_t __iomem *iop_data = (uint8_t __iomem *) pwbuffer->data;
	int32_t allxfer_len = 0;

	if (pACB->acb_flags & ACB_F_MESSAGE_WQBUFFER_READED) {
		pACB->acb_flags &= (~ACB_F_MESSAGE_WQBUFFER_READED);
		while ((pACB->wqbuf_firstindex != pACB->wqbuf_lastindex)
			&& (allxfer_len < 124)) {
			writeb(pACB->wqbuffer[pACB->wqbuf_firstindex], iop_data);
			pACB->wqbuf_firstindex++;
			pACB->wqbuf_firstindex %= ARCMSR_MAX_QBUFFER;
			iop_data++;
			allxfer_len++;
		}
		writel(allxfer_len, &pwbuffer->data_len);
		writel(ARCMSR_INBOUND_DRIVER_DATA_WRITE_OK
			, &reg->inbound_doorbell);
	}
}

static void arcmsr_stop_adapter_bgrb(struct AdapterControlBlock *pACB)
{
	struct MessageUnit __iomem *reg = pACB->pmu;

	pACB->acb_flags &= ~ACB_F_MSG_START_BGRB;
	writel(ARCMSR_INBOUND_MESG0_STOP_BGRB, &reg->inbound_msgaddr0);
	if (arcmsr_wait_msgint_ready(pACB))
		printk(KERN_NOTICE
			"arcmsr%d: wait 'stop adapter background rebulid' timeout \n"
			, pACB->host->host_no);
}

static void arcmsr_free_ccb_pool(struct AdapterControlBlock *pACB)
{
	dma_free_coherent(&pACB->pdev->dev
		, ARCMSR_MAX_FREECCB_NUM * sizeof (struct CommandControlBlock) + 0x20,
		pACB->dma_coherent,
		pACB->dma_coherent_handle);
}

irqreturn_t arcmsr_interrupt(struct AdapterControlBlock *pACB)
{
	struct MessageUnit __iomem *reg = pACB->pmu;
	struct CommandControlBlock *pCCB;
	uint32_t flag_ccb, outbound_intstatus, outbound_doorbell;

	outbound_intstatus = readl(&reg->outbound_intstatus)
		& pACB->outbound_int_enable;
	writel(outbound_intstatus, &reg->outbound_intstatus);
	if (outbound_intstatus & ARCMSR_MU_OUTBOUND_DOORBELL_INT) {
		outbound_doorbell = readl(&reg->outbound_doorbell);
		writel(outbound_doorbell, &reg->outbound_doorbell);
		if (outbound_doorbell & ARCMSR_OUTBOUND_IOP331_DATA_WRITE_OK) {
			struct QBUFFER __iomem * prbuffer =
				(struct QBUFFER __iomem *) &reg->message_rbuffer;
			uint8_t __iomem * iop_data = (uint8_t __iomem *)prbuffer->data;
			int32_t my_empty_len, iop_len, rqbuf_firstindex, rqbuf_lastindex;

			rqbuf_lastindex = pACB->rqbuf_lastindex;
			rqbuf_firstindex = pACB->rqbuf_firstindex;
			iop_len = readl(&prbuffer->data_len);
			my_empty_len = (rqbuf_firstindex - rqbuf_lastindex - 1)
					&(ARCMSR_MAX_QBUFFER - 1);
			if (my_empty_len >= iop_len) {
				while (iop_len > 0) {
					pACB->rqbuffer[pACB->rqbuf_lastindex] = readb(iop_data);
					pACB->rqbuf_lastindex++;
					pACB->rqbuf_lastindex %= ARCMSR_MAX_QBUFFER;
					iop_data++;
					iop_len--;
				}
				writel(ARCMSR_INBOUND_DRIVER_DATA_READ_OK,
					&reg->inbound_doorbell);
			} else
				pACB->acb_flags |= ACB_F_IOPDATA_OVERFLOW;
		}
		if (outbound_doorbell & ARCMSR_OUTBOUND_IOP331_DATA_READ_OK) {
			pACB->acb_flags |= ACB_F_MESSAGE_WQBUFFER_READED;
			if (pACB->wqbuf_firstindex != pACB->wqbuf_lastindex) {
				struct QBUFFER __iomem * pwbuffer =
						(struct QBUFFER __iomem *) &reg->message_wbuffer;
				uint8_t __iomem * iop_data = (uint8_t __iomem *) pwbuffer->data;
				int32_t allxfer_len = 0;

				pACB->acb_flags &= (~ACB_F_MESSAGE_WQBUFFER_READED);
				while ((pACB->wqbuf_firstindex != pACB->wqbuf_lastindex)
					&& (allxfer_len < 124)) {
					writeb(pACB->wqbuffer[pACB->wqbuf_firstindex], iop_data);
					pACB->wqbuf_firstindex++;
					pACB->wqbuf_firstindex %= ARCMSR_MAX_QBUFFER;
					iop_data++;
					allxfer_len++;
				}
				writel(allxfer_len, &pwbuffer->data_len);
				writel(ARCMSR_INBOUND_DRIVER_DATA_WRITE_OK,
					&reg->inbound_doorbell);
			}
			if (pACB->wqbuf_firstindex == pACB->wqbuf_lastindex)
				pACB->acb_flags |= ACB_F_MESSAGE_WQBUFFER_CLEARED;
		}
	}
	if (outbound_intstatus & ARCMSR_MU_OUTBOUND_POSTQUEUE_INT) {
		int id, lun;
		/*
		****************************************************************
		**               areca cdb command done
		****************************************************************
		*/
		while (1) {
			if ((flag_ccb = readl(&reg->outbound_queueport)) == 0xFFFFFFFF)
				break;/*chip FIFO no ccb for completion already*/
			/* check if command done with no error*/
			pCCB = (struct CommandControlBlock *)(pACB->vir2phy_offset +
				(flag_ccb << 5));
			if ((pCCB->pACB != pACB) || (pCCB->startdone != ARCMSR_CCB_START)) {
				if (pCCB->startdone == ARCMSR_CCB_ABORTED) {
					printk(KERN_NOTICE
						"arcmsr%d: ccb='0x%p' isr got aborted command \n"
						, pACB->host->host_no, pCCB);
					continue;
				}
				printk(KERN_NOTICE
					"arcmsr%d: isr get an illegal ccb command done acb='0x%p'"
					"ccb='0x%p' ccbacb='0x%p' startdone = 0x%x"
					" ccboutstandingcount=%d \n"
					, pACB->host->host_no
					, pACB
					, pCCB
					, pCCB->pACB
					, pCCB->startdone
					, atomic_read(&pACB->ccboutstandingcount));
				continue;
			}
			id = pCCB->pcmd->device->id;
			lun = pCCB->pcmd->device->lun;
			if (!(flag_ccb & ARCMSR_CCBREPLY_FLAG_ERROR)) {
				if (pACB->devstate[id][lun] == ARECA_RAID_GONE)
					pACB->devstate[id][lun] = ARECA_RAID_GOOD;
				pCCB->pcmd->result = DID_OK << 16;
				arcmsr_ccb_complete(pCCB, 1);
			} else {
				switch(pCCB->arcmsr_cdb.DeviceStatus) {
				case ARCMSR_DEV_SELECT_TIMEOUT: {
						pACB->devstate[id][lun] = ARECA_RAID_GONE;
						pCCB->pcmd->result = DID_TIME_OUT << 16;
						arcmsr_ccb_complete(pCCB, 1);
					}
					break;
				case ARCMSR_DEV_ABORTED:
				case ARCMSR_DEV_INIT_FAIL: {
						pACB->devstate[id][lun] = ARECA_RAID_GONE;
						pCCB->pcmd->result = DID_BAD_TARGET << 16;
						arcmsr_ccb_complete(pCCB, 1);
					}
					break;
				case ARCMSR_DEV_CHECK_CONDITION: {
						pACB->devstate[id][lun] = ARECA_RAID_GOOD;
						arcmsr_report_sense_info(pCCB);
						arcmsr_ccb_complete(pCCB, 1);
					}
					break;
				default:
 					printk(KERN_NOTICE
						"arcmsr%d: scsi id=%d lun=%d"
						" isr get command error done,"
						"but got unknown DeviceStatus = 0x%x \n"
						, pACB->host->host_no
						, id
						, lun
						, pCCB->arcmsr_cdb.DeviceStatus);
						pACB->devstate[id][lun] = ARECA_RAID_GONE;
						pCCB->pcmd->result = DID_NO_CONNECT << 16;
						arcmsr_ccb_complete(pCCB, 1);
					break;
				}
			}
		}/*drain reply FIFO*/
	}
	if (!(outbound_intstatus & ARCMSR_MU_OUTBOUND_HANDLE_INT))
		return IRQ_NONE;
	return IRQ_HANDLED;
}

static void arcmsr_iop_parking(struct AdapterControlBlock *pACB)
{
	if (pACB) {
		/* stop adapter background rebuild */
		if (pACB->acb_flags & ACB_F_MSG_START_BGRB) {
			pACB->acb_flags &= ~ACB_F_MSG_START_BGRB;
			arcmsr_stop_adapter_bgrb(pACB);
			arcmsr_flush_adapter_cache(pACB);
		}
	}
}

static int arcmsr_iop_message_xfer(struct AdapterControlBlock *pACB, struct scsi_cmnd *cmd)
{
	struct MessageUnit __iomem *reg = pACB->pmu;
	struct CMD_MESSAGE_FIELD *pcmdmessagefld;
	int retvalue = 0, transfer_len = 0;
	char *buffer;
	uint32_t controlcode = (uint32_t ) cmd->cmnd[5] << 24 |
						(uint32_t ) cmd->cmnd[6] << 16 |
						(uint32_t ) cmd->cmnd[7] << 8  |
						(uint32_t ) cmd->cmnd[8];
					/* 4 bytes: Areca io control code */
	if (cmd->use_sg) {
		struct scatterlist *sg = (struct scatterlist *)cmd->request_buffer;

		buffer = kmap_atomic(sg->page, KM_IRQ0) + sg->offset;
		if (cmd->use_sg > 1) {
			retvalue = ARCMSR_MESSAGE_FAIL;
			goto message_out;
		}
		transfer_len += sg->length;
	} else {
		buffer = cmd->request_buffer;
		transfer_len = cmd->request_bufflen;
	}
	if (transfer_len > sizeof(struct CMD_MESSAGE_FIELD)) {
		retvalue = ARCMSR_MESSAGE_FAIL;
		goto message_out;
	}
	pcmdmessagefld = (struct CMD_MESSAGE_FIELD *) buffer;
	switch(controlcode) {
	case ARCMSR_MESSAGE_READ_RQBUFFER: {
			unsigned long *ver_addr;
			dma_addr_t buf_handle;
			uint8_t *pQbuffer, *ptmpQbuffer;
			int32_t allxfer_len = 0;

			ver_addr = pci_alloc_consistent(pACB->pdev, 1032, &buf_handle);
			if (!ver_addr) {
				retvalue = ARCMSR_MESSAGE_FAIL;
				goto message_out;
			}
			ptmpQbuffer = (uint8_t *) ver_addr;
			while ((pACB->rqbuf_firstindex != pACB->rqbuf_lastindex)
				&& (allxfer_len < 1031)) {
				pQbuffer = &pACB->rqbuffer[pACB->rqbuf_firstindex];
				memcpy(ptmpQbuffer, pQbuffer, 1);
				pACB->rqbuf_firstindex++;
				pACB->rqbuf_firstindex %= ARCMSR_MAX_QBUFFER;
				ptmpQbuffer++;
				allxfer_len++;
			}
			if (pACB->acb_flags & ACB_F_IOPDATA_OVERFLOW) {
				struct QBUFFER __iomem * prbuffer = (struct QBUFFER __iomem *)
							&reg->message_rbuffer;
				uint8_t __iomem * iop_data = (uint8_t __iomem *)prbuffer->data;
				int32_t iop_len;

				pACB->acb_flags &= ~ACB_F_IOPDATA_OVERFLOW;
				iop_len = readl(&prbuffer->data_len);
				while (iop_len > 0) {
					pACB->rqbuffer[pACB->rqbuf_lastindex] = readb(iop_data);
					pACB->rqbuf_lastindex++;
					pACB->rqbuf_lastindex %= ARCMSR_MAX_QBUFFER;
					iop_data++;
					iop_len--;
				}
				writel(ARCMSR_INBOUND_DRIVER_DATA_READ_OK,
						&reg->inbound_doorbell);
			}
			memcpy(pcmdmessagefld->messagedatabuffer,
				(uint8_t *)ver_addr, allxfer_len);
			pcmdmessagefld->cmdmessage.Length = allxfer_len;
			pcmdmessagefld->cmdmessage.ReturnCode = ARCMSR_MESSAGE_RETURNCODE_OK;
			pci_free_consistent(pACB->pdev, 1032, ver_addr, buf_handle);
		}
		break;
	case ARCMSR_MESSAGE_WRITE_WQBUFFER: {
			unsigned long *ver_addr;
			dma_addr_t buf_handle;
			int32_t my_empty_len, user_len, wqbuf_firstindex, wqbuf_lastindex;
			uint8_t *pQbuffer, *ptmpuserbuffer;

			ver_addr = pci_alloc_consistent(pACB->pdev, 1032, &buf_handle);
			if (!ver_addr) {
				retvalue = ARCMSR_MESSAGE_FAIL;
				goto message_out;
			}
			ptmpuserbuffer = (uint8_t *)ver_addr;
			user_len = pcmdmessagefld->cmdmessage.Length;
			memcpy(ptmpuserbuffer, pcmdmessagefld->messagedatabuffer, user_len);
			wqbuf_lastindex = pACB->wqbuf_lastindex;
			wqbuf_firstindex = pACB->wqbuf_firstindex;
			if (wqbuf_lastindex != wqbuf_firstindex) {
				struct SENSE_DATA *sensebuffer =
					(struct SENSE_DATA *)cmd->sense_buffer;
				arcmsr_post_Qbuffer(pACB);
				/* has error report sensedata */
				sensebuffer->ErrorCode = 0x70;
				sensebuffer->SenseKey = ILLEGAL_REQUEST;
				sensebuffer->AdditionalSenseLength = 0x0A;
				sensebuffer->AdditionalSenseCode = 0x20;
				sensebuffer->Valid = 1;
				retvalue = ARCMSR_MESSAGE_FAIL;
			} else {
				my_empty_len = (wqbuf_firstindex-wqbuf_lastindex - 1)
						&(ARCMSR_MAX_QBUFFER - 1);
				if (my_empty_len >= user_len) {
					while (user_len > 0) {
						pQbuffer =
						&pACB->wqbuffer[pACB->wqbuf_lastindex];
						memcpy(pQbuffer, ptmpuserbuffer, 1);
						pACB->wqbuf_lastindex++;
						pACB->wqbuf_lastindex %= ARCMSR_MAX_QBUFFER;
						ptmpuserbuffer++;
						user_len--;
					}
					if (pACB->acb_flags & ACB_F_MESSAGE_WQBUFFER_CLEARED) {
						pACB->acb_flags &=
							~ACB_F_MESSAGE_WQBUFFER_CLEARED;
						arcmsr_post_Qbuffer(pACB);
					}
				} else {
					/* has error report sensedata */
					struct SENSE_DATA *sensebuffer =
						(struct SENSE_DATA *)cmd->sense_buffer;
					sensebuffer->ErrorCode = 0x70;
					sensebuffer->SenseKey = ILLEGAL_REQUEST;
					sensebuffer->AdditionalSenseLength = 0x0A;
					sensebuffer->AdditionalSenseCode = 0x20;
					sensebuffer->Valid = 1;
					retvalue = ARCMSR_MESSAGE_FAIL;
				}
			}
			pci_free_consistent(pACB->pdev, 1032, ver_addr, buf_handle);
		}
		break;
	case ARCMSR_MESSAGE_CLEAR_RQBUFFER: {
			uint8_t *pQbuffer = pACB->rqbuffer;

			if (pACB->acb_flags & ACB_F_IOPDATA_OVERFLOW) {
				pACB->acb_flags &= ~ACB_F_IOPDATA_OVERFLOW;
				writel(ARCMSR_INBOUND_DRIVER_DATA_READ_OK,
					&reg->inbound_doorbell);
			}
			pACB->acb_flags |= ACB_F_MESSAGE_RQBUFFER_CLEARED;
			pACB->rqbuf_firstindex = 0;
			pACB->rqbuf_lastindex = 0;
			memset(pQbuffer, 0, ARCMSR_MAX_QBUFFER);
			pcmdmessagefld->cmdmessage.ReturnCode = ARCMSR_MESSAGE_RETURNCODE_OK;
		}
		break;
	case ARCMSR_MESSAGE_CLEAR_WQBUFFER: {
			uint8_t *pQbuffer = pACB->wqbuffer;

			if (pACB->acb_flags & ACB_F_IOPDATA_OVERFLOW) {
				pACB->acb_flags &= ~ACB_F_IOPDATA_OVERFLOW;
				writel(ARCMSR_INBOUND_DRIVER_DATA_READ_OK
						, &reg->inbound_doorbell);
			}
			pACB->acb_flags |=
				(ACB_F_MESSAGE_WQBUFFER_CLEARED | ACB_F_MESSAGE_WQBUFFER_READED);
			pACB->wqbuf_firstindex = 0;
			pACB->wqbuf_lastindex = 0;
			memset(pQbuffer, 0, ARCMSR_MAX_QBUFFER);
			pcmdmessagefld->cmdmessage.ReturnCode = ARCMSR_MESSAGE_RETURNCODE_OK;
		}
		break;
	case ARCMSR_MESSAGE_CLEAR_ALLQBUFFER: {
			uint8_t *pQbuffer;

			if (pACB->acb_flags & ACB_F_IOPDATA_OVERFLOW) {
				pACB->acb_flags &= ~ACB_F_IOPDATA_OVERFLOW;
				writel(ARCMSR_INBOUND_DRIVER_DATA_READ_OK
						, &reg->inbound_doorbell);
			}
			pACB->acb_flags |=
				(ACB_F_MESSAGE_WQBUFFER_CLEARED
				| ACB_F_MESSAGE_RQBUFFER_CLEARED
				| ACB_F_MESSAGE_WQBUFFER_READED);
			pACB->rqbuf_firstindex = 0;
			pACB->rqbuf_lastindex = 0;
			pACB->wqbuf_firstindex = 0;
			pACB->wqbuf_lastindex = 0;
			pQbuffer = pACB->rqbuffer;
			memset(pQbuffer, 0, sizeof (struct QBUFFER));
			pQbuffer = pACB->wqbuffer;
			memset(pQbuffer, 0, sizeof (struct QBUFFER));
			pcmdmessagefld->cmdmessage.ReturnCode = ARCMSR_MESSAGE_RETURNCODE_OK;
		}
		break;
	case ARCMSR_MESSAGE_RETURN_CODE_3F: {
			pcmdmessagefld->cmdmessage.ReturnCode = ARCMSR_MESSAGE_RETURNCODE_3F;
		}
		break;
	case ARCMSR_MESSAGE_SAY_HELLO: {
			int8_t * hello_string = "Hello! I am ARCMSR";

			memcpy(pcmdmessagefld->messagedatabuffer, hello_string
				, (int16_t)strlen(hello_string));
			pcmdmessagefld->cmdmessage.ReturnCode = ARCMSR_MESSAGE_RETURNCODE_OK;
		}
		break;
	case ARCMSR_MESSAGE_SAY_GOODBYE:
		arcmsr_iop_parking(pACB);
		break;
	case ARCMSR_MESSAGE_FLUSH_ADAPTER_CACHE:
		arcmsr_flush_adapter_cache(pACB);
		break;
	default:
		retvalue = ARCMSR_MESSAGE_FAIL;
	}
message_out:
	if (cmd->use_sg) {
		struct scatterlist *sg;

		sg = (struct scatterlist *) cmd->request_buffer;
		kunmap_atomic(buffer - sg->offset, KM_IRQ0);
	}
	return retvalue;
}

static struct CommandControlBlock *arcmsr_get_freeccb(struct AdapterControlBlock *pACB)
{
	struct list_head *head = &pACB->ccb_free_list;
	struct CommandControlBlock *pCCB = NULL;

	if (!list_empty(head)) {
		pCCB = list_entry(head->next, struct CommandControlBlock, list);
		list_del(head->next);
	}
	return pCCB;
}

static int arcmsr_queue_command(struct scsi_cmnd *cmd, void (* done)(struct scsi_cmnd *))
{
	struct Scsi_Host *host = cmd->device->host;
	struct AdapterControlBlock *pACB = (struct AdapterControlBlock *) host->hostdata;
	struct CommandControlBlock *pCCB;
	int target = cmd->device->id;
	int lun = cmd->device->lun;
	uint8_t scsicmd = cmd->cmnd[0];

	cmd->scsi_done = done;
	cmd->host_scribble = NULL;
	cmd->result = 0;
	if (scsicmd == SYNCHRONIZE_CACHE) {
		/*
		** 0x35 avoid synchronizing disk cache cmd after
		** .remove : arcmsr_device_remove (linux bug)
		*/
		if (pACB->devstate[target][lun] == ARECA_RAID_GONE)
			cmd->result = (DID_NO_CONNECT << 16);
		cmd->scsi_done(cmd);
		return 0;
	}
	if (pACB->acb_flags & ACB_F_BUS_RESET) {
		printk(KERN_NOTICE "arcmsr%d: bus reset"
			" and return busy \n"
			, pACB->host->host_no);
		return SCSI_MLQUEUE_HOST_BUSY;
	}
	if(target == 16) {
		/* virtual device for iop message transfer */
		switch(scsicmd) {
		case INQUIRY: {
				unsigned char inqdata[36];
				char *buffer;

				if(lun != 0) {
					cmd->result = (DID_TIME_OUT << 16);
					cmd->scsi_done(cmd);
					return 0;
				}
				inqdata[0] = TYPE_PROCESSOR;
				/* Periph Qualifier & Periph Dev Type */
				inqdata[1] = 0;
				/* rem media bit & Dev Type Modifier */
				inqdata[2] = 0;
				/* ISO,ECMA,& ANSI versions */
				inqdata[4] = 31;
				/* length of additional data */
				strncpy(&inqdata[8], "Areca   ", 8);
				/* Vendor Identification */
				strncpy(&inqdata[16], "RAID controller ", 16);
				/* Product Identification */
				strncpy(&inqdata[32], "R001", 4); /* Product Revision */
				if (cmd->use_sg) {
					struct scatterlist *sg;

					sg = (struct scatterlist *) cmd->request_buffer;
					buffer = kmap_atomic(sg->page, KM_IRQ0) + sg->offset;
				} else {
					buffer = cmd->request_buffer;
				}
				memcpy(buffer, inqdata, sizeof(inqdata));
				if (cmd->use_sg) {
					struct scatterlist *sg;

					sg = (struct scatterlist *) cmd->request_buffer;
					kunmap_atomic(buffer - sg->offset, KM_IRQ0);
				}
				cmd->scsi_done(cmd);
				return 0;
			}
		case WRITE_BUFFER:
		case READ_BUFFER: {
				if (arcmsr_iop_message_xfer(pACB, cmd)) {
    					cmd->result = (DID_ERROR << 16);
				}
				cmd->scsi_done(cmd);
				return 0;
			}
		default:
			cmd->scsi_done(cmd);
			return 0;
		}
	}
	if (pACB->devstate[target][lun] == ARECA_RAID_GONE) {
		uint8_t block_cmd;

		block_cmd = cmd->cmnd[0] & 0x0f;
		if (block_cmd == 0x08 || block_cmd == 0x0a) {
			printk(KERN_NOTICE
				"arcmsr%d: block 'read/write'"
				"command with gone raid volume"
				" Cmd=%2x, TargetId=%d, Lun=%d \n"
				, pACB->host->host_no
				, cmd->cmnd[0]
				, target
				, lun);
			cmd->result = (DID_NO_CONNECT << 16);
			cmd->scsi_done(cmd);
			return 0;
		}
	}
	if (atomic_read(&pACB->ccboutstandingcount) < ARCMSR_MAX_OUTSTANDING_CMD) {
		pCCB = arcmsr_get_freeccb(pACB);
		if (pCCB) {
			arcmsr_build_ccb(pACB, pCCB, cmd);
			arcmsr_post_ccb(pACB, pCCB);
			return 0;
		}
	}
	return SCSI_MLQUEUE_HOST_BUSY;
}

static void arcmsr_get_firmware_spec(struct AdapterControlBlock *pACB)
{
	struct MessageUnit __iomem *reg = pACB->pmu;
	char *acb_firm_model = pACB->firm_model;
	char *acb_firm_version = pACB->firm_version;
	char __iomem *iop_firm_model = (char __iomem *) &reg->message_rwbuffer[15];
	char __iomem *iop_firm_version = (char __iomem *) &reg->message_rwbuffer[17];
	int count;

	writel(ARCMSR_INBOUND_MESG0_GET_CONFIG, &reg->inbound_msgaddr0);
	if (arcmsr_wait_msgint_ready(pACB))
		printk(KERN_NOTICE
			"arcmsr%d: wait "
			"'get adapter firmware miscellaneous data' timeout \n"
			, pACB->host->host_no);
	count = 8;
	while (count) {
		*acb_firm_model = readb(iop_firm_model);
		acb_firm_model++;
		iop_firm_model++;
		count--;
	}
	count = 16;
	while (count) {
		*acb_firm_version = readb(iop_firm_version);
		acb_firm_version++;
		iop_firm_version++;
		count--;
	}
	printk(KERN_INFO
		"ARECA RAID ADAPTER%d: FIRMWARE VERSION %s \n"
		, pACB->host->host_no
		, pACB->firm_version);
	pACB->firm_request_len = readl(&reg->message_rwbuffer[1]);
	pACB->firm_numbers_queue = readl(&reg->message_rwbuffer[2]);
	pACB->firm_sdram_size = readl(&reg->message_rwbuffer[3]);
	pACB->firm_hd_channels = readl(&reg->message_rwbuffer[4]);
}

static void arcmsr_start_adapter_bgrb(struct AdapterControlBlock *pACB)
{
	struct MessageUnit __iomem *reg = pACB->pmu;
	pACB->acb_flags |= ACB_F_MSG_START_BGRB;
	writel(ARCMSR_INBOUND_MESG0_START_BGRB, &reg->inbound_msgaddr0);
	if (arcmsr_wait_msgint_ready(pACB))
		printk(KERN_NOTICE
			"arcmsr%d: wait 'start adapter background rebulid' timeout \n"
			, pACB->host->host_no);
}

static void arcmsr_polling_ccbdone(struct AdapterControlBlock *pACB, struct CommandControlBlock 	*poll_ccb)
{
	struct MessageUnit __iomem *reg = pACB->pmu;
	struct CommandControlBlock *pCCB;
	uint32_t flag_ccb, outbound_intstatus, poll_ccb_done = 0, poll_count = 0;
	int id, lun;

polling_ccb_retry:
	poll_count++;
	outbound_intstatus = readl(&reg->outbound_intstatus)
					& pACB->outbound_int_enable;
	writel(outbound_intstatus, &reg->outbound_intstatus);/*clear interrupt*/
	while (1) {
		if ((flag_ccb = readl(&reg->outbound_queueport)) == 0xFFFFFFFF) {
			if (poll_ccb_done)
				break;
			else {
				msleep(25);
				if (poll_count > 100)
					break;
				goto polling_ccb_retry;
			}
		}
		pCCB = (struct CommandControlBlock *)(pACB->vir2phy_offset + (flag_ccb << 5));
		if ((pCCB->pACB != pACB) || (pCCB->startdone != ARCMSR_CCB_START)) {
			if ((pCCB->startdone == ARCMSR_CCB_ABORTED) && (pCCB == poll_ccb)) {
				printk(KERN_NOTICE
					"arcmsr%d: scsi id=%d lun=%d ccb='0x%p'"
					" poll command abort successfully \n"
					, pACB->host->host_no
					, pCCB->pcmd->device->id
					, pCCB->pcmd->device->lun
					, pCCB);
				pCCB->pcmd->result = DID_ABORT << 16;
				arcmsr_ccb_complete(pCCB, 1);
				poll_ccb_done = 1;
				continue;
			}
			printk(KERN_NOTICE
				"arcmsr%d: polling get an illegal ccb"
				" command done ccb='0x%p'"
				"ccboutstandingcount=%d \n"
				, pACB->host->host_no
				, pCCB
				, atomic_read(&pACB->ccboutstandingcount));
			continue;
		}
		id = pCCB->pcmd->device->id;
		lun = pCCB->pcmd->device->lun;
		if (!(flag_ccb & ARCMSR_CCBREPLY_FLAG_ERROR)) {
			if (pACB->devstate[id][lun] == ARECA_RAID_GONE)
				pACB->devstate[id][lun] = ARECA_RAID_GOOD;
			pCCB->pcmd->result = DID_OK << 16;
			arcmsr_ccb_complete(pCCB, 1);
		} else {
			switch(pCCB->arcmsr_cdb.DeviceStatus) {
			case ARCMSR_DEV_SELECT_TIMEOUT: {
					pACB->devstate[id][lun] = ARECA_RAID_GONE;
					pCCB->pcmd->result = DID_TIME_OUT << 16;
					arcmsr_ccb_complete(pCCB, 1);
				}
				break;
			case ARCMSR_DEV_ABORTED:
			case ARCMSR_DEV_INIT_FAIL: {
					pACB->devstate[id][lun] = ARECA_RAID_GONE;
					pCCB->pcmd->result = DID_BAD_TARGET << 16;
					arcmsr_ccb_complete(pCCB, 1);
				}
				break;
			case ARCMSR_DEV_CHECK_CONDITION: {
					pACB->devstate[id][lun] = ARECA_RAID_GOOD;
					arcmsr_report_sense_info(pCCB);
					arcmsr_ccb_complete(pCCB, 1);
				}
				break;
			default:
				printk(KERN_NOTICE
					"arcmsr%d: scsi id=%d lun=%d"
					" polling and getting command error done"
					"but got unknown DeviceStatus = 0x%x \n"
					, pACB->host->host_no
					, id
					, lun
					, pCCB->arcmsr_cdb.DeviceStatus);
				pACB->devstate[id][lun] = ARECA_RAID_GONE;
				pCCB->pcmd->result = DID_BAD_TARGET << 16;
				arcmsr_ccb_complete(pCCB, 1);
				break;
			}
		}
	}
}

static void arcmsr_iop_init(struct AdapterControlBlock *pACB)
{
	struct MessageUnit __iomem *reg = pACB->pmu;
	uint32_t intmask_org, mask, outbound_doorbell, firmware_state = 0;

	do {
		firmware_state = readl(&reg->outbound_msgaddr1);
	} while (!(firmware_state & ARCMSR_OUTBOUND_MESG1_FIRMWARE_OK));
	intmask_org = readl(&reg->outbound_intmask)
			| ARCMSR_MU_OUTBOUND_MESSAGE0_INTMASKENABLE;
	arcmsr_get_firmware_spec(pACB);
	arcmsr_start_adapter_bgrb(pACB);
	outbound_doorbell = readl(&reg->outbound_doorbell);
	writel(outbound_doorbell, &reg->outbound_doorbell);
	writel(ARCMSR_INBOUND_DRIVER_DATA_READ_OK, &reg->inbound_doorbell);
	mask = ~(ARCMSR_MU_OUTBOUND_POSTQUEUE_INTMASKENABLE
			| ARCMSR_MU_OUTBOUND_DOORBELL_INTMASKENABLE);
	writel(intmask_org & mask, &reg->outbound_intmask);
	pACB->outbound_int_enable = ~(intmask_org & mask) & 0x000000ff;
	pACB->acb_flags |= ACB_F_IOP_INITED;
}

static int arcmsr_bus_reset(struct scsi_cmnd *cmd)
{
	struct AdapterControlBlock *pACB;
	int retry = 0;

	pACB = (struct AdapterControlBlock *) cmd->device->host->hostdata;
	printk(KERN_NOTICE "arcmsr%d: bus reset ..... \n", pACB->host->host_no);
	pACB->num_resets++;
	pACB->acb_flags |= ACB_F_BUS_RESET;
	while (atomic_read(&pACB->ccboutstandingcount) != 0 && retry < 400) {
		arcmsr_interrupt(pACB);
		msleep(25);
		retry++;
	}
	arcmsr_iop_reset(pACB);
	pACB->acb_flags &= ~ACB_F_BUS_RESET;
	return SUCCESS;
}

static int arcmsr_seek_cmd2abort(struct scsi_cmnd *abortcmd)
{
	struct AdapterControlBlock *pACB = (struct AdapterControlBlock *)
			abortcmd->device->host->hostdata;
	struct MessageUnit __iomem *reg = pACB->pmu;
	struct CommandControlBlock *pCCB;
	uint32_t intmask_org, mask;
	int i = 0;

	pACB->num_aborts++;
	/*
	*****************************************************************
	** It is the upper layer do abort command this lock just prior
	** to calling us.
	** First determine if we currently own this command.
	** Start by searching the device queue. If not found
	** at all, and the system wanted us to just abort the
	** command return success.
	*****************************************************************
	*/
	if (atomic_read(&pACB->ccboutstandingcount) != 0) {
		for(i = 0; i < ARCMSR_MAX_FREECCB_NUM; i++) {
			pCCB = pACB->pccb_pool[i];
			if (pCCB->startdone == ARCMSR_CCB_START) {
				if (pCCB->pcmd == abortcmd) {
					pCCB->startdone = ARCMSR_CCB_ABORTED;
					printk(KERN_NOTICE
					"arcmsr%d: scsi id=%d lun=%d"
					" abort ccb '0x%p' outstanding command\n"
					, pACB->host->host_no
					, abortcmd->device->id
					, abortcmd->device->lun
					, pCCB);
					goto abort_outstanding_cmd;
				}
			}
		}
	}
	return SUCCESS;
abort_outstanding_cmd:
	/*wait for 3 sec for all command done*/
	msleep_interruptible(3000);
	/* disable all outbound interrupt */
	intmask_org = readl(&reg->outbound_intmask);
	writel(intmask_org | ARCMSR_MU_OUTBOUND_ALL_INTMASKENABLE
		, &reg->outbound_intmask);
	arcmsr_polling_ccbdone(pACB, pCCB);
	/* enable all outbound interrupt */
	mask = ~(ARCMSR_MU_OUTBOUND_POSTQUEUE_INTMASKENABLE
			| ARCMSR_MU_OUTBOUND_DOORBELL_INTMASKENABLE);
	writel(intmask_org & mask, &reg->outbound_intmask);
	return SUCCESS;
}

static int arcmsr_cmd_abort(struct scsi_cmnd *cmd)
{
	struct AdapterControlBlock *pACB = (struct AdapterControlBlock *)
			 cmd->device->host->hostdata;
	int error;

	printk(KERN_NOTICE
		"arcmsr%d: abort device command of scsi id=%d lun=%d \n"
		, pACB->host->host_no
		, cmd->device->id
		, cmd->device->lun);
	/*
	************************************************
	** the all interrupt service routine is locked
	** we need to handle it as soon as possible and exit
	************************************************
	*/
	error = arcmsr_seek_cmd2abort(cmd);
	if (error != SUCCESS)
		printk(KERN_NOTICE
		"arcmsr%d: abort command failed scsi id=%d lun=%d \n"
		, pACB->host->host_no
		, cmd->device->id
		, cmd->device->lun);
	return error;
}

static const char *arcmsr_info(struct Scsi_Host *host)
{
	static char buf[256];
	struct AdapterControlBlock *pACB;
	uint16_t device_id;

	pACB = (struct AdapterControlBlock *) host->hostdata;
	device_id = pACB->pdev->device;
	switch(device_id) {
	case PCI_DEVICE_ID_ARECA_1110: {
			sprintf(buf,
			"ARECA ARC1110 PCI-X 4 PORTS SATA RAID CONTROLLER "
			"\n        %s"
			, ARCMSR_DRIVER_VERSION);
			break;
		}
	case PCI_DEVICE_ID_ARECA_1120: {
			sprintf(buf,
			"ARECA ARC1120 PCI-X 8 PORTS SATA RAID CONTROLLER "
			"(RAID6-ENGINE Inside)\n        %s"
			, ARCMSR_DRIVER_VERSION);
			break;
		}
	case PCI_DEVICE_ID_ARECA_1130: {
			sprintf(buf,
			"ARECA ARC1130 PCI-X 12 PORTS SATA RAID CONTROLLER "
			"(RAID6-ENGINE Inside)\n        %s"
			, ARCMSR_DRIVER_VERSION);
			break;
		}
	case PCI_DEVICE_ID_ARECA_1160: {
			sprintf(buf,
			"ARECA ARC1160 PCI-X 16 PORTS SATA RAID CONTROLLER "
			"(RAID6-ENGINE Inside)\n        %s"
			, ARCMSR_DRIVER_VERSION);
			break;
		}
	case PCI_DEVICE_ID_ARECA_1170: {
			sprintf(buf,
			"ARECA ARC1170 PCI-X 24 PORTS SATA RAID CONTROLLER "
			"(RAID6-ENGINE Inside)\n        %s"
			, ARCMSR_DRIVER_VERSION);
			break;
		}
	case PCI_DEVICE_ID_ARECA_1210: {
			sprintf(buf,
			"ARECA ARC1210 PCI-EXPRESS 4 PORTS SATA RAID CONTROLLER "
			"\n        %s"
			, ARCMSR_DRIVER_VERSION);
			break;
		}
	case PCI_DEVICE_ID_ARECA_1220: {
			sprintf(buf,
			"ARECA ARC1220 PCI-EXPRESS 8 PORTS SATA RAID CONTROLLER "
			"(RAID6-ENGINE Inside)\n        %s"
			, ARCMSR_DRIVER_VERSION);
			break;
		}
	case PCI_DEVICE_ID_ARECA_1230: {
			sprintf(buf,
			"ARECA ARC1230 PCI-EXPRESS 12 PORTS SATA RAID CONTROLLER "
			"(RAID6-ENGINE Inside)\n        %s"
			, ARCMSR_DRIVER_VERSION);
			break;
		}
	case PCI_DEVICE_ID_ARECA_1260: {
			sprintf(buf,
			"ARECA ARC1260 PCI-EXPRESS 16 PORTS SATA RAID CONTROLLER "
			"(RAID6-ENGINE Inside)\n        %s"
			, ARCMSR_DRIVER_VERSION);
			break;
		}
	case PCI_DEVICE_ID_ARECA_1270: {
			sprintf(buf,
			"ARECA ARC1270 PCI-EXPRESS 24 PORTS SATA RAID CONTROLLER "
			"(RAID6-ENGINE Inside)\n        %s"
			, ARCMSR_DRIVER_VERSION);
			break;
		}
	case PCI_DEVICE_ID_ARECA_1280:
	default: {
			sprintf(buf,
			"ARECA X-TYPE SATA RAID CONTROLLER "
			"(RAID6-ENGINE Inside)\n        %s"
			, ARCMSR_DRIVER_VERSION);
			break;
		}
	}
	return buf;
}

static int arcmsr_initialize(struct AdapterControlBlock *pACB, struct pci_dev *pdev)
{
	struct MessageUnit __iomem *reg;
	uint32_t intmask_org, ccb_phyaddr_hi32;
	dma_addr_t dma_coherent_handle, dma_addr;
	uint8_t pcicmd;
	void *dma_coherent;
	void __iomem *page_remapped;
	int i, j;
	struct CommandControlBlock *pccb_tmp;

	pci_read_config_byte(pdev, PCI_COMMAND, &pcicmd);
	pci_write_config_byte(pdev, PCI_COMMAND,
		pcicmd | PCI_COMMAND_INVALIDATE | PCI_COMMAND_MASTER | PCI_COMMAND_MEMORY);
	page_remapped = ioremap(pci_resource_start(pdev, 0),
		pci_resource_len(pdev, 0));
	if ( !page_remapped ) {
		printk(KERN_NOTICE "arcmsr%d: memory"
			" mapping region fail \n", pACB->host->host_no);
		return -ENXIO;
	}
	pACB->pmu = (struct MessageUnit __iomem *)(page_remapped);
	pACB->acb_flags |=
		(ACB_F_MESSAGE_WQBUFFER_CLEARED
		| ACB_F_MESSAGE_RQBUFFER_CLEARED
		| ACB_F_MESSAGE_WQBUFFER_READED);
	pACB->acb_flags &= ~ACB_F_SCSISTOPADAPTER;
	INIT_LIST_HEAD(&pACB->ccb_free_list);
	dma_coherent = dma_alloc_coherent(&pdev->dev
		, ARCMSR_MAX_FREECCB_NUM * sizeof (struct CommandControlBlock) + 0x20
		, &dma_coherent_handle, GFP_KERNEL);
	if (!dma_coherent) {
		printk(KERN_NOTICE
			"arcmsr%d: dma_alloc_coherent got error \n"
			, pACB->host->host_no);
		return -ENOMEM;
	}
	pACB->dma_coherent = dma_coherent;
	pACB->dma_coherent_handle = dma_coherent_handle;
	memset(dma_coherent, 0, ARCMSR_MAX_FREECCB_NUM * sizeof (struct CommandControlBlock) +
			0x20);
	if (((unsigned long)dma_coherent & 0x1F) != 0) {
		dma_coherent = dma_coherent + (0x20 - ((unsigned long)dma_coherent & 0x1F));
		dma_coherent_handle = dma_coherent_handle
			+ (0x20 - ((unsigned long)dma_coherent_handle & 0x1F));
	}
	dma_addr = dma_coherent_handle;
	pccb_tmp = (struct CommandControlBlock *)dma_coherent;
	for(i = 0; i < ARCMSR_MAX_FREECCB_NUM; i++) {
		pccb_tmp->cdb_shifted_phyaddr = dma_addr >> 5;
		pccb_tmp->pACB = pACB;
		pACB->pccb_pool[i] = pccb_tmp;
		list_add_tail(&pccb_tmp->list, &pACB->ccb_free_list);
		dma_addr = dma_addr + sizeof (struct CommandControlBlock);
		pccb_tmp++;
	}
	pACB->vir2phy_offset = (unsigned long)pccb_tmp - (unsigned long)dma_addr;
	for(i = 0; i < ARCMSR_MAX_TARGETID; i++) {
		for(j = 0; j < ARCMSR_MAX_TARGETLUN; j++)
			pACB->devstate[i][j] = ARECA_RAID_GOOD;
	}
	reg = pACB->pmu;
	/*
	*************************************************************
	** here we need to tell iop 331 our pccb_tmp.HighPart
	** if pccb_tmp.HighPart is not zero
	*************************************************************
	*/
	ccb_phyaddr_hi32 = (uint32_t) ((dma_coherent_handle >> 16) >> 16);
	if (ccb_phyaddr_hi32 != 0) {
		writel(ARCMSR_SIGNATURE_SET_CONFIG, &reg->message_rwbuffer[0]);
		writel(ccb_phyaddr_hi32, &reg->message_rwbuffer[1]);
		writel(ARCMSR_INBOUND_MESG0_SET_CONFIG, &reg->inbound_msgaddr0);
		if (arcmsr_wait_msgint_ready(pACB))
			printk(KERN_NOTICE
				"arcmsr%d: 'set ccb high part physical address' timeout \n"
				, pACB->host->host_no);
	}
	intmask_org = readl(&reg->outbound_intmask);
	writel(intmask_org | ARCMSR_MU_OUTBOUND_ALL_INTMASKENABLE
			, &reg->outbound_intmask);
	return 0;
}

static void arcmsr_pcidev_disattach(struct AdapterControlBlock *pACB)
{
	struct MessageUnit __iomem *reg = pACB->pmu;
	struct pci_dev *pdev;
	struct CommandControlBlock *pCCB;
	struct Scsi_Host *host;
	uint32_t intmask_org;
	int i = 0, poll_count = 0;

	arcmsr_stop_adapter_bgrb(pACB);
	arcmsr_flush_adapter_cache(pACB);
	intmask_org = readl(&reg->outbound_intmask);
	writel(intmask_org | ARCMSR_MU_OUTBOUND_ALL_INTMASKENABLE
			, &reg->outbound_intmask);
	pACB->acb_flags |= ACB_F_SCSISTOPADAPTER;
	pACB->acb_flags &= ~ACB_F_IOP_INITED;
	if (atomic_read(&pACB->ccboutstandingcount) != 0) {
		while (atomic_read(&pACB->ccboutstandingcount) != 0 && (poll_count < 256)) {
			arcmsr_interrupt(pACB);
			msleep(25);
			poll_count++;
		}
		if (atomic_read(&pACB->ccboutstandingcount) != 0) {
			arcmsr_abort_allcmd(pACB);
			for(i = 0; i < ARCMSR_MAX_OUTSTANDING_CMD; i++)
				readl(&reg->outbound_queueport);
			for(i = 0; i < ARCMSR_MAX_FREECCB_NUM; i++) {
				pCCB = pACB->pccb_pool[i];
				if (pCCB->startdone == ARCMSR_CCB_START) {
					pCCB->startdone = ARCMSR_CCB_ABORTED;
					pCCB->pcmd->result = DID_ABORT << 16;
					arcmsr_ccb_complete(pCCB, 1);
				}
			}
		}
	}
	host = pACB->host;
	pdev = pACB->pdev;
	iounmap(pACB->pmu);
	arcmsr_free_ccb_pool(pACB);
	scsi_remove_host(host);
	scsi_host_put(host);
	free_irq(pdev->irq, pACB);
	pci_release_regions(pdev);
	pci_disable_device(pdev);
	pci_set_drvdata(pdev, NULL);
}

