/*
 * Line6 PODxt Pro USB driver - 0.5
 *
 * Copyright (C) 2004, 2005 Markus Grabner (grabner@icg.tu-graz.ac.at)
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation, version 2.
 *
 */

#include <linux/config.h>
#ifdef CONFIG_USB_DEBUG
#define DEBUG 1
#endif
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/module.h>
#include <linux/usb.h>

#include "audio.h"
#include "capture.h"
#include "config.h"
#include "control.h"
#include "driver.h"
#include "midi.h"
#include "playback.h"
#include "usbdefs.h"


#if DO_DEBUG_MESSAGES
#define DEBUG_MESSAGES(x) (x)
#else
#define DEBUG_MESSAGES(x)
#endif


#define DRIVER_AUTHOR  "Markus Grabner <grabner@icg.tu-graz.ac.at>"
#define DRIVER_DESC    "Line6 POD series USB Driver"
#define DRIVER_VERSION "0.5.2"


/* table of devices that work with this driver */
static struct usb_device_id id_table [] = {
	{ USB_DEVICE(LINE6_VENDOR_ID, LINE6_DEVID_BASSPODXT) },
	{ USB_DEVICE(LINE6_VENDOR_ID, LINE6_DEVID_BASSPODXTLIVE) },
	{ USB_DEVICE(LINE6_VENDOR_ID, LINE6_DEVID_BASSPODXTPRO) },
	{ USB_DEVICE(LINE6_VENDOR_ID, LINE6_DEVID_PODXT) },
	{ USB_DEVICE(LINE6_VENDOR_ID, LINE6_DEVID_PODXTLIVE) },
	{ USB_DEVICE(LINE6_VENDOR_ID, LINE6_DEVID_PODXTPRO) },
	{ },
};
MODULE_DEVICE_TABLE (usb, id_table);

static const char *name_table[] = {
	"BassPODxt",
	"BassPODxt Live",
	"BassPODxt Pro",
	"PODxt",
	"PODxt Live",
	"PODxt Pro",
	0
};

/* sysex related stuff: */
static const unsigned char sysex_header[] = { 0x00, 0x01, 0x0c, 0x03 };

static const int SYSEX_DATA_OFS = sizeof(sysex_header) + 2;
static const int SYSEX_EXTRA_SIZE = sizeof(sysex_header) + 3;

static struct usb_podxtpro *podxtpro_devices[PODXTPRO_MAX_DEVICES];


enum {
	PODXTPRO_SYSEX_SAVE      = 0x24,
	PODXTPRO_SYSEX_SYSTEM    = 0x56,
	PODXTPRO_SYSEX_SYSTEMREQ = 0x57,
	/* PODXTPRO_SYSEX_UPDATE    = 0x6c, */  /* software update! */
	PODXTPRO_SYSEX_STORE     = 0x71,
	PODXTPRO_SYSEX_FINISH    = 0x72,
	PODXTPRO_SYSEX_DUMPMEM   = 0x73,
	PODXTPRO_SYSEX_DUMP      = 0x74,
	PODXTPRO_SYSEX_DUMPREQ   = 0x75
	/* PODXTPRO_SYSEX_DUMPMEM2  = 0x76 */   /* dumps entire internal memory of PODxt Pro */
};

enum {
	PODXTPRO_monitor_level  = 0x04,
	PODXTPRO_routing        = 0x05,
	PODXTPRO_tuner_mute     = 0x13,
	PODXTPRO_tuner_freq     = 0x15,
	PODXTPRO_tuner_note     = 0x16,
	PODXTPRO_tuner_pitch    = 0x17,
	PODXTPRO_system_invalid = 0x7fff
};

enum {
	PODXTPRO_DUMP_NONE,
	PODXTPRO_DUMP_CURRENT,
	PODXTPRO_DUMP_MEMORY
};


/*
	Forward declarations.
*/
static void data_received(struct urb *urb, struct pt_regs *regs);
static void podxtpro_dump_request_delayed(struct usb_podxtpro *podxtpro, int seconds);

/*
	Start to listen on endpoint.
*/
static int start_listen(struct usb_podxtpro *podxtpro)
{
	usb_fill_bulk_urb(podxtpro->urb_listen,
										podxtpro->usbdev,
										usb_rcvbulkpipe(podxtpro->usbdev, PODXTPRO_EP_READ_MIDI),
										podxtpro->buffer_listen, PODXTPRO_BUFSIZE_LISTEN,
										data_received,
										podxtpro);

	podxtpro->urb_listen->actual_length = 0;
	return usb_submit_urb(podxtpro->urb_listen, GFP_KERNEL);
}

/*
	Set "dump in progress" flag.
*/
static void dump_started(struct usb_podxtpro *podxtpro, int dest)
{
	podxtpro->dump_in_progress = dest;
}

/*
	Invalidate current channel, i.e., set "dump in progress" flag.
	Reading from the "dump" special file blocks until dump is completed.
*/
void podxtpro_invalidate_current(struct usb_podxtpro *podxtpro)
{
	dump_started(podxtpro, PODXTPRO_DUMP_CURRENT);
}

/*
	Clear "dump in progress" flag and notify waiting processes.
*/
static void dump_finished(struct usb_podxtpro *podxtpro)
{
	podxtpro->dump_in_progress = PODXTPRO_DUMP_NONE;
	wake_up_interruptible(&podxtpro->dump_wait);
}

/*
	Notification of completion of dump request transmission.
*/
static void dump_request_sent(struct urb *urb, struct pt_regs *regs)
{
	usb_free_urb(urb);
}

/*
	Mark all parameters as dirty and notify waiting processes.
*/
static void mark_batch_all_dirty(struct usb_podxtpro *podxtpro)
{
	int i;

	for(i = PODXTPRO_CONTROL_SIZE; i--;)
		set_bit(i, podxtpro->param_dirty);

	wake_up_interruptible(&podxtpro->batch_wait);
}

#if DO_URB_DUMP_SEND || DO_URB_DUMP_RECEIVE
/*
	Write hexdump to syslog.
*/
void podxtpro_write_hexdump(struct usb_podxtpro *podxtpro, char dir, const unsigned char *buffer, int size)
{
	static const int BYTES_PER_LINE = 8;
	char hexdump[100];
	char asc[BYTES_PER_LINE + 1];
	int i, j;

	for(i = 0; i < size; i += BYTES_PER_LINE) {
		int hexdumpsize = sizeof(hexdump);
		char *p = hexdump;
		int n = min(size - i, BYTES_PER_LINE);
		asc[n] = 0;

		for(j = 0; j < BYTES_PER_LINE; ++j) {
			int bytes;

			if(j < n) {
				unsigned char val = buffer[i + j];
				bytes = snprintf(p, hexdumpsize, " %02X", val);
				asc[j] = ((val >= 0x20) && (val < 0x7f)) ? val : '.';
			}
			else
				bytes = snprintf(p, hexdumpsize, "   ");
		
			if(bytes > hexdumpsize)
				break;  /* buffer overflow */

			p += bytes;
			hexdumpsize -= bytes;
		}

		dev_info(podxtpro->msgdev, "%c%04X:%s %s\n", dir, i, hexdump, asc);
	}
}
#endif

#if DO_URB_DUMP_RECEIVE
/*
	Dump URB data to syslog.
*/
static void dump_urb(struct urb *urb)
{
	struct usb_podxtpro *podxtpro = (struct usb_podxtpro *)urb->context;

	if(urb->status < 0)
		return;

	podxtpro_write_hexdump(podxtpro, 'R', (unsigned char *)urb->transfer_buffer, urb->actual_length);
}
#endif

/*
	Send raw message in pieces of 32 bytes.
*/
static int podxtpro_send_raw_message(struct usb_podxtpro *podxtpro, const char *buffer, int size)
{
	int i, done = 0;
	
#if DO_URB_DUMP_SEND
	podxtpro_write_hexdump(podxtpro, 'S', buffer, size);
#endif

	for(i = 0; i < size; i += PODXTPRO_RAW_CHUNK_SIZE)	{
		int partial;
		const char *frag_buf = buffer + i;
		int frag_size = min(PODXTPRO_RAW_CHUNK_SIZE, size - i);
		int retval = usb_bulk_msg(podxtpro->usbdev,
															usb_sndbulkpipe(podxtpro->usbdev, PODXTPRO_EP_WRITE_MIDI),
															(char *)frag_buf, frag_size, &partial, PODXTPRO_TIMEOUT * HZ);

		if(retval) {
			dev_err(podxtpro->msgdev, "usb_bulk_msg failed (%d)\n", retval);
			break;
		}

		done += frag_size;
	}

	return done;
}

/*
	Send sysex message in pieces of 32 bytes.
*/
static int send_sysex_message(struct usb_podxtpro *podxtpro, const char *buffer, int size)
{
	return podxtpro_send_raw_message(podxtpro, buffer, size + SYSEX_EXTRA_SIZE) - SYSEX_EXTRA_SIZE;
}

/*
	Send an asynchronous channel dump request.
*/
static int podxtpro_dump_request_async(struct usb_podxtpro *podxtpro)
{
	struct urb *urb_dumpreq;
	int retval;

	urb_dumpreq = usb_alloc_urb(0, GFP_ATOMIC);

	if(urb_dumpreq == NULL) {
		dev_err(podxtpro->msgdev, "Out of memory\n");
		return -ENOMEM;
	}

	podxtpro_invalidate_current(podxtpro);
	usb_fill_bulk_urb(urb_dumpreq,
										podxtpro->usbdev,
										usb_sndbulkpipe(podxtpro->usbdev, PODXTPRO_EP_WRITE_MIDI),
										podxtpro->buffer_dumpreq, PODXTPRO_BUFSIZE_DUMPREQ,
										dump_request_sent, podxtpro);

#if DO_URB_DUMP_SEND
	podxtpro_write_hexdump(podxtpro, 'S', podxtpro->buffer_dumpreq, PODXTPRO_BUFSIZE_DUMPREQ);
#endif

	urb_dumpreq->actual_length = 0;
	retval = usb_submit_urb(urb_dumpreq, GFP_ATOMIC);

	if(retval < 0) {
		dev_err(podxtpro->msgdev, "usb_submit_urb failed\n");
		usb_free_urb(urb_dumpreq);
		dump_finished(podxtpro);
		return -EINVAL;
	}

	return 0;
}

static void podxtpro_dumpreq_timeout(unsigned long arg)
{
	struct usb_podxtpro *podxtpro = (struct usb_podxtpro *)arg;
	podxtpro_dump_request_async(podxtpro);
	podxtpro_dump_request_delayed(podxtpro, 1);
}

/*
	Send an asynchronous channel dump request after a given interval.
*/
static void podxtpro_dump_request_delayed(struct usb_podxtpro *podxtpro, int seconds)
{
	if(podxtpro->dumpreq_ok)
		return;

	podxtpro->dumpreq_timer.expires = jiffies + seconds * HZ;
	podxtpro->dumpreq_timer.function = podxtpro_dumpreq_timeout;
	podxtpro->dumpreq_timer.data = (unsigned long)podxtpro;
	add_timer(&podxtpro->dumpreq_timer);
}

/*
	Allocate buffer for sysex message and prepare header.
	@param code sysex message code
	@param size number of bytes between code and sysex end
*/
static char *alloc_sysex_buffer(struct usb_podxtpro *podxtpro, int code, int size)
{
	char *buffer = kmalloc(size + SYSEX_EXTRA_SIZE, GFP_KERNEL);

	if(!buffer) {
		dev_err(podxtpro->msgdev, "out of memory\n");
		return 0;
	}

	buffer[0] = PODXTPRO_SYSEX_BEGIN;
	memcpy(buffer + 1, sysex_header, sizeof(sysex_header));
	buffer[1 + sizeof(sysex_header)] = code;
	buffer[size + sizeof(sysex_header) + 2] = PODXTPRO_SYSEX_END;
	return buffer;
}

/*
	Send channel dump data to the PODxt Pro.
*/
static void podxtpro_dump(struct usb_podxtpro *podxtpro, const unsigned char *data)
{
	int size = 1 + sizeof(podxtpro->prog_data);
	char *sysex = alloc_sysex_buffer(podxtpro, PODXTPRO_SYSEX_DUMP, size);
	if(!sysex) return;
	sysex[SYSEX_DATA_OFS] = 5;  /* Don't know what this is good for, but PODxt Pro transmits it, so we also do... */
	memcpy(sysex + SYSEX_DATA_OFS + 1, data, sizeof(podxtpro->prog_data));
	send_sysex_message(podxtpro, sysex, size);
	memcpy(&podxtpro->prog_data, data, sizeof(podxtpro->prog_data));
	mark_batch_all_dirty(podxtpro);
	kfree(sysex);
}

/*
	Store parameter value in driver memory and mark it as dirty.
*/
static void store_parameter(struct usb_podxtpro *podxtpro, int param, int value)
{
	podxtpro->prog_data.control[param] = value;
	set_bit(param, podxtpro->param_dirty);
	podxtpro->dirty = 1;
	wake_up_interruptible(&podxtpro->batch_wait);
}

/*
	Handle SAVE button
*/
static void save_button_pressed(struct usb_podxtpro *podxtpro, int type, int index)
{
	podxtpro->dirty = 0;
	set_bit(PODXTPRO_SAVE_PRESSED, &podxtpro->atomic_flags);
	wake_up_interruptible(&podxtpro->batch_wait);
}

/*
	Process a completely received message.
*/
static void process_message(struct usb_podxtpro *podxtpro)
{
	const unsigned char *buf = podxtpro->buffer_message;

	switch(buf[0])
		{
		case PODXTPRO_PARAM_CHANGE | PODXTPRO_OFFSET_ORIG:
			store_parameter(podxtpro, buf[1], buf[2]);
			/* intentionally no break here! */

		case PODXTPRO_PARAM_CHANGE | PODXTPRO_OFFSET_ECHO:
			if((buf[1] == PODXTPRO_amp_model_setup) || (buf[1] == PODXTPRO_effect_setup))  /* these also affect other settings */
				podxtpro_dump_request_async(podxtpro);

			break;

		case PODXTPRO_PROGRAM_CHANGE | PODXTPRO_OFFSET_ORIG:
		case PODXTPRO_PROGRAM_CHANGE | PODXTPRO_OFFSET_ECHO:
			podxtpro->channel_num = buf[1];
			podxtpro->dirty = 0;
			set_bit(PODXTPRO_CHANNEL_DIRTY, &podxtpro->atomic_flags);
			podxtpro_dump_request_async(podxtpro);
			break;

		case PODXTPRO_SYSEX_BEGIN | PODXTPRO_OFFSET_ORIG:
		case PODXTPRO_SYSEX_BEGIN | PODXTPRO_OFFSET_UNKNOWN:
			if(memcmp(buf + 1, sysex_header, sizeof(sysex_header)) == 0) {
				switch(buf[5]) {
				case PODXTPRO_SYSEX_DUMP:
					if(podxtpro->message_pos == sizeof(podxtpro->prog_data) + 8) {
						switch(podxtpro->dump_in_progress) {
						case PODXTPRO_DUMP_CURRENT:
							memcpy(&podxtpro->prog_data, buf + 7, sizeof(podxtpro->prog_data));
							mark_batch_all_dirty(podxtpro);
							podxtpro->dumpreq_ok = 1;
							break;

						case PODXTPRO_DUMP_MEMORY:
							memcpy(&podxtpro->prog_data_buf, buf + 7, sizeof(podxtpro->prog_data_buf));
							break;

						default:
							DEBUG_MESSAGES(dev_err(podxtpro->msgdev, "unknown dump code %02X\n", podxtpro->dump_in_progress));
						}

						dump_finished(podxtpro);
					}
					else
						DEBUG_MESSAGES(dev_err(podxtpro->msgdev, "wrong size of channel dump message (%d instead of %d)\n",
																	 podxtpro->message_pos, (int)sizeof(podxtpro->prog_data) + 8));
							
					break;

				case PODXTPRO_SYSEX_SYSTEM: {
					short value = ((int)buf[7] << 12) | ((int)buf[8] << 8) | ((int)buf[9] << 4) | (int)buf[10];

#define PROCESS_SYSTEM_PARAM(x) \
					case PODXTPRO_ ## x: \
						podxtpro->x.value = value; \
						wake_up_interruptible(&podxtpro->x.wait); \
						break;

					switch(buf[6]) {
						PROCESS_SYSTEM_PARAM(monitor_level);
						PROCESS_SYSTEM_PARAM(routing);
						PROCESS_SYSTEM_PARAM(tuner_mute);
						PROCESS_SYSTEM_PARAM(tuner_freq);
						PROCESS_SYSTEM_PARAM(tuner_note);
						PROCESS_SYSTEM_PARAM(tuner_pitch);

#undef PROCESS_SYSTEM_PARAM

					default:
						DEBUG_MESSAGES(dev_err(podxtpro->msgdev, "unknown tuner/system response %02X\n", buf[6]));
					}

					break;
				}
				
				case PODXTPRO_SYSEX_FINISH:
					/* do we need to respond to this? */
					break;

				case PODXTPRO_SYSEX_SAVE:
					save_button_pressed(podxtpro, buf[6], buf[7]);
					break;

				case PODXTPRO_SYSEX_STORE:
					DEBUG_MESSAGES(dev_err(podxtpro->msgdev, "message %02X not yet implemented\n", buf[5]));
					break;

				default:
					DEBUG_MESSAGES(dev_err(podxtpro->msgdev, "unknown sysex message %02X\n", buf[5]));
				}
			}
			else
				DEBUG_MESSAGES(dev_err(podxtpro->msgdev, "unknown sysex header\n"));
			
			break;

		default:
			DEBUG_MESSAGES(dev_err(podxtpro->msgdev, "unknown message %02X\n", buf[0]));
		}
}

/*
	Notification of data received from the PODxt Pro.
*/
static void data_received(struct urb *urb, struct pt_regs *regs)
{
	unsigned char *data;
	int length = 0;
	struct usb_podxtpro *podxtpro = (struct usb_podxtpro *)urb->context;

	if(urb->status == -ESHUTDOWN)
		return;

#if DO_URB_DUMP_RECEIVE
	dump_urb(urb);
#endif

	data = urb->transfer_buffer;
	length = urb->actual_length;
	podxtpro_midi_receive(podxtpro, data, length);

	while(length > 0) {
		int bytes = 1, complete = 0;

		if(podxtpro->message_length == 0) {
			if(data[0] < 0x80) {
				DEBUG_MESSAGES(dev_err(podxtpro->msgdev, "message header missed\n"));
				goto finish;
			}
			
			/* determine message length: */
			switch(data[0]) {
			case PODXTPRO_PARAM_CHANGE | PODXTPRO_OFFSET_ORIG:
			case PODXTPRO_PARAM_CHANGE | PODXTPRO_OFFSET_ECHO:
				podxtpro->message_length = 3;
				break;

			case PODXTPRO_PROGRAM_CHANGE | PODXTPRO_OFFSET_ORIG:
			case PODXTPRO_PROGRAM_CHANGE | PODXTPRO_OFFSET_ECHO:
				podxtpro->message_length = 2;
				break;

			case PODXTPRO_SYSEX_BEGIN | PODXTPRO_OFFSET_ORIG:
			case PODXTPRO_SYSEX_BEGIN | PODXTPRO_OFFSET_UNKNOWN:
				podxtpro->message_length = -1;
				break;

			case PODXTPRO_SYSEX_END:
				podxtpro->message_pos = 0;
				podxtpro->message_length = 0;
				goto finish;

			default:
				DEBUG_MESSAGES(dev_err(podxtpro->msgdev, "unknown message %02X\n", (int)data[0]));
				goto finish;
			}
		}

		/* find number of bytes in the URB that contribute to the current message: */
		if(podxtpro->message_length > 0) {
			/* fixed length message: */
			bytes = min(podxtpro->message_length, length);

			if(podxtpro->message_pos + bytes == podxtpro->message_length)
				complete = 1;
		}
		else {
			/* variable length message (sysex): */
			for(bytes = 0; bytes < length; ++bytes)
				if(data[bytes] == PODXTPRO_SYSEX_END) {
					complete = 1;
					++bytes;  /* include sysex end */
					break;
				}
		}

		if(podxtpro->message_pos + bytes > PODXTPRO_MESSAGE_MAXLEN) {
			DEBUG_MESSAGES(dev_err(podxtpro->msgdev, "buffer overflow - message skipped\n"));
			podxtpro->message_pos = 0;
			podxtpro->message_length = 0;
		}
		else {
			memcpy(podxtpro->buffer_message + podxtpro->message_pos, data, length);
			podxtpro->message_pos += bytes;

			if(complete) {
				process_message(podxtpro);
				podxtpro->message_pos = 0;
				podxtpro->message_length = 0;
			}
		}
	
	finish:	
		data += bytes;
		length -= bytes;
	}

	start_listen(podxtpro);
}

/*
	Set channel number (i.e., switch to a different sound).
*/
void podxtpro_set_channel(struct usb_podxtpro *podxtpro, int value)
{
	int retval;
	unsigned char *buffer;
	unsigned int partial;

	buffer = kmalloc(2, GFP_KERNEL);

	if(!buffer) {
		dev_err(podxtpro->msgdev, "out of memory\n");
		return;
	}

	buffer[0] = PODXTPRO_PROGRAM_CHANGE;
	buffer[1] = value;

#if DO_URB_DUMP_SEND
	podxtpro_write_hexdump(podxtpro, 'S', buffer, 2);
#endif

	podxtpro_invalidate_current(podxtpro);
	retval = usb_bulk_msg(podxtpro->usbdev,
												usb_sndbulkpipe(podxtpro->usbdev, PODXTPRO_EP_WRITE_MIDI),
												buffer, 2, &partial, PODXTPRO_TIMEOUT * HZ);

	if(retval) {
		dev_err(podxtpro->msgdev, "usb_bulk_msg failed (%d)\n", retval);
		dump_finished(podxtpro);
	}
	else
		podxtpro->channel_num = value;

	kfree(buffer);
}

/*
	Set PODxt Pro control parameter.
*/
void podxtpro_set_parameter(struct usb_podxtpro *podxtpro, int param, int value)
{
	int retval;
	unsigned char *buffer;
	unsigned int partial;

	buffer = kmalloc(3, GFP_KERNEL);

	if(!buffer) {
		dev_err(podxtpro->msgdev, "out of memory\n");
		return;
	}

	buffer[0] = PODXTPRO_PARAM_CHANGE;
	buffer[1] = param;
	buffer[2] = value;

#if DO_URB_DUMP_SEND
	podxtpro_write_hexdump(podxtpro, 'S', buffer, 3);
#endif

	retval = usb_bulk_msg(podxtpro->usbdev,
												usb_sndbulkpipe(podxtpro->usbdev, PODXTPRO_EP_WRITE_MIDI),
												buffer, 3, &partial, PODXTPRO_TIMEOUT * HZ);

	if(retval)
		dev_err(podxtpro->msgdev, "usb_bulk_msg failed (%d)\n", retval);
	else
		store_parameter(podxtpro, param, value);

	kfree(buffer);

	if((param == PODXTPRO_amp_model_setup) || (param == PODXTPRO_effect_setup))  /* these also affect other settings */
		podxtpro_invalidate_current(podxtpro);
}

static void podxtpro_read_serial_number(struct usb_podxtpro *podxtpro)
{
	struct usb_device *usbdev = podxtpro->usbdev;
	int ret;

	/* for some reason, this message needs to be sent before querying the serial number: */
	ret = usb_control_msg(usbdev, usb_sndctrlpipe(usbdev, 0), 0x67,
												USB_TYPE_VENDOR | USB_RECIP_DEVICE | USB_DIR_OUT,
												0x0421, 0x80d0, 0, 0, PODXTPRO_TIMEOUT * HZ);

	if(ret < 0) {
		dev_err(podxtpro->msgdev, "reading of serial number failed (step 1, error %d)\n", ret);
		return;
	}

  /* just waiting for USB completion is not sufficient :-( */
	msleep(10);

	/* now query the serial number: */
	ret = usb_control_msg(usbdev, usb_rcvctrlpipe(usbdev, 0), 0x67,
												USB_TYPE_VENDOR | USB_RECIP_DEVICE | USB_DIR_IN,
												0x0013, 0x0000, &podxtpro->serial_number, 4, PODXTPRO_TIMEOUT * HZ);

	if(ret < 0) {
		dev_err(podxtpro->msgdev, "reading of serial number failed (step 2, error %d)\n", ret);
		return;
	}
}

#if USE_CHARACTER_DEVICE

/*
	Determine message length.
*/
static ssize_t message_length(unsigned char cmd)
{
	switch(cmd) {
	case PODXTPRO_PARAM_CHANGE: return 3;
	case PODXTPRO_PROGRAM_CHANGE: return 2;
	default: return -1;
	}
}

/*
	Copy data to user space for read call.
*/
static void read_copy(struct usb_podxtpro *podxtpro, unsigned char **bufp, size_t *count)
{
	size_t mlen = podxtpro->chrdev_msglen;
	size_t len = min(*count, mlen - podxtpro->chrdev_readpos);
	int bytes_failed = copy_to_user(*bufp, podxtpro->chrdev_readbuf + podxtpro->chrdev_readpos, len);

	if(bytes_failed > 0) {
		DEBUG_MESSAGES(dev_err(podxtpro->msgdev, "couldn't copy data\n"));
		len -= bytes_failed;
	}

	*bufp += len;
	*count -= len;

	if((podxtpro->chrdev_readpos += len) >= mlen)
		podxtpro->chrdev_readpos = 0;
}

static void check_chrdev_readbuf(struct usb_podxtpro *podxtpro)
{
	if(podxtpro->chrdev_readpos != 0)
		DEBUG_MESSAGES(dev_err(podxtpro->msgdev, "read buffer not empty\n"));
}

/*
	Read from PODxt Pro character device.
*/
static ssize_t podxtpro_read(struct file *filp, char *buff, size_t count, loff_t *offp)
{
	struct usb_podxtpro *podxtpro = (struct usb_podxtpro *)filp->private_data;
	unsigned char *bufp = (unsigned char *)buff;
	size_t retval = 0;
	int i;
	DECLARE_WAITQUEUE(wait, current);
	add_wait_queue(&podxtpro->batch_wait, &wait);
	current->state = TASK_INTERRUPTIBLE;

	if(podxtpro->chrdev_readpos > 0)
		read_copy(podxtpro, &bufp, &count);

	while(count > 0) {
		if(test_and_clear_bit(PODXTPRO_CHANNEL_DIRTY, &podxtpro->atomic_flags)) {
			check_chrdev_readbuf(podxtpro);
			podxtpro->chrdev_readbuf[0] = PODXTPRO_PROGRAM_CHANGE;
			podxtpro->chrdev_readbuf[1] = podxtpro->channel_num;
			podxtpro->chrdev_msglen = 2;
			read_copy(podxtpro, &bufp, &count);
		}

		if(test_and_clear_bit(PODXTPRO_SAVE_PRESSED, &podxtpro->atomic_flags)) {
			static const char save_pressed[] = { 0xf5, 0x00, 0x01, 0x0c, 0x03, 0x24, 0x01, 0x05, 0xf7 };
			check_chrdev_readbuf(podxtpro);
			memcpy(podxtpro->chrdev_readbuf, save_pressed, sizeof(save_pressed));
			podxtpro->chrdev_msglen = sizeof(save_pressed);
			read_copy(podxtpro, &bufp, &count);
		}

		for(i = 0; (i < PODXTPRO_CONTROL_SIZE) && (count > 0); ++i) {
			if(!test_and_clear_bit(i, podxtpro->param_dirty))
				continue;

			check_chrdev_readbuf(podxtpro);
			podxtpro->chrdev_readbuf[0] = PODXTPRO_PARAM_CHANGE;
			podxtpro->chrdev_readbuf[1] = i;
			podxtpro->chrdev_readbuf[2] = podxtpro->prog_data.control[i];
			podxtpro->chrdev_msglen = 3;
			read_copy(podxtpro, &bufp, &count);
		}

		if((char *)bufp > buff)
			break;

		if(filp->f_flags & O_NONBLOCK) {
			retval = -EAGAIN;
			break;
		}

		if(signal_pending(current)) {
			retval = -ERESTARTSYS;
			break;
		}
		
		schedule();
	}

	current->state = TASK_RUNNING;
	remove_wait_queue(&podxtpro->batch_wait, &wait);
	return (retval < 0) ? retval : ((char *)bufp - buff);
}

/*
	Write to PODxt Pro character device.
*/
static ssize_t podxtpro_write(struct file *filp, const char *buff, size_t count, loff_t *offp)
{
	struct usb_podxtpro *podxtpro = (struct usb_podxtpro *)filp->private_data;
	size_t size = podxtpro->chrdev_writepos + count;
	size_t retval;
	unsigned char *bufp0, *bufp;
	int bytes_failed;

	/* check/wait for unfinished dump: */
	retval = podxtpro_wait_dump(podxtpro, filp->f_flags & O_NONBLOCK);

	if(retval < 0)
		return retval;

	/* copy data to kernel space: */
	bufp0 = kmalloc(size, GFP_KERNEL);

	if(!bufp0) {
		dev_err(podxtpro->msgdev, "out of memory\n");
		return -ENOMEM;
	}

	bufp = bufp0;

	if(podxtpro->chrdev_writepos > 0)
		memcpy(bufp, podxtpro->chrdev_writebuf, podxtpro->chrdev_writepos);

	bytes_failed = copy_from_user(bufp + podxtpro->chrdev_writepos, buff, count);

	if(bytes_failed > 0) {
		DEBUG_MESSAGES(dev_err(podxtpro->msgdev, "couldn't copy data\n"));
		count -= bytes_failed;
	}

	podxtpro->chrdev_writepos = 0;
	
	/* process requests: */
	while(size > 0) {
		unsigned char cmd = bufp[0];
		size_t mlen = message_length(cmd);

		if(mlen < 0) {
			DEBUG_MESSAGES(dev_err(podxtpro->msgdev, "unknown command %02X\n", cmd));
			++bufp;
			continue;
		}

		if(mlen > size) {
			memcpy(podxtpro->chrdev_writebuf, bufp, size);
			podxtpro->chrdev_writepos = size;
			size = 0;
			break;
		}

		size -= mlen;

		switch(cmd) {
		case PODXTPRO_PROGRAM_CHANGE:
			podxtpro_set_channel(podxtpro, bufp[1]);

			if(filp->f_flags & O_NONBLOCK)
				goto exit;  /* we don't need to report an error here since we already have written at least one byte */

			podxtpro_wait_dump(podxtpro, 0);
			break;

		case PODXTPRO_PARAM_CHANGE:
			podxtpro_set_parameter(podxtpro, bufp[1], bufp[2]);
			break;
		}

		bufp += mlen;
	}

 exit:
	kfree(bufp0);
	return count - size;
}

/*
	Open PODxt Pro character device.
*/
static int podxtpro_open(struct inode *inode, struct file *filp)
{
	struct usb_podxtpro *podxtpro;

	int num = MINOR(inode->i_rdev);

	if(num > PODXTPRO_MAX_DEVICES)
		return -ENODEV;

	podxtpro = podxtpro_devices[num];

	if(podxtpro == 0)
		return -ENODEV;

	/* check if device is busy: */
	switch(filp->f_flags & O_ACCMODE) {
	case O_RDONLY:
		if(test_and_set_bit(PODXTPRO_BUSY_READ, &podxtpro->atomic_flags))
			return -EBUSY;

		break;

	case O_WRONLY:
		if(test_and_set_bit(PODXTPRO_BUSY_WRITE, &podxtpro->atomic_flags))
			return -EBUSY;

		break;

	case O_RDWR:
		if(test_and_set_bit(PODXTPRO_BUSY_READ, &podxtpro->atomic_flags))
			return -EBUSY;

		if(test_and_set_bit(PODXTPRO_BUSY_WRITE, &podxtpro->atomic_flags)) {
			clear_bit(PODXTPRO_BUSY_READ, &podxtpro->atomic_flags);  /* not nice, can't we do an atomic operation on two bits? */
			return -EBUSY;
		}
	}

	filp->private_data = podxtpro;
	return 0;
}

/*
	Release PODxt Pro character device.
*/
static int podxtpro_release(struct inode *inode, struct file *filp)
{
	struct usb_podxtpro *podxtpro = (struct usb_podxtpro *)filp->private_data;

	/* clear busy flags: */
	switch(filp->f_flags & O_ACCMODE) {
	case O_RDONLY:
		clear_bit(PODXTPRO_BUSY_READ, &podxtpro->atomic_flags);
		break;

	case O_WRONLY:
		clear_bit(PODXTPRO_BUSY_WRITE, &podxtpro->atomic_flags);
		break;

	case O_RDWR:
		clear_bit(PODXTPRO_BUSY_READ, &podxtpro->atomic_flags);
		clear_bit(PODXTPRO_BUSY_WRITE, &podxtpro->atomic_flags);
	}

	return 0;
}

/* character device file operations: */
static struct file_operations podxtpro_fops = {
	read: podxtpro_read,
	write: podxtpro_write,
	open: podxtpro_open,
	release: podxtpro_release,
	owner: THIS_MODULE
};

#endif  /* USE_CHARACTER_DEVICE */

/*
	Resolve value to memory location.
*/
static void resolve(const char *buf, short block0, short block1, unsigned char *location)
{
	int value = simple_strtoul(buf, NULL, 10);
	short block = (value < 0x40) ? block0 : block1;
	value &= 0x3f;
	location[0] = block >> 7;
	location[1] = value | (block & 0x7f);
}

/*
	Send command to store channel/effects setup/amp setup to PODxt Pro.
*/
static ssize_t send_store_command(struct device *dev, const char *buf, size_t count, short block0, short block1)
{
	struct usb_interface *interface = to_usb_interface(dev);
	struct usb_podxtpro *podxtpro = usb_get_intfdata(interface);

	int size = 3 + sizeof(podxtpro->prog_data_buf);
	char *sysex = alloc_sysex_buffer(podxtpro, PODXTPRO_SYSEX_STORE, size);
	if(!sysex) return 0;

	sysex[SYSEX_DATA_OFS] = 5;  /* see podxtpro_dump() */
	resolve(buf, block0, block1, sysex + SYSEX_DATA_OFS + 1);
	memcpy(sysex + SYSEX_DATA_OFS + 3, &podxtpro->prog_data_buf, sizeof(podxtpro->prog_data_buf));

	send_sysex_message(podxtpro, sysex, size);
	kfree(sysex);
	/* needs some delay here on AMD64 platform */
	return count;
}

/*
	Send command to retrieve channel/effects setup/amp setup to PODxt Pro.
*/
static ssize_t send_retrieve_command(struct device *dev, const char *buf, size_t count, short block0, short block1)
{
	struct usb_interface *interface = to_usb_interface(dev);
	struct usb_podxtpro *podxtpro = usb_get_intfdata(interface);

	int size = 4;
	char *sysex = alloc_sysex_buffer(podxtpro, PODXTPRO_SYSEX_DUMPMEM, size);
	if(!sysex) return 0;

	resolve(buf, block0, block1, sysex + SYSEX_DATA_OFS);
	sysex[SYSEX_DATA_OFS + 2] = 0;
	sysex[SYSEX_DATA_OFS + 3] = 0;
	dump_started(podxtpro, PODXTPRO_DUMP_MEMORY);

	if(send_sysex_message(podxtpro, sysex, size) < size)
		dump_finished(podxtpro);

	kfree(sysex);
	/* needs some delay here on AMD64 platform */
	return count;
}

/*
	Generic get name function.
*/
static ssize_t get_name_generic(struct usb_podxtpro *podxtpro, const char *str, char *buf)
{
	int length = 0;
	const char *p1;
	char *p2;
	char *last_non_space = buf;

	int retval = podxtpro_wait_dump(podxtpro, 0);
	if(retval < 0) return retval;

	for(p1 = str, p2 = buf; *p1; ++p1, ++p2) {
		*p2 = *p1;
		if(*p2 != ' ') last_non_space = p2;
		if(++length == PODXTPRO_NAME_LENGTH) break;
	}

	*(last_non_space + 1) = '\n';
	return last_non_space - buf + 2;
}

/*
	"read" request on "channel" special file.
*/
static ssize_t get_channel(struct device *dev, DEVICE_ATTRIBUTE char *buf)
{
	struct usb_interface *interface = to_usb_interface(dev);
	struct usb_podxtpro *podxtpro = usb_get_intfdata(interface);
	return sprintf(buf, "%d\n", podxtpro->channel_num);
}

/*
	"write" request on "channel" special file.
*/
static ssize_t set_channel(struct device *dev, DEVICE_ATTRIBUTE const char *buf, size_t count)
{
	struct usb_interface *interface = to_usb_interface(dev);
	struct usb_podxtpro *podxtpro = usb_get_intfdata(interface);
	int value = simple_strtoul(buf, NULL, 10);
	podxtpro_set_channel(podxtpro, value);
	return count;
}

/*
	"read" request on "name" special file.
*/
static ssize_t get_name(struct device *dev, DEVICE_ATTRIBUTE char *buf)
{
	struct usb_interface *interface = to_usb_interface(dev);
	struct usb_podxtpro *podxtpro = usb_get_intfdata(interface);
	return get_name_generic(podxtpro, podxtpro->prog_data.header + PODXTPRO_NAME_OFFSET, buf);
}

/*
	"read" request on "name" special file.
*/
static ssize_t get_name_buf(struct device *dev, DEVICE_ATTRIBUTE char *buf)
{
	struct usb_interface *interface = to_usb_interface(dev);
	struct usb_podxtpro *podxtpro = usb_get_intfdata(interface);
	return get_name_generic(podxtpro, podxtpro->prog_data_buf.header + PODXTPRO_NAME_OFFSET, buf);
}

/*
	No operation (i.e., unsupported).
*/
ssize_t podxtpro_nop_read(struct device *dev, DEVICE_ATTRIBUTE char *buf)
{
	return 0;
}

/*
	No operation (i.e., unsupported).
*/
ssize_t podxtpro_nop_write(struct device *dev, DEVICE_ATTRIBUTE const char *buf, size_t count)
{
	return count;
}

/*
	"read" request on "dump" special file.
*/
static ssize_t get_dump(struct device *dev, DEVICE_ATTRIBUTE char *buf)
{
	struct usb_interface *interface = to_usb_interface(dev);
	struct usb_podxtpro *podxtpro = usb_get_intfdata(interface);
	int retval = podxtpro_wait_dump(podxtpro, 0);
	if(retval < 0) return retval;
	memcpy(buf, &podxtpro->prog_data, sizeof(podxtpro->prog_data));
	return sizeof(podxtpro->prog_data);
}

/*
	"write" request on "dump" special file.
*/
static ssize_t set_dump(struct device *dev, DEVICE_ATTRIBUTE const char *buf, size_t count)
{
	struct usb_interface *interface = to_usb_interface(dev);
	struct usb_podxtpro *podxtpro = usb_get_intfdata(interface);

	if(count != sizeof(podxtpro->prog_data)) {
		dev_err(podxtpro->msgdev,
						"data block must be exactly %d bytes\n",
						(int)sizeof(podxtpro->prog_data));
		return -EINVAL;
	}

	podxtpro_dump(podxtpro, buf);
	return sizeof(podxtpro->prog_data);
}

/*
	Wait for completion.
*/
int podxtpro_wait_dump(struct usb_podxtpro *podxtpro, int nonblock)
{
	int retval = 0;
	DECLARE_WAITQUEUE(wait, current);
	add_wait_queue(&podxtpro->dump_wait, &wait);
	current->state = TASK_INTERRUPTIBLE;

	while(podxtpro->dump_in_progress) {
		if(nonblock) {
			retval = -EAGAIN;
			break;
		}

		if(signal_pending(current)) {
			retval = -ERESTARTSYS;
			break;
		}
		else
			schedule();
	}

	current->state = TASK_RUNNING;
	remove_wait_queue(&podxtpro->dump_wait, &wait);
	return retval;
}

/*
	Request system parameter.
	@param tuner non-zero, if code refers to a tuner parameter
*/
static ssize_t get_system_param(struct usb_podxtpro *podxtpro, char *buf, int code, struct ValueWait *param, int tuner, int sign)
{
	char *sysex;
	int value;
	static const int size = 1;
	int retval = 0;
	DECLARE_WAITQUEUE(wait, current);

	if(((podxtpro->prog_data.control[PODXTPRO_tuner] & 0x40) == 0) && tuner)
		return -ENODEV;

  /* send value request to tuner: */
	param->value = PODXTPRO_system_invalid;
	sysex = alloc_sysex_buffer(podxtpro, PODXTPRO_SYSEX_SYSTEMREQ, size);
	if(!sysex) return 0;
	sysex[SYSEX_DATA_OFS] = code;
	send_sysex_message(podxtpro, sysex, size);
	kfree(sysex);

	/* wait for tuner to respond: */
	add_wait_queue(&param->wait, &wait);
	current->state = TASK_INTERRUPTIBLE;

	while(param->value == PODXTPRO_system_invalid) {
		if(signal_pending(current)) {
			retval = -ERESTARTSYS;
			break;
		}
		else
			schedule();
	}

	current->state = TASK_RUNNING;
	remove_wait_queue(&param->wait, &wait);

	if(retval < 0)
		return retval;

	value = sign ? (int)(signed short)param->value : (int)(unsigned short)param->value;
	return sprintf(buf, "%d\n", value);
}

/*
	Send system parameter.
	@param tuner non-zero, if code refers to a tuner parameter
*/
static ssize_t set_system_param(struct usb_podxtpro *podxtpro, const char *buf, int count, int code, unsigned short mask, int tuner)
{
	char *sysex;
	static const int size = 5;
	unsigned short value;

	if(((podxtpro->prog_data.control[PODXTPRO_tuner] & 0x40) == 0) && tuner)
		return -EINVAL;

  /* send value to tuner: */
	sysex = alloc_sysex_buffer(podxtpro, PODXTPRO_SYSEX_SYSTEM, size);
	if(!sysex) return 0;
	value = simple_strtoul(buf, NULL, 10) & mask;
	sysex[SYSEX_DATA_OFS] = code;
	sysex[SYSEX_DATA_OFS + 1] = (value >> 12) & 0x0f;
	sysex[SYSEX_DATA_OFS + 2] = (value >>  8) & 0x0f;
	sysex[SYSEX_DATA_OFS + 3] = (value >>  4) & 0x0f;
	sysex[SYSEX_DATA_OFS + 4] = (value      ) & 0x0f;
	send_sysex_message(podxtpro, sysex, size);
	kfree(sysex);
	return count;
}

/*
	"read" request on "dump_buf" special file.
*/
static ssize_t get_dump_buf(struct device *dev, DEVICE_ATTRIBUTE char *buf)
{
	struct usb_interface *interface = to_usb_interface(dev);
	struct usb_podxtpro *podxtpro = usb_get_intfdata(interface);
	int retval = podxtpro_wait_dump(podxtpro, 0);
	if(retval < 0) return retval;
	memcpy(buf, &podxtpro->prog_data_buf, sizeof(podxtpro->prog_data_buf));
	return sizeof(podxtpro->prog_data_buf);
}

/*
	"write" request on "dump_buf" special file.
*/
static ssize_t set_dump_buf(struct device *dev, DEVICE_ATTRIBUTE const char *buf, size_t count)
{
	struct usb_interface *interface = to_usb_interface(dev);
	struct usb_podxtpro *podxtpro = usb_get_intfdata(interface);

	if(count != sizeof(podxtpro->prog_data)) {
		dev_err(podxtpro->msgdev,
						"data block must be exactly %d bytes\n",
						(int)sizeof(podxtpro->prog_data));
		return -EINVAL;
	}
	
	memcpy(&podxtpro->prog_data_buf, buf, sizeof(podxtpro->prog_data));
	return sizeof(podxtpro->prog_data);
}

/*
	"write" request on "finish" special file.
*/
static ssize_t set_finish(struct device *dev, DEVICE_ATTRIBUTE const char *buf, size_t count)
{
	struct usb_interface *interface = to_usb_interface(dev);
	struct usb_podxtpro *podxtpro = usb_get_intfdata(interface);
	int size = 0;
	char *sysex = alloc_sysex_buffer(podxtpro, PODXTPRO_SYSEX_FINISH, size);
	if(!sysex) return 0;
	send_sysex_message(podxtpro, sysex, size);
	kfree(sysex);
	return count;
}

/*
	"write" request on "store_channel" special file.
*/
static ssize_t set_store_channel(struct device *dev, DEVICE_ATTRIBUTE const char *buf, size_t count)
{
	return send_store_command(dev, buf, count, 0x0000, 0x00c0);
}

/*
	"write" request on "store_effects_setup" special file.
*/
static ssize_t set_store_effects_setup(struct device *dev, DEVICE_ATTRIBUTE const char *buf, size_t count)
{
	return send_store_command(dev, buf, count, 0x0080, 0x0080);
}

/*
	"write" request on "store_amp_setup" special file.
*/
static ssize_t set_store_amp_setup(struct device *dev, DEVICE_ATTRIBUTE const char *buf, size_t count)
{
	return send_store_command(dev, buf, count, 0x0040, 0x0100);
}

/*
	"write" request on "retrieve_channel" special file.
*/
static ssize_t set_retrieve_channel(struct device *dev, DEVICE_ATTRIBUTE const char *buf, size_t count)
{
	return send_retrieve_command(dev, buf, count, 0x0000, 0x00c0);
}

/*
	"write" request on "retrieve_effects_setup" special file.
*/
static ssize_t set_retrieve_effects_setup(struct device *dev, DEVICE_ATTRIBUTE const char *buf, size_t count)
{
	return send_retrieve_command(dev, buf, count, 0x0080, 0x0080);
}

/*
	"write" request on "retrieve_amp_setup" special file.
*/
static ssize_t set_retrieve_amp_setup(struct device *dev, DEVICE_ATTRIBUTE const char *buf, size_t count)
{
	return send_retrieve_command(dev, buf, count, 0x0040, 0x0100);
}

/*
	"read" request on "dirty" special file.
*/
static ssize_t get_dirty(struct device *dev, DEVICE_ATTRIBUTE char *buf)
{
	struct usb_interface *interface = to_usb_interface(dev);
	struct usb_podxtpro *podxtpro = usb_get_intfdata(interface);
	buf[0] = podxtpro->dirty ? '1' : '0';
	buf[1] = '\n';
	return 2;
}

/*
	"read" request on "serial_number" special file.
*/
static ssize_t get_serial_number(struct device *dev, DEVICE_ATTRIBUTE char *buf)
{
	struct usb_interface *interface = to_usb_interface(dev);
	struct usb_podxtpro *podxtpro = usb_get_intfdata(interface);
	sprintf(buf, "%d\n", podxtpro->serial_number);
	return strlen(buf);
}

#define GET_SYSTEM_PARAM(code, tuner, sign) \
static ssize_t get_ ## code(struct device *dev, DEVICE_ATTRIBUTE char *buf) \
{ \
	struct usb_interface *interface = to_usb_interface(dev); \
	struct usb_podxtpro *podxtpro = usb_get_intfdata(interface); \
	return get_system_param(podxtpro, buf, PODXTPRO_ ## code, &podxtpro->code, tuner, sign); \
}

#define GET_SET_SYSTEM_PARAM(code, mask, tuner, sign) \
GET_SYSTEM_PARAM(code, tuner, sign) \
static ssize_t set_ ## code(struct device *dev, DEVICE_ATTRIBUTE const char *buf, size_t count) \
{ \
	struct usb_interface *interface = to_usb_interface(dev); \
	struct usb_podxtpro *podxtpro = usb_get_intfdata(interface); \
	return set_system_param(podxtpro, buf, count, PODXTPRO_ ## code, mask, tuner); \
}

GET_SET_SYSTEM_PARAM(monitor_level, 0xffff, 0, 0);
GET_SET_SYSTEM_PARAM(routing, 0x0003, 0, 0);
GET_SET_SYSTEM_PARAM(tuner_mute, 0x0001, 1, 0);
GET_SET_SYSTEM_PARAM(tuner_freq, 0xffff, 1, 0);
GET_SYSTEM_PARAM(tuner_note, 1, 1);
GET_SYSTEM_PARAM(tuner_pitch, 1, 1);

#undef GET_SET_SYSTEM_PARAM
#undef GET_SYSTEM_PARAM

/*
	"write" request on "raw" special file.
*/
#if CREATE_RAW_FILE
static ssize_t set_raw(struct device *dev, DEVICE_ATTRIBUTE const char *buf, size_t count)
{
	struct usb_interface *interface = to_usb_interface(dev);
	struct usb_podxtpro *podxtpro = usb_get_intfdata(interface);
	podxtpro_send_raw_message(podxtpro, buf, count);
	return count;
}
#endif

/* special files: */
static DEVICE_ATTR(channel, S_IWUGO | S_IRUGO, get_channel, set_channel);
static DEVICE_ATTR(name, S_IRUGO, get_name, podxtpro_nop_write);
static DEVICE_ATTR(name_buf, S_IRUGO, get_name_buf, podxtpro_nop_write);
static DEVICE_ATTR(dump, S_IWUGO | S_IRUGO, get_dump, set_dump);
static DEVICE_ATTR(dump_buf, S_IWUGO | S_IRUGO, get_dump_buf, set_dump_buf);
static DEVICE_ATTR(store_channel, S_IWUGO, podxtpro_nop_read, set_store_channel);
static DEVICE_ATTR(store_effects_setup, S_IWUGO, podxtpro_nop_read, set_store_effects_setup);
static DEVICE_ATTR(store_amp_setup, S_IWUGO, podxtpro_nop_read, set_store_amp_setup);
static DEVICE_ATTR(finish, S_IWUGO, podxtpro_nop_read, set_finish);
static DEVICE_ATTR(retrieve_channel, S_IWUGO, podxtpro_nop_read, set_retrieve_channel);
static DEVICE_ATTR(retrieve_effects_setup, S_IWUGO, podxtpro_nop_read, set_retrieve_effects_setup);
static DEVICE_ATTR(retrieve_amp_setup, S_IWUGO, podxtpro_nop_read, set_retrieve_amp_setup);
static DEVICE_ATTR(dirty, S_IRUGO, get_dirty, podxtpro_nop_write);
static DEVICE_ATTR(monitor_level, S_IWUGO | S_IRUGO, get_monitor_level, set_monitor_level);
static DEVICE_ATTR(routing, S_IWUGO | S_IRUGO, get_routing, set_routing);
static DEVICE_ATTR(tuner_mute, S_IWUGO | S_IRUGO, get_tuner_mute, set_tuner_mute);
static DEVICE_ATTR(tuner_freq, S_IWUGO | S_IRUGO, get_tuner_freq, set_tuner_freq);
static DEVICE_ATTR(tuner_note, S_IRUGO, get_tuner_note, podxtpro_nop_write);
static DEVICE_ATTR(tuner_pitch, S_IRUGO, get_tuner_pitch, podxtpro_nop_write);
static DEVICE_ATTR(serial_number, S_IRUGO, get_serial_number, podxtpro_nop_write);
#if CREATE_RAW_FILE
static DEVICE_ATTR(raw, S_IWUGO, podxtpro_nop_read, set_raw);
#endif

/*
	Main destructor.
*/
static void destruct(struct usb_interface *interface)
{
	struct usb_podxtpro *podxtpro = usb_get_intfdata(interface);

	if(podxtpro) {
		podxtpro_cleanup_audio(podxtpro);

#if USE_CHARACTER_DEVICE
		/* unregister character device: */
		if(podxtpro->chrdev >= 0) unregister_chrdev(PODXTPRO_CHRDEV_MAJOR, PODXTPRO_CHRDEV_NAME);
#endif

		/* free buffer memory first: */
		if(podxtpro->buffer_dumpreq) kfree(podxtpro->buffer_dumpreq);
		if(podxtpro->buffer_message) kfree(podxtpro->buffer_message);
		if(podxtpro->buffer_listen) kfree(podxtpro->buffer_listen);

		/* then free URBs: */
		if(podxtpro->urb_listen) usb_free_urb(podxtpro->urb_listen);

		kfree(podxtpro);
		usb_set_intfdata(interface, NULL);  /* make sure it isn't executed twice */
		
		/* remove timer: */
		podxtpro->dumpreq_ok = 1;
		del_timer_sync(&podxtpro->dumpreq_timer);
	}
}

/*
	Probe USB device.
*/
static int podxtpro_probe(struct usb_interface *interface, const struct usb_device_id *id)
{
	int devtype;
	const char *devname;
	struct usb_device *usbdev = interface_to_usbdev(interface);
	struct usb_podxtpro *podxtpro;
	unsigned char *p;
	int devnum;
	int alternate;

	/* we don't handle multiple configurations */
	if(usbdev->descriptor.bNumConfigurations != 1)
		return -ENODEV;

	/* check vendor and product id */
	for(devtype = sizeof(id_table) / sizeof(id_table[0]) - 1; devtype--;)
		if((le16_to_cpu(usbdev->descriptor.idVendor) == id_table[devtype].idVendor) &&
			 (le16_to_cpu(usbdev->descriptor.idProduct) == id_table[devtype].idProduct))
			break;

	if(devtype < 0)
		return -ENODEV;

	/* find free slot in device table: */
	for(devnum = 0; devnum < PODXTPRO_MAX_DEVICES; ++devnum)
		if(podxtpro_devices[devnum] == 0)
			break;

	if(devnum == PODXTPRO_MAX_DEVICES)
		return -ENODEV;

	/* initialize device info: */
	devname = name_table[devtype];
	dev_info(&interface->dev, "Line6 %s found\n", devname);

	if(usb_reset_configuration(usbdev) < 0) {
		dev_err(&interface->dev, "reset_configuration failed\n");
		return -EINVAL;
	}

	alternate =
		(usbdev->descriptor.idProduct == LINE6_DEVID_PODXTLIVE ||
		 usbdev->descriptor.idProduct == LINE6_DEVID_BASSPODXTLIVE)
		? LINE6_ALTERNATE_PODXTLIVE : LINE6_ALTERNATE; 

	if(usb_set_interface(usbdev, LINE6_INTERFACE, alternate) < 0) {
		dev_err(&interface->dev, "set_interface failed\n");
		return -EINVAL;
	}

	/* initialize device data: */
	podxtpro = kmalloc(sizeof(struct usb_podxtpro), GFP_KERNEL);

	if(podxtpro == NULL) {
		dev_err(&interface->dev, "Out of memory\n");
		return -ENOMEM;
	}

	podxtpro_devices[devnum] = podxtpro;
	memset(podxtpro, 0, sizeof(*podxtpro));
	podxtpro->devname = devname;
	podxtpro->usbdev = usb_get_dev(usbdev);
	podxtpro->msgdev = &interface->dev;
	usb_set_intfdata(interface, podxtpro);
	podxtpro->channel_num = 255;
#if USE_CHARACTER_DEVICE
	podxtpro->chrdev = -1;
#endif

	init_waitqueue_head(&podxtpro->dump_wait);
	init_waitqueue_head(&podxtpro->monitor_level.wait);
	init_waitqueue_head(&podxtpro->routing.wait);
	init_waitqueue_head(&podxtpro->tuner_mute.wait);
	init_waitqueue_head(&podxtpro->tuner_freq.wait);
	init_waitqueue_head(&podxtpro->tuner_note.wait);
	init_waitqueue_head(&podxtpro->tuner_pitch.wait);
	init_timer(&podxtpro->dumpreq_timer);

	/* initialize batch data system: */
	init_waitqueue_head(&podxtpro->batch_wait);
	memset(podxtpro->param_dirty, 0xff, sizeof(podxtpro->param_dirty));

	/* initialize USB buffers: */
	podxtpro->buffer_listen = kmalloc(PODXTPRO_BUFSIZE_LISTEN, GFP_KERNEL);

	if(podxtpro->buffer_listen == NULL) {
		dev_err(&interface->dev, "Out of memory\n");
		destruct(interface);
		return -ENOMEM;
	}

	podxtpro->buffer_message = kmalloc(PODXTPRO_MESSAGE_MAXLEN, GFP_KERNEL);

	if(podxtpro->buffer_message == NULL) {
		dev_err(&interface->dev, "Out of memory\n");
		destruct(interface);
		return -ENOMEM;
	}

	podxtpro->urb_listen = usb_alloc_urb(0, GFP_KERNEL);

	if(podxtpro->urb_listen == NULL) {
		dev_err(&interface->dev, "Out of memory\n");
		destruct(interface);
		return -ENOMEM;
	}

	podxtpro->buffer_dumpreq = kmalloc(PODXTPRO_BUFSIZE_DUMPREQ, GFP_KERNEL);

	if(podxtpro->buffer_dumpreq == NULL) {
		dev_err(&interface->dev, "Out of memory\n");
		destruct(interface);
		return -ENOMEM;
	}

	p = podxtpro->buffer_dumpreq;
	*(p++) = PODXTPRO_SYSEX_BEGIN;
	memcpy(p, sysex_header, sizeof(sysex_header)); p += sizeof(sysex_header);
	*(p++) = PODXTPRO_SYSEX_DUMPREQ;
	*(p++) = PODXTPRO_SYSEX_END;

	if(start_listen(podxtpro) < 0) {
		dev_err(&interface->dev, "usb_submit_urb failed\n");
		destruct(interface);
		return -EINVAL;
	}

#if USE_CHARACTER_DEVICE
	if((podxtpro->chrdev = register_chrdev(PODXTPRO_CHRDEV_MAJOR, PODXTPRO_CHRDEV_NAME, &podxtpro_fops)) < 0) {
		destruct(interface);
		return -EINVAL;
	}
#endif

	/* create sysfs entries: */
	podxtpro_create_files(&interface->dev);
	device_create_file(&interface->dev, &dev_attr_channel);
	device_create_file(&interface->dev, &dev_attr_name);
	device_create_file(&interface->dev, &dev_attr_name_buf);
	device_create_file(&interface->dev, &dev_attr_dump);
	device_create_file(&interface->dev, &dev_attr_dump_buf);
	device_create_file(&interface->dev, &dev_attr_store_channel);
	device_create_file(&interface->dev, &dev_attr_store_effects_setup);
	device_create_file(&interface->dev, &dev_attr_store_amp_setup);
	device_create_file(&interface->dev, &dev_attr_finish);
	device_create_file(&interface->dev, &dev_attr_retrieve_channel);
	device_create_file(&interface->dev, &dev_attr_retrieve_effects_setup);
	device_create_file(&interface->dev, &dev_attr_retrieve_amp_setup);
	device_create_file(&interface->dev, &dev_attr_dirty);
	device_create_file(&interface->dev, &dev_attr_monitor_level);
	device_create_file(&interface->dev, &dev_attr_routing);
	device_create_file(&interface->dev, &dev_attr_tuner_mute);
	device_create_file(&interface->dev, &dev_attr_tuner_freq);
	device_create_file(&interface->dev, &dev_attr_tuner_note);
	device_create_file(&interface->dev, &dev_attr_tuner_pitch);
	device_create_file(&interface->dev, &dev_attr_serial_number);
#if CREATE_RAW_FILE
	device_create_file(&interface->dev, &dev_attr_raw);
#endif

	/* initialize audio system: */
	if(podxtpro_init_audio(podxtpro) < 0) {
		destruct(interface);
		return -EINVAL;
	}

	podxtpro_dump_request_delayed(podxtpro, 3);
	podxtpro_read_serial_number(podxtpro);
	dev_info(&interface->dev, "Line6 %s now attached\n", devname);
	return 0;
}

/*
	PODxt Pro disconnected.
*/
static void podxtpro_disconnect(struct usb_interface *interface)
{
	struct usb_podxtpro *podxtpro = usb_get_intfdata(interface);
	int i;

	usb_kill_urb(podxtpro->urb_listen);
	unlink_wait_clear_audio_out_urbs(podxtpro->chip);
	unlink_wait_clear_audio_in_urbs(podxtpro->chip);

	/* remove sysfs entries: */
	podxtpro_remove_files(&interface->dev);
	device_remove_file(&interface->dev, &dev_attr_channel);
	device_remove_file(&interface->dev, &dev_attr_name);
	device_remove_file(&interface->dev, &dev_attr_name_buf);
	device_remove_file(&interface->dev, &dev_attr_dump);
	device_remove_file(&interface->dev, &dev_attr_dump_buf);
	device_remove_file(&interface->dev, &dev_attr_store_channel);
	device_remove_file(&interface->dev, &dev_attr_store_effects_setup);
	device_remove_file(&interface->dev, &dev_attr_store_amp_setup);
	device_remove_file(&interface->dev, &dev_attr_finish);
	device_remove_file(&interface->dev, &dev_attr_retrieve_channel);
	device_remove_file(&interface->dev, &dev_attr_retrieve_effects_setup);
	device_remove_file(&interface->dev, &dev_attr_retrieve_amp_setup);
	device_remove_file(&interface->dev, &dev_attr_dirty);
	device_remove_file(&interface->dev, &dev_attr_monitor_level);
	device_remove_file(&interface->dev, &dev_attr_routing);
	device_remove_file(&interface->dev, &dev_attr_tuner_mute);
	device_remove_file(&interface->dev, &dev_attr_tuner_freq);
	device_remove_file(&interface->dev, &dev_attr_tuner_note);
	device_remove_file(&interface->dev, &dev_attr_tuner_pitch);
	device_remove_file(&interface->dev, &dev_attr_serial_number);
#if CREATE_RAW_FILE
	device_remove_file(&interface->dev, &dev_attr_raw);
#endif

	usb_put_dev(podxtpro->usbdev);
	dev_info(&interface->dev, "Line6 %s now disconnected\n", podxtpro->devname);
	destruct(interface);

	for(i = PODXTPRO_MAX_DEVICES; i--;)
		if(podxtpro_devices[i] == podxtpro)
			podxtpro_devices[i] = 0;
}

static struct usb_driver podxtpro_driver = {
	.owner =	THIS_MODULE,
	.name =		"podxtpro",
	.probe =	podxtpro_probe,
	.disconnect =	podxtpro_disconnect,
	.id_table =	id_table,
};

/*
	Module initialization.
*/
static int __init podxtpro_init(void)
{
	int i, retval;

	for(i = PODXTPRO_MAX_DEVICES; i--;)
		podxtpro_devices[i] = 0;

	retval = usb_register(&podxtpro_driver);

	if(retval)
		err("usb_register failed. Error number %d", retval);

	return retval;
}

/*
	Module cleanup.
*/
static void __exit podxtpro_exit(void)
{
	usb_deregister(&podxtpro_driver);
}

module_init(podxtpro_init);
module_exit(podxtpro_exit);

MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_LICENSE("GPL");
MODULE_VERSION(DRIVER_VERSION);

EXPORT_SYMBOL(podxtpro_set_parameter);
