#ifndef _ASM_GENERIC_COMPAT_SIGINFO_H
#define _ASM_GENERIC_COMPAT_SIGINFO_H

#include <linux/config.h>
#include <linux/compat.h>

#ifndef CONFIG_COMPAT

#define is_compat_task(x) 0

#else

#include <linux/compiler.h>
#include <asm/siginfo.h>

#define is_compat_task(x) (personality((x)->personality) == PER_LINUX32)

/*
 * This is the size (including padding) of the part of the
 * struct siginfo that is before the union.
 */
#ifndef __ARCH_SI_COMPAT_PREAMBLE_SIZE
#define __ARCH_SI_COMPAT_PREAMBLE_SIZE	(3 * sizeof(compat_int_t))
#endif

#define SI_COMPAT_MAX_SIZE	128
#ifndef SI_COMPAT_PAD_SIZE
#define SI_COMPAT_PAD_SIZE \
  ((SI_COMPAT_MAX_SIZE - __ARCH_SI_COMPAT_PREAMBLE_SIZE) / sizeof(compat_int_t))
#endif

/* 32-bit view of si.uid_t */
#ifndef __ARCH_SI_COMPAT_UID_T
#define __ARCH_SI_COMPAT_UID_T compat_uid_t
#endif

/* 32-bit view of si.band_t */
#ifndef __ARCH_SI_COMPAT_BAND_T
#define __ARCH_SI_COMPAT_BAND_T compat_int_t
#endif

#ifndef HAVE_ARCH_COMPAT_SIGINFO_T

/* Compat view of siginfo_t */
typedef struct compat_siginfo {
	compat_int_t si_signo;
	compat_int_t si_errno;
	compat_int_t si_code;

	union {
		compat_int_t _pad[SI_COMPAT_PAD_SIZE];

		/* kill() */
		struct {
			compat_pid_t _pid;	/* sender's pid */
			__ARCH_SI_COMPAT_UID_T _uid;	/* sender's uid */
		} _kill;

		/* POSIX.1b timers */
		struct {
			compat_timer_t _tid;	/* timer id */
			compat_int_t _overrun;		/* overrun count */
			char _pad[sizeof(__ARCH_SI_COMPAT_UID_T) - sizeof(compat_int_t)];
			compat_sigval_t _sigval;	/* same as below */
			compat_int_t _sys_private;       /* not to be passed to user */
		} _timer;

		/* POSIX.1b signals */
		struct {
			compat_pid_t _pid;		/* sender's pid */
			__ARCH_SI_COMPAT_UID_T _uid;	/* sender's uid */
			compat_sigval_t _sigval;
		} _rt;

		/* SIGCHLD */
		struct {
			compat_pid_t _pid;		/* which child */
			__ARCH_SI_COMPAT_UID_T _uid;	/* sender's uid */
			compat_int_t _status;		/* exit code */
			compat_clock_t _utime;
			compat_clock_t _stime;
		} _sigchld;

		/* SIGILL, SIGFPE, SIGSEGV, SIGBUS */
		struct {
			compat_uptr_t _addr; /* faulting insn/memory ref. */
#ifdef __ARCH_SI_COMPAT_TRAPNO
			compat_int_t _trapno;	/* TRAP # which caused the signal */
#endif
		} _sigfault;

		/* SIGPOLL */
		struct {
			__ARCH_SI_COMPAT_BAND_T _band;	/* POLL_IN, POLL_OUT, POLL_MSG */
			compat_int_t _fd;
		} _sigpoll;
	} _sifields;
} compat_siginfo_t;
#endif /* !HAVE_ARCH_COMPAT_SIGINFO_T */

#ifdef __ARCH_SI_COMPAT_TRAPNO
#define si_trapno	_sifields._sigfault._trapno
#endif

/*
 * sigevent definitions
 * 
 * It seems likely that SIGEV_THREAD will have to be handled from 
 * userspace, libpthread transmuting it to SIGEV_SIGNAL, which the
 * thread manager then catches and does the appropriate nonsense.
 * However, everything is written out here so as to not get lost.
 */

#ifndef __ARCH_SIGEV_COMPAT_PREAMBLE_SIZE
#define __ARCH_SIGEV_COMPAT_PREAMBLE_SIZE (sizeof(compat_int_t) * 2 + sizeof(compat_sigval_t))
#endif

#include <linux/string.h>

static inline void compat_copy_siginfo(struct compat_siginfo *to, struct compat_siginfo *from)
{
	if (from->si_code < 0)
		memcpy(to, from, sizeof(*to));
	else
		/* _sigchld is currently the largest know union member */
		memcpy(to, from, __ARCH_SI_COMPAT_PREAMBLE_SIZE + sizeof(from->_sifields._sigchld));
}

extern int compat_copy_sigevent_from_user(struct sigevent *to, compat_sigevent_t __user *from);

#endif /* CONFIG_COMPAT */
#endif /* _ASM_GENERIC_COMPAT_SIGINFO_H */

