/*************************************************************************** 
 * RT2400 SourceForge Project - http://rt2400.sourceforge.net              * 
 *                                                                         * 
 *   This program is free software; you can redistribute it and/or modify  * 
 *   it under the terms of the GNU General Public License as published by  * 
 *   the Free Software Foundation; either version 2 of the License, or     * 
 *   (at your option) any later version.                                   * 
 *                                                                         * 
 *   This program is distributed in the hope that it will be useful,       * 
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        * 
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         * 
 *   GNU General Public License for more details.                          * 
 *                                                                         * 
 *   You should have received a copy of the GNU General Public License     * 
 *   along with this program; if not, write to the                         * 
 *   Free Software Foundation, Inc.,                                       * 
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             * 
 *                                                                         * 
 *   Licensed under the GNU GPL                                            * 
 *   Original code supplied under license from RaLink Inc, 2003.           * 
 ***************************************************************************/ 

 /*************************************************************************** 
 *      Module Name: rtmp_init.c 
 *              
 *      Abstract: Miniport initialisation 
 *              
 *      Revision History: 
 *      Who             When            What 
 *      --------        -----------     ----------------------------- 
 *      PaulL           1st  Aug 02     Created
 *      MarkW           9th  Feb 04     Baseline of code
 *      MarkW           16th Aug 04     Promisc enable code
 *      Thor Johansen   08th Mar 05     Allocate DMA in smaller portions.
 *      Robert Harris   12th Apr 05     Fix Module unloading crash
 ***************************************************************************/ 

 
#include	"rt_config.h"

//
// BBP register initialization set
//
ULONG	BBPRegTable[] = {
	0x00018100, 		// R1 
	0x00018327,			// R3 
	0x00018408,         // R4 
	0x00018a0f,			// R10
	0x00018d08,			// R13
	0x00018f72,			// R15
	0x00019074,			// R16
	0x00019120,			// R17
	0x00019272,			// R18
	0x0001930b,			// R19
	0x00019400,			// R20
	0x00019c11,			// R28
	0x00019d04,         // R29
	0x00019e21,			// R30
	0x00019f00,			// R31
};	
//
// ASIC register initialization sets
//
RTMP_REG_PAIR	MACRegTable[] =	{
	{PSCSR0,    0x00020002},			// 0xc8
	{PSCSR1,    0x00000002},			// 0xcc
	{PSCSR2,    0x00023f20},			// 0xd0
	{PSCSR3,    0x00000002},			// 0xd4
	{TIMECSR,   0x00003f21},     // 1/8 BEACON interval
	{CSR9,      0x00000800},			// 0x24
	{CSR18,     0x001e000a},			// kevin change PIFS,SIFS
	{CSR11,     0x070414a5},			// 0x2C, retry count	
	{CNT3,      0x3f080000},		    // Backoff_CCA_Th, RX_&_TX_CCA_Th

	{TXCSR1,    0x01614562},			// 0x64, ACK as 1Mb time
	{ARCSR0,    0x87888685},			// 0x98, ACK/CTS BBP register ID
	{ARCSR1,    0x0000000f},			// 0x9c
    {ARCSR2,    0x00700400},            // 0x13c, ACK/CTS PLCP at 1 Mbps
    {ARCSR3,    0x00380401},            // 0x140, ACK/CTS PLCP at 2 Mbps
    {ARCSR4,    0x00150402},            // 0x144, ACK/CTS PLCP at 5.5 Mbps
    {ARCSR5,    0x000b8403},            // 0x148, ACK/CTS PLCP at 11 Mbps
	{RXCSR0,    0xffffffff},			// 0x80	
	{RXCSR3,    0x0083a4a0},			// RT2430 BBP R36:RSSI, R32:SIGNAL, R3:TXPOWER
	{PWRCSR0,   0x3f3b3100},			// 0xC4
	{PSCSR2,    0x00023f20},			// 0xd4
	{PSCSR3,    0x00000002},			// 0xd4
	{PWRCSR1,   0x000001ff},			// 0xd8		
	{MACCSR0,   0x00217223},			// 0xE0, Enable Tx dribble mode			
	{MACCSR1,   0x00235518},			// 0xE4, Disable Rx Reset, tx dribble count, 2x30x16 = 960n,
	{MACCSR2, 	0x00000040},			// 0x0134, 64*33ns = 2us
 	{RALINKCSR, 0x9a009a11},			// 0xE8	
	{CSR7,      0xffffffff},			// 0x1C, Clear all pending interrupt source
    {LEDCSR,    0x00001E46},             // default both LEDs off
};
		
#define	NUM_BBP_REG_PARMS	(sizeof(BBPRegTable) / sizeof(ULONG))
#define	NUM_MAC_REG_PARMS	(sizeof(MACRegTable) / sizeof(RTMP_REG_PAIR))


/*
	========================================================================
	
	Routine Description:
		Allocate all DMA releated resources

	Arguments:
		Adapter			Pointer to our adapter

	Return Value:
		None

	Note:

	========================================================================
*/
NDIS_STATUS RTMPAllocDMAMemory(
	IN  PRTMP_ADAPTER   pAd)
{
	INT				index;
	VOID			*ring;			// VA of ring descriptor
	VOID			*ring_data;		// VA of DMA data buffer
	dma_addr_t		ring_dma;		// PA of ring descriptor
	dma_addr_t		ring_data_dma;	// PA of DMA data buffer
	PTXD_STRUC		pTxD;			// Tx type ring descriptor
	PRXD_STRUC		pRxD;			// Rx type ring descriptor

	DBGPRINT(RT_DEBUG_INFO, "--> RTMPAllocDMAMemory\n");
	
	// Allocate Ring descriptors DMA block
	ring = pci_alloc_consistent(pAd->pPci_Dev, DESCRIPTOR_REQUIRED * RING_DESCRIPTOR_SIZE, &ring_dma);
	if (!ring) {
		printk(KERN_ERR DRV_NAME ": Could not allocate DMA ring descriptor memory.\n");
		return -ENOMEM;
	}

	// Zero init ring descriptors
	memset(ring, 0, (DESCRIPTOR_REQUIRED * RING_DESCRIPTOR_SIZE));

	// Start with Tx ring & DMA buffer
	for (index = 0; index < TX_RING_SIZE; index++)
	{
		// Init Tx Ring Size, Va, Pa variables
		pAd->TxRing[index].size = RING_DESCRIPTOR_SIZE;
		pAd->TxRing[index].va_addr = ring;
		pAd->TxRing[index].pa_addr = ring_dma;
		ring     += RING_DESCRIPTOR_SIZE;
		ring_dma += RING_DESCRIPTOR_SIZE;
		
		// Initialize Tx DMA buffer
		pAd->TxRing[index].data_size = TX_BUFFER_SIZE;

		// Allocate buffer
		ring_data = pci_alloc_consistent(pAd->pPci_Dev, TX_BUFFER_SIZE, &ring_data_dma);
                
		// On failure, print error message, free buffers and exit.
		if(!ring_data)
		{
			printk(KERN_ERR DRV_NAME ": Could not allocate Tx DMA buffer.\n");
			RTMPFreeDMAMemory(pAd);
			return -ENOMEM;
		}
		
		// Insert buffer pointers into Tx ring

		pAd->TxRing[index].va_data_addr = ring_data;
		pAd->TxRing[index].pa_data_addr = ring_data_dma;

		// Write TxD buffer address & allocated buffer length
		pTxD = (PTXD_STRUC) pAd->TxRing[index].va_addr;
		pTxD->BufferAddressPa = pAd->TxRing[index].pa_data_addr;
		pTxD->BufferLen       = pAd->TxRing[index].data_size;

		DBGPRINT(RT_DEBUG_INFO, "TxRing[%d] va = 0x%p, pa = 0x%lx, size = 0x%x\n",
			index, pAd->TxRing[index].va_addr, (unsigned long)pAd->TxRing[index].pa_addr, pAd->TxRing[index].size);
		DBGPRINT(RT_DEBUG_INFO, "TxRing[%d] va_data = 0x%p, pa_data = 0x%lx, size = 0x%x\n",
			index, pAd->TxRing[index].va_data_addr, (unsigned long)pAd->TxRing[index].pa_data_addr, pAd->TxRing[index].data_size);
	}

	// Second with Prio ring & DMA buffer
	for (index = 0; index < PRIO_RING_SIZE; index++)
	{
		// Init Prio Ring Size, Va, Pa variables
		pAd->PrioRing[index].size = RING_DESCRIPTOR_SIZE;
		pAd->PrioRing[index].va_addr = ring;
		pAd->PrioRing[index].pa_addr = ring_dma;
		ring     += RING_DESCRIPTOR_SIZE;
		ring_dma += RING_DESCRIPTOR_SIZE;
		
		// Initialize Prio DMA buffer
		pAd->PrioRing[index].data_size = PRIO_BUFFER_SIZE;

		// Allocate buffer
		ring_data = pci_alloc_consistent(pAd->pPci_Dev, PRIO_BUFFER_SIZE, &ring_data_dma);

		// On failure, print error, free buffers and exit
		if(!ring_data)
		{
			printk(KERN_ERR DRV_NAME ": Could not allocate Prio DMA buffer.\n");
			RTMPFreeDMAMemory(pAd);
			return -ENOMEM;
		}
		
		// Insert buffer pointers into Prio ring
		pAd->PrioRing[index].va_data_addr = ring_data;
		pAd->PrioRing[index].pa_data_addr = ring_data_dma;

		// Write TxD buffer address & allocated buffer length for priority ring
		pTxD = (PTXD_STRUC) pAd->PrioRing[index].va_addr;
		pTxD->BufferAddressPa = pAd->PrioRing[index].pa_data_addr;
		pTxD->BufferLen       = pAd->PrioRing[index].data_size;
		
		DBGPRINT(RT_DEBUG_INFO, "PrioRing[%d] va = 0x%p, pa = 0x%lx, size = 0x%x\n",
			index, pAd->PrioRing[index].va_addr, (unsigned long)pAd->PrioRing[index].pa_addr, pAd->PrioRing[index].size);
		DBGPRINT(RT_DEBUG_INFO, "PrioRing[%d] va_data = 0x%p, pa_data = 0x%lx, size = 0x%x\n",
			index, pAd->PrioRing[index].va_data_addr, (unsigned long)pAd->PrioRing[index].pa_data_addr, pAd->PrioRing[index].data_size);
	}
	
	// Atim ring & DMA buffer
	for (index = 0; index < ATIM_RING_SIZE; index++)
	{
		// Init Atim Ring Size, Va, Pa variables
		pAd->AtimRing[index].size = RING_DESCRIPTOR_SIZE;
		pAd->AtimRing[index].va_addr = ring;
		pAd->AtimRing[index].pa_addr = ring_dma;
		ring     += RING_DESCRIPTOR_SIZE;
		ring_dma += RING_DESCRIPTOR_SIZE;
		
		// Initialize Atim DMA buffer
		pAd->AtimRing[index].data_size = ATIM_BUFFER_SIZE;
		// Allocate buffer
		ring_data =
			pci_alloc_consistent(pAd->pPci_Dev,
				ATIM_BUFFER_SIZE, &ring_data_dma);

		// On failure, print error, free buffers and exit
		if(!ring_data)
		{
			printk(KERN_ERR DRV_NAME ": Could not allocate Atim DMA buffer.\n");
			RTMPFreeDMAMemory(pAd);
			return -ENOMEM;
		}

		// Insert buffer pointers into Atim ring
		pAd->AtimRing[index].va_data_addr = ring_data;
		pAd->AtimRing[index].pa_data_addr = ring_data_dma;

		// Write TxD buffer address & allocated buffer length
		pTxD = (PTXD_STRUC) pAd->AtimRing[index].va_addr;
		pTxD->BufferAddressPa = pAd->AtimRing[index].pa_data_addr;
		pTxD->BufferLen       = pAd->AtimRing[index].data_size;

		DBGPRINT(RT_DEBUG_INFO, "AtimRing[%d] va = 0x%p, pa = 0x%lx, size = 0x%x\n",
			index, pAd->AtimRing[index].va_addr, (unsigned long)pAd->AtimRing[index].pa_addr, pAd->AtimRing[index].size);
		DBGPRINT(RT_DEBUG_INFO, "AtimRing[%d] va_data = 0x%p, pa_data = 0x%lx, size = 0x%x\n",
			index, pAd->AtimRing[index].va_data_addr, (unsigned long)pAd->AtimRing[index].pa_data_addr, pAd->AtimRing[index].data_size);
	}
	
	// Rx ring & DMA buffer
	for (index = 0; index < RX_RING_SIZE; index++)
	{
		// Init Rx Ring Size, Va, Pa variables
		pAd->RxRing[index].size = RING_DESCRIPTOR_SIZE;
		pAd->RxRing[index].va_addr = ring;
		pAd->RxRing[index].pa_addr = ring_dma;
		ring     += RING_DESCRIPTOR_SIZE;
		ring_dma += RING_DESCRIPTOR_SIZE;
		
		// Initialize Rx DMA buffer
		pAd->RxRing[index].data_size = RX_BUFFER_SIZE;

		// Allocate buffer
		ring_data =
			pci_alloc_consistent(pAd->pPci_Dev,
				RX_BUFFER_SIZE, &ring_data_dma);

		// On failure, print error, free up buffers and exit
		if(!ring_data)
		{
			printk(KERN_ERR DRV_NAME ": Could not allocate Rx DMA buffer.\n");
			RTMPFreeDMAMemory(pAd);
			return -ENOMEM;
		}

		// Insert buffer pointers into ring
		pAd->RxRing[index].va_data_addr = ring_data;
		pAd->RxRing[index].pa_data_addr = ring_data_dma;

		// Write RxD buffer address & allocated buffer length
		pRxD = (PRXD_STRUC) pAd->RxRing[index].va_addr;
		pRxD->BufferAddressPa = pAd->RxRing[index].pa_data_addr;
		pRxD->BufferLen       = pAd->RxRing[index].data_size;
		// Rx owner bit assign to NIC immediately
		pRxD->Owner = DESC_OWN_NIC;

		DBGPRINT(RT_DEBUG_INFO, "RxRing[%d] va = 0x%p, pa = 0x%lx, size = 0x%x\n",
			index, pAd->RxRing[index].va_addr, (unsigned long)pAd->RxRing[index].pa_addr, pAd->RxRing[index].size);
		DBGPRINT(RT_DEBUG_INFO, "RxRing[%d] va_data = 0x%p, pa_data = 0x%lx, size = 0x%x\n",
			index, pAd->RxRing[index].va_data_addr, (unsigned long)pAd->RxRing[index].pa_data_addr, pAd->RxRing[index].data_size);
	}

	// Now for beacon ring
	// Init Rx Ring Size, Va, Pa variables
	pAd->BeaconRing.size = RING_DESCRIPTOR_SIZE;
	pAd->BeaconRing.va_addr = ring;
	pAd->BeaconRing.pa_addr = ring_dma;
		
	// Initialize Beacon DMA buffer
	pAd->BeaconRing.data_size = BEACON_BUFFER_SIZE;
	// Allocate buffer
	ring_data =
		pci_alloc_consistent(pAd->pPci_Dev,
			BEACON_BUFFER_SIZE, &ring_data_dma);

	// On failure, print error message, free buffers and exit
	if(!ring_data)
	{
		printk(KERN_ERR DRV_NAME ": Could not allocate Beacon DMA buffer.\n");
		RTMPFreeDMAMemory(pAd);
		return -ENOMEM;
	}

	// Insert buffer pointers into ring.
	pAd->BeaconRing.va_data_addr = ring_data;
	pAd->BeaconRing.pa_data_addr = ring_data_dma;
		
	// Write RxD buffer address & allocated buffer length
	pTxD = (PTXD_STRUC) pAd->BeaconRing.va_addr;
	pTxD->BufferAddressPa = pAd->BeaconRing.pa_data_addr;
	pTxD->BufferLen       = pAd->BeaconRing.data_size;

	DBGPRINT(RT_DEBUG_INFO, "BeaconRing va = 0x%p, pa = 0x%lx, size = 0x%x\n",
		pAd->BeaconRing.va_addr, (unsigned long)pAd->BeaconRing.pa_addr, pAd->BeaconRing.size);
	DBGPRINT(RT_DEBUG_INFO, "BeaconRing va_data = 0x%p, pa_data = 0x%lx, size = 0x%x\n",
		pAd->BeaconRing.va_data_addr, (unsigned long)pAd->BeaconRing.pa_data_addr, pAd->BeaconRing.data_size);
	DBGPRINT(RT_DEBUG_INFO, "<-- RTMPAllocDMAMemory\n");
	return 0;
}

/*
	========================================================================
	
	Routine Description:
		Free all DMA memory.

	Arguments:
		Adapter			Pointer to our adapter

	Return Value:
		None

	Note:

	========================================================================
*/
VOID	RTMPFreeDMAMemory(
	IN  PRTMP_ADAPTER   pAd)
{
        int index;

	DBGPRINT(RT_DEBUG_INFO, "--> RTMPFreeDMAMemory\n");
	
	// Free All allocated data DMA blocks first.
	// Free Tx ring
	for(index = 0; pAd->TxRing[index].va_data_addr && index < TX_RING_SIZE; index++)
		pci_free_consistent(pAd->pPci_Dev, TX_BUFFER_SIZE,
			pAd->TxRing[index].va_data_addr, pAd->TxRing[index].pa_data_addr);
		
	// Free Prio ring
	for(index = 0; pAd->PrioRing[index].va_data_addr && index < PRIO_RING_SIZE; index++)
		pci_free_consistent(pAd->pPci_Dev, PRIO_BUFFER_SIZE,
			pAd->PrioRing[index].va_data_addr, pAd->PrioRing[index].pa_data_addr);
	
	// Free Atim ring
	for(index = 0; pAd->AtimRing[index].va_data_addr && index < ATIM_RING_SIZE; index++)
		pci_free_consistent(pAd->pPci_Dev, ATIM_BUFFER_SIZE,
			pAd->AtimRing[index].va_data_addr, pAd->AtimRing[index].pa_data_addr);
	
	// Free Rx ring
	for(index = 0; pAd->RxRing[index].va_data_addr && index < RX_RING_SIZE; index++)
		pci_free_consistent(pAd->pPci_Dev, RX_BUFFER_SIZE,
			pAd->RxRing[index].va_data_addr, pAd->RxRing[index].pa_data_addr);       
	
	// Free Beacon ring
	if(pAd->BeaconRing.va_data_addr != 0)
		pci_free_consistent(pAd->pPci_Dev, BEACON_BUFFER_SIZE,
			pAd->BeaconRing.va_data_addr, pAd->BeaconRing.pa_data_addr);

	// Free ring descriptor afterwards
	// the start address is the same as TxRing first elment
	pci_free_consistent(pAd->pPci_Dev, DESCRIPTOR_REQUIRED * RING_DESCRIPTOR_SIZE,
		pAd->TxRing[0].va_addr, pAd->TxRing[0].pa_addr);

	DBGPRINT(RT_DEBUG_INFO, "<-- RTMPFreeDMAMemory\n");
}

/*
	========================================================================
	
	Routine Description:
		Initialize transmit data structures

	Arguments:
		Adapter						Pointer to our adapter

	Return Value:
		None

	Note:
		Initialize all transmit releated private buffer, include those define
		in RTMP_ADAPTER structure and all private data structures.
		
	========================================================================
*/
VOID	NICInitTransmit(
	IN	PRTMP_ADAPTER	pAd)
{
	DBGPRINT(RT_DEBUG_TRACE, "--> NICInitTransmit\n");

	// Initialize all Transmit releated queues
	skb_queue_head_init(&pAd->SendTxWaitQueue);
	
	// Init Ring index pointer
	pAd->CurRxIndex          = 0;
	pAd->CurTxIndex          = 0;
	pAd->CurAtimIndex        = 0;
	pAd->CurPrioIndex        = 0;
	pAd->NextTxDoneIndex     = 0;
	pAd->NextAtimDoneIndex   = 0;
	pAd->NextPrioDoneIndex   = 0;
	pAd->PushMgmtIndex       = 0;
	pAd->PopMgmtIndex        = 0;
	pAd->MgmtQueueSize       = 0;
	
	pAd->PrivateInfo.TxRingFullCnt       = 0;
	
	DBGPRINT(RT_DEBUG_TRACE, "<-- NICInitTransmit\n");
}

/*
	========================================================================
	
	Routine Description:
		Read additional information from EEPROM, such as MAC address

	Arguments:
		Adapter						Pointer to our adapter

	Return Value:
		NDIS_STATUS_SUCCESS
		NDIS_STATUS_FAILURE

	Note:
	
	========================================================================
*/
NDIS_STATUS	NICReadAdapterInfo(
	IN	PRTMP_ADAPTER		pAd)
{
	CSR3_STRUC		StaMacReg0;
	CSR4_STRUC		StaMacReg1;
	NDIS_STATUS		Status = NDIS_STATUS_SUCCESS;

	// 
	// Read MAC address from CSR3 & CSR4, these CSRs reflects real value
	// stored with EEPROM.
	//
	RTMP_IO_READ32(pAd, CSR3, &StaMacReg0.word);
	RTMP_IO_READ32(pAd, CSR4, &StaMacReg1.word);
	pAd->PermanentAddress[0] = StaMacReg0.field.Byte0;
	pAd->PermanentAddress[1] = StaMacReg0.field.Byte1;
	pAd->PermanentAddress[2] = StaMacReg0.field.Byte2;
	pAd->PermanentAddress[3] = StaMacReg0.field.Byte3;
	pAd->PermanentAddress[4] = StaMacReg1.field.Byte4;
	pAd->PermanentAddress[5] = StaMacReg1.field.Byte5;
	// Set Current address to permanet address.
	// TODO: put current address check within registery read out
	pAd->CurrentAddress[0] = pAd->PermanentAddress[0];
	pAd->CurrentAddress[1] = pAd->PermanentAddress[1];
	pAd->CurrentAddress[2] = pAd->PermanentAddress[2];
	pAd->CurrentAddress[3] = pAd->PermanentAddress[3];
	pAd->CurrentAddress[4] = pAd->PermanentAddress[4];
	pAd->CurrentAddress[5] = pAd->PermanentAddress[5];

	return Status;
}

/*
	========================================================================
	
	Routine Description:
		Read initial parameters from EEPROM
		
	Arguments:
		Adapter						Pointer to our adapter

	Return Value:
		None

	Note:
		
	========================================================================
*/
VOID	NICReadEEPROMParameters(
	IN	PRTMP_ADAPTER	pAd)
{
	ULONG			data;
	USHORT			i, value;
	EEPROM_TX_PWR_STRUC	Power;

	DBGPRINT(RT_DEBUG_TRACE, "--> NICReadEEPROMParameters\n");

	// Init EEPROM Address Number, before access EEPROM; if 93c46, EEPROMAddressNum=6, else if 93c66, EEPROMAddressNum=8
	RTMP_IO_READ32(pAd, CSR21, &data);

	if(data & 0x20)
		pAd->EEPROMAddressNum = 6;		
	else
		pAd->EEPROMAddressNum = 8;

	// Read BBP default value from EEPROM and store to array(EEPROMDefaultValue) in pAdapter
	for(i = 0; i < NUM_EEPROM_BBP_PARMS; i++)
	{
		value = RTMP_EEPROM_READ16(pAd, EEPROM_BBP_BASE_OFFSET + i*2);
		
		pAd->EEPROMDefaultValue[i] = value;
	}

	// Read Tx power value for all 14 channels
	// Value from 1 - 0x7f. Default value is 0x27.
	for (i = 0; i < NUM_EEPROM_TX_PARMS; i++)
	{
		Power.word = RTMP_EEPROM_READ16(pAd, EEPROM_TX_PWR_OFFSET + i*2);
		pAd->PortCfg.ChannelTxPower[i * 2]     = ((Power.field.Byte0 > 0x7f) ? pAd->PortCfg.TxPower : Power.field.Byte0);
		pAd->PortCfg.ChannelTxPower[i * 2 + 1] = ((Power.field.Byte1 > 0x7f) ? pAd->PortCfg.TxPower : Power.field.Byte1);
	}

	DBGPRINT(RT_DEBUG_TRACE, "<-- NICReadEEPROMParameters\n");
}

/*
	========================================================================
	
	Routine Description:
		Set default value from EEPROM
		
	Arguments:
		Adapter						Pointer to our adapter

	Return Value:
		None

	Note:
		
	========================================================================
*/
VOID	NICInitAsicFromEEPROM(
	IN	PRTMP_ADAPTER	pAd)
{
	ULONG					data;
	USHORT					i, value;
	UCHAR					TxValue,RxValue;
	EEPROM_ANTENNA_STRUC	Antenna;

	DBGPRINT(RT_DEBUG_TRACE, "--> NICInitAsicFromEEPROM\n");
	
	for(i = 1; i < NUM_EEPROM_BBP_PARMS; i++)
	{
		value = pAd->EEPROMDefaultValue[i];
		
		if((value != 0xFFFF) && (value != 0))
		{
			data = value | 0x18000;
			RTMP_BBP_IO_WRITE32(pAd, data);
		}
	}

	Antenna.word = pAd->EEPROMDefaultValue[0];

	if((Antenna.word != 0xFFFF) && ((Antenna.field.NumOfAntenna == 1) || (Antenna.field.NumOfAntenna == 2)) 
		&& (Antenna.field.TxDefaultAntenna <= 2) && (Antenna.field.RxDefaultAntenna <= 2))
	{
		pAd->PortCfg.NumberOfAntenna = (UCHAR)Antenna.field.NumOfAntenna;
		pAd->PortCfg.CurrentTxAntenna = (UCHAR)Antenna.field.TxDefaultAntenna;
		pAd->PortCfg.CurrentRxAntenna = (UCHAR)Antenna.field.RxDefaultAntenna;

		RTMP_BBP_IO_READ32_BY_REG_ID(pAd, BBP_Tx_Configure, &TxValue);
		RTMP_BBP_IO_READ32_BY_REG_ID(pAd, BBP_Rx_Configure, &RxValue);

		// Tx antenna select
		if(Antenna.field.TxDefaultAntenna == 0)
		{// Antenna Diversity
			TxValue = (TxValue & 0xFC) | 0x01;
		}
		else if(Antenna.field.TxDefaultAntenna == 1)
		{// Antenna A
			TxValue = (TxValue & 0xFC) | 0x00;
		}
		else if(Antenna.field.TxDefaultAntenna == 2)
		{// Antenna B
			TxValue = (TxValue & 0xFC) | 0x02;
		}

		// Rx antenna select
		if(Antenna.field.RxDefaultAntenna == 0)
		{// Antenna Diversity
			RxValue = (RxValue & 0xF9) | 0x02;
		}
		else if(Antenna.field.RxDefaultAntenna == 1)
		{// Antenna A
			RxValue = (RxValue & 0xF9) | 0x00;
		}
		else if(Antenna.field.RxDefaultAntenna == 2)
		{// Antenna B
			RxValue = (RxValue & 0xF9) | 0x04;
		}
		// Change to match microsoft definition, 0xff: diversity, 0: A, 1: B
		pAd->PortCfg.CurrentTxAntenna--;
		pAd->PortCfg.CurrentRxAntenna--;

		RTMP_BBP_IO_WRITE32_BY_REG_ID(pAd, BBP_Tx_Configure, TxValue);
		RTMP_BBP_IO_WRITE32_BY_REG_ID(pAd, BBP_Rx_Configure, RxValue);

		if (Antenna.field.RfType == RFIC_2420)
			pAd->PortCfg.RfType = RFIC_2420;
		else
			pAd->PortCfg.RfType = RFIC_2421;
		
		if (Antenna.field.LedMode == LED_MODE_TXRX_ACTIVITY)
			pAd->PortCfg.LedMode = LED_MODE_TXRX_ACTIVITY;
		else if (Antenna.field.LedMode == LED_MODE_SINGLE)
		{
			pAd->PortCfg.LedMode = LED_MODE_SINGLE;
			ASIC_LED_ACT_ON(pAd);
		}
		else if (Antenna.field.LedMode == LED_MODE_ASUS)
		{
			pAd->PortCfg.LedMode = LED_MODE_ASUS;
			RTMP_IO_WRITE32(pAd, LEDCSR, 0x0002461E);
		}
		else			
			pAd->PortCfg.LedMode = LED_MODE_DEFAULT;

		pAd->PortCfg.EnableRxAgcVgcTuning = (BOOLEAN)Antenna.field.EnableRxAgcVgcTuning;
	}

	// Read Hardware controlled Radio state enable bit
	if (Antenna.field.HardwareRadioControl == 1)
	{
		pAd->PortCfg.bHardwareRadio = TRUE;

		// Read GPIO pin0 as Hardware controlled radio state
		RTMP_IO_READ32(pAd, GPIOCSR0, &data);
		if ((data & 0x01) == 0)
		{
			pAd->PortCfg.bHwRadio = FALSE;
			pAd->PortCfg.bRadio = FALSE;
			RTMP_IO_WRITE32(pAd, PWRCSR0, 0x00000000);
			RTMP_SET_FLAG(pAd, fRTMP_ADAPTER_RADIO_OFF);
			if (pAd->PortCfg.LedMode == LED_MODE_ASUS)
			{
				// Turn bit 17 for Radio OFF
				RTMP_IO_WRITE32(pAd, LEDCSR, 0x0000461E);
			}
		}
	}
	else
		pAd->PortCfg.bHardwareRadio = FALSE;

	DBGPRINT(RT_DEBUG_TRACE, "RF IC=%d, LED mode=%d, R13 tuning=%d\n", 
	    pAd->PortCfg.RfType, pAd->PortCfg.LedMode, pAd->PortCfg.EnableRxAgcVgcTuning);
	DBGPRINT(RT_DEBUG_TRACE, "<-- NICInitAsicFromEEPROM\n");
}

void NICInitializeAdapter(IN	PRTMP_ADAPTER	pAd)
{
	TXCSR2_STRUC	TxCSR2;
	RXCSR1_STRUC	RxCSR1;
	ULONG			Value;

	DBGPRINT(RT_DEBUG_TRACE, "--> NICInitializeAdapter\n");

	// Init spin locks
	spin_lock_init(&pAd->TxRingLock);
	spin_lock_init(&pAd->PrioRingLock);
	spin_lock_init(&pAd->AtimRingLock);
	spin_lock_init(&pAd->RxRingLock);
	spin_lock_init(&pAd->SendTxWaitQueueLock);
		
	// Write TXCSR2 register
	TxCSR2.field.TxDSize = RING_DESCRIPTOR_SIZE;
	TxCSR2.field.NumTxD  = TX_RING_SIZE;
	TxCSR2.field.NumAtimD  = ATIM_RING_SIZE;
	TxCSR2.field.NumPrioD  = PRIO_RING_SIZE;	
	RTMP_IO_WRITE32(pAd, TXCSR2, TxCSR2.word);

	// Write TXCSR3 register
	Value = pAd->TxRing[0].pa_addr;
	RTMP_IO_WRITE32(pAd, TX_RING_BASE_REG, Value);
	
	// Write TXCSR4 register
	Value = pAd->PrioRing[0].pa_addr;
	RTMP_IO_WRITE32(pAd, PRIO_RING_BASE_REG, Value);
	
	// Write TXCSR5 register
	Value = pAd->AtimRing[0].pa_addr;
	RTMP_IO_WRITE32(pAd, ATIM_RING_BASE_REG, Value);

	// Write TXCSR6 register
	Value = pAd->BeaconRing.pa_addr;
	RTMP_IO_WRITE32(pAd, BEACON_BASE_REG, Value);
	
	// Write RXCSR1 register
	RxCSR1.field.RxDSize = RING_DESCRIPTOR_SIZE;
	RxCSR1.field.NumRxD  = RX_RING_SIZE;
	RTMP_IO_WRITE32(pAd, RXCSR1, RxCSR1.word);
	
	// Write RXCSR2 register
	Value = pAd->RxRing[0].pa_addr;
	RTMP_IO_WRITE32(pAd, RX_RING_BASE_REG, Value);

	// Write CSR1 for host ready
	// Move Host reay to end of ASIC initialization 
	// to ensure no Rx will perform before ASIC init
	// RTMP_IO_WRITE32(pAd, CSR1, 0x4);

	// Initialze ASIC for TX & Rx operation
	NICInitializeAsic(pAd);
	
	DBGPRINT(RT_DEBUG_TRACE, "<-- NICInitializeAdapter\n");
}	

void	NICInitializeAsic(IN	PRTMP_ADAPTER	pAd)
{
	ULONG			Index;
	UCHAR			Value = 0xff;

	DBGPRINT(RT_DEBUG_TRACE, "--> NICInitializeAsic\n");

	// Initialize MAC register to default value
	for (Index = 0; Index < NUM_MAC_REG_PARMS; Index++)
	{
		RTMP_IO_WRITE32(pAd, MACRegTable[Index].Register, MACRegTable[Index].Value);
	}
	
	// Set Host ready before kicking Rx
	RTMP_IO_WRITE32(pAd, CSR1, 0x1); // reset MAC state machine, requested by Kevin 2003-2-11
	RTMP_IO_WRITE32(pAd, CSR1, 0x4);

	// Read BBP register, make sure BBP is up and running before write new data
	Index = 0;
	while (((Value == 0xff) || (Value == 0x00)) && Index < 10)
	{
		RTMP_BBP_IO_READ32_BY_REG_ID(pAd, BBP_Version, &Value);
		mdelay(1);
		Index++;
	}
	if(Index >= 10){
		DBGPRINT(RT_DEBUG_TRACE, "BBP register was never ready!\n");
		return;
	}

	// Initialize BBP register to default value
	for (Index = 0; Index < NUM_BBP_REG_PARMS; Index++)
	{
		RTMP_BBP_IO_WRITE32(pAd, BBPRegTable[Index]);
	}

	// Initialize RF register to default value
	AsicSwitchChannel(pAd, pAd->PortCfg.Channel);
	AsicLockChannel(pAd, pAd->PortCfg.Channel);
	
	// Add radio off control
	if (pAd->PortCfg.bRadio == FALSE)
	{
		RTMP_IO_WRITE32(pAd, PWRCSR0, 0x00000000);
		RTMP_SET_FLAG(pAd, fRTMP_ADAPTER_RADIO_OFF);
	}
	
	// Kick Rx
	if (pAd->bAcceptPromiscuous == TRUE)
        {
                // Register bits with "drop unicast not to me disabled"
                RTMP_IO_WRITE32(pAd, RXCSR0, 0x6e);
        }
        else
        {
                // Standard default register bits.
                RTMP_IO_WRITE32(pAd, RXCSR0, 0x7e);
        }


	// Clear old FCS jitter before init ASIC
	RTMP_IO_READ32(pAd, CNT0, &Index);
	// Clear old Rx FIFO error jitter before init ASIC
	RTMP_IO_READ32(pAd, CNT4, &Index);
	
	DBGPRINT(RT_DEBUG_TRACE, "<-- NICInitializeAsic\n");
}

/*
	========================================================================
	
	Routine Description:
		Reset NIC Asics

	Arguments:
		Adapter						Pointer to our adapter

	Return Value:
		None

	Note:
		Reset NIC to initial state AS IS system boot up time.
		
	========================================================================
*/
VOID	NICIssueReset(
	IN	PRTMP_ADAPTER	pAd)
{
	DBGPRINT(RT_DEBUG_TRACE, "--> NICIssueReset\n");

	// Abort Tx, prevent ASIC from writing to Host memory
	RTMP_IO_WRITE32(pAd, TXCSR0, 0x08);
	
	// Disable Rx, register value supposed will remain after reset
	RTMP_IO_WRITE32(pAd, RXCSR0, 0x1);

	// Issue reset and clear from reset state
	RTMP_IO_WRITE32(pAd, CSR1, 0x01);
	RTMP_IO_WRITE32(pAd, CSR1, 0x00);

	DBGPRINT(RT_DEBUG_TRACE, "<-- NICIssueReset\n");
}

/*
	========================================================================
	
	Routine Description:
		Check ASIC registers and find any reason the system might hang

	Arguments:
		Adapter						Pointer to our adapter

	Return Value:
		None

	Note:
		
		
	========================================================================
*/
BOOLEAN	NICCheckForHang(
	IN	PRTMP_ADAPTER	pAd)
{
	ULONG	Value;

	// Update FCS counters
 	RTMP_IO_READ32(pAd, CNT0, &Value);
	pAd->WlanCounters.FCSErrorCount += Value;
		
	// Update RxNoBuffer counters
 	RTMP_IO_READ32(pAd, CNT4, &Value);
	pAd->Counters.RxNoBuffer += Value;
	
	// Read CCA error count
	RTMP_IO_READ32(pAd, CNT3, &Value);
	pAd->PrivateInfo.CCAErrCnt = (Value & 0x0000ffff);
	DBGPRINT(RT_DEBUG_INFO, "CCA error count -> 0x%x\n", pAd->PrivateInfo.CCAErrCnt);

	// Read PLCP error count
	RTMP_IO_READ32(pAd, CNT1, &Value);
	pAd->PrivateInfo.PhyRxErrCnt = (Value & 0x0000ffff);
	DBGPRINT(RT_DEBUG_INFO, "PHY Rx error count -> 0x%x\n", pAd->PrivateInfo.PhyRxErrCnt);
	
	pAd->PrivateInfo.PhyTxErrCnt = (Value & 0xffff0000) >> 16;
	DBGPRINT(RT_DEBUG_INFO, "PHY Tx error count -> 0x%x\n", pAd->PrivateInfo.PhyTxErrCnt);
	
	return (FALSE);
}

/*
	========================================================================
	
	Routine Description:
		Reset NIC from error

	Arguments:
		Adapter						Pointer to our adapter

	Return Value:
		None

	Note:
		Reset NIC from error state
		
	========================================================================
*/
VOID	NICResetFromError(
	IN	PRTMP_ADAPTER	pAd)
{
	// Reset BBP (according to alex, reset ASIC will force reset BBP
	// Therefore, skip the reset BBP
	// RTMP_IO_WRITE32(pAdapter, CSR1, 0x2);
	// Release BBP reset
	// RTMP_IO_WRITE32(pAdapter, CSR1, 0x0);
		
	RTMP_IO_WRITE32(pAd, CSR1, 0x1);
	// Remove ASIC from reset state
	RTMP_IO_WRITE32(pAd, CSR1, 0x0);

	// Init send data structures and related parameters
	NICInitTransmit(pAd);

	NICInitializeAdapter(pAd);	
	NICInitAsicFromEEPROM(pAd);
	
	// Switch to current channel, since during reset process, the connection should remains on.	
	AsicSwitchChannel(pAd, pAd->PortCfg.Channel);
	AsicLockChannel(pAd, pAd->PortCfg.Channel);
}

/*
	========================================================================
	
	Routine Description:
		Reset NIC Asics

	Arguments:
		Adapter						Pointer to our adapter

	Return Value:
		None

	Note:
		Reset NIC to initial state AS IS system boot up time.
		
	========================================================================
*/
VOID	RTMPRingCleanUp(
	IN	PRTMP_ADAPTER	pAd,
	IN	UCHAR			RingType)
{
	ULONG			Count;
	PTXD_STRUC		pTxD;
	PRXD_STRUC		pRxD;
	PMGMT_STRUC		pMgmt;
	struct sk_buff	*skb;

	Count = 0;
	switch (RingType)
	{
		case TX_RING:
			// We have to clean all descriptos in case some error happened with reset
			do 
			{
				pTxD  = (PTXD_STRUC) pAd->TxRing[pAd->NextTxDoneIndex].va_addr;
						
				pTxD->Owner = DESC_OWN_HOST;
				pTxD->Valid = FALSE;

				pAd->NextTxDoneIndex++;
				Count++;
				if (pAd->NextTxDoneIndex >= TX_RING_SIZE)
				{
					pAd->NextTxDoneIndex = 0;
				}
		
			}	while (Count < TX_RING_SIZE);	// We have to scan all TX ring

			// Check for packet in send tx wait waiting queue
			while (!skb_queue_empty(&pAd->SendTxWaitQueue))
			{
				skb = skb_dequeue(&pAd->SendTxWaitQueue);
				dev_kfree_skb_irq(skb);
			}			
			break;

		case PRIO_RING:
			// We have to clean all descriptos in case some error happened with reset
			do 
			{
				pTxD  = (PTXD_STRUC) pAd->PrioRing[pAd->NextPrioDoneIndex].va_addr;
				
				// We just re-claim these ring spaces.		
				pTxD->Owner = DESC_OWN_HOST;
				pTxD->Valid = FALSE;

				pAd->NextPrioDoneIndex++;
				Count++;
				if (pAd->NextPrioDoneIndex >= PRIO_RING_SIZE)
				{
					pAd->NextPrioDoneIndex = 0;
				}
		
			}	while (Count < PRIO_RING_SIZE);		// We have to scan all Priority Ring

			// Clear managemt buffer ring
			while ((pAd->PushMgmtIndex != pAd->PopMgmtIndex) || (pAd->MgmtQueueSize != 0))
			{
				pMgmt = (PMGMT_STRUC) &pAd->MgmtRing[pAd->PopMgmtIndex];
				if (pMgmt->Valid == TRUE)
				{
					if (pMgmt->pBuffer)		// Prevent null pointer
					{
						MlmeConfirm((PVOID)pAd, pMgmt->pBuffer, 0);
					}
					pMgmt->Valid = FALSE;
					pMgmt->pBuffer = NULL;
					pAd->PopMgmtIndex++;
					pAd->MgmtQueueSize--;
					if (pAd->PopMgmtIndex >= MGMT_RING_SIZE)
					{
						pAd->PopMgmtIndex = 0;
					}
				}
			}
			pAd->RalinkCounters.MgmtRingFullCount = 0;
			break;
			
		case RX_RING:
			// We have to clean all descriptos in case some error happened with reset
			do 
			{
				pRxD  = (PRXD_STRUC) pAd->RxRing[pAd->CurRxIndex].va_addr;
				
				// Re-initial Rx ring cell to owned by NIC.
				pRxD->Owner = DESC_OWN_NIC;

				pAd->CurRxIndex++;
				Count++;
				if (pAd->CurRxIndex >= RX_RING_SIZE)
				{
					pAd->CurRxIndex = 0;
				}
		
			}	while (Count < RX_RING_SIZE);		// We have to scan all Rx Ring
			break;
			
		default:
			break;
			
	}
}

/*
	========================================================================
	
	Routine Description:
		In kernel mode read parameters from file

	Arguments:
		src						the location of the file.
		dest						put the parameters to the destination.
		Length					size to read.

	Return Value:
		None

	Note:
		
	========================================================================
*/

VOID RTMPReadParametersFromFile(
	IN	PRTMP_ADAPTER pAd)
{
	RTMP_PROFILE_SETTING	profile;
	PUCHAR					src;
	struct file				*srcf;
	INT 					retval, orgfsuid, orgfsgid;
	UINT					Length;
   	mm_segment_t			orgfs;
   	
   	src = PROFILE_PATH;
   	Length = sizeof(RTMP_PROFILE_SETTING);
   	
	// Save uid and gid used for filesystem access.
	// Set user and group to 0 (root)	
	orgfsuid=current->fsuid;
	orgfsgid=current->fsgid;
	current->fsuid=current->fsgid=0;
    orgfs=get_fs();
    set_fs(KERNEL_DS);

	if (src && *src) 
	{
		srcf = filp_open(src, O_RDONLY, 0);
		if (IS_ERR(srcf)) 
		{
			DBGPRINT(RT_DEBUG_TRACE, "--> Error %ld opening %s\n", -PTR_ERR(srcf),src);    
		}
		else 
		{
			/* The object must have a read method */
			if (srcf->f_op && srcf->f_op->read) 
			{
				retval=srcf->f_op->read(srcf, (PUCHAR)&profile, Length, &srcf->f_pos);
				if (retval < 0)
				{
					DBGPRINT(RT_DEBUG_TRACE, "--> Read %s error %d\n", src, -retval);
				}
				else
				{
					// set file parameter to portcfg
					DBGPRINT(RT_DEBUG_TRACE, "Ssid=%s SsidLen=%d\n", profile.SSID, profile.SsidLen);
					pAd->PortCfg.SsidLen = profile.SsidLen;
					memcpy(pAd->PortCfg.Ssid, profile.SSID, pAd->PortCfg.SsidLen);
					pAd->PortCfg.Channel = profile.Channel;
					pAd->PortCfg.AuthAlgorithm= profile.Authentication;
					pAd->PortCfg.PrivacyInvoked = profile.Encryption;
					pAd->PortCfg.BssType= profile.NetworkType;
					pAd->PortCfg.DefaultKeyId= profile.KeyIndex;
					pAd->PortCfg.SharedKey[0].KeyLen = profile.KeyLength;
					memcpy(pAd->PortCfg.SharedKey[0].Key, &profile.Key1, profile.KeyLength);
					pAd->PortCfg.SharedKey[1].KeyLen = profile.KeyLength;
					memcpy(pAd->PortCfg.SharedKey[1].Key, &profile.Key2, profile.KeyLength);
					pAd->PortCfg.SharedKey[2].KeyLen = profile.KeyLength;
					memcpy(pAd->PortCfg.SharedKey[2].Key, &profile.Key3, profile.KeyLength);
					pAd->PortCfg.SharedKey[3].KeyLen = profile.KeyLength;
					memcpy(pAd->PortCfg.SharedKey[3].Key, &profile.Key4, profile.KeyLength);
					pAd->PortCfg.FragmentThreshold = profile.Fragment;
					pAd->PortCfg.RtsThreshold = profile.RTS;
					pAd->PortCfg.WindowsTxPreamble = profile.PreamType;
					pAd->PortCfg.TxRate= profile.TransRate;
					pAd->PortCfg.TxRateFixed= 0;
					pAd->PortCfg.TxPower= profile.TransPower;
					pAd->PortCfg.Psm= profile.PSmode;
					pAd->PortCfg.CurrentTxAntenna= profile.AntennaTx;
					pAd->PortCfg.CurrentRxAntenna= profile.AntennaRx;
					pAd->PortCfg.CountryRegion = profile.CountryRegion;
				}
			}
			else
			{
				DBGPRINT(RT_DEBUG_TRACE, "--> %s does not have a write method\n", src);
			}
			retval=filp_close(srcf,NULL);
			if (retval) 
				DBGPRINT(RT_DEBUG_TRACE, "--> Error %d closing %s\n",-retval,src);				
		}
	}
	set_fs(orgfs);
	current->fsuid=orgfsuid;
	current->fsgid=orgfsgid;
}	

/*
	========================================================================
	
	Routine Description:
		Initialize port configuration structure

	Arguments:
		Adapter						Pointer to our adapter

	Return Value:
		None

	Note:
		
	========================================================================
*/
VOID	PortCfgInit(
	IN	PRTMP_ADAPTER pAd)
{
    UINT i;

    DBGPRINT(RT_DEBUG_TRACE, "--> PortCfgInit\n");    
    
    pAd->PortCfg.AuthRspTimeout = AUTH_KEY_TIMEOUT;   // in msec
    pAd->PortCfg.CapabilityInfo = 0x0000;
    pAd->PortCfg.Psm = PWR_ACTIVE;

    pAd->PortCfg.SupportedRates[0] = 0x02; // in units of 0.5 Mbps
    pAd->PortCfg.SupportedRates[1] = 0x04; // in units of 0.5 Mbps
    pAd->PortCfg.SupportedRates[2] = 0x0B; // in units of 0.5 Mbps
    pAd->PortCfg.SupportedRates[3] = 0x16; // in units of 0.5 Mbps
    pAd->PortCfg.SupportedRatesLen = 4;

    pAd->PortCfg.DesiredRates[0] = 2;  // in units of 0.5 Mbps
    pAd->PortCfg.DesiredRates[1] = 4;  // in units of 0.5 Mbps
    pAd->PortCfg.DesiredRates[2] = 11; // in units of 0.5 Mbps
    pAd->PortCfg.DesiredRates[3] = 22; // in units of 0.5 Mbps

    pAd->PortCfg.MaxBasicRate = RATE_2;
    pAd->PortCfg.MaxTxRate = RATE_11;  // intersection of DesiredRates[] and SupportedRates[]
    pAd->PortCfg.TxRate = RATE_11;
    pAd->PortCfg.TxRateFixed = 0;

    pAd->PortCfg.BeaconPeriod = 100;     // in mSec
    pAd->PortCfg.DtimPeriod = 5;         // never mind, decided by AP later
    pAd->PortCfg.AssocRspTimeout = ASSOC_TIMEOUT;  // in mSec

    pAd->PortCfg.CfpMaxDuration = 0;     // never mind, decided by AP later
    pAd->PortCfg.CfpDurRemain = 0;       // never mind, decided by AP later
    pAd->PortCfg.CfpCount = 0;           // never mind, decided by AP later
    pAd->PortCfg.CfpPeriod = 0;          // never mind, decided by AP later
    pAd->PortCfg.AuthAlgorithm = AUTH_MODE_OPEN;
    
    for(i = 0; i < SHARE_KEY_NO; i++) {
        pAd->PortCfg.SharedKey[i].KeyLen = 0;
    }

    for(i = 0; i < PEER_KEY_NO; i++) {
        pAd->PortCfg.PeerKey[i].KeyLen = 0;
    }

    pAd->PortCfg.PrivacyInvoked = FALSE;
    pAd->PortCfg.DefaultKeyId = 0;
    pAd->PortCfg.ExcludeUnencrypt = FALSE;

    //memcpy(pAd->PortCfg.StaAddr.Octet, pAd->CurrentAddress, MAC_ADDR_LEN);

    pAd->PortCfg.RtsThreshold = 2347;
    pAd->PortCfg.ShortRetryLimit = 7;
    pAd->PortCfg.LongRetryLimit = 4;
    pAd->PortCfg.FragmentThreshold = 2346;
    pAd->PortCfg.bFragmentZeroDisable = FALSE;

    pAd->PortCfg.CurrentTxAntenna = 1;
    pAd->PortCfg.CurrentRxAntenna = 1;
    pAd->PortCfg.NumberOfAntenna = 2;

    pAd->PortCfg.TxPowerLevel[0] = 1; 
    pAd->PortCfg.TxPowerLevel[1] = 5;
    pAd->PortCfg.TxPowerLevel[2] = 20;
    pAd->PortCfg.TxPowerLevel[3] = 30;
    pAd->PortCfg.TxPowerLevel[4] = 50;
    pAd->PortCfg.TxPowerLevel[5] = 100;
    pAd->PortCfg.NumOfTxPowerLevel = 6;
    pAd->PortCfg.TxPower = 0x27; //mW

    pAd->PortCfg.CwMin = CW_MIN_IN_BITS;
    pAd->PortCfg.CwMax = CW_MAX_IN_BITS;
    pAd->PortCfg.RecvDtim = TRUE;
    pAd->PortCfg.DtimCount = 0;
    memset(&pAd->PortCfg.Bssid, 0, MAC_ADDR_LEN);
    memset(&pAd->PortCfg.Broadcast, 0xff, MAC_ADDR_LEN);
    pAd->PortCfg.Pss = PWR_ACTIVE;
    pAd->PortCfg.RssiTrigger = 0;
    pAd->PortCfg.LastRssi = 0;
    pAd->PortCfg.RssiTriggerMode = RSSI_TRIGGERED_UPON_BELOW_THRESHOLD;
    pAd->PortCfg.AtimWin = 0;
    pAd->PortCfg.Channel = 1;

    pAd->PortCfg.Aid = 1;

    pAd->PortCfg.DefaultListenCount = 3;//default listen count;
    pAd->PortCfg.CurrListenCount = 0;  // in units of Beacon interval
    pAd->PortCfg.MinChnlTime = MIN_CHANNEL_TIME;   // minimum channel time (in mSEC)
    pAd->PortCfg.MaxChnlTime = MAX_CHANNEL_TIME; // maximum channel time (in mSEC)
    pAd->PortCfg.BssType = BSS_INFRA;  // BSS_INFRA or BSS_INDEP

    pAd->PortCfg.SsidLen = 0;
    memset(pAd->PortCfg.Ssid, 0, MAX_LEN_OF_SSID);  // NOT NULL-terminated

    // global variables mXXXX used in MAC protocol state machines
    pAd->PortCfg.Mibss = FALSE;
    pAd->PortCfg.Massoc = FALSE;
    pAd->PortCfg.Mauth = FALSE;

    // PHY specification
    pAd->PortCfg.PhyMode = PHY_11B;    // PHY_11A, PHY_11G, PHY_11B
    pAd->PortCfg.Drxtx = 0;      // RF RX to TX turn around time, in units of usec
    pAd->PortCfg.Dswchnl = 0;    // in units of usec, time to switch to a new channel 
    pAd->PortCfg.Dpreamble = 144;  // in units of usec 
    pAd->PortCfg.Dplcp = 48;      // in units of usec 
    pAd->PortCfg.Dslot = 20;      // in units of usec 
    pAd->PortCfg.Dsifs = 10;      // in units of usec 
    pAd->PortCfg.Dpifs = 30;      // in units of usec 
    pAd->PortCfg.Ddifs = 50;      // in units of usec 
//    pAd->PortCfg.Deifs = 364;     // in units of usec, change when BasicRateSet changes
    pAd->PortCfg.TxPreamble = Rt802_11PreambleLong; // use Long preamble on TX by defaut

    // user desired power mode
    pAd->PortCfg.WindowsPowerMode = Ndis802_11PowerModeCAM; // Ndis802_11PowerModeFast_PSP;
    pAd->PortCfg.WindowsBatteryPowerMode = Ndis802_11PowerModeCAM; // Ndis802_11PowerModeFast_PSP;
    pAd->PortCfg.WindowsTxPreamble = Rt802_11PreambleAuto; // use Long preamble on TX by defaut
    pAd->PortCfg.WindowsACCAMEnable = FALSE;
    pAd->PortCfg.PacketFilter = NDIS_PACKET_TYPE_ALL_MULTICAST | NDIS_PACKET_TYPE_DIRECTED | NDIS_PACKET_TYPE_BROADCAST | NDIS_PACKET_TYPE_PROMISCUOUS;
	pAd->bAcceptDirect = TRUE;
	pAd->bAcceptMulticast = FALSE;
	pAd->bAcceptBroadcast = TRUE;
	pAd->bAcceptAllMulticast = TRUE;
	pAd->bAcceptPromiscuous = FALSE;
	
#ifdef NDIS51_MINIPORT
    pAd->PortCfg.WindowsPowerProfile = NdisPowerProfileAcOnLine; // Ndis802_11PowerModeFast_PSP;
#endif
    // parameters to be used when this STA starts a new ADHOC network
    pAd->PortCfg.IbssConfig.BeaconPeriod = 100;
    pAd->PortCfg.IbssConfig.AtimWin = 0;
    pAd->PortCfg.IbssConfig.Channel = 1;
    pAd->PortCfg.IbssConfig.SupportedRatesLen = 4;
    pAd->PortCfg.IbssConfig.SupportedRates[0] = 0x02; // basic
    pAd->PortCfg.IbssConfig.SupportedRates[1] = 0x04; // basic
    pAd->PortCfg.IbssConfig.SupportedRates[2] = 0x0b; // not basic
    pAd->PortCfg.IbssConfig.SupportedRates[3] = 0x16; // not basic

    pAd->PortCfg.RfType = RFIC_2420;
    pAd->PortCfg.LedMode = LED_MODE_DEFAULT;
    
    init_timer(&pAd->PortCfg.RfTuningTimer);
    pAd->PortCfg.RfTuningTimer.data = (unsigned long)pAd;
    pAd->PortCfg.RfTuningTimer.function = &AsicRfTuningExec;				/* timer handler */

    // Patch for Ndtest
    pAd->PortCfg.IgnoredScanNumber = 0;

	pAd->PortCfg.bHwRadio  = TRUE;
	pAd->PortCfg.bSwRadio  = TRUE;
    pAd->PortCfg.bRadio  = TRUE;
	pAd->PortCfg.bHardwareRadio = FALSE;		// Default is OFF

    pAd->bRaConfig = TRUE;
    pAd->PortCfg.AutoReconnect = TRUE;

    pAd->PortCfg.CountryRegion = REGION_MKK1;

    // Save the init time as last scan time, the system should do scan after 2 seconds.
    // This patch is for driver wake up from standby mode, system will do scan right away.
    pAd->PortCfg.LastScanTime = 0;

    DBGPRINT(RT_DEBUG_TRACE, "<-- PortCfgInit\n");
}

