
/*
 *   pata-legacy.c - Legacy port PATA/SATA controller driver.
 *   Copyright 2005/2006 Red Hat Inc <alan@redhat.com>, all rights reserved.
 *
 *   An ATA driver for the legacy ATA ports.
 *
 *   Data Sources:
 *	Opti 82C465/82C611 support: Data sheets at opti-inc.com
 *	HT6560 series:
 *	Promise 20230/20620:
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/init.h>
#include <linux/blkdev.h>
#include <linux/delay.h>
#include <scsi/scsi_host.h>
#include <linux/ata.h>
#include <linux/libata.h>
#include <linux/platform_device.h>

#define DRV_NAME "legacy"
#define DRV_VERSION "0.4.1"

#define NR_HOST 6

static int legacy_port[NR_HOST] = { 0x1f0, 0x170, 0x1e8, 0x168, 0x1e0, 0x160 };
static int legacy_irq[NR_HOST] = { 15, 14, 11, 10, 8, 12 };

static struct ata_host_set *legacy_host[NR_HOST];
static struct platform_device *legacy_platform[NR_HOST];
static int nr_legacy_host;
static int legacy_all;
static int ht6560a;
static int ht6560b;
static int opti82c611a;
static int opti82c465mv;

/**
 *	legacy_set_mode		-	mode setting
 *	@ap: IDE interface
 *
 *	Use a non standard set_mode function. We don't want to be tuned.
 *
 *	The BIOS configured everything. Our job is not to fiddle. Just use
 *	whatever PIO the hardware is using and leave it at that. When we
 *	get some kind of nice user driven API for control then we can
 *	expand on this as per hdparm in the base kernel.
 */
 
static void legacy_set_mode(struct ata_port *ap)
{
	int i;

	for (i = 0; i < ATA_MAX_DEVICES; i++) {
		struct ata_device *dev = &ap->device[i];
		if (ata_dev_present(dev)) {
			dev->pio_mode = XFER_PIO_0;
			dev->xfer_mode = XFER_PIO_0;
			dev->xfer_shift = ATA_SHIFT_PIO;
			dev->flags |= ATA_DFLAG_PIO;
		}
	}
}

static void legacy_pio_data_xfer_noirq(struct ata_port *ap, struct ata_device *adev,
				    unsigned char *buf, unsigned int buflen,
				    int write_data)
{
	unsigned long flags;
	local_irq_save(flags);
	ata_pio_data_xfer(ap, adev, buf, buflen, write_data);
	local_irq_restore(flags);
}


static struct scsi_host_template legacy_sht = {
	.module			= THIS_MODULE,
	.name			= DRV_NAME,
	.ioctl			= ata_scsi_ioctl,
	.queuecommand		= ata_scsi_queuecmd,
	.can_queue		= ATA_DEF_QUEUE,
	.this_id		= ATA_SHT_THIS_ID,
	.sg_tablesize		= LIBATA_MAX_PRD,
	.max_sectors		= ATA_MAX_SECTORS,
	.cmd_per_lun		= ATA_SHT_CMD_PER_LUN,
	.emulated		= ATA_SHT_EMULATED,
	.use_clustering		= ATA_SHT_USE_CLUSTERING,
	.proc_name		= DRV_NAME,
	.dma_boundary		= ATA_DMA_BOUNDARY,
	.slave_configure	= ata_scsi_slave_config,
	.bios_param		= ata_std_bios_param,
//	.ordered_flush		= 1,
};

static struct ata_port_operations legacy_port_ops = {
	.set_mode	= legacy_set_mode,
	
	.port_disable	= ata_port_disable,
	.tf_load	= ata_tf_load,
	.tf_read	= ata_tf_read,
	.check_status 	= ata_check_status,
	.exec_command	= ata_exec_command,
	.dev_select 	= ata_std_dev_select,
	
	.probe_reset	= ata_std_probe_reset,

	.qc_prep 	= ata_qc_prep,
	.qc_issue	= ata_qc_issue_prot,
	.eng_timeout	= ata_eng_timeout,
	.data_xfer	= legacy_pio_data_xfer_noirq,

	.irq_handler	= ata_interrupt,
	.irq_clear	= ata_bmdma_irq_clear,
	
	.port_start	= ata_port_start,
	.port_stop	= ata_port_stop,
	.host_stop	= ata_host_stop
};	

/*
 *	Promise 20230C and 20620 support
 *
 *	This controller supports PIO0 to PIO2. We set PIO timings conservatively to
 *	allow for 50MHz Vesa Local Bus. The 20620 DMA support is weird being DMA to
 *	controller and PIO'd to the host and not supported. 
 */

static void pdc20230_set_piomode(struct ata_port *ap, struct ata_device *adev)
{
	int tries = 5;
	int pio = adev->pio_mode - XFER_PIO_0;
	u8 rt;
	
	do
	{
		inb(0x1F5);
		outb(inb(0x1F2) | 0x80, 0x1F2);
		inb(0x1F2);
		inb(0x3F6);
		inb(0x3F6);
		inb(0x1F2);
		inb(0x1F2);
	}
	while((inb(0x1F2) & 0x80) && --tries);
	
	outb(inb(0x1F4) & 0x07, 0x1F4);
	
	rt = inb(0x1F3);
	rt &= 0x07 << (3 * adev->devno);
	rt |= (3 * pio) << (3 * adev->devno);
	
	udelay(100);
	outb(inb(0x1F2) | 0x01, 0x1F2);
	udelay(100);
	inb(0x1F5);
	
}

static void pdc_data_xfer_vlb(struct ata_port *ap, struct ata_device *adev, unsigned char *buf, unsigned int buflen, int write_data)
{
	int slop = buflen & 3;
	unsigned long flags;

	if (ata_id_has_dword_io(adev->id)) {
		local_irq_save(flags);

		/* Perform the 32bit I/O synchronization sequence */	
		inb(ap->ioaddr.nsect_addr);
		inb(ap->ioaddr.nsect_addr);
		inb(ap->ioaddr.nsect_addr);
	
		/* Now the data */
	
		if (write_data)
			outsl(ap->ioaddr.data_addr, buf, buflen >> 2);
		else
			insl(ap->ioaddr.data_addr, buf, buflen >> 2);
	
		if (unlikely(slop)) {
			u32 pad;
			if (write_data) {
				memcpy(&pad, buf + buflen - slop, slop);
				outl(le32_to_cpu(pad), ap->ioaddr.data_addr);
			} else {
				pad = cpu_to_le16(inl(ap->ioaddr.data_addr));
				memcpy(buf + buflen - slop, &pad, slop);
			}
		}
		local_irq_restore(flags);
	}
	else
		legacy_pio_data_xfer_noirq(ap, adev, buf, buflen, write_data);
}

static struct ata_port_operations pdc20230_port_ops = {
	.set_piomode	= pdc20230_set_piomode,	

	.port_disable	= ata_port_disable,
	.tf_load	= ata_tf_load,
	.tf_read	= ata_tf_read,
	.check_status 	= ata_check_status,
	.exec_command	= ata_exec_command,
	.dev_select 	= ata_std_dev_select,

	.probe_reset	= ata_std_probe_reset,

	.qc_prep 	= ata_qc_prep,
	.qc_issue	= ata_qc_issue_prot,
	.eng_timeout	= ata_eng_timeout,
	.data_xfer	= pdc_data_xfer_vlb,

	.irq_handler	= ata_interrupt,
	.irq_clear	= ata_bmdma_irq_clear,
	
	.port_start	= ata_port_start,
	.port_stop	= ata_port_stop,
	.host_stop	= ata_host_stop
};	

/*
 *	Holtek 6560A support
 *
 *	This controller supports PIO0 to PIO2 (no IORDY even though higher timings
 *	can be loaded).
 */

static void ht6560a_set_piomode(struct ata_port *ap, struct ata_device *adev)
{
	u8 active, recover;
	struct ata_timing t;

	/* Get the timing data in cycles. For now play safe at 50Mhz */
	ata_timing_compute(adev, adev->pio_mode, &t, 20000, 1000);
	
	active = FIT(t.active, 2, 15);
	recover = FIT(t.recover, 4, 15);

	inb(0x3E6);
	inb(0x3E6);
	inb(0x3E6);
	inb(0x3E6);
	
	outb(recover << 4 | active, ap->ioaddr.device_addr);
	inb(ap->ioaddr.status_addr);
}

static struct ata_port_operations ht6560a_port_ops = {
	.set_piomode	= ht6560a_set_piomode,	

	.port_disable	= ata_port_disable,
	.tf_load	= ata_tf_load,
	.tf_read	= ata_tf_read,
	.check_status 	= ata_check_status,
	.exec_command	= ata_exec_command,
	.dev_select 	= ata_std_dev_select,

	.probe_reset	= ata_std_probe_reset,

	.qc_prep 	= ata_qc_prep,
	.qc_issue	= ata_qc_issue_prot,
	.eng_timeout	= ata_eng_timeout,
	.data_xfer	= ata_pio_data_xfer,	/* Check vlb/noirq */

	.irq_handler	= ata_interrupt,
	.irq_clear	= ata_bmdma_irq_clear,
	
	.port_start	= ata_port_start,
	.port_stop	= ata_port_stop,
	.host_stop	= ata_host_stop
};	

/*
 *	Holtek 6560B support
 *
 *	This controller supports PIO0 to PIO4. We honour the BIOS/jumper FIFO setting
 *	unless we see an ATAPI device in which case we force it off.
 *
 *	FIXME: need to implement 2nd channel support.
 */

static void ht6560b_set_piomode(struct ata_port *ap, struct ata_device *adev)
{
	u8 active, recover;
	struct ata_timing t;
	
	/* Get the timing data in cycles. For now play safe at 50Mhz */
	ata_timing_compute(adev, adev->pio_mode, &t, 20000, 1000);
	
	active = FIT(t.active, 2, 15);
	recover = FIT(t.recover, 2, 16);
	recover &= 0x15;

	inb(0x3E6);
	inb(0x3E6);
	inb(0x3E6);
	inb(0x3E6);
	
	outb(recover << 4 | active, ap->ioaddr.device_addr);
	
	if (adev->class != ATA_DEV_ATA) {
		u8 rconf = inb(0x3E6);
		if (rconf & 0x24) {
			rconf &= ~ 0x24;
			outb(rconf, 0x3E6);
		}
	}
	inb(ap->ioaddr.status_addr);
}

static struct ata_port_operations ht6560b_port_ops = {
	.set_piomode	= ht6560b_set_piomode,	

	.port_disable	= ata_port_disable,
	.tf_load	= ata_tf_load,
	.tf_read	= ata_tf_read,
	.check_status 	= ata_check_status,
	.exec_command	= ata_exec_command,
	.dev_select 	= ata_std_dev_select,

	.probe_reset	= ata_std_probe_reset,

	.qc_prep 	= ata_qc_prep,
	.qc_issue	= ata_qc_issue_prot,
	.eng_timeout	= ata_eng_timeout,
	.data_xfer	= ata_pio_data_xfer,	/* FIXME: Check 32bit and noirq */

	.irq_handler	= ata_interrupt,
	.irq_clear	= ata_bmdma_irq_clear,
	
	.port_start	= ata_port_start,
	.port_stop	= ata_port_stop,
	.host_stop	= ata_host_stop
};	

/*
 *	Opti core chipset helpers
 */
 
static u8 opti_syscfg(u8 reg)
{
	outb(reg, 0x22);
	return inb(0x24);
}

/*
 *	Opti 82C611A
 *
 *	This controller supports PIO0 to PIO3.
 */

static void opti82c611a_set_piomode(struct ata_port *ap, struct ata_device *adev)
{
	u8 active, recover, setup;
	struct ata_timing t;
	struct ata_device *pair = ata_dev_pair(ap, adev);
	int clock;
	int khz[4] = { 50000, 40000, 33000, 25000 };
	u8 rc;
	
	/* Enter configuration mode */
	inw(ap->ioaddr.error_addr);
	inw(ap->ioaddr.error_addr);
	outb(3, ap->ioaddr.nsect_addr);
	
	/* Read VLB clock strapping */
	clock = 1000000000 / khz[inb(ap->ioaddr.lbah_addr) & 0x03];
	
	/* Get the timing data in cycles */
	ata_timing_compute(adev, adev->pio_mode, &t, clock, 1000);
	
	/* Setup timing is shared */
	if (pair) {
		struct ata_timing tp;
		ata_timing_compute(pair, pair->pio_mode, &tp, clock, 1000);
		
		ata_timing_merge(&t, &tp, &t, ATA_TIMING_SETUP);
	}
	
	active = FIT(t.active, 2, 17) - 2;
	recover = FIT(t.recover, 1, 16) - 1;
	setup = FIT(t.setup, 1, 4) - 1;
	
	/* Select the right timing bank for write timing */
	rc = inb(ap->ioaddr.lbal_addr);
	rc &= 0x7F;
	rc |= (adev->devno << 7);
	outb(rc, ap->ioaddr.lbal_addr);

	/* Write the timings */	
	outb(active << 4 | recover, ap->ioaddr.error_addr);
	
	/* Select the right bank for read timings, also
	   load the shared timings for address */
	rc = inb(ap->ioaddr.device_addr);
	rc &= 0xC0;
	rc |= adev->devno;	/* Index select */
	rc |= (setup << 4) | 0x04;
	outb(rc, ap->ioaddr.device_addr);
	
	/* Load the read timings */
	outb(active << 4 | recover, ap->ioaddr.data_addr);
	
	/* Ensure the timing register mode is right */
	rc = inb (ap->ioaddr.lbal_addr);
	rc &= 0x73;
	rc |= 0x84;
	outb(rc, ap->ioaddr.lbal_addr);

	/* Exit command mode */
	outb(0x83,  ap->ioaddr.nsect_addr);	
}


static struct ata_port_operations opti82c611a_port_ops = {
	.set_piomode	= opti82c611a_set_piomode,	

	.port_disable	= ata_port_disable,
	.tf_load	= ata_tf_load,
	.tf_read	= ata_tf_read,
	.check_status 	= ata_check_status,
	.exec_command	= ata_exec_command,
	.dev_select 	= ata_std_dev_select,

	.probe_reset	= ata_std_probe_reset,

	.qc_prep 	= ata_qc_prep,
	.qc_issue	= ata_qc_issue_prot,
	.eng_timeout	= ata_eng_timeout,
	.data_xfer	= ata_pio_data_xfer,

	.irq_handler	= ata_interrupt,
	.irq_clear	= ata_bmdma_irq_clear,
	
	.port_start	= ata_port_start,
	.port_stop	= ata_port_stop,
	.host_stop	= ata_host_stop
};	

/*
 *	Opti 82C465MV
 *
 *	This controller supports PIO0 to PIO3. Unlike the 611A the MVB
 *	version is dual channel but doesn't have a lot of unique registers.
 */

static void opti82c465mv_set_piomode(struct ata_port *ap, struct ata_device *adev)
{
	u8 active, recover, setup;
	struct ata_timing t;
	struct ata_device *pair = ata_dev_pair(ap, adev);
	int clock;
	int khz[4] = { 50000, 40000, 33000, 25000 };
	u8 rc;
	u8 sysclk;
	
	/* Get the clock */
	sysclk = opti_syscfg(0xAC) & 0xC0;	/* BIOS set */
	
	/* Enter configuration mode */
	inw(ap->ioaddr.error_addr);
	inw(ap->ioaddr.error_addr);
	outb(3, ap->ioaddr.nsect_addr);
	
	/* Read VLB clock strapping */
	clock = 1000000000 / khz[sysclk];
	
	/* Get the timing data in cycles */
	ata_timing_compute(adev, adev->pio_mode, &t, clock, 1000);
	
	/* Setup timing is shared */
	if (pair) {
		struct ata_timing tp;
		ata_timing_compute(pair, pair->pio_mode, &tp, clock, 1000);
		
		ata_timing_merge(&t, &tp, &t, ATA_TIMING_SETUP);
	}
	
	active = FIT(t.active, 2, 17) - 2;
	recover = FIT(t.recover, 1, 16) - 1;
	setup = FIT(t.setup, 1, 4) - 1;
	
	/* Select the right timing bank for write timing */
	rc = inb(ap->ioaddr.lbal_addr);
	rc &= 0x7F;
	rc |= (adev->devno << 7);
	outb(rc, ap->ioaddr.lbal_addr);

	/* Write the timings */	
	outb(active << 4 | recover, ap->ioaddr.error_addr);
	
	/* Select the right bank for read timings, also
	   load the shared timings for address */
	rc = inb(ap->ioaddr.device_addr);
	rc &= 0xC0;
	rc |= adev->devno;	/* Index select */
	rc |= (setup << 4) | 0x04;
	outb(rc, ap->ioaddr.device_addr);
	
	/* Load the read timings */
	outb(active << 4 | recover, ap->ioaddr.data_addr);
	
	/* Ensure the timing register mode is right */
	rc = inb (ap->ioaddr.lbal_addr);
	rc &= 0x73;
	rc |= 0x84;
	outb(rc, ap->ioaddr.lbal_addr);

	/* Exit command mode */
	outb(0x83,  ap->ioaddr.nsect_addr);	
	
	/* We need to know this for quad device on the MVB */
	ap->host_set->private_data = ap;
}

/**
 *	opt82c465mv_qc_issue_prot	-	command issue
 *	@qc: command pending
 *
 *	Called when the libata layer is about to issue a command. We wrap
 *	this interface so that we can load the correct ATA timings. The
 *	MVB has a single set of timing registers and these are shared
 *	across channels. As there are two registers we really ought to
 *	track the last two used values as a sort of register window. For
 *	now we just reload on a channel switch. On the single channel
 *	setup this condition never fires so we do nothing extra.
 *
 *	FIXME: dual channel needs ->serialize support
 */

static unsigned int opti82c465mv_qc_issue_prot(struct ata_queued_cmd *qc)
{
	struct ata_port *ap = qc->ap;
	struct ata_device *adev = qc->dev;

	/* If timings are set and for the wrong channel (2nd test is
	   due to a libata shortcoming and will eventually go I hope) */
	if (ap->host_set->private_data != ap->host_set
	    && ap->host_set->private_data != NULL)
		opti82c465mv_set_piomode(ap, adev);
		
	return ata_qc_issue_prot(qc);
}

static struct ata_port_operations opti82c465mv_port_ops = {
	.set_piomode	= opti82c465mv_set_piomode,	

	.port_disable	= ata_port_disable,
	.tf_load	= ata_tf_load,
	.tf_read	= ata_tf_read,
	.check_status 	= ata_check_status,
	.exec_command	= ata_exec_command,
	.dev_select 	= ata_std_dev_select,

	.probe_reset	= ata_std_probe_reset,

	.qc_prep 	= ata_qc_prep,
	.qc_issue	= opti82c465mv_qc_issue_prot,
	.eng_timeout	= ata_eng_timeout,
	.data_xfer	= ata_pio_data_xfer,

	.irq_handler	= ata_interrupt,
	.irq_clear	= ata_bmdma_irq_clear,
	
	.port_start	= ata_port_start,
	.port_stop	= ata_port_stop,
	.host_stop	= ata_host_stop
};	


/**
 *	legacy_init_one		-	attach a legacy interface
 *	@io: I/O port start
 *	@ctrl: control port
 *	@irq: interrupt line
 *
 *	Register an ISA bus IDE interface. Such interfaces are PIO and we
 *	assume do not support IRQ sharing.
 */
 
static __init int legacy_init_one(unsigned long io, unsigned long ctrl, int irq)
{
	struct ata_probe_ent ae;
	struct platform_device *pdev;
	int ret = -EBUSY;
	struct ata_port_operations *ops = &legacy_port_ops;
	int pio_mask = 0x1F;
	
	if (request_region(io, 8, "pata_legacy") == NULL)
		return -EBUSY;
	if (request_region(ctrl, 1, "pata_legacy") == NULL)
		goto fail_io;
		
	pdev = platform_device_register_simple(DRV_NAME, nr_legacy_host, NULL, 0);
	if (pdev == NULL)
		goto fail_dev;

	if (ht6560a == 1 && (io == 0x1F0 || io == 0x170)) {
		ops = &ht6560a_port_ops;
		pio_mask = 0x07;
	}
	if (ht6560b == 1 && (io == 0x1F0 || io == 0x170)) {
		ops = &ht6560b_port_ops;
		pio_mask = 0x1F;
	}
	if (opti82c611a == 1 && (io == 0x1F0 || io == 0x170)) {
		ops = &opti82c611a_port_ops;
		pio_mask = 0x0F;
	}
	/* Same as the 82C611 for now but has differences that may
	   matter later so keep separate */
	if (opti82c465mv == 1) {
		static const char *optis[4] = { 
			"3/463MV", "5MV",
			"5MVA", "5MVB"
		};
		
		u8 ctrl = (opti_syscfg(0x30) & 0xC0) >> 6;
		u16 mv_port;
		u8 chans = 1;
		printk(KERN_INFO DRV_NAME ": Opti 82C46%s chipset support.\n", optis[ctrl]);
		if (ctrl == 3)
			chans = (opti_syscfg(0x3F) & 0x20) ? 2 : 1;
		ctrl = opti_syscfg(0xAC);
		/* Check enabled and this port is the 465MV port. On the
		   MVB we may have two channels */
		if (ctrl & 8) {
			int alt_port = mv_port = (ctrl & 4) ? 0x170 : 0x1F0;
			if (chans == 2)
				alt_port = mv_port ^ 0x80;
				
			if (mv_port == io || alt_port == io) {
				ops = &opti82c465mv_port_ops;
				pio_mask = 0x0F;
			}
		}
	}
	
	/* Probe for automatically detectable controllers */
	
	if (io == 0x1F0 && ops == &legacy_port_ops) {
		/* Probes */
		inb(0x1F5);
		outb(inb(0x1F2) | 0x80, 0x1F2);
		inb(0x1F2);
		inb(0x3F6);
		inb(0x3F6);
		inb(0x1F2);
		inb(0x1F2);
		
		if ((inb(0x1F2) & 0x80) == 0) {
			/* PDC20230 or 20630 ? */
			printk(KERN_INFO "PDC20230-C/20630 VLB ATA controller detected.\n");
				pio_mask = 0x07;
			ops = &pdc20230_port_ops;
			udelay(100);
			inb(0x1F5);
		} else {
			outb(0x55, 0x1F2);
			inb(0x1F2);
			inb(0x1F2);
			if (inb(0x1F2) == 0x00) {
				printk(KERN_INFO "PDC20230-B VLB ATA controller detected.\n");
			}
		}
	}
	memset(&ae, 0, sizeof(struct ata_probe_ent));
	INIT_LIST_HEAD(&ae.node);
	ae.dev = &pdev->dev;
	ae.port_ops = ops;
	ae.sht = &legacy_sht;
	ae.n_ports = 1;
	ae.pio_mask = pio_mask;
	ae.irq = irq;
	ae.irq_flags = 0;
	ae.host_flags = ATA_FLAG_SLAVE_POSS | ATA_FLAG_IRQ_MASK;
	ae.port[0].cmd_addr = io;
	ae.port[0].altstatus_addr = ctrl;
	ae.port[0].ctl_addr =	ctrl;
	ata_std_ports(&ae.port[0]);

	ret = ata_device_add(&ae);
	if (ret == 0) {
		ret = -ENODEV;
		goto fail;
	}		
	legacy_host[nr_legacy_host++] = dev_get_drvdata(&pdev->dev);
	legacy_platform[nr_legacy_host]	= pdev;
	return 0;	

fail:
	platform_device_unregister(pdev);
fail_dev:
	release_region(ctrl, 1);
fail_io:
	release_region(io, 8);
	return ret;
}

/**
 *	legacy_check_special_cases	-	ATA special cases
 *	@p: PCI device to check
 *	@master: set this if we find an ATA master
 *	@master: set this if we find an ATA secondary
 *
 *	A small number of vendors implemented early PCI ATA interfaces on bridge logic
 *	without the ATA interface being PCI visible. Where we have a matching PCI driver
 *	we must skip the relevant device here. If we don't know about it then the legacy
 *	driver is the right driver anyway.
 */

static void legacy_check_special_cases(struct pci_dev *p, int *primary, int *secondary)
{
	/* Cyrix CS5510 pre SFF MWDMA ATA on the bridge */
	if (p->vendor == 0x1078 && p->device == 0x0000) {
		*primary = *secondary = 1;
		return;
	}
	/* Cyrix CS5520 pre SFF MWDMA ATA on the bridge */
	if (p->vendor == 0x1078 && p->device == 0x0002) {
		*primary = *secondary = 1;
		return;
	}
	/* Intel MPIIX - PIO ATA on non PCI side of bridge */
	if (p->vendor == 0x8086 && p->device == 0x1234) {
		u16 r;
		pci_read_config_word(p, 0x6C, &r);
		if (r & 0x8000) {	/* ATA port enabled */
			if (r & 0x4000)
				*secondary = 1;
			else
				*primary = 1;
		}
		return;
	}
}

/**
 *	legacy_init		-	attach legacy interfaces
 *
 *	Attach legacy IDE interfaces by scanning the usual IRQ/port suspects.
 *	Right now we do not scan the ide0 and ide1 address but should do so
 *	for non PCI systems or systems with no PCI IDE legacy mode devices.
 *	If you fix that note there are special cases to consider like VLB
 *	drivers and CS5510/20.
 */
 
static __init int legacy_init(void)
{
	int i;
	int ct = 0;
	int primary = 0;
	int secondary = 0;
	int last_port = NR_HOST;
	
	struct pci_dev *p = NULL;

	for_each_pci_dev(p) {	
		int r;
		/* Check for any overlap of the system ATA mappings. Native mode controllers
		   stuck on these addresses or some devices in 'raid' mode won't be found by
		   the storage class test */
		for (r = 0; r < 6; r++) {
			if (pci_resource_start(p, r) == 0x1f0)
				primary = 1;
			if (pci_resource_start(p, r) == 0x170)
				secondary = 1;
		}
		/* Check for special cases */
		legacy_check_special_cases(p, &primary, &secondary);
		
		/* If PCI bus is present then don't probe for tertiary legacy ports */
		if (legacy_all == 0)
			last_port = 2;
	}
	
	
	for (i = 0; i < last_port; i++) {
		/* Skip primary if we have seen a PCI one */
		if (i == 0 && primary == 1)
			continue;
		/* Skip secondary if we have seen a PCI one */
		if (i == 1 && secondary == 1)
			continue;
		if (legacy_init_one(legacy_port[i], 
				   legacy_port[i] + 0x0206, 
				   legacy_irq[i]) == 0)
			ct++;
	}
	if (ct != 0)
		return 0;
	return -ENODEV;
}

static __exit void legacy_exit(void)
{
	int i;

	for (i = 0; i < nr_legacy_host; i++) {
		struct ata_port *ap =legacy_host[i]->ports[0];
		unsigned long io = ap->ioaddr.cmd_addr;
		unsigned long ctrl = ap->ioaddr.ctl_addr;
		ata_host_set_remove(legacy_host[i]);
		platform_device_unregister(legacy_platform[i]);
		release_region(io, 8);
		release_region(ctrl, 1);
	}	
}

MODULE_AUTHOR("Alan Cox");
MODULE_DESCRIPTION("low-level driver for legacy ATA");
MODULE_LICENSE("GPL");
MODULE_VERSION(DRV_VERSION);

/**
 *	legacy_probe_all	-	setup argument handler
 *	@unused: unused
 *
 *	Called when the probe_all argument is passed to this driver. This indicates
 *	that we should check all the legacy ATA addresses even if PCI is present. Should
 *	only ever be needed on very strange PCI/legacy combinations
 */
 
static int __init legacy_probe_all(char *unused)
{
	legacy_all = 1;
	return 1;
}

__setup("probe-all", legacy_probe_all);

/**
 *	legacy_ht6560a	-	setup argument handler
 *	@unused: unused
 *
 *	Called when the ht6560a argument is passed to this driver. This indicates
 *	that we should check all the legacy ATA addresses even if PCI is present. Should
 *	only ever be needed on very strange PCI/legacy combinations
 */
 
static int __init legacy_ht6560a(char *unused)
{
	ht6560a = 1;
	return 1;
}

__setup("ht6560a", legacy_ht6560a);

/**
 *	legacy_ht6560b	-	setup argument handler
 *	@unused: unused
 *
 *	Called when the ht6560b argument is passed to this driver. This indicates
 *	that we should check all the legacy ATA addresses even if PCI is present. Should
 *	only ever be needed on very strange PCI/legacy combinations
 */
 
static int __init legacy_ht6560b(char *unused)
{
	ht6560b = 1;
	return 1;
}

__setup("ht6560b", legacy_ht6560b);

/**
 *	legacy_opti82c611a	-	setup argument handler
 *	@unused: unused
 *
 *	Called when the 82c611a argument is passed to this driver. This indicates
 *	that we should check all the legacy ATA addresses even if PCI is present. Should
 *	only ever be needed on very strange PCI/legacy combinations
 */
 
static int __init legacy_opti82c611a(char *unused)
{
	opti82c611a = 1;
	return 1;
}

__setup("opti82c611a", legacy_opti82c611a);

/**
 *	legacy_opti82c465mv	-	setup argument handler
 *	@unused: unused
 *
 *	Called when the opti82c465mv argument is passed to this driver. This indicates
 *	that we should check all the legacy ATA addresses even if PCI is present. Should
 *	only ever be needed on very strange PCI/legacy combinations
 */
 
static int __init legacy_opti82c465mv(char *unused)
{
	opti82c465mv = 1;
	return 1;
}

__setup("opti82c465mv", legacy_opti82c465mv);

module_init(legacy_init);
module_exit(legacy_exit);

