/**************************************************************************
 * Copyright (c) Intel Corp. 2007.
 * All Rights Reserved.
 *
 * Intel funded Tungsten Graphics (http://www.tungstengraphics.com) to
 * develop this driver.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sub license, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDERS, AUTHORS AND/OR ITS SUPPLIERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
 * USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 **************************************************************************/
/*
 */

#include "drmP.h"
#include "psb_drv.h"

struct psb_use_reg {
	struct drm_reg reg;
	struct drm_psb_private *dev_priv;
	uint32_t reg_seq;
	uint32_t base;
	uint32_t data_master;
};

struct psb_use_reg_data {
	uint32_t base;
	uint32_t size;
	uint32_t data_master;
};

static int psb_use_reg_reusable(const struct drm_reg *reg, const void *data)
{
	struct psb_use_reg *use_reg = 
		container_of(reg, struct psb_use_reg, reg);
	struct psb_use_reg_data *use_data = 
		(struct psb_use_reg_data *) data;

	return ((use_reg->base <= use_data->base) &&
		(use_reg->base + PSB_USE_OFFSET_SIZE > 
		 use_data->base + use_data->size) &&
		use_reg->data_master == use_data->data_master);
}

		
static int psb_use_reg_set(struct psb_use_reg *use_reg, 
			   const struct psb_use_reg_data *use_data)
{
	struct drm_psb_private *dev_priv = use_reg->dev_priv;

	if (use_reg->reg.fence == NULL)
		use_reg->data_master = use_data->data_master;

	if (use_reg->reg.fence == NULL && 
	    !psb_use_reg_reusable(&use_reg->reg, (const void *)use_data)) {

		use_reg->base = use_data->base & ~PSB_USE_OFFSET_MASK;
		use_reg->data_master = use_data->data_master;

		if (!psb_use_reg_reusable(&use_reg->reg, 
					  (const void *)use_data)) {
			DRM_ERROR("USE base mechanism didn't support "
				  "buffer size or alignment\n");
			return -EINVAL;
		}

		PSB_WSGX32(PSB_ALPL(use_reg->base, _PSB_CUC_BASE_ADDR) | 
			   (use_reg->data_master << _PSB_CUC_BASE_DM_SHIFT), 
			   PSB_CR_USE_CODE_BASE(use_reg->reg_seq));
	}
	return 0;
			   
}
	
int psb_grab_use_base(struct drm_psb_private *dev_priv,
		      unsigned long base,
		      unsigned long size,
		      unsigned int data_master,
		      uint32_t fence_class,
		      uint32_t fence_type,
		      int no_wait,
		      int interruptible,
		      int *r_reg,
		      u32 *r_offset)
{
	struct psb_use_reg_data use_data = {
		.base = base,
		.size = size,
		.data_master = data_master};
	int ret;

	struct drm_reg *reg;
	struct psb_use_reg *use_reg;

	ret = drm_regs_alloc(&dev_priv->use_manager,
			     (const void *) &use_data,
			     fence_class,
			     fence_type,
			     interruptible,
			     no_wait,
			     &reg);
	if (ret)
		return ret;

	use_reg = container_of(reg, struct psb_use_reg, reg);
	ret = psb_use_reg_set(use_reg, &use_data);

	if (ret)
		return ret;

	*r_reg = use_reg->reg_seq;
	*r_offset = base - use_reg->base;

	return 0;
};
	
	
static void psb_use_reg_destroy(struct drm_reg *reg)
{
	struct psb_use_reg *use_reg = 
		container_of(reg, struct psb_use_reg, reg);
	struct drm_psb_private *dev_priv = use_reg->dev_priv;
	
	PSB_WSGX32(PSB_ALPL(0, _PSB_CUC_BASE_ADDR),
		   PSB_CR_USE_CODE_BASE(use_reg->reg_seq));

	drm_free(use_reg, sizeof(*use_reg), DRM_MEM_DRIVER);
}

int psb_init_use_base(struct drm_psb_private *dev_priv, 
		      unsigned int reg_start,
		      unsigned int reg_num)
{
	struct psb_use_reg *use_reg;
	int i;
	int ret = 0;

	mutex_lock(&dev_priv->cmdbuf_mutex);

	drm_regs_init(&dev_priv->use_manager,
		      &psb_use_reg_reusable,
		      &psb_use_reg_destroy);

	for (i=reg_start; i<reg_start + reg_num; ++i) {
		use_reg = drm_calloc(1, sizeof(*use_reg), DRM_MEM_DRIVER);
		if (!use_reg) {
			ret = -ENOMEM;
			goto out;
		}

		use_reg->dev_priv = dev_priv;
		use_reg->reg_seq = i;
		use_reg->base = 0;
		use_reg->data_master = _PSB_CUC_DM_PIXEL;

		PSB_WSGX32(PSB_ALPL(use_reg->base, _PSB_CUC_BASE_ADDR) | 
			   (use_reg->data_master << _PSB_CUC_BASE_DM_SHIFT), 
			   PSB_CR_USE_CODE_BASE(use_reg->reg_seq));
		
		drm_regs_add(&dev_priv->use_manager, &use_reg->reg);
	}
out:
	mutex_unlock(&dev_priv->cmdbuf_mutex);

	return ret;
	
}

void psb_takedown_use_base(struct drm_psb_private *dev_priv)
{
	mutex_unlock(&dev_priv->cmdbuf_mutex);
	drm_regs_free(&dev_priv->use_manager);
	mutex_unlock(&dev_priv->cmdbuf_mutex);
}
