/*
 * Parser input reading gzipped files using zlib
 *
 * Copyright (C) 2003,2004,2005  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <apt-front/utils/zlibparserinput.h>

#include <zlib.h>
#include <errno.h>

using namespace std;
using namespace Tagcoll;

namespace aptFront {
namespace utils {

struct ZlibParserInputData
{
	gzFile in;
	int backbuf;

	ZlibParserInputData() throw ()
		: in(0), backbuf(ParserInput::Eof) {}
};

ZlibParserInput::ZlibParserInput(const std::string& fname) throw (Tagcoll::FileException)
	: _data(new ZlibParserInputData), _file(fname), _line(1)
{
	_data->in = gzopen(fname.c_str(), "r");
	if (!_data->in)
	{
		delete _data;
		_data = 0;
		throw FileException(errno, "opening compressed file " + fname + " for reading");
	}
}

ZlibParserInput::~ZlibParserInput() throw ()
{
	if (_data)
	{
		gzclose(_data->in);
		delete _data;
	}
}

int ZlibParserInput::nextChar() throw (Tagcoll::ParserInputException)
{
	// See if there's a previously pushed char
	if (_data->backbuf != ParserInput::Eof)
	{
		int res = _data->backbuf;
		_data->backbuf = ParserInput::Eof;
		return res;
	}

	int res = gzgetc(_data->in);
	if (res == -1)
		return ParserInput::Eof;

	if (res == '\n')
		_line++;
	return res;
}

void ZlibParserInput::pushChar(int c) throw (Tagcoll::ParserInputException)
{
	_data->backbuf = c;
}

}
}

// vim:set ts=4 sw=4:
