/*
 * Ready-made utility classes for Debtags
 *
 * Copyright (C) 2003,2004,2005  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <apt-front/cache/entity/tag.h>

#ifndef APTFRONT_CACHE_COMPONENT_DEBTAGS_UTILS_H
#define APTFRONT_CACHE_COMPONENT_DEBTAGS_UTILS_H

#include <tagcoll/CardinalityStore.h>

#if 0
#include <debtags/Tags.h>
#include <debtags/Filters.h>
#include <debtags/Status.h>
#include <map>
#include <debtags/Tag.h>
#include <tagcoll/Filter.h>
#endif

namespace aptFront {
namespace cache {
namespace component {
namespace debtags {

/**
 * Convert a collection of ITEMs tagged with Tags to a collection of
 * ITEMs tagged with only the facets.
 */
template<class ITEM>
class TagToFacet : public Tagcoll::Consumer<ITEM, entity::Tag>
{
protected:
	Tagcoll::Consumer<ITEM, entity::Facet>* consumer;

	Tagcoll::OpSet<entity::Facet> strip(const Tagcoll::OpSet<entity::Tag>& tags)
	{
		Tagcoll::OpSet<entity::Facet> res;
		for (Tagcoll::OpSet<entity::Tag>::const_iterator i = tags.begin();
				i != tags.end(); i++)
			res += i->facet();
		return res;
	}

	virtual void consumeItemUntagged(const ITEM& item) { consumer->consume(item); }
	virtual void consumeItem(const ITEM& item, const Tagcoll::OpSet<entity::Tag>& tags)
	{
		consumer->consume(item, strip(tags));
	}
	virtual void consumeItemsUntagged(const Tagcoll::OpSet<ITEM>& items) { consumer->consume(items); }
	virtual void consumeItems(const Tagcoll::OpSet<ITEM>& items, const Tagcoll::OpSet<entity::Tag>& tags)
	{
		consumer->consume(items, strip(tags));
	}
	
public:
	TagToFacet() : consumer(0) {}
	TagToFacet(Tagcoll::Consumer<ITEM, entity::Facet>& consumer) : consumer(&consumer) {}

	Tagcoll::Consumer<ITEM, entity::Facet>& getConsumer() const { return *consumer; }
	void setConsumer(Tagcoll::Consumer<ITEM, entity::Facet>& consumer) { this->consumer = &consumer; }
};



/**
 * This class should help in implementing full-text + debtags searches, but
 * it's still very much in the making
 */
template <class ITEM>
class SearchHelper
{
protected:
	static const int dist = 7;

	class CardCompute : public std::map<entity::Tag, int>, public Tagcoll::Consumer<ITEM, entity::Tag>
	{
		void consumeItemUntagged(const ITEM&) {}
		void consumeItem(const ITEM&, const Tagcoll::OpSet<entity::Tag>& tags)
		{
			for (Tagcoll::OpSet<entity::Tag>::const_iterator i = tags.begin();
					i != tags.end(); i++)
				(*this)[*i]++;
		}
		void consumeItems(const Tagcoll::OpSet<ITEM>& items, const Tagcoll::OpSet<entity::Tag>& tags)
		{
			for (Tagcoll::OpSet<entity::Tag>::const_iterator i = tags.begin();
					i != tags.end(); i++)
				(*this)[*i] += items.size();
		}
	public:
		virtual ~CardCompute() {}
	};

	component::PackageTags& debtags;
	Tagcoll::OpSet<ITEM> startPackages;
	Tagcoll::OpSet<entity::Tag> wanted;
	Tagcoll::OpSet<entity::Tag> unwanted;
	Tagcoll::OpSet<ITEM> found;
	Tagcoll::OpSet<entity::Tag> pivotTags;
	CardCompute cardComputer;
	int cards[dist];
	entity::Tag tags[dist];

	void storeTag(const entity::Tag& tag, int card = -1, int idx = 0);

public:
	SearchHelper(component::PackageTags& debtags) : debtags(debtags) {}

	void clearStartPackages() { startPackages.clear(); }
	void addStartPackage(const ITEM& item) { startPackages += item; }

	void clearWanted() { wanted.clear(); }
	void addWanted(const entity::Tag& tag) { wanted += tag; }
	const Tagcoll::OpSet<entity::Tag>& getWanted() const { return wanted; }

	void clearUnwanted() { unwanted.clear(); }
	void addUnwanted(const entity::Tag& tag) { unwanted += tag; }
	const Tagcoll::OpSet<entity::Tag>& getUnwanted() const { return unwanted; }

	const Tagcoll::OpSet<entity::Tag>& getPivot() const { return pivotTags; }

	const Tagcoll::OpSet<ITEM>& getFound() const { return found; }

	void compute();
};


/**
 * Compute "Special-case packages"
 *
 * Compute a list of those packages whose tag set is 'strange' compared to the
 * other packages.
 *
 * The list corresponds to those packages reported by 
 * debtags facetcoll | tagcoll findspecials
 */
template <class ITEM>
class Specials : public Tagcoll::Consumer<ITEM, entity::Tag>
{
protected:
	Tagcoll::CardinalityStore<ITEM, entity::Facet> coll;
	TagToFacet<ITEM> tagStripper;
	unsigned int maxPerGroup;
	bool computed;

	virtual void consumeItemUntagged(const ITEM& item)
	{
		tagStripper.consume(item);
	}
	virtual void consumeItem(const ITEM& item, const Tagcoll::OpSet<entity::Tag>& tags)
	{
		tagStripper.consume(item, tags);
	}
	virtual void consumeItemsUntagged(const Tagcoll::OpSet<ITEM>& items)
	{
		tagStripper.consume(items);
	}
	virtual void consumeItems(const Tagcoll::OpSet<ITEM>& items, const Tagcoll::OpSet<entity::Tag>& tags)
	{
		tagStripper.consume(items, tags);
	}

public:
	Specials(unsigned int maxPerGroup = 0) :
		tagStripper(coll), maxPerGroup(maxPerGroup), computed(false) {}
	virtual ~Specials() {}
	
	bool isComputed() const { return computed; }
	void compute(/*Status* tracker = 0*/);
	void clear()
	{
		coll = Tagcoll::CardinalityStore<ITEM, entity::Facet>(); specials.clear();
		computed = false;
	}

	std::map< entity::Facet, Tagcoll::OpSet<ITEM> > specials;
};

}
}
}
}

// vim:set ts=4 sw=4:
#endif
