/**
 * @file cache/component/packagetags.cpp
 * @author Enrico Zini (enrico) <enrico@enricozini.org>
 */

/*
 * System tag database
 *
 * Copyright (C) 2003-2006  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <apt-front/cache/component/packagetags.h>
#include <apt-front/cache/component/debtags/update.h>
#include <apt-front/utils/paths.h>

#include <tagcoll/StdioParserInput.h>
#include <tagcoll/InputMerger.h>
#include <tagcoll/TextFormat.h>
#if 0
#include <debtags/Tags.h>
#include <debtags/Paths.h>

#include <tagcoll/Patches.h>
#include <tagcoll/Filters.h>
#endif

#include <sys/wait.h>	// WIFEXITED WEXITSTATUS
#include <sys/types.h>	// getpwuid, stat, mkdir, getuid
#include <sys/stat.h>	// stat, mkdir
#include <pwd.h>		// getpwuid
#include <unistd.h>		// stat, getuid

#if 0
#include <iostream> // std::cerr
#endif

#include <errno.h>

using namespace std;
using namespace Tagcoll;
using namespace aptFront;
using namespace cache;
using namespace component;

std::string PackageTags::componentName() { return "PackageTags"; }

static string get_rcdir(bool editable)
{
	std::string rcdir;
	struct passwd* udata = getpwuid(getuid());
	rcdir = udata->pw_dir;
	rcdir += "/.debtags";

	//printf("Init debtags environment; rcdir is %.*s\n", PFSTR(rcdir));

	struct stat rcdir_stat;
	if (stat(rcdir.c_str(), &rcdir_stat) == -1)
	{
		//printf("Creating profile directory %.*s.\n", PFSTR(rcdir));
		if (editable && mkdir(rcdir.c_str(), 0777) == -1)
			throw SystemException(errno, "creating directory " + rcdir);
	} else {
		if (S_ISDIR(rcdir_stat.st_mode) == 0)
			if (editable)
				throw ConsistencyCheckException(rcdir + " already exists and is not a directory");
			else
				fprintf(stderr, "warning: %.*s already exists and is not a directory", PFSTR(rcdir));
		
		if (!utils::Path::access(rcdir, R_OK | W_OK | X_OK))
			if (editable)
				throw ConsistencyCheckException(rcdir + " already exists and has insufficient permissions");
			else
				fprintf(stderr, "%.*s already exists and has insufficient permissions", PFSTR(rcdir));
	}

	return rcdir;
}

static void generate(Cache& c)
{
	vector<debtags::source> sources = debtags::readSources();

	MasterMMapIndexer master(utils::Path::tagdbIndex());

	// Read and merge tag data
	IntDiskIndexer<entity::Package, entity::Tag> tagindexer(
			c.packageintconverter(), c.tagintconverter());

	InputMerger<entity::Package, entity::Tag> merger;
	int found = 0;
	for (vector<debtags::source>::const_iterator i = sources.begin();
			i != sources.end(); i++)
		found += readTags(c, merger, *i);
	if (!found)
		throw ConsistencyCheckException("Unable to use any data source (not even previously cached ones)");

	// Setup the string->int conversion
	merger.output(tagindexer);

	// 3: pkg->tag
	master.append(tagindexer.pkgIndexer());
	// 4: tag->pkg
	master.append(tagindexer.tagIndexer());

	// Write the tag database in text format
	// fprintf(stdout, "Writing merged tag database...\n");
	string tagdb = utils::Path::tagdb();
	string tmpdb = tagdb + ".tmp";
	FILE* out = fopen(tmpdb.c_str(), "wt");
	if (!out) throw FileException(errno, "opening " + tmpdb);
	TextFormat<entity::Package, entity::Tag> writer(
			c.packagestringconverter(), c.tagstringconverter(), out);
	merger.output(writer);
	fclose(out);

	// Perform "atomic" update of the tag database
	// FIXME: cannot be atomic because race conditions happening between file
	// renames
	if (rename(tmpdb.c_str(), tagdb.c_str()) == -1)
		throw FileException(errno, "renaming " + tmpdb + " to " + tagdb);

	master.commit();
}

PackageTags::PackageTags () : m_coll(m_rocoll) {}

void PackageTags::init(Cache& cache, bool editable)
{
	if (debtags::dbNeedsReindex(cache))
		generate(cache);

	m_timestamp = utils::Path::timestamp(utils::Path::tagdbIndex());

	mastermmap.init(utils::Path::tagdbIndex());

	// Initialize the readonly index
	m_rocoll.init(mastermmap, 0, 1,
			&cache.packageintconverter(),
			&cache.tagintconverter(),
			&cache.packageintconverter(),
			&cache.tagintconverter());
	
	// Initialize the patch collection layer
	rcdir = get_rcdir(editable);

	string patchFile = rcdir + "/patch";
	if (utils::Path::access(patchFile, F_OK))
	{
		StdioParserInput in(patchFile);
		PatchList<entity::Package, entity::Tag> patch =
			TextFormat<entity::Package, entity::Tag>::parsePatch(
				cache.packagestringconverter(),
				cache.tagstringconverter(), in);
		m_coll.setChanges(patch);
	}
}


bool PackageTags::hasTagDatabase()
{
	using namespace utils;

	if (!Path::access(Path::tagdb(), R_OK))
	{
		std::cerr << "Missing tag database " << Path::tagdb() << std::endl;
		return false;
	}
	if (!Path::access(Path::tagdbIndex(), R_OK))
	{
		std::cerr << "Missing tag database index " << Path::tagdbIndex() << std::endl;
		return false;
	}
	if (!Path::access(Path::vocabulary(), R_OK))
	{
		std::cerr << "Missing tag vocabulary " << Path::vocabulary() << std::endl;
		return false;
	}
	if (!Path::access(Path::vocabularyIndex(), R_OK))
	{
		std::cerr << "Missing index for tag vocabulary " << Path::vocabularyIndex() << std::endl;
		return false;
	}
	return true;
}


void PackageTags::savePatch()
{
	savePatch(m_coll.getChanges());
}

void PackageTags::savePatch(const PatchList<entity::Package, entity::Tag>& patch)
{
	string patchFile = rcdir + "/patch";
	string backup = patchFile + "~";

	if (access(patchFile.c_str(), F_OK))
		if (rename(patchFile.c_str(), backup.c_str()) == -1)
			throw SystemException(errno, "Can't rename " + patchFile + " to " + backup);

	try {
		FILE* out = fopen(patchFile.c_str(), "w");
		if (out == 0)
			throw SystemException(errno, "Can't write to " + patchFile);
		
		TextFormat<entity::Package, entity::Tag>::outputPatch(
				cache().packagestringconverter(),
				cache().tagstringconverter(),
				patch, out);

		fclose(out);
	} catch (Exception& e) {
		if (rename(backup.c_str(), patchFile.c_str()) == -1)
			fprintf(stderr, "Warning: Impossible to restore previous backup copy: %.*s (%s)\n", PFSTR(e.desc()), e.type());
		throw;
	}
}

void PackageTags::sendPatch()
{
	PatchList<entity::Package, entity::Tag> patch = m_coll.getChanges();
	if (!patch.empty())
		sendPatch(m_coll.getChanges());
}

void PackageTags::sendPatch(const PatchList<entity::Package, entity::Tag>& patch)
{
	static const char* cmd = "/usr/sbin/sendmail -t";
	FILE* out = popen(cmd, "w");
	if (out == 0)
		throw SystemException(errno, string("trying to run `") + cmd + "'");

	struct passwd* udata = getpwuid(getuid());

	fprintf(out,
			"To: debtags-submit@vitavonni.de\n"
			"Bcc: %s\n"
			"Subject: Tag patch\n"
			"Mime-Version: 1.0\n"
			"Content-Type: multipart/mixed; boundary=\"9amGYk9869ThD9tj\"\n"
			"Content-Disposition: inline\n"
			"X-Mailer: debtags-edit\n\n"
			"This mail contains a Debtags patch for the central archive\n\n"
			"--9amGYk9869ThD9tj\n"
			"Content-Type: text/plain; charset=utf-8\n"
			"Content-Disposition: inline\n\n"
			"-- DEBTAGS DIFF V0.1 --\n", udata->pw_name);

	TextFormat<entity::Package, entity::Tag>::outputPatch(
			cache().packagestringconverter(),
			cache().tagstringconverter(),
			patch, out);

	fprintf(out, "\n--9amGYk9869ThD9tj\n");

	int res = pclose(out);
	if (!WIFEXITED(res) || WEXITSTATUS(res) != 0)
		throw ConsistencyCheckException("sendmail returned nonzero (" + stringf::fmt(res) + "): the mail may have not been sent");
}


void PackageTags::outputSystem(Tagcoll::Consumer<std::string, std::string>& cons)
{
	Tagcoll::ConversionFilter<entity::Package, entity::Tag, std::string, std::string> conv(
			cache().packagestringconverter(),
			cache().tagstringconverter(),
			cons);
	m_rocoll.output(conv);
}


void PackageTags::outputSystem(Tagcoll::Consumer<entity::Package, entity::Tag>& cons)
{
	m_rocoll.output(cons);
}

void PackageTags::outputPatched(Tagcoll::Consumer<std::string, std::string>& cons)
{
	Tagcoll::ConversionFilter<entity::Package, entity::Tag, std::string, std::string> conv(
			cache().packagestringconverter(),
			cache().tagstringconverter(),
			cons);
	m_coll.output(conv);
}

void PackageTags::outputPatched(Tagcoll::Consumer<entity::Package, entity::Tag>& cons)
{
	m_coll.output(cons);
}





#ifdef COMPILE_TESTSUITE
#define SAVED_COMPILE_TESTSUITE
#undef COMPILE_TESTSUITE
#endif

#define INSTANTIATING_TEMPLATES

/* Instantiate templates */
#include <apt-front/cache/cache.h>
#include <apt-front/cache/entity/tag.h>

#include <tagcoll/OpSet.cc>
#include <tagcoll/Patches.cc>
#include <tagcoll/PatchCollection.cc>
#include <tagcoll/InputMerger.cc>
#include <tagcoll/ItemGrouper.cc>
#include <tagcoll/CardinalityStore.cc>
#include <tagcoll/SmartHierarchy.cc>
#include <tagcoll/IntDiskIndex.cc>
//#include <tagcoll/TDBDiskIndex.cc>
//#include <tagcoll/TDBReadonlyDiskIndex.h>
#include <tagcoll/TextFormat.cc>

template class OpSet<entity::Package>;
template class Patch<entity::Package, entity::Tag>;
template class PatchList<entity::Package, entity::Tag>;
template class PatchCollection<entity::Package, entity::Tag>;
template class InputMerger<entity::Package, entity::Tag>;
template class ItemGrouper<entity::Package, entity::Tag>;
template class IntDiskIndex<entity::Package, entity::Tag>;
template class IntDiskIndexer<entity::Package, entity::Tag>;
#if 0
template class CardinalityStore<entity::Package, entity::Facet>;
template class SmartHierarchyNode<entity::Package, entity::Facet>;
template class CardinalityStore<entity::Package, entity::Tag>;
template class SmartHierarchyNode<entity::Package, entity::Tag>;
template class TDBDiskIndex<entity::Package, entity::Tag>;
template class TDBReadonlyDiskIndex<entity::Package, entity::Tag>;
#endif
template class TextFormat<entity::Package, entity::Tag>;



#ifdef SAVED_COMPILE_TESTSUITE
#define COMPILE_TESTSUITE
#endif

#ifdef COMPILE_TESTSUITE
#include "test-utils.h"
#include <apt-front/cache/component/debtags/update.h>

namespace tut {

struct cache_component_debtags_shar {
    cache_component_debtags_shar () {
        aptInit ();
        c.open( Cache::OpenDefault |
                Cache::OpenReadOnly );
    }
    Cache c;
};

TESTGRP( cache_component_debtags );

template<> template<>
void to::test<1>()
{
    c.reopen();
}


#if 0

template<> template<>
void to::test<1> ()
{
    /* Get the 'debtags' package */
    entity::Package p = c.packages().packageByName( "debtags" );
    ensure(p.valid());

    /* Get its tags */
    Tagcoll::OpSet<entity::Tag> tags = c.debtags().tagdb().getTags(p);
    ensure(!tags.empty());

    /* Get the items for the tagset of 'debtags' */
    Tagcoll::OpSet<entity::Package> packages = c.debtags().tagdb().getItems(tags);
    ensure(!packages.empty());
    /* They should at least contain 'debtags' */
    ensure(packages.contains(p));

    /* Get one of the tags of 'debtags' */
    entity::Tag tag = *tags.begin();
    ensure(tag);

    /* Get its items */
    {
        /* Need this workaround until I figure out how to tell the new GCC
         * that TagDB is a TDBReadonlyDiskIndex and should behave as such
         */
        Tagcoll::OpSet<entity::Tag> ts;
	ts += tag;
        packages = c.debtags().tagdb().getItems(ts);
    }
    //packages = c.debtags().tagdb().getItems(tag);
    ensure(!packages.empty());
    /* They should at least contain 'debtags' */
    ensure(packages.contains(p));

    //c.debtags().getTags(""); // XXX HACK AWW!
}

template<> template<>
void to::test<2>()
{
}
#endif

}

// vim:set ts=4 sw=4:
#endif
