// -*- C++ -*-
#include <ept/cache/debtags/update.h>
#include <ept/cache/debtags/vocabularymerger.h>
#include <ept/cache/debtags/serializer.h>

#include <tagcoll/input/zlib.h>
#include <tagcoll/input/stdio.h>
#include <tagcoll/coll/intdiskindex.h>
#include <tagcoll/coll/simple.h>
#include <tagcoll/stream/filters.h>
#include <tagcoll/TextFormat.h>

#include <wibble/sys/fs.h>

#include <iostream>

#ifndef EPT_CACHE_DEBTAGS_UPDATE_TCC
#define EPT_CACHE_DEBTAGS_UPDATE_TCC

namespace ept {
namespace t {
namespace cache {
namespace debtags {

template<typename OUT>
void SourceDir::readTags(OUT out)
{
	if (!valid()) return;

	for (const_iterator d = begin(); d != end(); ++d)
	{
		FileType type = fileType(d->d_name);
		if (type == TAG)
		{
			// Read uncompressed data
			tagcoll::input::Stdio in(path() + "/" + d->d_name);

			// Read the collection
			tagcoll::textformat::parse(in, out);
		}
		else if (type == TAGGZ)
		{
			// Read compressed data
			tagcoll::input::Zlib in(path() + "/" + d->d_name);

			// Read the collection
			tagcoll::textformat::parse(in, out);
		}
	}
}

//#include <iostream>


template<typename P>
IndexManager<P>::Vocabulary::Vocabulary()
	: mainSource(P::debtagsSourceDir()), userSource(P::debtagsUserSourceDir())
{
	rescan();
}

template<typename P>
void IndexManager<P>::Vocabulary::rescan()
{
	ts_main_src = mainSource.vocTimestamp();
	ts_user_src = userSource.vocTimestamp();
	ts_main_voc = P::timestamp(P::vocabulary());
	ts_main_idx = P::timestamp(P::vocabularyIndex());
	ts_user_voc = P::timestamp(P::userVocabulary());
	ts_user_idx = P::timestamp(P::userVocabularyIndex());
}

template<typename P>
bool IndexManager<P>::Vocabulary::needsRebuild() const
{
	// If there are no indexes of any kind, then we need rebuilding
	if (ts_user_voc == 0 && ts_user_idx == 0 && ts_main_voc == 0 && ts_main_idx == 0)
		return true;

	// If the user index is ok, then we are fine
	if (ts_user_voc >= sourceTimestamp() || ts_user_idx >= sourceTimestamp())
		return false;

	// If there are user sources, then we cannot use the system index
	if (ts_user_src > 0)
		return true;

	// If there are no user sources, then we can fallback on the system
	// indexes in case the user indexes are not up to date
	if (ts_main_voc >= sourceTimestamp() || ts_main_idx >= sourceTimestamp())
		return false;

	return true;
}

template<typename P>
bool IndexManager<P>::Vocabulary::userIndexIsRedundant() const
{
	// If there is no user index, then it is not redundant
	if (ts_user_voc == 0 && ts_user_idx == 0)
		return false;

	// If we have user sources, then the user index is never redundant
	if (ts_user_src > 0)
		return false;

	// If the system index is not up to date, then the user index is not
	// redundant
	if (ts_main_voc < sourceTimestamp() || ts_main_idx < sourceTimestamp())
		return false;

	return true;
}

template<typename P>
void IndexManager<P>::Vocabulary::rebuild(const std::string& vocfname, const std::string& idxfname)
{
	using namespace tagcoll;

	// Create the master MMap index
	diskindex::MasterMMapIndexer master(idxfname);

	// Read and merge vocabulary data
	VocabularyMerger voc;
	mainSource.readVocabularies(voc);
	userSource.readVocabularies(voc);

	if (voc.empty())
		throw wibble::exception::Consistency("Reading debtags sources from " + mainSource.path() + " and " + userSource.path(), "Unable to find any vocabulary data");

	// Write the merged vocabulary, and generate tag and facet IDs as a side
	// effect
	std::string tmpvocfname = vocfname + ".tmp";
	voc.write(tmpvocfname);

	// Add the indexed vocabulary data to the master index
	// 0: facets
	master.append(voc.facetIndexer());
	// 1: tags
	master.append(voc.tagIndexer());

	if (rename(tmpvocfname.c_str(), vocfname.c_str()) == -1)
		throw wibble::exception::System("renaming " + tmpvocfname + " to " + vocfname);

	master.commit();
}

template<typename P>
bool IndexManager<P>::Vocabulary::rebuildIfNeeded()
{
	if (needsRebuild())
	{
		// Decide if we rebuild the user index or the system index

		if (ts_user_src == 0 && P::access(P::debtagsIndexDir(), W_OK) == 0)
		{
			// There are no user sources and we can write to the system index
			// directory: rebuild the system index
			rebuild(P::vocabulary(), P::vocabularyIndex());
			ts_main_voc = P::timestamp(P::vocabulary());
			ts_main_idx = P::timestamp(P::vocabularyIndex());
		} else {
			wibble::sys::fs::mkFilePath(P::userVocabulary());
			wibble::sys::fs::mkFilePath(P::userVocabularyIndex());
			rebuild(P::userVocabulary(), P::userVocabularyIndex());
			ts_user_voc = P::timestamp(P::userVocabulary());
			ts_user_idx = P::timestamp(P::userVocabularyIndex());
		}
		return true;
	}
	return false;
}

template<typename P>
bool IndexManager<P>::Vocabulary::deleteRedundantUserIndex()
{
	if (userIndexIsRedundant())
	{
		// Delete the user indexes if they exist
		unlink(P::userVocabulary().c_str());
		unlink(P::userVocabularyIndex().c_str());
		ts_user_voc = 0;
		ts_user_idx = 0;
		return true;
	}
	return false;
}

template<typename P>
bool IndexManager<P>::Vocabulary::getUpToDateVocabulary(std::string& vocfname, std::string& idxfname)
{
	// If there are no indexes of any kind, then we have nothing to return
	if (ts_user_voc == 0 && ts_user_idx == 0 && ts_main_voc == 0 && ts_main_idx == 0)
		return false;

	// If the user index is up to date, use it
	if (ts_user_voc >= sourceTimestamp() &&
		ts_user_idx >= sourceTimestamp())
	{
		vocfname = P::userVocabulary();
		idxfname = P::userVocabularyIndex();
		return true;
	}

	// If the user index is not up to date and we have user sources, we cannot
	// fall back to the system index
	if (ts_user_src != 0)
		return false;
	
	// Fallback to the system index
	if (ts_main_voc >= sourceTimestamp() &&
		ts_main_idx >= sourceTimestamp())
	{
		vocfname = P::vocabulary();
		idxfname = P::vocabularyIndex();
		return true;
	}
	
	return false;
}

template<typename P>
void IndexManager<P>::obtainWorkingVocabulary(std::string& vocfname, std::string& idxfname)
{
	Vocabulary v;

	v.rebuildIfNeeded();
	v.deleteRedundantUserIndex();
	v.getUpToDateVocabulary(vocfname, idxfname);
}






template<typename P> template<typename C>
IndexManager<P>::Tagdb<C>::Tagdb(typename C::Aggregator& agg)
	: agg(agg), mainSource(P::debtagsSourceDir()), userSource(P::debtagsUserSourceDir())
{
	rescan();
}

template<typename P> template<typename C>
void IndexManager<P>::Tagdb<C>::rescan()
{
	PkgIdx<C> pi(agg);
	ts_pkgidx = pi.timestamp();
	ts_main_src = mainSource.timestamp();
	ts_user_src = userSource.timestamp();
	ts_main_tag = P::timestamp(P::tagdb());
	ts_main_idx = P::timestamp(P::tagdbIndex());
	ts_user_tag = P::timestamp(P::userTagdb());
	ts_user_idx = P::timestamp(P::userTagdbIndex());
}

template<typename P> template<typename C>
bool IndexManager<P>::Tagdb<C>::needsRebuild() const
{
	// If there are no indexes of any kind, then we need rebuilding
	if (ts_user_tag == 0 && ts_user_idx == 0 && ts_main_tag == 0 && ts_main_idx == 0)
		return true;

	// If the user index is ok, then we are fine
	if (ts_user_tag >= sourceTimestamp() || ts_user_idx >= sourceTimestamp())
		return false;

	// If there are user sources, then we cannot use the system index
	if (ts_user_src > 0)
		return true;

	// If there are no user sources, then we can fallback on the system
	// indexes in case the user indexes are not up to date
	if (ts_main_tag >= sourceTimestamp() || ts_main_idx >= sourceTimestamp())
		return false;

	return true;
}

template<typename P> template<typename C>
bool IndexManager<P>::Tagdb<C>::userIndexIsRedundant() const
{
	// If there is no user index, then it is not redundant
	if (ts_user_tag == 0 && ts_user_idx == 0)
		return false;

	// If we have user sources, then the user index is never redundant
	if (ts_user_src > 0)
		return false;

	// If the system index is not up to date, then the user index is not
	// redundant
	if (ts_main_tag < sourceTimestamp() || ts_main_idx < sourceTimestamp())
		return false;

	return true;
}

template<typename P> template<typename C>
void IndexManager<P>::Tagdb<C>::rebuild(const std::string& tagfname, const std::string& idxfname)
{
	using namespace tagcoll;

	typedef typename C::Package Package;
	typedef typename C::Tag Tag;

	diskindex::MasterMMapIndexer master(idxfname);

	// Read and merge tag data
	coll::IntDiskIndexer tagindexer;
	coll::Simple<Package, Tag> mergedCopy;

	mainSource.readTags(stringToEpt<C>(agg, agg.vocabulary(), stream::teeFilter(toInt(inserter(tagindexer)), inserter(mergedCopy))));
	userSource.readTags(stringToEpt<C>(agg, agg.vocabulary(), stream::teeFilter(toInt(inserter(tagindexer)), inserter(mergedCopy))));

	if (mergedCopy.empty())
		throw wibble::exception::Consistency("Reading debtags sources from " + P::debtagsSourceDir() + " and " + P::debtagsUserSourceDir(), "Unable to find any tag data");

	// 3: pkg->tag
	master.append(tagindexer.pkgIndexer());
	// 4: tag->pkg
	master.append(tagindexer.tagIndexer());

	// Write the tag database in text format
	std::string tmpdb = tagfname + ".tmp";
	FILE* out = fopen(tmpdb.c_str(), "wt");
	if (!out) throw wibble::exception::File(tmpdb, "creating temporary copy of tag index");
	mergedCopy.output(toString(textformat::StdioWriter(out)));
	fclose(out);

	// Perform "atomic" update of the tag database
	// FIXME: cannot be atomic because race conditions happening between file
	// renames
	if (rename(tmpdb.c_str(), tagfname.c_str()) == -1)
		throw wibble::exception::System("Renaming " + tmpdb + " to " + tagfname);

	master.commit();
}

template<typename P> template<typename C>
bool IndexManager<P>::Tagdb<C>::rebuildIfNeeded()
{
	if (needsRebuild())
	{
		// Decide if we rebuild the user index or the system index

		if (ts_user_src == 0 && P::access(P::debtagsIndexDir(), W_OK) == 0)
		{
			// There are no user sources and we can write to the system index
			// directory: rebuild the system index
			rebuild(P::tagdb(), P::tagdbIndex());
			ts_main_tag = P::timestamp(P::tagdb());
			ts_main_idx = P::timestamp(P::tagdbIndex());
		} else {
			wibble::sys::fs::mkFilePath(P::userTagdb());
			wibble::sys::fs::mkFilePath(P::userTagdbIndex());
			rebuild(P::userTagdb(), P::userTagdbIndex());
			ts_user_tag = P::timestamp(P::userTagdb());
			ts_user_idx = P::timestamp(P::userTagdbIndex());
		}
		return true;
	}
	return false;
}

template<typename P> template<typename C>
bool IndexManager<P>::Tagdb<C>::deleteRedundantUserIndex()
{
	if (userIndexIsRedundant())
	{
		// Delete the user indexes if they exist
		unlink(P::userTagdb().c_str());
		unlink(P::userTagdbIndex().c_str());
		ts_user_tag = 0;
		ts_user_idx = 0;
		return true;
	}
	return false;
}

template<typename P> template<typename C>
bool IndexManager<P>::Tagdb<C>::getUpToDateTagdb(std::string& tagfname, std::string& idxfname)
{
	// If there are no indexes of any kind, then we have nothing to return
	if (ts_user_tag == 0 && ts_user_idx == 0 && ts_main_tag == 0 && ts_main_idx == 0)
		return false;

	// If the user index is up to date, use it
	if (ts_user_tag >= sourceTimestamp() &&
		ts_user_idx >= sourceTimestamp())
	{
		tagfname = P::userTagdb();
		idxfname = P::userTagdbIndex();
		return true;
	}

	// If the user index is not up to date and we have user sources, we cannot
	// fall back to the system index
	if (ts_user_src != 0)
		return false;
	
	// Fallback to the system index
	if (ts_main_tag >= sourceTimestamp() &&
		ts_main_idx >= sourceTimestamp())
	{
		tagfname = P::tagdb();
		idxfname = P::tagdbIndex();
		return true;
	}
	
	return false;
}



template<typename P> template<typename C>
void IndexManager<P>::obtainWorkingTagdb(typename C::Aggregator& ag, std::string& tagfname, std::string& idxfname)
{
	Tagdb<C> t(ag);

	t.rebuildIfNeeded();
	t.deleteRedundantUserIndex();
	t.getUpToDateTagdb(tagfname, idxfname);
}

template<typename C>
class Generator : public tagcoll::diskindex::MMapIndexer
{
	typedef typename C::Aggregator Aggregator;
	typedef typename C::Index Index;
	Aggregator& m_pkgs;

public:
	Generator(Aggregator& p) : m_pkgs(p) {}

	int encodedSize() const // currently packageCount() + 1 hardcoded,
							// FIXME
	{
		int size = ( m_pkgs.index().packageCount() + 1 ) * sizeof(int);
		typename Index::iterator e = m_pkgs.index().end();
		for (typename Index::iterator i = m_pkgs.index().begin(); i != e; ++i)
			size += i->name().size() + 1;
		size += std::string( "-invalid-" ).size() + 1; // for invalid id
		return size;
	}

	void encode(char* buf) const // currently packageCount() + 1 hardcoded,
                                 // FIXME
	{
		typedef typename ept::t::cache::Package<C> Package;

		// First sort the packages by ID
        std::vector<Package> pkgs;
		pkgs.resize(m_pkgs.index().packageCount() + 1);
		for (typename Index::iterator i = m_pkgs.index().begin(); i != m_pkgs.index().end(); ++i)
			pkgs[i->ondiskId()] = *i;

		// Then write them out
		int pos = pkgs.size() * sizeof(int);
		int idx = 0;
		for (typename std::vector<Package>::const_iterator i = pkgs.begin(); i != pkgs.end(); ++i)
		{
			((int*)buf)[idx++] = pos;
			memcpy(buf + pos, i->name( std::string( "invalid" ) ).c_str(),
                   i->name( std::string( "invalid" ) ).size() + 1);
			pos += i->name( std::string( "invalid" ) ).size() + 1;
		}
	}
};





template<typename P> template<typename C>
IndexManager<P>::Pkgidx<C>::Pkgidx(typename C::Aggregator& agg)
	: agg(agg)
{
	rescan();
}

template<typename P> template<typename C>
void IndexManager<P>::Pkgidx<C>::rescan()
{
	ts_pkgs = agg.index().timestamp();
	ts_main_idx = P::timestamp(P::pkgidx());
	ts_user_idx = P::timestamp(P::userPkgidx());
}

template<typename P> template<typename C>
bool IndexManager<P>::Pkgidx<C>::needsRebuild() const
{
	// If there are no indexes of any kind, then we need rebuilding
	if (ts_user_idx == 0 && ts_main_idx == 0)
		return true;

	// If the user index is ok, then we are fine
	if (ts_user_idx >= sourceTimestamp())
		return false;

	// Else we can fallback on the system indexes in case the user indexes are
	// not up to date
	if (ts_main_idx >= sourceTimestamp())
		return false;

	return true;
}

template<typename P> template<typename C>
bool IndexManager<P>::Pkgidx<C>::userIndexIsRedundant() const
{
	// If there is no user index, then it is not redundant
	if (ts_user_idx == 0)
		return false;

	// If the system index is not up to date, then the user index is not
	// redundant
	if (ts_main_idx < sourceTimestamp())
		return false;

	return true;
}

template<typename P> template<typename C>
void IndexManager<P>::Pkgidx<C>::rebuild(const std::string& idxfname)
{
	tagcoll::diskindex::MasterMMapIndexer master(idxfname);

	Generator<C> gen(agg);
	master.append(gen);

	master.commit();
}

template<typename P> template<typename C>
bool IndexManager<P>::Pkgidx<C>::rebuildIfNeeded()
{
	if (needsRebuild())
	{
		// Decide if we rebuild the user index or the system index
		if (P::access(P::debtagsIndexDir(), W_OK) == 0)
		{
			// Since we can write on the system index directory, we rebuild
			// the system index
			rebuild(P::pkgidx());
			ts_main_idx = P::timestamp(P::pkgidx());
		} else {
			wibble::sys::fs::mkFilePath(P::userPkgidx());
			rebuild(P::userPkgidx());
			ts_user_idx = P::timestamp(P::userPkgidx());
		}
		return true;
	}
	return false;
}

template<typename P> template<typename C>
bool IndexManager<P>::Pkgidx<C>::deleteRedundantUserIndex()
{
	if (userIndexIsRedundant())
	{
		// Delete the user indexes if they exist
		unlink(P::userPkgidx().c_str());
		ts_user_idx = 0;
		return true;
	}
	return false;
}

template<typename P> template<typename C>
bool IndexManager<P>::Pkgidx<C>::getUpToDatePkgidx(std::string& idxfname)
{
	// If there are no indexes of any kind, then we have nothing to return
	if (ts_user_idx == 0 && ts_main_idx == 0)
		return false;

	// If the user index is up to date, use it
	if (ts_user_idx >= sourceTimestamp())
	{
		idxfname = P::userPkgidx();
		return true;
	}

	// Fallback to the system index
	if (ts_main_idx >= sourceTimestamp())
	{
		idxfname = P::pkgidx();
		return true;
	}
	
	return false;
}





















template<typename P> template<typename C>
void IndexManager<P>::obtainWorkingPkgidx(typename C::Aggregator& ag, std::string& idxfname)
{
	Pkgidx<C> t(ag);

	t.rebuildIfNeeded();
	t.deleteRedundantUserIndex();
	t.getUpToDatePkgidx(idxfname);
}


}
}
}
}

#include <ept/cache/debtags/serializer.tcc>
#include <ept/cache/apt/records.tcc>

#endif

// vim:set ts=4 sw=4:
