/*
 * Copyright (C) 2007  Enrico Zini <enrico@enricozini.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <tests/test-utils.h>
#include <ept/apt/recordparser.h>

//#include <iostream>

namespace tut {
using namespace std;
using namespace ept;
using namespace ept::apt;

struct ept_apt_recordparser_shar {
	std::string record;
	ept_apt_recordparser_shar()
	{
		record =
			"A:\n"
			"D: da de di do du\n"
			"B: b\n"
			"C: c \n"
			"Desc: this is the beginning\n"
			" this is the continuation\n"
			" this is the end\n";
	}
};
TESTGRP(ept_apt_recordparser);

// Check that the fields are identified and broken up correctly
template<> template<>
void to::test<1>()
{
	RecordParser p(record);

	ensure_equals(p.record(), record);
	ensure_equals(p.size(), 5u);
}

template<> template<>
void to::test<2>()
{
	RecordParser p(record);
	ensure_equals(p.field(0), "A:\n");
	ensure_equals(p.field(1), "D: da de di do du\n");
	ensure_equals(p.field(2), "B: b\n");
	ensure_equals(p.field(3), "C: c \n");
	ensure_equals(p.field(4), "Desc: this is the beginning\n this is the continuation\n this is the end\n");
}

template<> template<>
void to::test<3>()
{
	RecordParser p(record);
	ensure_equals(p.name(0), "A");
	ensure_equals(p.name(1), "D");
	ensure_equals(p.name(2), "B");
	ensure_equals(p.name(3), "C");
	ensure_equals(p.name(4), "Desc");
}

template<> template<>
void to::test<4>()
{
	RecordParser p(record);
	ensure_equals(p[0], "");
	ensure_equals(p[1], "da de di do du");
	ensure_equals(p[2], "b");
	ensure_equals(p[3], "c");
	ensure_equals(p[4], "this is the beginning\n this is the continuation\n this is the end");
}

// Check that the field search by name finds all the fields
template<> template<>
void to::test<5>()
{
	RecordParser p(record);

	ensure_equals(p.index("A"), 0u);
	ensure_equals(p.index("D"), 1u);
	ensure_equals(p.index("B"), 2u);
	ensure_equals(p.index("C"), 3u);
	ensure_equals(p.index("Desc"), 4u);

	ensure_equals(p.name(p.index("A")), "A");
	ensure_equals(p.name(p.index("B")), "B");
	ensure_equals(p.name(p.index("C")), "C");
	ensure_equals(p.name(p.index("D")), "D");
	ensure_equals(p.name(p.index("Desc")), "Desc");
}

template<> template<>
void to::test<6>()
{
	RecordParser p(record);
	ensure_equals(p["A"], "");
	ensure_equals(p["B"], "b");
	ensure_equals(p["C"], "c");
	ensure_equals(p["D"], "da de di do du");
	ensure_equals(p["Desc"], "this is the beginning\n this is the continuation\n this is the end");
}

template<> template<>
void to::test<7>()
{
	RecordParser p(record);
	// Missing fields give empty strings
	ensure_equals(p.field(100), "");
	ensure_equals(p.name(100), "");
	ensure_equals(p[100], "");
	ensure_equals(p["Missing"], "");
}

// Check that scanning twice replaces the old fields
template<> template<>
void to::test<8>()
{
	std::string record =
		"A: a\n"
		"B: b\n"
		"C: c\n";

	RecordParser p(record);
	ensure_equals(p.size(), 3u);
	ensure_equals(p["A"], "a");
	ensure_equals(p["B"], "b");
	ensure_equals(p["C"], "c");

	std::string record1 =
		"Foo: bar\n"
		"A: different\n";

	p.scan(record1);

	//for (size_t i = 0; i < p.size(); ++i)
	//	cerr << ">> " << i << "==" << p.index(p.name(i)) << " " << p.name(i) << " " << p[i] << endl;

	ensure_equals(p.size(), 2u);
	ensure_equals(p["A"], "different");
	ensure_equals(p["B"], "");
	ensure_equals(p["C"], "");
	ensure_equals(p["Foo"], "bar");
}

// Real-life example
template<> template<>
void to::test<9>()
{
	string record = 
		"Package: apt\n"
		"Priority: important\n"
		"Section: admin\n"
		"Installed-Size: 4368\n"
		"Maintainer: APT Development Team <deity@lists.debian.org>\n"
		"Architecture: amd64\n"
		"Version: 0.6.46.4-0.1\n"
		"Replaces: libapt-pkg-doc (<< 0.3.7), libapt-pkg-dev (<< 0.3.7)\n"
		"Provides: libapt-pkg-libc6.3-6-3.11\n"
		"Depends: libc6 (>= 2.3.5-1), libgcc1 (>= 1:4.1.1-12), libstdc++6 (>= 4.1.1-12), debian-archive-keyring\n"
		"Suggests: aptitude | synaptic | gnome-apt | wajig, dpkg-dev, apt-doc, bzip2\n"
		"Filename: pool/main/a/apt/apt_0.6.46.4-0.1_amd64.deb\n"
		"Size: 1436478\n"
		"MD5sum: 1776421f80d6300c77a608e77a9f4a15\n"
		"SHA1: 1bd7337d2df56d267632cf72ac930c0a4895898f\n"
		"SHA256: b92442ab60046b4d0728245f39cc932f26e17db9f7933a5ec9aaa63172f51fda\n"
		"Description: Advanced front-end for dpkg\n"
		" This is Debian's next generation front-end for the dpkg package manager.\n"
		" It provides the apt-get utility and APT dselect method that provides a\n"
		" simpler, safer way to install and upgrade packages.\n"
		" .\n"
		" APT features complete installation ordering, multiple source capability\n"
		" and several other unique features, see the Users Guide in apt-doc.\n"
		"Build-Essential: yes\n"
		"Tag: admin::package-management, filetransfer::ftp, filetransfer::http, hardware::storage:cd, interface::commandline, network::client, protocol::{ftp,http,ipv6}, role::program, suite::debian, use::downloading, use::searching, works-with::software:package\n";
	RecordParser p(record);

	ensure_equals(p.size(), 19u);

	string rec1;
	for (size_t i = 0; i < p.size(); ++i)
		rec1 += p.field(i);
	ensure_equals(record, rec1);
}

// Various buffer termination patterns
template<> template<>
void to::test<10>()
{
	std::string record =
		"A: a\n"
		"B: b";

	RecordParser p(record);
	ensure_equals(p.size(), 2u);
	ensure_equals(p["A"], "a");
	ensure_equals(p["B"], "b");
}

template<> template<>
void to::test<11>()
{
	std::string record =
		"A: a\n"
		"B: b\n\n";

	RecordParser p(record);
	ensure_equals(p.size(), 2u);
	ensure_equals(p["A"], "a");
	ensure_equals(p["B"], "b");
}

template<> template<>
void to::test<12>()
{
	std::string record =
		"A: a\n"
		"B: b\n\n"
		"C: c\n";

	RecordParser p(record);
	ensure_equals(p.size(), 2u);
	ensure_equals(p["A"], "a");
	ensure_equals(p["B"], "b");
}

}

// vim:set ts=4 sw=4:
