/*
 * Tag vocabulary access
 *
 * Copyright (C) 2003--2007  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <ept/debtags/vocabulary.h>
#include <ept/debtags/maint/vocabularymerger.h>
#include <ept/debtags/maint/path.h>
#include <ept/tests/test-utils.h>
#include <tagcoll/utils/set.h>
#include <tagcoll/input/stdio.h>

// This is not exported by default
namespace ept {
namespace debtags {
int tagcmp(const char* tag1, const char* tag2);
}
}

namespace tut {

using namespace std;
using namespace tagcoll::utils;
using namespace ept::debtags;

struct ept_debtags_vocabulary_shar {
	Path::OverrideDebtagsSourceDir odsd;
	Path::OverrideDebtagsIndexDir odid;
	Path::OverrideDebtagsUserSourceDir odusd;
	Path::OverrideDebtagsUserIndexDir oduid;
	Vocabulary  m_tags;

    ept_debtags_vocabulary_shar()
		: odsd("./"), odid("./"), odusd("./"), oduid("./"), m_tags() {}

	Vocabulary& tags() { return m_tags; }
};
TESTGRP(ept_debtags_vocabulary);

template<> template<>
void to::test< 1 >()
{
    tags(); // this will throw if the open above didn't work
}

template<> template<>
void to::test< 2 >()
{
    ensure( tags().hasFacet( "works-with" ) );
    ensure( !tags().hasFacet( "blah" ) );
}

template<> template<>
void to::test< 3 >()
{
    ensure( tags().hasTag( "works-with::people" ) );
    ensure( !tags().hasTag( "works-with::midgets" ) );
}

template<> template<>
void to::test< 4 >()
{
    Tag people = tags().tagByName( "works-with::people" ),
                midgets = tags().tagByName( "works-with::midgets" ),
                blahg = tags().tagByName( "works-with::blahg" ),
                text = tags().tagByName( "works-with::text" ),
                people2 = tags().tagByName( "works-with::people" );
    ensure( people != midgets );
    ensure( people != text );
    ensure( people != blahg );
    ensure( midgets == blahg );
    ensure( midgets == midgets );
    ensure( people == people2 );
    ensure( people == people );
}

template<> template<>
void to::test< 5 >()
{
    Tag a = tags().tagByName( "works-with::people" ),
                b = tags().tagByName( "works-with::midgets" );
	std::set< Tag > s = tags().tags(),
                         f = tags().tags( "works-with" ),
                         n = tags().tags( "nonsense" );
    ensure( set_contains(s, a) );
    ensure( set_contains(f, a) );
    ensure( set_contains(s, f) );
    ensure( !set_contains(s, b) );
    ensure( !set_contains(f, b) );
    ensure( n.empty() );
}

template<> template<>
void to::test< 6 >()
{
	Facet f = tags().facetByName( "works-with" );
    Tag t = tags().tagByName( "works-with::people" );
	ensure_equals(f.name(), "works-with");
	ensure_equals(t.name(), "people");
	ensure_equals(t.fullname(), "works-with::people");
}

template<> template<>
void to::test< 7 >()
{
    Facet f = tags().facetByName( "works-with" );
	std::set< Tag > x = tags().tags( "works-with" );
    ensure( x == f.tags() );
}

template<> template<>
void to::test< 8 >()
{
    Facet f = tags().facetByName( "does-not-work-with" );
    int x = 1;
    try {
        f.tags();
        x = 2;
    } catch (...) {
        x = 3;
    }
    ensure_equals( x, 3 );
}

template<> template<>
void to::test< 9 >()
{
    Facet f = tags().facetByName( "legacy" );
    ensure_equals(f.shortDescription(), "");
    ensure_equals(f.longDescription(), "");
    //ensure_equals(f.shortDescription( "weehee" ), "weehee");
}

template<> template<>
void to::test< 10 >()
{
	// ensure that one-character tag names are parsed correctly
	ensure( tags().hasTag( "implemented-in::c" ) );
}

template<> template<>
void to::test< 11 >()
{
	// ensure that all tags are somehow working
	std::set<Facet> facets = tags().facets();

	for (std::set<Facet>::const_iterator i = facets.begin();
			i != facets.end(); i++)
	{
		i->name(string("foo"));
		i->shortDescription(string("foo"));
		i->longDescription(string("foo"));
		i->tags();
	}
}

template<> template<>
void to::test< 12 >()
{
	// ensure that all tags are somehow working
	std::set<Tag> tags = this->tags().tags();

	for (std::set<Tag>::const_iterator i = tags.begin();
			i != tags.end(); i++)
	{
		i->name(string("foo"));
		i->fullname(string("foo"));
		i->shortDescription(string("foo"));
		i->longDescription(string("foo"));
	}
}

// Check for correctness of the first and last tag in the vocabulary
template<> template<>
void to::test< 13 >()
{
	Vocabulary& tags = this->tags();

	Tag first = tags.tagByName("accessibility::TODO");
	ensure(first != Tag());
	ensure_equals(first.fullname(), string("accessibility::TODO"));
	ensure_equals(first.name(), string("TODO"));
	ensure_equals(first.shortDescription(), string("Need an extra tag"));

	Tag last = tags.tagByName("x11::xserver");
	ensure(last != Tag());
	ensure_equals(last.fullname(), string("x11::xserver"));
	ensure_equals(last.name(), string("xserver"));
	ensure_equals(last.shortDescription(), string("X Server"));
}

template<> template<>
void to::test< 14 >()
{
	// ensure that it's possible to go from facet to ID and back
	std::set<Facet> facets = tags().facets();

	for (std::set<Facet>::const_iterator i = facets.begin();
			i != facets.end(); i++)
	{
		Facet f = tags().facetByID(i->id());
		ensure_equals(*i, f);
		ensure_equals(i->name(), f.name());
		ensure_equals(i->shortDescription(), f.shortDescription());
		ensure_equals(i->longDescription(), f.longDescription());
		ensure_equals(i->tags().size(), f.tags().size());
	}
}

template<> template<>
void to::test< 15 >()
{
	// ensure that it's possible to go from tag to ID and back
	std::set<Tag> tags = this->tags().tags();

	for (std::set<Tag>::const_iterator i = tags.begin();
			i != tags.end(); i++)
	{
		Tag t = this->tags().tagByID(i->id());
		ensure_equals(*i, t);
		ensure_equals(i->name(), t.name());
		ensure_equals(i->fullname(), t.fullname());
		ensure_equals(i->shortDescription(), t.shortDescription());
		ensure_equals(i->longDescription(), t.longDescription());
	}
}

template<> template<>
void to::test< 16 >()
{
	// ensure that facet IDs are distinct
	std::set<Facet> facets = tags().facets();
	std::set<int> ids;
	for (std::set<Facet>::const_iterator i = facets.begin();
			i != facets.end(); i++)
		ids.insert(i->id());

	ensure_equals(facets.size(), ids.size());
}

template<> template<>
void to::test< 17 >()
{
	// ensure that tag IDs are distinct
	std::set<Tag> tags = this->tags().tags();
	std::set<int> ids;
	for (std::set<Tag>::const_iterator i = tags.begin();
			i != tags.end(); i++)
		ids.insert(i->id());

	ensure_equals(tags.size(), ids.size());
}

template<> template<>
void to::test< 18 >()
{
	// ensure that all the tags are indexed
	ept::debtags::VocabularyMerger voc;
	tagcoll::input::Stdio in(ept::debtags::Path::vocabulary());
	voc.read(in);
	std::set<std::string> all = voc.tagNames();
	for (std::set<std::string>::const_iterator i = all.begin();
			i != all.end(); ++i)
		ensure(this->tags().hasTag(*i));

	// There should be the same amount of tags in both
	std::set<Tag> allTags = this->tags().tags();
	ensure_equals(all.size(), allTags.size());
}

template<> template<>
void to::test< 19 >()
{
	// test the tagcmp function

	// If unfaceted, same as strcmp
	ensure(ept::debtags::tagcmp("antani", "blinda") < 0);
	ensure(ept::debtags::tagcmp("blinda", "antani") > 0);
	ensure_equals(ept::debtags::tagcmp("antani", "antani"), 0);

	// If the same and faceted, should work
	ensure_equals(ept::debtags::tagcmp("antani::blinda", "antani::blinda"), 0);

	// With different facet names, work just as strcmp
	ensure(ept::debtags::tagcmp("antani::blinda", "blinda::blinda") < 0);
	ensure(ept::debtags::tagcmp("blinda::blinda", "antani::blinda") > 0);
	ensure(ept::debtags::tagcmp("anta::blinda", "antani::blinda") < 0);
	ensure(ept::debtags::tagcmp("antani::blinda", "anta::blinda") > 0);
	ensure(ept::debtags::tagcmp("anta::blinda", "anta-ni::blinda") < 0);
	ensure(ept::debtags::tagcmp("anta-ni::blinda", "anta::blinda") > 0);

	// With same facet names, work just as strcmp on the tags
	ensure(ept::debtags::tagcmp("a::antani", "a::blinda") < 0);
	ensure(ept::debtags::tagcmp("a::blinda", "a::antani") > 0);
	ensure(ept::debtags::tagcmp("a::anta", "a::antani") < 0);
	ensure(ept::debtags::tagcmp("a::antani", "a::anta") > 0);
	ensure(ept::debtags::tagcmp("a::anta", "a::anta-ni") < 0);
	ensure(ept::debtags::tagcmp("a::anta-ni", "a::anta") > 0);
}

template<> template<>
void to::test< 20 >()
{
	// check that we're seeing all the tags for a facet
	std::set<Tag> t = tags().tags("accessibility");
	ensure_equals(t.size(), 10u);

	t = tags().tags("works-with-format");
	ensure_equals(t.size(), 33u);
}

// If there is no data, Vocabulary should work as an empty vocabulary
template<> template<>
void to::test< 21 >()
{
	Path::OverrideDebtagsSourceDir odsd("./empty");
	Path::OverrideDebtagsIndexDir odid("./empty");
	Path::OverrideDebtagsUserSourceDir odusd("./empty");
	Path::OverrideDebtagsUserIndexDir oduid("./empty");
	Vocabulary empty;

	ensure(!empty.hasData());

	set<Facet> facets = empty.facets();
	ensure_equals(facets.size(), 0u);

	set<Tag> tags = empty.tags();
	ensure_equals(tags.size(), 0u);
}

}

// vim:set ts=4 sw=4:
