// -*- C++ -*-
#include <xapian.h>
#include <ept/core/apt.h>

#ifndef EPT_XAPIAN_H
#define EPT_XAPIAN_H

namespace ept {
namespace core {
namespace xapian {

// Allocate value indexes for known values
const Xapian::valueno VAL_APT_INSTALLED_SIZE      =  1;
const Xapian::valueno VAL_APT_PACKAGE_SIZE        =  2;
const Xapian::valueno VAL_POPCON                  = 10;
const Xapian::valueno VAL_ITERATING_RATING        = 20;
const Xapian::valueno VAL_ITERATING_FUNCTIONALITY = 21;
const Xapian::valueno VAL_ITERATING_USABILITY     = 22;
const Xapian::valueno VAL_ITERATING_SECURITY      = 23;
const Xapian::valueno VAL_ITERATING_PERFORMANCE   = 24;
const Xapian::valueno VAL_ITERATING_QUALITY       = 25;
const Xapian::valueno VAL_ITERATING_SUPPORT       = 26;
const Xapian::valueno VAL_ITERATING_ADOPTION      = 27;

struct List {
    char m_enqPlace[sizeof(Xapian::Enquire)];
    Xapian::MSet m_matches;
    Xapian::MSet::const_iterator m_iter;
    int m_pos;
    typedef List Type;

    static const int chunkSize = 20;

    List head() {
        seek();
        return *this;
    }

    Token token() {
        Token t;
        t._id = m_iter.get_document().get_data();
        return t;
    }

    void seek() {
        if ( m_matches.size() == chunkSize && m_iter == m_matches.end() ) {
            m_matches = enq().get_mset( m_pos, chunkSize );
            m_iter = m_matches.begin();
            m_pos += chunkSize;
        }
    }

    bool empty() {
        if ( m_pos == -1 )
            return true;
        seek();
        return m_matches.size() < 30 && m_iter == m_matches.end();
    }

    List tail() {
        List t = *this;
        t.seek();
        t.m_iter ++;
        return t;
    }

    Xapian::Enquire &enq() {
        return *reinterpret_cast< Xapian::Enquire * >( m_enqPlace );
    }

    List( Xapian::Enquire _enq )
    {
        Xapian::Enquire *e = new (m_enqPlace) Xapian::Enquire( _enq );
        assert_eq( e, &enq() );
        m_matches = enq().get_mset( 0, chunkSize );
        m_iter = m_matches.begin();
        m_pos = chunkSize;
    }

    List() {}
};

struct Source
{
protected:
    time_t m_timestamp;
    Xapian::Database m_db;
    Xapian::Stem m_stem;

    /// Return a lowercased copy of the string
    static std::string toLower(const std::string& str);

    /**
     * Add normalised tokens computed from the string to the document doc.
     *
     * pos is used as a sequence generator for entering the token position in
     * the document.
     */
    void normalize_and_add(Xapian::Document& doc, const std::string& term,
                           int& pos) const;

public:
    Source();

    /// Access the Xapian database
    Xapian::Database& db() { return m_db; }

    /// Access the Xapian database
    const Xapian::Database& db() const { return m_db; }

    /// Timestamp of when the Xapian database was last updated
    time_t timestamp() const { return m_timestamp; }

    /// Returns true if the index has data
    bool hasData() const { return m_timestamp > 0; }

    List query( const std::string &s, int qualityCutoff = 50 ) {
        Xapian::Enquire enq( db() );
        enq.set_query( makeORQuery( s ) );
        Xapian::MSet first = enq.get_mset(0, 1, 0, 0, 0);
        Xapian::MSetIterator ifirst = first.begin();
        if ( ifirst != first.end() ) {
            Xapian::percent cutoff = ifirst.get_percent() * qualityCutoff / 100;
            enq.set_cutoff(cutoff);
        }
        return List( enq );
    }

    List partialQuery( const std::string &s ) {
        Xapian::Enquire enq( db() );
        enq.set_query( makePartialORQuery( s ) );
        return List( enq );
    }

    /// Returns true if the index is older than the Apt database information
    // bool needsRebuild(apt::Apt& apt);

    Xapian::docid docidByName(const std::string& pkgname) const;

    /**
     * Tokenize the string and build an OR query with the resulting keywords
     */
    Xapian::Query makeORQuery(const std::string& keywords) const;

    /**
     * Tokenize the string and build an OR query with the resulting keywords.
     *
     * The last token in keywords is considered to be typed only partially, to
     * implement proper search-as-you-type.
     */
    Xapian::Query makePartialORQuery(const std::string& keywords) const;

    /**
     * Build a query with the given keywords, specified as iterators of strings
     */
    template<typename ITER>
    Xapian::Query makeORQuery(const ITER& begin, const ITER& end) const
    {
        std::vector<std::string> terms;
        // Insert both the lowercased and the stemmed lowercased query terms
        for (ITER i = begin; i != end; ++i)
        {
            std::string t = toLower(*i);
            std::string s = m_stem(t);
            terms.push_back(t);
            if (s != t)
                terms.push_back(s);
        }
        return Xapian::Query(Xapian::Query::OP_OR, terms.begin(), terms.end());
    }

    /// Return a list of tag-based terms that can be used to expand an OR query
    std::vector<std::string> expand(Xapian::Enquire& enq) const;

//	std::vector<std::string> similar(const std::string& pkg);

    /**
     * Create a query to look for packages similar to the given one
     */
    Xapian::Query makeRelatedQuery(const std::string& pkgname) const;

    /**
     * Get the integer value for
     */
    double getDoubleValue(const std::string& pkgname,
                          Xapian::valueno val_id) const;

    /**
     * Get the integer value for
     */
    int getIntValue(const std::string& pkgname, Xapian::valueno val_id) const;
};

}
}
}

#endif
