/*
 * Copyright(C) 2010 Neil Jagdish Patel
 * Copyright(C) 2010 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 * Authored by Ken VaDine <ken.vandine@canonical.com>
 */

[DBus (name = "com.Gwibber.Accounts")]
private interface AccountsInterface : Object {
	public signal void Updated (string account);
	public signal void Created (string account);
	public signal void Deleted (string account);
	public abstract string List () throws DBus.Error;
	public abstract string Get (string id) throws DBus.Error;
	public abstract void SendEnabled (string id) throws DBus.Error;
}

namespace Gwibber
{


    public class Accounts : Object
    {
        private const string accounts_name  = "com.Gwibber.Accounts";
        private const string accounts_path  = "/com/gwibber/Accounts";
        private const string accounts_iface = "com.Gwibber.Accounts";

        private AccountsInterface accounts_service;
        private HashTable<string,HashTable> accounts_table;
        private HashTable<string,Value?> account_table;
	private Gwibber.Utils utils;

	/** 
            Accounts::is_available:
            @arg0: The current state

            Emitted when com.Gwibber.Accounts availability state changes
	*/
	public signal void is_available(bool is_up);

        /** 
            Accounts::account_updated:
            @arg0: HashTable of an account

            Emitted when an account has changed
        */
        public signal void account_updated(HashTable<string,Value?> account_table);


	public Accounts()
	{
            try
            {
                var dbus_conn = DBus.Bus.get(DBus.BusType.SESSION);

                accounts_service = (AccountsInterface) dbus_conn.get_object(this.accounts_name,
                                                           this.accounts_path,
                                                           this.accounts_iface);

		
		this.utils = new Gwibber.Utils();
		this.utils.setup(this.accounts_name);
		this.utils.available.connect(this.accounts_available);

		accounts_service.Updated.connect((source) => {
			this.updated (source);});

            }
            catch(GLib.Error e)
            {
                warning ("Unable to get Gwibber accounts");
            }
        }

        public GLib.HashTable? list()
        {
            accounts_table = new HashTable<string,HashTable>.full(str_hash, str_equal, g_free, GLib.HashTable.remove_all);
            try {
                string accounts = accounts_service.List();
                var parser = new Json.Parser();
                try {
                    parser.load_from_data(accounts, -1);
                    Json.Array nodeArray = parser.get_root().get_array();
                    for(int i = 0; i < nodeArray.get_length(); i++) {
                        string id = nodeArray.get_element(i).get_object().get_string_member("id");
                        account_table = new HashTable<string,Value?>(str_hash, str_equal);
                        foreach(var member in nodeArray.get_element(i).get_object().get_members()) {
                            if((member != "_conflicts") &&(member != "_deleted_conflicts")) {
                                GLib.Value value;
                                if(nodeArray.get_element(i).get_object().get_member(member).is_null()) {
                                    continue;
                                } else if(nodeArray.get_element(i).get_object().get_member(member).type_name() == "gboolean") {
                                    value = nodeArray.get_element(i).get_object().get_member(member).get_boolean();
                                } else {
                                    nodeArray.get_element(i).get_object().get_member(member).get_value(out value);
                                }
                                account_table.insert(member, value);
                            }
                        }
                        accounts_table.insert(id, account_table);
                    }
                    return accounts_table;
                } catch (GLib.Error e) {
                    warning (e.message);
                    return null;
                }
            } catch (DBus.Error e) {
                warning (e.message);
                return null;
            }
        }

        public GLib.HashTable? lookup_by_id (string id)
        {
            try {
                string account = accounts_service.Get(id);
                account_table = new HashTable<string,Value?>(str_hash, str_equal);
                var parser = new Json.Parser();
                try {
                    parser.load_from_data(account, -1);
                    var root_object = parser.get_root().get_object();
                    foreach(var param in root_object.get_members()) {
                        GLib.Value? value = null;
                        if(root_object.get_member(param).is_null()) {
                            continue;
                        } else if(root_object.get_member(param).type_name() == "gboolean") {
                            value = root_object.get_member(param).get_boolean();
                        } else {
                            root_object.get_member(param).get_value(out value);
                        }
                        account_table.insert(param, value);
                    }
                    return account_table;
                } catch (GLib.Error e) {
                    warning (e.message);
                    return null;
                }
            } catch(DBus.Error e) {
                warning (e.message);
                return null;
            }
        }


        /**
            send_enabled:

            Toggle the value of send_enabled for an account.
        */
        public void send_enabled(string id)
        {
            try {
                this.accounts_service.SendEnabled(id);
            } catch(DBus.Error e) {
                warning (e.message);
            }
        }

	public void updated(string account_in)
	{
            string account = (string)account_in;
            try {
                account_table = new HashTable<string,Value?>(str_hash, str_equal);
                var parser = new Json.Parser();
                parser.load_from_data(account, -1);
                var root_object = parser.get_root().get_object();
                foreach(var param in root_object.get_members()) {
                    GLib.Value? value = null;
                    if(root_object.get_member(param).is_null()) {
                        continue;
                    } else if(root_object.get_member(param).type_name() == "gboolean") {
                        value = root_object.get_member(param).get_boolean();
                    } else {
                        root_object.get_member(param).get_value(out value);
                    }
                    account_table.insert(param, value);
                }
                this.account_updated(account_table);
            } catch(GLib.Error e) {
                warning (e.message);
            }
        }

	public void accounts_available(bool is_up)
	{
		this.is_available(is_up);
	}

    }
}
