/*
 * Copyright (C) 2010 Neil Jagdish Patel
 * Copyright(C) 2010 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 * Authored by Ken VaDine <ken.vandine@canonical.com>
 */

namespace GwibberGtk
{
    public class Entry : Gtk.VBox
    {
        public Gwibber.Service service;

        private InputTextView text_view;
        private AccountTargetBar target_bar;

        public Entry ()
        {
            Object ();
        }

        construct
        {
            service = new Gwibber.Service ();

            this.text_view = new InputTextView (service);
            this.target_bar = new AccountTargetBar ();
            this.add (this.text_view);
            this.add_with_properties(this.target_bar,"expand", false);
            this.text_view.show ();
            this.text_view.submit.connect((source) => {
                on_send(this.text_view.buffer.text);});
            this.target_bar.show ();
            this.target_bar.send.clicked.connect((source) => {
                this.on_send(this.text_view.buffer.text);});
            // Stick a couple pixels at the bottom to keep the buttons from hitting the edge
            var spacer = new Gtk.HBox(false, 0);
            this.add_with_properties (spacer, "padding", 2, "expand", false);
        }

        private void on_send(string msg)
        {
            service.send_message(msg);
            debug ("Message: %s posted", msg);
            this.text_view.clear();
        }

    }

    public class SimpleEntry : Gtk.VBox
    {
        public Gwibber.Service service;

        private InputTextView text_view;
        private SendBar send_bar;

        public SimpleEntry ()
        {
            Object ();
        }

        construct
        {
            service = new Gwibber.Service ();
            
            this.text_view = new InputTextView (service);
            this.send_bar = new SendBar (service);
            this.add (this.text_view);
            this.add_with_properties(this.send_bar,"expand", false);
            this.text_view.show ();
            this.send_bar.show ();
            this.text_view.submit.connect((source) => {
                on_send(this.text_view.buffer.text);});
            this.send_bar.send.clicked.connect((source) => {
                on_send(this.text_view.buffer.text);});
        }


        private void on_send(string msg)
        {
            service.send_message(msg);
            debug ("Message: %s posted", msg);
            this.text_view.clear();
        }

    }

    public class InputTextView : Gtk.TextView
    {
        public Gwibber.Service service;
        public Gwibber.Connection conn_service;
        public Gwibber.URLShorten urlshorten;

        private Gdk.Color base_color;
        private Gdk.Color error_color;
        public string overlay_text;
        public string overlay_color;
        public Gdk.Drawable drawable;
        public Pango.Layout pango_overlay;

        bool last_was_shortened = false;

        private static const int MAX_MESSAGE_LENGTH = 140;

        public InputTextView (Gwibber.Service service)
        {
            Object (border_width:5,
                    accepts_tab:true,
                    editable:true,
                    cursor_visible:true,
                    wrap_mode:Gtk.WrapMode.WORD_CHAR,
                    left_margin:2,
                    right_margin:2,
                    pixels_above_lines:2,
                    pixels_below_lines:2);
        }

        construct
        {
            service = new Gwibber.Service ();
            conn_service = new Gwibber.Connection ();
            urlshorten = new Gwibber.URLShorten ();

            /* FIXME: We should get the color from the theme and modify it */
            //this.overlay_color = style.text[Gtk.StateType.NORMAL].to_string();
            this.overlay_color = "#aeab9f";
            this.overlay_text = "<span weight=\"bold\" size=\"xx-large\" foreground=\"%s\">%s</span>";

            unowned Gtk.BindingSet binding_set;
            binding_set = Gtk.BindingSet.by_class (typeof (InputTextView).class_ref ());
            Gtk.BindingEntry.add_signal (binding_set, Gdk.keyval_from_name ("Return"), 0, "submit", 0);
            Gtk.BindingEntry.add_signal (binding_set, Gdk.keyval_from_name ("KP_Enter"), 0, "submit", 0);
            this.base_color = this.get_style ().base[Gtk.StateType.NORMAL];

            Gdk.Color.parse ("indianred", out this.error_color);

            this.get_buffer ().changed.connect (this.on_text_changed);
            this.get_buffer ().insert_text.connect (this.on_text_inserted);

            this.set_sensitive (conn_service.is_connected ());
            conn_service.connection_changed.connect((source) => {
                    this.on_connection_changed (source);});
        }

  	public void clear() {
		this.buffer.set_text("");
	}

        [Signal (action=true)]
        public virtual signal void submit () {
        }

        private void on_connection_changed (bool is_connected)
	{
		this.set_sensitive(is_connected);
	}

        private void on_text_changed ()
        {
            var chars = this.get_buffer ().get_char_count ();
            this.set_overlay_text ((MAX_MESSAGE_LENGTH - chars).to_string());
            this.modify_base (Gtk.StateType.NORMAL,
                        chars > MAX_MESSAGE_LENGTH ? this.error_color : this.base_color);

        }

        private void on_text_inserted (Gtk.TextIter iter, string text, int len)
        {
            if (this.last_was_shortened == false
                && len > 30
                && text != null
                && text[0:4] == "http")
            {
                var buf = this.get_buffer ();
                Signal.stop_emission_by_name (buf, "insert-text") ;
                var shrt = this.urlshorten.shorten (text);
                this.last_was_shortened = true;
                buf.insert (iter, shrt, -1);
            }
            else
            {
                this.last_was_shortened = false;
            }
        }

        public override bool expose_event (Gdk.EventExpose event)
	{
            if (this.drawable == null) {
                this.drawable = this.get_window(Gtk.TextWindowType.TEXT);
                this.pango_overlay = this.create_pango_layout("");
                this.set_overlay_text(MAX_MESSAGE_LENGTH.to_string());
            }
            Gdk.GC gc = new Gdk.GC(drawable);
            int ww, wh, tw, th;
            this.drawable.get_size(out ww, out wh);
            this.pango_overlay.get_pixel_size(out tw, out th);

            Gdk.draw_layout(this.drawable, gc, ww - tw - 2, wh - th, this.pango_overlay);

            return false;
         }


	private void set_overlay_text(string text)
	{
		if (this.pango_overlay == null) {
                    this.pango_overlay = this.create_pango_layout("");
                }
		this.pango_overlay.set_markup((this.overlay_text).printf(this.overlay_color, text), -1);
	}

    }

    public class AccountToggleButton : Gtk.ToggleButton
    {
        public string color {  get; construct;}
        public AccountToggleButton (string color)
        {
            Object (color:color);
        }

        construct
        {
        }

        public override bool expose_event (Gdk.EventExpose event) 
        {
            double factor = 0.3;
            Gdk.Color gdk_color;

            if (this.get_active())
            {
                factor = 1.0;
            } else if (this.get_state () == 2) {
                factor = 0.3;
            } else if (this.get_state () == 1) {
                factor = 0.5;
            }
            var context = Gdk.cairo_create (this.get_window ());
            Gdk.Color.parse (this.color, out gdk_color);

            double left = event.area.x;
            double top = event.area.y;
            double right = event.area.x + event.area.width;
            double bottom = event.area.y + event.area.height;
            double radius = 10;

            // Create the clipping region starting from the top left curve
            context.move_to(left, top + radius);
            context.curve_to(left, top, left, top, left + radius, top);
            context.line_to(right - radius, top);
            context.curve_to(right, top, right, top, right, top + radius);
            context.line_to(right, bottom - radius);
            context.curve_to(right, bottom, right, bottom, right - radius, bottom);
            context.line_to(left + radius, bottom);
            context.curve_to(left, bottom, left, bottom, left, bottom - radius);
            context.clip();
            Gdk.Color.parse (this.color, out gdk_color);
            /* this pattern could be used to create a gradient
            var pat = new Cairo.Pattern.linear(event.area.x, event.area.y, event.area.x, event.area.y + event.area.height);
	    pat.add_color_stop_rgba(0.0, gdk_color.red/65535.0, gdk_color.green/65535.0, gdk_color.blue/65535.0, factor); 
	    pat.add_color_stop_rgb(1.0, gdk_color.red/65535.0, gdk_color.green/65535.0, gdk_color.blue/65535.0);
            context.set_source(pat);
            */


            context.set_source_rgba (gdk_color.red/65535.0, gdk_color.green/65535.0, gdk_color.blue/65535.0, factor);
	    context.paint();

            this.propagate_expose (this.get_child (), event);

            return true;
        }
    }

    public class AccountTargetBar : Gtk.HBox
    {
        public Gwibber.Accounts accounts_service;
        public Gwibber.Connection conn_service;
        public Gtk.Button send;
	public HashTable<string,AccountToggleButton> accounts_buttons_table;

        public AccountTargetBar ()
        {
            Object (spacing:0);
        }

        construct
        {
            accounts_service = new Gwibber.Accounts();
            conn_service = new Gwibber.Connection ();
            // Add buttons to button area at the bottom
            var box = new Gtk.HBox(false, 0);

            var send_with_label = new Gtk.Label(_("Send with:"));
            box.pack_start(send_with_label, false, false, 5);

            var service_icon_path = GLib.Path.build_path(
                GLib.Path.DIR_SEPARATOR_S, GLib.Path.DIR_SEPARATOR_S, "usr", "share",
                "gwibber", "ui", "icons", "breakdance", "22x22", GLib.Path.DIR_SEPARATOR_S);

            if (GLib.FileUtils.test(GLib.Path.build_path(GLib.Path.DIR_SEPARATOR_S, GLib.Path.DIR_SEPARATOR_S, "usr", "local", "share", "gwibber", "ui", "icons", "breakdance", "22x22"), GLib.FileTest.EXISTS)) {
                service_icon_path = GLib.Path.build_path(
                    GLib.Path.DIR_SEPARATOR_S, GLib.Path.DIR_SEPARATOR_S, "usr", "local", "share",
                    "gwibber", "ui", "icons", "breakdance", "22x22", GLib.Path.DIR_SEPARATOR_S);
            }
            
            accounts_buttons_table = new HashTable<string,AccountToggleButton>(str_hash, str_equal);

            var accounts_list = accounts_service.list ();
            
            //foreach (var account in accounts_service.list()) {

            for (int i = 0; i < accounts_list.length(); i++) {
                Gwibber.Account account = accounts_list.nth_data(i);
                if (account == null) {
                    continue;
                }

                // If there is no send_enabled key, don't display an icon
                if (account.send_enabled == null) {
                    continue;
                }

                Gtk.Image icon_service = new Gtk.Image();
                icon_service.set_from_file(service_icon_path + account.service + ".png");
   
                icon_service.show();
                AccountToggleButton account_button = new AccountToggleButton(account.color);
                account_button.set_active((account.send_enabled == "1")?true:false);
	    	if (account.send_enabled != "1") {
                    account_button.tooltip_text = (account.service + " (" + account.username + ") - " + _("Disabled"));
		} else {
                    account_button.tooltip_text = (account.service + " (" + account.username + ")");
		}
                account_button.set_image(icon_service);

                account_button.clicked.connect((source) => {
                this.on_account_toggled(account_button, account.id);});

                box.pack_start(account_button, false, false, 2);
                accounts_buttons_table.insert(account.id, account_button);
            }

            this.send = new Gtk.Button.with_label(_("Send"));
            box.pack_end(this.send, false, false, 5);
            this.add(box);
            this.send.set_sensitive (conn_service.is_connected ());
            conn_service.connection_changed.connect((source) => {
                    this.on_connection_changed (source);});
            accounts_service.account_updated.connect((source) => {
                this.account_updated(accounts_buttons_table, source);});
        }

        void on_account_toggled(AccountToggleButton account_button, string id) {
            Gwibber.Account account = accounts_service.lookup_by_id (id);
	    if (((account.send_enabled == "1")?true:false) !=  (account_button.get_active()))
            {
                accounts_service.send_enabled(id);
            }
        }

        static void account_updated(GLib.HashTable<string,AccountToggleButton> accounts_buttons_table, Gwibber.Account account) 
        {
            debug ("ACCOUNT: %s %s %s", account.username, account.service, account.id);
            AccountToggleButton account_button = accounts_buttons_table.lookup(account.id);
            if (account.send_enabled != "1") {
                account_button.tooltip_text = (account.service + " (" + account.username + ") - " + _("Disabled"));
            } else {
                account_button.tooltip_text = (account.service + " (" + account.username + ")");
            }
	    if (((account.send_enabled == "1")?true:false) !=  (account_button.get_active())) {
                account_button.set_active((account.send_enabled == "1")?true:false);
            }
        }

        private void on_connection_changed (bool is_connected) {
                this.send.set_sensitive(is_connected);
        } 
    }

    public class SendBar : Gtk.HBox
    {
        public Gwibber.Service service { get; construct; }
        public Gtk.Button send;
        public SendBar (Gwibber.Service service)
        {
            Object (service:service,
                    spacing:5);
        }

        construct
        {
            // Add buttons to button area at the bottom
            var box = new Gtk.HBox(false, 0);
            send = new Gtk.Button.with_label(_("Send"));
            box.pack_end(send, false, false, 0);
            this.add(box);
        }
    }
}
