/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributors: Jason Day <jasonday@worldnet.att.net>
 *               Reinhold Kainhofer (reinhold@kainhofer.com)
 *
 */

#include <stdio.h>
#include <signal.h>

#ifndef _WIN32
#include <unistd.h>
#endif

#ifndef _HPUX_SOURCE
#include <dlfcn.h>
#endif

#include <AGNet.h>
#include <AGUtil.h>
#include <AGClientProcessor.h>
#include <AGProtocol.h>
#include <AGBufferReader.h>
#include <AGPalmProtocol.h>
#include <AGUserConfig.h>
#include <AGServerConfig.h>
#include <AGSyncCommon.h>
#include <AGCommandProcessor.h>
#include <AGDesktopInfoPalm.h>
#include <AGTimeoutMessageBox.h>
#include <AGMobileLinkSettings.h>
#include <MAL31UserConfig.h>

#include <pi-source.h>
#include <pi-socket.h>
#include <pi-file.h>
#include <pi-dlp.h>
#include <pi-version.h>
#include <pi-header.h>

#include "libmal.h"
#include "config.h"



#ifdef _DEBUG
#define debug_print(x) printf("%s\n", (x));
#else
#define debug_print(x)
#endif

static int verbose = 0;
static const char *default_device = "/dev/pilot";

int lm_errno = 0;

static void fill_in_versioninfo(int sd, AGExpansionVersion *vi);
static int32 initAndOpenDatabase(void *out, AGDBConfig *theDatabase, 
                                 int32 *errCode);
static int32 getNextModifiedRecord(void * out, AGRecord ** record, 
                                   int32 *errCode);
static int32 getNextRecord(void * out, AGRecord ** record,
                           int32 *errCode);

static int32 cmdTASK(void *out, int32 *returnErrorCode, char *currentTask,
                     AGBool bufferable);
static int32 cmdITEM(void *out, int32 *returnErrorCode,
                                   int32 currentItemNumber,
                                   int32 totalItemCount,
                                   char *currentItem);
static int32 cmdDELETEDATABASE(void *out, int32 *returnErrorCode,
                                             char *dbname);
static int32 cmdOPENDATABASE(void *out, int32 *returnErrorCode,
                                           char *dbname);
static int32 cmdCLOSEDATABASE(void *out, int32 *returnErrorCode);
static int32 cmdCLEARMODS(void *out, int32 *returnErrorCode);
static int32 cmdGOODBYE(void *out, int32 *returnErrorCode,
                        AGSyncStatus syncStatus,
                        int32 errorCode,
                        char *errorMessage);
static int32 cmdRECORD(void *out, int32 *returnErrorCode,
                       int32 *newUID,
                       int32 uid,
                       AGRecordStatus mod,
                       int32 recordDataLength,
                       void *recordData,
                       int32 platformDataLength,
                       void *platformData);


/*----------------------------------------------------------------------------*/
#ifdef __UNUSED
static int
loadSecLib(AGNetCtx **ctx)
{
    char *seclib;
    
    seclib = getenv("MALSYNC_SECURITYLIB");

    if (!seclib) {
        if (verbose) {
            printf("MALSYNC_SECURITYLIB env variable not set\n");
        }
        return 0;
    }

#ifdef _WIN32
    {
        HINSTANCE h = LoadLibrary(seclib);
        if (h) {
            secnetinit = (netInitFunc)GetProcAddress(h, "NetInit");
            secnetclose = (netCloseFunc)GetProcAddress(h, "NetClose");
            secctxsize = (netCtxSizeFunc)GetProcAddress(h, "NetGetCtxSize");
            secnetpostsync = 
                (netPostSyncHook)GetProcAddress(h, "NetPostSyncHook");
            secnetpresync = (netPreSyncHook)GetProcAddress(h, "NetPreSyncHook");
        }
    }
#else /* _WIN32 */
    {   
#ifndef _HPUX_SOURCE
        void *h;
#ifdef __linux__
        h  = dlopen(seclib, RTLD_GLOBAL|RTLD_NOW);
#else /* __linux__ */
        h  = dlopen(seclib, RTLD_NOW);
#endif /* __linux */
        
        if (h) {
            secnetinit = dlsym(h, "NetInit");
            secnetclose = dlsym(h, "NetClose");
            secctxsize = dlsym(h, "NetGetCtxSize");
            secnetpostsync = dlsym(h, "NetPostSyncHook");
            secnetpresync = dlsym(h, "NetPreSyncHook");
        } else {
            if (verbose)
                printf("%s\n", dlerror());
        }
#endif
    }
#endif /* !_WIN32 */

    if (secnetinit && secnetclose && secctxsize ) {
        if (verbose) {
            printf("Found security library, initalizing\n");
        }
        *ctx = calloc(1, (*secctxsize)());
        (*secnetinit)(*ctx);
        return 1;
    }
    return 0;
}
#endif

/*---------------------------------------------------------------------------*/
void syncInfoFree(PalmSyncInfo * pInfo) {


    if (NULL != pInfo) {

        if (NULL != pInfo->platform)
            free(pInfo->platform);

        if (NULL != pInfo->userConfig)
            AGUserConfigFree(pInfo->userConfig);

        if (NULL != pInfo->pilot_buffer)
            free(pInfo->pilot_buffer);

        if (NULL != pInfo->commandProcessor)
            AGCommandProcessorFree(pInfo->commandProcessor);

        free(pInfo);
    }
}

/*---------------------------------------------------------------------------*/
PalmSyncInfo *syncInfoNew() {
    PalmSyncInfo * pInfo;

    /* Allocate the PalmSyncInfo structure. */
    pInfo = (PalmSyncInfo *)malloc(sizeof(PalmSyncInfo));
    if (NULL != pInfo) {

        const int pbs = 65535;

        bzero(pInfo, sizeof(PalmSyncInfo));

        pInfo->pilot_buffer_size    = pbs;
        pInfo->pilot_buffer         = (uint8 *)malloc(pbs);
        if (NULL == pInfo->pilot_buffer)
            goto fail;

        /* Allocate the platform calls record. */
        pInfo->platform = (AGPlatformCalls *)malloc(sizeof(AGPlatformCalls));
        bzero(pInfo->platform, sizeof(AGPlatformCalls));
        if (NULL == pInfo->platform)
            goto fail;

        pInfo->device = (char *)default_device;

        return pInfo;
    }

fail:
    if (NULL != pInfo) {
        if (verbose) {
            printf("Error in syncInfoNew\n");
        }
        syncInfoFree(pInfo);
    }

    lm_errno = LM_SYNC_INFO;
    return NULL;
}
/*---------------------------------------------------------------------------*/
static AGBool setupPlatformCalls(PalmSyncInfo * pInfo)
{
    pInfo->platform->out = pInfo;
    pInfo->platform->nextModifiedRecordFunc = getNextModifiedRecord;
    pInfo->platform->nextRecordFunc = getNextRecord;
    pInfo->platform->openDatabaseFunc = initAndOpenDatabase;
    return FALSE;
}
/*---------------------------------------------------------------------------*/
static int32 
readInt32(uint8 *p)
{
    return (((int32)p[0]) << 24)
        + (((int32)p[1]) << 16)
        + (((int32)p[2]) << 8)
        + ((int32)p[3]);
}
/*---------------------------------------------------------------------------*/
static 
int16 readInt16(uint8 * p)
{
    return (((int16)p[0]) << 8) + ((int16)p[1]);
}
/*---------------------------------------------------------------------------*/
static void 
readAndUseDeviceInfoDatabase(AGDeviceInfo *devInfo,
                             int sd,
                             uint8 *dev_db_info_buffer,
                             uint32 dev_db_info_buffer_size)
{
    int database_id = 0;
    long result;

    if (verbose) 
        printf("Entering readAndUseDeviceInfoDatabase\n");

    /*  Open the device info database on the device.  It may or may not
        be present.  If it isn't, that's ok -- just return and don't
        change anything.
    */
    /* 11/24/00 - Changed name of the database that the color info is stored 
     * in. This should clear up some bug reports from people with color 
     * devices. 
     */
    result = dlp_OpenDB(sd, DEFAULT_CARD_NUM, dlpOpenRead,  
                        "AvGoDeviceInfo", &database_id);
    
    if (result < 0) {
        if (verbose) 
            printf("Unable to open MBlnDevice Info\n");

    }
    else {

        int attr = 0;
        int cat  = 0;
        recordid_t id;
        int rc;

#ifdef PILOT_LINK_0_12
        rc = dlp_ReadRecordByIndex(sd, database_id, 0,
                                   (void *)dev_db_info_buffer,
                                   &id, &attr, &cat);
#else
        rc = dlp_ReadRecordByIndex(sd, database_id, 0,
                                   (void *)dev_db_info_buffer,
                                   &id, &dev_db_info_buffer_size,
                                   &attr, &cat);
#endif

        if (rc >= 0) {
            uint8 *p = dev_db_info_buffer;
            /*int16 dev_db_info_version = readInt16(p);*/
            /* make gcc happy */
            readInt16(p);
            p+=sizeof(int16);
            devInfo->colorDepth = readInt32(p);
            p+=sizeof(int32);
            devInfo->screenWidth = readInt32(p);
            p+=sizeof(int32);
            devInfo->screenHeight = readInt32(p);
            p+=sizeof(int32);

            if (devInfo->serialNumber) 
                free(devInfo->serialNumber);

            /*  Next line is safe because we know p isn't null
                at this point and we know the creator of the
                record wrote at least a zero here.
            */
            devInfo->serialNumber = strdup((char*)p);

            if (verbose)
                printf("MBlnDeviceInfo sez: colorDepth = %d,"
                       " serial number is %s\n", 
                       devInfo->colorDepth, devInfo->serialNumber);

        }
        
        dlp_CloseDB(sd, database_id);
    }
}
/*---------------------------------------------------------------------------*/
static int readDeviceInfo(PalmSyncInfo *pInfo) {
    AGDeviceInfo *devInfo;
    struct SysInfo sysInfo;
    struct CardInfo cardInfo;
    char osverstring[24];
    int err;
    int majorVersion, minorVersion, bugfixVersion, build, state;
    
    /* Start with clean slate. */
    devInfo = pInfo->deviceInfo;

    err = dlp_ReadSysInfo(pInfo->sd, &sysInfo);
    if (err < 0) {
        if (verbose) {
            fprintf(stderr, "dlp_ReadSysInfo failed with err %d\n", err);
        }
        /*
         * FIXME
         * It would be nice to get the original error somehow...
         */
        lm_errno = LM_READ_SYS_INFO;
        return -1;
    }

    cardInfo.card = DEFAULT_CARD_NUM;
    err = dlp_ReadStorageInfo(pInfo->sd, DEFAULT_CARD_NUM, &cardInfo);
    if (err < 0) {
        /* FIXME */
        if (verbose) {
            fprintf(stderr, "dlp_ReadStorageInfo failed with err %d\n", err);
        }
        /*
         * FIXME
         * It would be nice to get the original error somehow...
         */
        lm_errno = LM_READ_STORAGE_INFO;
        return -1;
    }

    majorVersion = (((sysInfo.romVersion >> 28) & 0xf) * 10) + 
        ((sysInfo.romVersion >> 24) & 0xf);
    minorVersion = ((sysInfo.romVersion >> 20) & 0xf);
    bugfixVersion = ((sysInfo.romVersion >> 16) & 0xf);
    state = ((sysInfo.romVersion >> 12) & 0xf);
    build = (((sysInfo.romVersion >> 8) & 0xf) * 10) + 
        (((sysInfo.romVersion >> 4) & 0xf) * 10)+ (sysInfo.romVersion  & 0xf);
    

    snprintf(osverstring, 24, "%d.%d", majorVersion, minorVersion);

    if (verbose) {
        printf("OS Version = %s\n", osverstring);
    }

    devInfo->availableBytes   = cardInfo.ramFree;
    devInfo->serialNumber     = strdup("");
    devInfo->osName           = strdup("PALM_OS");
    devInfo->osVersion        = strdup(osverstring);
    devInfo->screenWidth      = 150;
    devInfo->screenHeight     = 150;

    if((majorVersion > 3) || (majorVersion == 3 && minorVersion >= 5))
        devInfo->colorDepth = 8;
    else {
        if(majorVersion > 2)
            devInfo->colorDepth = 2;
        else
            devInfo->colorDepth = 1;
    }

    if (verbose) {
        printf("Setting colordepth: devInfo->colorDepth = %d\n",
               devInfo->colorDepth);
    }

    readAndUseDeviceInfoDatabase(devInfo,
                                 pInfo->sd,
                                 pInfo->pilot_buffer,
                                 pInfo->pilot_buffer_size);

    /* Override the color depth if the user wants low res images. */
    if (pInfo->lowres) {
        if (verbose) {
            printf("Overriding colordepth: devInfo->colorDepth = 1\n");
        }        
        devInfo->colorDepth = 1;
    }


    return 0;
}
/*---------------------------------------------------------------------------*/
static AGBool 
getPalmDatabaseCreationInfo(AGDBConfig *db, uint32 *creator, uint32 *flags, 
                            uint32 *type)
{
    AGBufferReader * r = NULL;
    
    if (verbose)
        printf("GetPalmDatabaseCreationInfo()\n");

    if (NULL == db)
        return FALSE;

    if (0 == db->platformDataLength || NULL == db->platformData)
        return FALSE;

    r = AGBufferReaderNew((uint8*)db->platformData);
    if (NULL == r)
        return FALSE;
    
    AGPalmReadDBConfigPlatformData((AGReader*)r, creator, type, flags);
    AGBufferReaderFree(r);
    return TRUE;
}

/*---------------------------------------------------------------------------*/
static int createDatabase(int sd, AGDBConfig *db)
{
    int dbhandle;
    long creator;
    int flags;
    int cardNo = DEFAULT_CARD_NUM;
    long type;
    int version = 0;
    int err;
    
    if (!db)
        return 0;

    if (verbose)
        printf("createDatabase\n");
    
    getPalmDatabaseCreationInfo(db, (uint32*)&creator, 
                                (uint32*)&flags, (uint32*)&type);

    if ((err = dlp_CreateDB(sd,  creator,  type, cardNo, flags,  
                            version, db->dbname, &dbhandle)) < 0)
    {
        if (verbose)
            printf("createDatabase: dlp_CreateDB failed err = %d\n", err);
        dbhandle = 0;
        /*
         * FIXME
         * It would be nice to get the original error somehow...
         */
        lm_errno = LM_CREATE_DB;
    }
    
    return dbhandle;
}

/*---------------------------------------------------------------------------*/
static long openDatabase(PalmSyncInfo *pInfo, char *dbname, AGBool create)
{

    long result;
    
    if (!dbname || !pInfo) {
        if (verbose)
            printf("\n");
        return -1;
    }
    
    if (verbose) {
        printf("... opening '%s' ...", dbname);
    }

    pInfo->currentDb =
        AGServerConfigGetDBConfigNamed(pInfo->currentServerConfig, dbname);
    
    result = dlp_OpenDB(pInfo->sd, DEFAULT_CARD_NUM, dlpOpenRead|dlpOpenWrite,
                        dbname, &pInfo->pilot_rHandle);
    
    if ((result < 0) && create)
        pInfo->pilot_rHandle = createDatabase(pInfo->sd, pInfo->currentDb);

    if (pInfo->pilot_rHandle) {
        uint32 creator, flags, type;
        if (getPalmDatabaseCreationInfo(pInfo->currentDb, &creator, &flags, 
                                        &type) && (flags & 0x01) != 0) {
            pInfo->currentDBIsResourceType = TRUE;
        } else
            pInfo->currentDBIsResourceType = FALSE;
        if (verbose)
            printf("successfully.\n");
    }
    else {
        if (verbose)
            printf("unsuccessfully.\n");
        pInfo->currentDBIsResourceType = FALSE;
        pInfo->currentDb = NULL;
    }

    return result;
}

/*---------------------------------------------------------------------------*/
static long closeDatabase(PalmSyncInfo * pInfo)
{
    int result;

    result = dlp_CloseDB(pInfo->sd, pInfo->pilot_rHandle);
    pInfo->pilot_rHandle = 0;
    pInfo->currentDb = NULL;
    pInfo->currentDBIsResourceType = FALSE;
    return result;
}

/*---------------------------------------------------------------------------*/
static void deleteDatabase(int sd, char *dbname)
{
    if (verbose)
        printf("deleteDatabase(%s)\n", dbname);
    dlp_DeleteDB(sd, DEFAULT_CARD_NUM, dbname);
}
/*---------------------------------------------------------------------------*/
static void clearMods(int sd, long dbHandle)
{
    if (verbose)
        printf("clearMods()\n");
    dlp_CleanUpDatabase(sd, dbHandle);
    dlp_ResetSyncFlags(sd, dbHandle);

}
#define DEVICE_USERCONFIG_DB_NAME "MBlnUserConfig"
/*---------------------------------------------------------------------------*/
static long openUserConfigDatabase(int sd, int *threeone)
{
    long result;
    int userConfigDBHandle = 0;
    
    *threeone = 0;
    result = dlp_OpenDB(sd, DEFAULT_CARD_NUM, dlpOpenRead|dlpOpenWrite,  
                        DEVICE_PROFILE_DB_NAME, 
                        &userConfigDBHandle);

    if (result < 0) {
        
        if (verbose) {
            printf("Failed to locate MBlnProfile database. Lets look for"
                   " a MBlnUserConfig database\n");
        }
        /* OK Now lets look for a 3.1 client database */
        
        result = dlp_OpenDB(sd, DEFAULT_CARD_NUM, dlpOpenRead|dlpOpenWrite,  
                            DEVICE_USERCONFIG_DB_NAME, 
                            &userConfigDBHandle);
        if (result < 0) {
            
            result = dlp_CreateDB(sd, DEVICE_PROFILE_DB_CREATOR, 
                                  DEVICE_PROFILE_DB_TYPE, DEFAULT_CARD_NUM,
                                  0, 0, DEVICE_PROFILE_DB_NAME, 
                                  &userConfigDBHandle);
            if (result < 0) {
                if (verbose) {
                    fprintf(stderr, "Unable to create user Config Databage\n");
                }
                lm_errno = LM_USER_CONFIG;
                userConfigDBHandle = 0;
            }        
        } else {
            if (verbose) {
                printf("Found a MBlnUserConfig, this must be MobileLink"
                       "3.1 or older\n");
            }
            *threeone = 1;
        }

    }
    return userConfigDBHandle;
}
#define BUFFERSIZE 0xFFFF
/*---------------------------------------------------------------------------*/
static int32 
readDeviceUserConfig32(int sd, int userConfigDBHandle, AGUserConfig **deviceUserConfig)
{
    recordid_t id;
    int bufferSize = BUFFERSIZE;
    int attr = 0;
    int cat  = 0;
    int rc;
    uint8 buffer[BUFFERSIZE];
    AGBufferReader * r = NULL;

#ifdef PILOT_LINK_0_12
    rc = dlp_ReadRecordByIndex(sd, userConfigDBHandle, 0, (void *)buffer,
                               &id, &attr, &cat);
#else
    rc = dlp_ReadRecordByIndex(sd, userConfigDBHandle, 0, (void *)buffer,
                               &id, &bufferSize, &attr, &cat);
#endif

    if (rc < 0) {
        if (verbose)
            printf("readDeviceUserConfig: dlp_ReadRecordByIndex , err = %d\n",
                   rc);
        /*
         * FIXME
         * It would be nice to get the original error somehow...
         */
        lm_errno = LM_READ_RECORD;
        return 0;
    }
    
    r = AGBufferReaderNew(buffer);
    if (r) {
        *deviceUserConfig = AGUserConfigNew();
        AGUserConfigReadData(*deviceUserConfig, (AGReader*)r);
        AGBufferReaderFree(r);
        return id;
    }
    else
        return 0;
}
#define BUFFERSIZE 0xFFFF
/*---------------------------------------------------------------------------*/
static int32 readDeviceUserConfig31(int sd, int userConfigDBHandle,
                                    AGUserConfig **deviceUserConfig)
{
    recordid_t id;
    int bufferSize = BUFFERSIZE;
    int attr = 0;
    int cat  = 0;
    int rc;
    uint8 buffer[BUFFERSIZE];
    AGBufferReader * r = NULL;

#ifdef PILOT_LINK_0_12
    rc = dlp_ReadRecordByIndex(sd, userConfigDBHandle, 0, (void *)buffer,
                               &id, &attr, &cat);
#else
    rc = dlp_ReadRecordByIndex(sd, userConfigDBHandle, 0, (void *)buffer,
                               &id, &bufferSize, &attr, &cat);
#endif

    if (rc < 0) {
        if (verbose)
            printf("readDeviceUserConfig: dlp_ReadRecordByIndex , err = %d\n",
                   rc);
        /*
         * FIXME
         * It would be nice to get the original error somehow...
         */
        lm_errno = LM_READ_RECORD;
        return 0;
    }
    
    r = AGBufferReaderNew(buffer);
    if (r) {
        *deviceUserConfig = AGUserConfigNew();
        MAL31ReadUserData(*deviceUserConfig, (AGReader*)r);
        AGBufferReaderFree(r);
        return id;
    } else
        return 0;
}
/*---------------------------------------------------------------------------*/
static int32 readDeviceUserConfig(int sd, int userConfigDBHandle,
                                  AGUserConfig **deviceUserConfig,
                                  int *threeone)
{ 
    int32 ret;

    if (*threeone)
        ret = readDeviceUserConfig31(sd, userConfigDBHandle, deviceUserConfig);
    else 
        ret = readDeviceUserConfig32(sd, userConfigDBHandle, deviceUserConfig);
    return ret;
}
/*---------------------------------------------------------------------------*/
static void writeDeviceUserConfig(int sd,
                                  int userConfigDBHandle,
                                  AGUserConfig * deviceUserConfig,
                                  recordid_t *recID, int threeone)
{

    recordid_t id;
    int bufferSize = BUFFERSIZE;
    int attr = 0;
    int cat  = 0;
    uint8 buffer[BUFFERSIZE];
    AGBufferWriter * w = NULL;
    w = AGBufferWriterNew(0);
    if (w) {
        long result;
        
        if (threeone) {
            MAL31WriteUserData(deviceUserConfig, (AGWriter*)w);
        } else {
            AGUserConfigWriteData(deviceUserConfig, (AGWriter*)w);
        }

#ifdef PILOT_LINK_0_12
        result = dlp_ReadRecordByIndex(sd, userConfigDBHandle, 0, (void *)buffer,
                                       &id, &attr, &cat);
#else
        result = dlp_ReadRecordByIndex(sd, userConfigDBHandle, 0, (void *)buffer,
                                       &id, &bufferSize, &attr, &cat);
#endif

        if (result < 0)
            id = 0;

        result =  dlp_WriteRecord(sd, userConfigDBHandle, 0,
                                  id, 0, (void *)AGBufferWriterGetBuffer(w), 
                                  AGBufferWriterGetBufferSize(w), 
                                  &id);
        AGBufferWriterFree(w);

    }
}

/*---------------------------------------------------------------------------*/
static AGUserConfig *getUserConfig(int sd, uint32 *pilotID, int *threeone)
{
    
    /* The device is the truth. There is no way to set these values on 
       the desktop */
    AGUserConfig *deviceUserConfig = NULL;
    int userConfigDBHandle = 0;

    /* Get device's record. */
    userConfigDBHandle = openUserConfigDatabase(sd, threeone);

    if (userConfigDBHandle) {
        
        
        /* Retrieve device's idea of current userConfig. */
        *pilotID = readDeviceUserConfig(sd, userConfigDBHandle,
                                        &deviceUserConfig, 
                                        threeone);
        
        /* Done with database for now, so close it. */
        dlp_CloseDB(sd, userConfigDBHandle);
        

    }
    return deviceUserConfig;
}

/*---------------------------------------------------------------------------*/
static void storeDeviceUserConfig(int sd, AGUserConfig *userConfig, recordid_t id)
{
    int threeone;
    int userConfigDBHandle = openUserConfigDatabase(sd, &threeone);
    if (0 != userConfigDBHandle) {
        writeDeviceUserConfig(sd, userConfigDBHandle,
                              userConfig, &id, threeone);
        dlp_CloseDB(sd, userConfigDBHandle);
    }
}

/*---------------------------------------------------------------------------*/
static void doStartServer(PalmSyncInfo *pInfo,
                          AGServerConfig *sc,
                          int32 *errCode)
{
    pInfo->currentServerConfig = sc;
    if(pInfo->commandProcessor) {
        AGCommandProcessorFree(pInfo->commandProcessor);
        pInfo->commandProcessor = NULL;
    }
    pInfo->commandProcessor = AGCommandProcessorNew(sc);
    pInfo->platform->performCommandOut = pInfo->commandProcessor;
    pInfo->platform->performCommandFunc = 
                    AGCommandProcessorGetPerformFunc(pInfo->commandProcessor);

    pInfo->commandProcessor->commands.out = pInfo;

    if (pInfo->taskFunc) {
        pInfo->commandProcessor->commands.performTaskFunc = pInfo->taskFunc;
    }
    else {
        pInfo->commandProcessor->commands.performTaskFunc = cmdTASK;
    }

    if (pInfo->itemFunc) {
        pInfo->commandProcessor->commands.performItemFunc = pInfo->itemFunc;
    }
    else {
        pInfo->commandProcessor->commands.performItemFunc = cmdITEM;
    }

    pInfo->commandProcessor->commands.performDeleteDatabaseFunc
        = cmdDELETEDATABASE;
    pInfo->commandProcessor->commands.performOpenDatabaseFunc 
        = cmdOPENDATABASE;
    pInfo->commandProcessor->commands.performCloseDatabaseFunc 
        = cmdCLOSEDATABASE;
    pInfo->commandProcessor->commands.performClearModsFunc = cmdCLEARMODS;
    pInfo->commandProcessor->commands.performGoodbyeFunc = cmdGOODBYE;
    pInfo->commandProcessor->commands.performRecordFunc = cmdRECORD;
    
}

/*---------------------------------------------------------------------------*/
static void doEndServer(PalmSyncInfo *pInfo, int32 *errCode) {
    pInfo->currentServerConfig = NULL;
    if (pInfo->commandProcessor) {
        AGCommandProcessorFree(pInfo->commandProcessor);
        pInfo->commandProcessor = NULL;
    }
}
/*---------------------------------------------------------------------------*/
static int32 cmdTASK(void *out, int32 *returnErrorCode, char *currentTask,
                     AGBool bufferable)
{
    if (currentTask)
        printf("%s\n", currentTask);
    return AGCLIENT_CONTINUE;
}
/*---------------------------------------------------------------------------*/
static int32 
cmdITEM(void *out, int32 *returnErrorCode, int32 currentItemNumber,
        int32 totalItemCount, char *currentItem)
{
    printf(".");
    fflush(stdout);
    if (currentItemNumber == totalItemCount)
        printf("\n");
    
    return AGCLIENT_CONTINUE;
}
/*---------------------------------------------------------------------------*/
static int32 
cmdDELETEDATABASE(void *out, int32 *returnErrorCode, char *dbname)
{
    PalmSyncInfo *pInfo = (PalmSyncInfo *)out;

    if (verbose)
        printf("doCmdAG_DELETEDATABASE_CMD()\n");

    if (NULL != dbname) {
        if (verbose)
            printf("... trying to delete database %s from device\n", 
                  dbname);
        deleteDatabase(pInfo->sd, dbname);
    }
    return AGCLIENT_CONTINUE;
}
/*---------------------------------------------------------------------------*/
static int32 
cmdOPENDATABASE(void *out, int32 *returnErrorCode, char *dbname)
{
    PalmSyncInfo *pInfo = (PalmSyncInfo *)out;
    if (verbose)
        printf("doCmdAG_OPENDATABASE_CMD(%s)\n", dbname);

    if (NULL != dbname) {
        long result;
        /* We assign a result code here only for debugging.  It's ok for an
        open to fail here. */
        result = openDatabase(pInfo, dbname, TRUE);
    }
    return AGCLIENT_CONTINUE;
}
/*---------------------------------------------------------------------------*/
static int32 
cmdCLOSEDATABASE(void *out, int32 *returnErrorCode)
{
    PalmSyncInfo *pInfo = (PalmSyncInfo *)out;
    if (verbose)
        printf("doCmdAG_CLOSEDATABASE_CMD()\n");
    closeDatabase(pInfo);
    return AGCLIENT_CONTINUE;
}
/*---------------------------------------------------------------------------*/
static int32 
cmdCLEARMODS(void *out, int32 *returnErrorCode)
{
    PalmSyncInfo *pInfo = (PalmSyncInfo *)out;
    if (verbose)
        printf("doCmdAG_CLEARMODS_CMD()\n");
    clearMods(pInfo->sd, pInfo->pilot_rHandle);
    return AGCLIENT_CONTINUE;
}
/*---------------------------------------------------------------------------*/
static int32 
cmdGOODBYE(void *out, int32 *returnErrorCode, AGSyncStatus syncStatus,
           int32 errorCode, char *errorMessage)
{
    if (verbose)
        printf("doCmdAG_GOODBYE_CMD()\n");

    /* FIXME */
    if (errorMessage && verbose)
        printf("%s\n", errorMessage);
    
    return AGCLIENT_CONTINUE;
}
/*---------------------------------------------------------------------------*/
/*
 * FIXME
 * This function isn't used anywhere, so it's commented out to make gcc
 * happy.
 *
static int
getIndexFromPlatformData(uint8 *platformData)
{
    int16 recIndex;
    AGBufferReader reader;

    if (!platformData)
        return 0;

    AGBufferReaderInit(&reader, platformData);
    AGPalmReadRecordPlatformData((AGReader *)&reader, &recIndex);
    AGBufferReaderFinalize(&reader);
    return (int)recIndex;
}
*/
/*---------------------------------------------------------------------------*/
static int32 
cmdRECORD(void *out, int32 *returnErrorCode, int32 *newUID,
          int32 uid, AGRecordStatus mod, int32 recordDataLength,
          void *recordData, int32 platformDataLength, void *platformData)
{
    PalmSyncInfo *pInfo = (PalmSyncInfo *)out;
    
    if (verbose)
        printf("doCmdAG_RECORD_CMD()\n");

    if (mod == AG_RECORD_NEW_TEMPORARY_UID) {
        uid = 0; 
    }

    if (AG_RECORD_DELETED == mod) {

        dlp_DeleteRecord (pInfo->sd, pInfo->pilot_rHandle, 0, uid);

    }
    else if (recordDataLength <= 0x0000ffff) {

        if (pInfo->currentDBIsResourceType) {
            
            dlp_WriteRecord(pInfo->sd, pInfo->pilot_rHandle, dlpDBFlagResource,
                            uid, 0, (void *)recordData,
                            recordDataLength, (recordid_t *)newUID);
            if (verbose)
                printf("doCmdAG_RECORD_CMD()\n");

        }
        else {

            dlp_WriteRecord(pInfo->sd, pInfo->pilot_rHandle, 0,
                            uid, 0, (void *)recordData,
                            recordDataLength, (recordid_t *)newUID);
        }

    }

    return AGCLIENT_CONTINUE;
}
/*---------------------------------------------------------------------------*/
static int32 
initAndOpenDatabase(void *_pInfo, AGDBConfig *db, int32 *errCode)
{
    long result;
    PalmSyncInfo * pInfo = (PalmSyncInfo *)_pInfo;

    if (NULL == db->dbname) {
        *errCode = AGCLIENT_OPEN_ERR;
        return AGCLIENT_ERR;
    }

    result = openDatabase(pInfo, db->dbname, FALSE);
    if (result < 0) {
        if (result == dlpErrNotFound)
            *errCode = AGCLIENT_OPEN_ERR;
        else
            *errCode = AGCLIENT_UNKNOWN_ERR;
        return AGCLIENT_ERR;
    }

    pInfo->pilot_RecIndex = 0;
    pInfo->record = AGRecordNew(0, AG_RECORD_UNMODIFIED, 0, 0, 0, 0);
    if (!pInfo->record) {
        *errCode = AGCLIENT_OPEN_ERR;
        return AGCLIENT_ERR;
    }

    return AGCLIENT_IDLE;
}
/*---------------------------------------------------------------------------*/
static int32 
leaveGetRecord(PalmSyncInfo *pInfo, int32 result)
{
    if (pInfo->record) {

        /* Set recordData to NULL so that AGRecordFree doesn't try
        to free it (we own that buffer and will release it when the
        program ends). */
        pInfo->record->recordData = NULL;

        AGRecordFree(pInfo->record);
        pInfo->record = NULL;
    }
    return result;
}
/*---------------------------------------------------------------------------*/
static int32 
getRecordBase(PalmSyncInfo *pInfo, AGBool modonly,  AGRecord **record,
              int32 *errCode)
{
    int32 result;
    int att = 0;
    int cat = 0;
    int size = pInfo->pilot_buffer_size;
    int idx   = pInfo->pilot_RecIndex++;

    result = (modonly) ?
#ifdef PILOT_LINK_0_12
        dlp_ReadNextModifiedRec (pInfo->sd, pInfo->pilot_rHandle, pInfo->pilot_buffer,
                                 &pInfo->id, &idx, &att, &cat)
        :
        dlp_ReadRecordByIndex(pInfo->sd, pInfo->pilot_rHandle, idx,
                              (void *)pInfo->pilot_buffer,
                              &pInfo->id, &att, &cat);
#else
        dlp_ReadNextModifiedRec (pInfo->sd, pInfo->pilot_rHandle, pInfo->pilot_buffer,
                                 &pInfo->id, &idx,
                                 &size, &att, &cat)
        :
        dlp_ReadRecordByIndex(pInfo->sd, pInfo->pilot_rHandle, idx,
                              pInfo->pilot_buffer, &pInfo->id,
                              &size, &att, &cat);
#endif

    if (result < 0) {
        closeDatabase(pInfo);
        if (result == dlpErrNotFound) {
            if (verbose)
                printf("(successfully reached end of records ...)\n");
            return leaveGetRecord(pInfo, AGCLIENT_IDLE);
        }
        else {
            *errCode = AGCLIENT_UNKNOWN_ERR;
            return leaveGetRecord(pInfo, AGCLIENT_ERR);
        }
    }
    pInfo->record = AGRecordInit(pInfo->record, pInfo->id,
                                 AGPalmPilotAttribsToMALMod((uint8)att),
                                 size, pInfo->pilot_buffer, 0, NULL);

    *record = pInfo->record;
    return AGCLIENT_CONTINUE;
}
/*---------------------------------------------------------------------------*/
static int32 
getNextModifiedRecord(void *out, AGRecord **record, int32 *errCode)
{
    if (verbose)
        printf("GetNextModifiedRecord()\n");

    return getRecordBase((PalmSyncInfo *)out, TRUE, record, errCode);
}
/*---------------------------------------------------------------------------*/
static int32 
getNextRecord (void *out, AGRecord **record, int32 *errCode)
{
    if (verbose)
        printf("GetNextRecord()\n");

    return getRecordBase((PalmSyncInfo *)out, FALSE, record, errCode);
}
/*---------------------------------------------------------------------------*/
static AGBool doClientProcessorLoop(PalmSyncInfo *pInfo, AGNetCtx *ctx) {
    int32 dummyError;
    int32 cpResult;
    int32 syncCount;
    int32 i, n;
    AGBool cancelled = FALSE;
    AGLocationConfig *lc = NULL;
    int migrate = 0;

    n = AGUserConfigCount(pInfo->userConfig);

    /* Lets check for 3.1 to 3.x migration here */
    if (n == 1) {
        AGServerConfig * sc =
            AGUserConfigGetServerByIndex(pInfo->userConfig, 0);
        
        /* Yup, no server name */
        if (!sc->serverName) {
            /* Lets look for the 3.1 config database */
            long result;
            int userConfigDBHandle = 0;
            result = dlp_OpenDB(pInfo->sd,
                                DEFAULT_CARD_NUM,
                                dlpOpenRead | dlpOpenWrite,
                                DEVICE_USERCONFIG_DB_NAME, 
                                &userConfigDBHandle);
            if (result > 0) {
                char response[2];

                /* FIXME
                 * Commented out for now, until we can figure out
                 * a way to deal with this
                 *
                printf("It looks like you recently upgraded your client"
                       ". Would you\nlike to migrate your old settings"
                       "?[y/n] ");
                if ((fgets(response, 2, stdin) != 0)
                */
                if (0
                    && ((response[0] == 'y') || (response[0] == 'Y'))) {
           
                    pInfo->threeone = 1;
                    /* Retrieve device's idea of current userConfig. */
                    result = readDeviceUserConfig(pInfo->sd,
                                                  userConfigDBHandle,
                                                  &pInfo->userConfig, 
                                                  &pInfo->threeone);

                    /* This will cause us to write out the 3.1 data 
                       as 3.2 when we finish syncing */
                    pInfo->threeone = 0;
                    
                    /* Done with database for now, so close it. */
                    dlp_CloseDB(pInfo->sd, userConfigDBHandle);
                    

                    n = AGUserConfigCount(pInfo->userConfig);
                    
                    /* set a flag to delete the old database */
                    migrate = 1;

                }
            }
        }
    }
                    
    for (i = 0; i < n; ++i) {

        AGServerConfig * sc =
            AGUserConfigGetServerByIndex(pInfo->userConfig, i);
        
        if (cancelled)
            continue;

        if (NULL == sc)
            continue;

        if (sc->disabled)
            continue;

        if (NULL == sc->serverName || sc->serverPort <= 0) 
            continue;

        syncCount = 0;
        doStartServer(pInfo, sc, &dummyError);
 
        do {
            AGCommandProcessorStart(pInfo->commandProcessor);
            
            pInfo->deviceInfo = AGDeviceInfoNew();
            /* If this fails, we're totally dead.  Exit with fail code. */
            if(!pInfo->deviceInfo)
                return FALSE;
                
            readDeviceInfo(pInfo);

            if (pInfo->httpProxy && pInfo->httpProxyPort) {
                if (verbose) {
                    printf("Setting proxy to %s and port to %d\n",
                           pInfo->httpProxy, pInfo->httpProxyPort);
                }   
                lc = AGLocationConfigNew();
                lc->HTTPUseProxy = 1;
                lc->HTTPName = pInfo->httpProxy;
                lc->HTTPPort = pInfo->httpProxyPort;
                if (pInfo->proxyUsername && pInfo->proxyPassword) {
                    if (verbose) {
                        printf("Setting proxy user to %s and password to %s\n",
                               pInfo->proxyUsername, pInfo->proxyPassword);
                    }   
                    lc->HTTPUseAuthentication = 1;
                    lc->HTTPUsername = pInfo->proxyUsername;
                    lc->HTTPPassword = pInfo->proxyPassword;
                }
            }

            if (pInfo->socksProxy && pInfo->socksProxyPort) {
                if (verbose) {
                    printf("Setting socks proxy to %s and port to %d\n",
                           pInfo->socksProxy, pInfo->socksProxyPort);
                }   
                if (!lc) {
                    lc = AGLocationConfigNew();
                }
                lc->SOCKSUseProxy = 1;
                lc->SOCKSName = pInfo->socksProxy;
                lc->SOCKSPort = pInfo->socksProxyPort;
            }
            
            pInfo->clientProcessor =
                AGClientProcessorNew(pInfo->currentServerConfig,
                                     pInfo->deviceInfo, 
                                     lc,
                                     pInfo->platform, 
                                     TRUE,
                                     ctx);
                                        
            /* If this fails, we're totally dead.  Exit with fail code. */
            if (NULL == pInfo->clientProcessor) {
                AGDeviceInfoFree(pInfo->deviceInfo);
                return FALSE;
            }

            pInfo->clientProcessor->version_info = 
                malloc(sizeof(AGExpansionVersion));
            if (NULL == pInfo->clientProcessor->version_info) 
                exit(-1);

            fill_in_versioninfo(pInfo->sd, pInfo->clientProcessor->version_info);

            AGClientProcessorSetBufferServerCommands(pInfo->clientProcessor,
                                                     FALSE);
            
            AGClientProcessorSync(pInfo->clientProcessor);


            cpResult = AGCLIENT_CONTINUE;
            while (AGCLIENT_CONTINUE == cpResult) {
                cpResult = AGClientProcessorProcess(pInfo->clientProcessor);
                
                if (AGCLIENT_CONTINUE == cpResult && pInfo->quit) {
                    cancelled = TRUE;
                    cpResult = AGCLIENT_IDLE;
                }
                if (dlp_OpenConduit(pInfo->sd) < 0) {
                    lm_errno = LM_EXIT_CANCEL;
                    /*
                     * FIXME
                     * This used to be an exit() call
                     * Investigate what else we need to do to report an
                     * error and clean up.
                     */
                    return 1;
                }
                
            }


            if (cpResult == AGCLIENT_ERR) {
                /* FIXME */
                char *msg = AGGetMsg(pInfo->clientProcessor->errStringId);
                if (verbose) {
                    if (msg)
                        fprintf(stderr,"%s\n", msg);
                    else
                        fprintf(stderr,"Unknown error\n");
                }
                lm_errno = LM_CLIENT;
            }

            if (pInfo->clientProcessor->version_info)
                free(pInfo->clientProcessor->version_info);

            AGClientProcessorFree(pInfo->clientProcessor);
            AGDeviceInfoFree(pInfo->deviceInfo);

        } while (!cancelled
                 && AGCommandProcessorShouldSyncAgain(pInfo->commandProcessor) 
                 && syncCount++ < MAX_SYNCS);

        doEndServer(pInfo, &dummyError);

        /* It's possible to cancel in the middle of an open database,
           so if that happened, close it. */
        if (pInfo->pilot_rHandle) {
            closeDatabase(pInfo);
        }

        /* If we migrated from 3.1 to 3.2 delete the 3.1 database */
        if (migrate) {
            dlp_DeleteDB (pInfo->sd, DEFAULT_CARD_NUM, DEVICE_USERCONFIG_DB_NAME);
        }
    }

    return TRUE; /* success */
}
/*---------------------------------------------------------------------------*/
void Disconnect (PalmSyncInfo *pi) {
    if (pi->sd == 0) {
        return;
    }
    
    dlp_EndOfSync(pi->sd, 0);
    pi_close(pi->sd);
    pi->sd = 0;
}

void Connect(PalmSyncInfo *pi) {
    if (pi->sd != 0)
        return;

    if ((pi->sd = pilot_connect(pi->device)) < 0) {
        perror("pilot_connect");
        exit(1);
    }
    
    if( verbose )
      puts("Connected");

}

#define AG_EXPANSION_VERSION_NONE (-1)
#define AVANTGO_VERSION_DATABASE_NAME   "AvGoVersion"
#define AVANTGO_AGVERSION_DATABASE_NAME "AGVersion"

typedef struct ClientVersion {
    uint32 dataCompatibility;
    uint32 major;
    uint32 minor;
    uint32 buildno;
} ClientVersion;

void ClientVersionReadData(ClientVersion *cversion, uint32 len, uint8 *buf)
{
    int32 version;
    AGBufferReader r;

    bzero(cversion, sizeof(ClientVersion));

    AGBufferReaderInit(&r, buf);
    AGReadInt32((AGReader *)&r); /* old data compatibility field */
    if (len > sizeof(int32)) {
        version = AGReadInt32((AGReader *)&r);
        cversion->dataCompatibility = AGReadInt32((AGReader *)&r);
        cversion->major = AGReadInt32((AGReader *)&r);
        cversion->minor = AGReadInt32((AGReader *)&r);
        cversion->buildno = AGReadInt32((AGReader *)&r);
    }
    else {
        cversion->dataCompatibility = 1;
    }
    AGBufferReaderFinalize(&r);
}

/*----------------------------------------------------------------------------*/
static void 
fill_in_versioninfo(int sd, AGExpansionVersion *vi)
{
    int database_id = 0;
    long result = -1;

    /* Assume failure of device portion and write reasonable values. */
    vi->device_major = AG_EXPANSION_VERSION_NONE;
    vi->device_minor = AG_EXPANSION_VERSION_NONE;
    vi->device_build = AG_EXPANSION_VERSION_NONE;
    vi->conn_agent_major = 1;
    vi->conn_agent_minor = 0;
    vi->conn_agent_build = 0;

    result = dlp_OpenDB(sd, 
                        DEFAULT_CARD_NUM, 
                        dlpOpenRead,  
                        AVANTGO_AGVERSION_DATABASE_NAME,
                        &database_id);
    
    /* 4.x and earlier client has AvGoVersion db, try that */
    if (result < 0)
        result = dlp_OpenDB(sd, 
                            DEFAULT_CARD_NUM, 
                            dlpOpenRead,  
                            AVANTGO_VERSION_DATABASE_NAME,
                            &database_id);

    
    if (result >= 0) {
        int attr = 0;
        int cat  = 0;
        recordid_t id;
        int rc;
        int version_buffer_size = 0x1000;
        uint8 version_buffer[version_buffer_size];


#ifdef PILOT_LINK_0_12
        rc = dlp_ReadRecordByIndex(sd, database_id, 
                                   0, 
                                   (void *)version_buffer, 
                                   &id, 
                                   &attr, 
                                   &cat);
#else
        rc = dlp_ReadRecordByIndex(sd, database_id, 
                                   0, 
                                   (void *)version_buffer, 
                                   &id, 
                                   &version_buffer_size,
                                   &attr, 
                                   &cat);
#endif

        if (rc >= 0) {
            ClientVersion cv;
            ClientVersionReadData(&cv,
                                  version_buffer_size,
                                  version_buffer);
            vi->device_major = cv.major;
            vi->device_minor = cv.minor;
            vi->device_build = cv.buildno;

            vi->conn_agent_major = cv.major;
            vi->conn_agent_minor = cv.minor;
            vi->conn_agent_build = cv.buildno;
        }
        dlp_CloseDB(sd, database_id);

    }

}

int malsync (PalmSyncInfo *pInfo) {
    uint32 pilotID;
    AGNetCtx *ctx;

    if (dlp_OpenConduit (pInfo->sd) < 0) {
        lm_errno = LM_EXIT_CANCEL;
        return 1;
    }

    /*
     * FIXME
     * seclib stuff disabled for now
     */
    ctx = (AGNetCtx *)malloc(sizeof(AGNetCtx));
    AGNetInit(ctx);

    if (setupPlatformCalls(pInfo)) {
        /*
         * FIXME
         * set lm_errno to something appropriate
         */
        return -1;
    }

    pInfo->userConfig = getUserConfig(pInfo->sd, &pilotID, &pInfo->threeone);
    if (pInfo->userConfig) {
        doClientProcessorLoop (pInfo, ctx);
        storeDeviceUserConfig (pInfo->sd, pInfo->userConfig, pilotID);
    }
    else {
        lm_errno = LM_USER_CONFIG;
    }

    AGNetClose(ctx);

    free(ctx);

    return 0;
}
