/*
 * lib/route/route.c	Routes
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 *
 * Copyright (c) 2003-2005 Thomas Graf <tgraf@suug.ch>
 */

/**
 * @ingroup rtnl
 * @defgroup route Routing
 * @brief
 * @{
 */

#include <netlink-local.h>
#include <netlink/netlink.h>
#include <netlink/cache.h>
#include <netlink/utils.h>
#include <netlink/data.h>
#include <netlink/route/rtnl.h>
#include <netlink/route/route.h>
#include <netlink/route/link.h>

/** @cond SKIP */
#define ROUTE_ATTR_FAMILY    0x000001
#define ROUTE_ATTR_DST_LEN   0x000002
#define ROUTE_ATTR_SRC_LEN   0x000004
#define ROUTE_ATTR_TOS       0x000008
#define ROUTE_ATTR_TABLE     0x000010
#define ROUTE_ATTR_PROTOCOL  0x000020
#define ROUTE_ATTR_SCOPE     0x000040
#define ROUTE_ATTR_TYPE      0x000080
#define ROUTE_ATTR_FLAGS     0x000100
#define ROUTE_ATTR_DST       0x000200
#define ROUTE_ATTR_SRC       0x000400
#define ROUTE_ATTR_IIF       0x000800
#define ROUTE_ATTR_OIF       0x001000
#define ROUTE_ATTR_GATEWAY   0x002000
#define ROUTE_ATTR_PRIO      0x004000
#define ROUTE_ATTR_PREF_SRC  0x008000
#define ROUTE_ATTR_METRICS   0x010000
#define ROUTE_ATTR_MULTIPATH 0x020000
#define ROUTE_ATTR_REALMS    0x040000
#define ROUTE_ATTR_CACHEINFO 0x080000
#define ROUTE_ATTR_PROTOINFO 0x100000
#define ROUTE_ATTR_MP_ALGO   0x200000

#define NEXTHOP_HAS_FLAGS   0x000001
#define NEXTHOP_HAS_HOPS    0x000002
#define NEXTHOP_HAS_IFINDEX 0x000004
#define NEXTHOP_HAS_GATEWAY 0x000008

static struct nl_cache_ops rtnl_route_ops;
/** @endcond */

static void route_free_data(struct nl_object *c)
{
	struct rtnl_route *r = (struct rtnl_route *) c;

	if (r == NULL)
		return;

	nl_data_free(r->rt_protoinfo);
	nl_addr_put(r->rt_dst);
	nl_addr_put(r->rt_src);
	nl_addr_put(r->rt_gateway);
	nl_addr_put(r->rt_pref_src);

	while (r->rt_nexthops) {
		struct rtnl_nexthop *nh = r->rt_nexthops;
		r->rt_nexthops = nh->rtnh_next;
		nl_addr_put(nh->rtnh_gateway);
		free(nh);
	}
}

static struct nla_policy route_policy[RTA_MAX+1] = {
	[RTA_IIF]	= { .type = NLA_STRING,
			    .maxlen = IFNAMSIZ, },
	[RTA_OIF]	= { .type = NLA_U32 },
	[RTA_PRIORITY]	= { .type = NLA_U32 },
	[RTA_FLOW]	= { .type = NLA_U32 },
	[RTA_MP_ALGO]	= { .type = NLA_U32 },
	[RTA_CACHEINFO]	= { .minlen = sizeof(struct rta_cacheinfo) },
	[RTA_METRICS]	= { .type = NLA_NESTED },
	[RTA_MULTIPATH]	= { .type = NLA_NESTED },
};

static int route_msg_parser(struct sockaddr_nl *who, struct nlmsghdr *n,
			    void *arg)
{
	struct nl_parser_param *pp = arg;
	struct rtnl_route *route;
	struct nlattr *tb[RTA_MAX + 1];
	struct rtmsg *r = nlmsg_data(n);
	int family, err;

	route = rtnl_route_alloc();
	route->ce_msgtype = n->nlmsg_type;

	err = nlmsg_parse(n, sizeof(*r), tb, RTA_MAX, route_policy);
	if (err < 0)
		goto errout;

	route->rt_family = family = r->rtm_family;
	route->rt_dst_len = r->rtm_dst_len;
	route->rt_src_len = r->rtm_src_len;
	route->rt_tos = r->rtm_tos;
	route->rt_table = r->rtm_table;
	route->rt_type = r->rtm_type;
	route->rt_scope = r->rtm_scope;
	route->rt_protocol = r->rtm_protocol;
	route->rt_flags = r->rtm_flags;

	route->rt_mask = (ROUTE_ATTR_FAMILY | ROUTE_ATTR_DST_LEN | 
			  ROUTE_ATTR_SRC_LEN | ROUTE_ATTR_TABLE |
			  ROUTE_ATTR_PROTOCOL| ROUTE_ATTR_SCOPE |
			  ROUTE_ATTR_TYPE | ROUTE_ATTR_FLAGS);

	if (route->rt_tos)
		route->rt_mask |= ROUTE_ATTR_TOS;

	if (tb[RTA_DST]) {
		route->rt_dst = nla_get_addr(tb[RTA_DST], family);
		if (!route->rt_dst)
			goto errout_inherit;
		nl_addr_set_prefixlen(route->rt_dst, route->rt_dst_len);
		route->rt_mask |= ROUTE_ATTR_DST;
	}

	if (tb[RTA_SRC]) {
		route->rt_src = nla_get_addr(tb[RTA_SRC], family);
		if (!route->rt_src)
			goto errout_inherit;
		nl_addr_set_prefixlen(route->rt_src, route->rt_src_len);
		route->rt_mask |= ROUTE_ATTR_SRC;
	}

	if (tb[RTA_IIF]) {
		nla_strlcpy(route->rt_iif, tb[RTA_IIF], IFNAMSIZ);
		route->rt_mask |= ROUTE_ATTR_IIF;
	}

	if (tb[RTA_OIF]) {
		route->rt_oif = nla_get_u32(tb[RTA_OIF]);
		route->rt_mask |= ROUTE_ATTR_OIF;
	}

	if (tb[RTA_GATEWAY]) {
		route->rt_gateway = nla_get_addr(tb[RTA_GATEWAY], family);
		if (!route->rt_gateway)
			goto errout_inherit;
		route->rt_mask |= ROUTE_ATTR_GATEWAY;
	}

	if (tb[RTA_PRIORITY]) {
		route->rt_prio = nla_get_u32(tb[RTA_PRIORITY]);
		route->rt_mask |= ROUTE_ATTR_PRIO;
	}

	if (tb[RTA_PREFSRC]) {
		route->rt_pref_src = nla_get_addr(tb[RTA_PREFSRC], family);
		if (!route->rt_pref_src)
			goto errout_inherit;
		route->rt_mask |= ROUTE_ATTR_PREF_SRC;
	}

	if (tb[RTA_METRICS]) {
		struct nlattr *mtb[RTAX_MAX + 1];
		int i;

		err = nla_parse_nested(mtb, RTAX_MAX, tb[RTA_METRICS], NULL);
		if (err < 0)
			goto errout;

		for (i = 1; i <= RTAX_MAX; i++) {
			if (mtb[i] && nla_len(mtb[i]) >= sizeof(uint32_t)) {
				uint32_t m = nla_get_u32(mtb[i]);
				route->rt_metrics[i-1] = m;
				route->rt_metrics_mask |= (1<<(i-1));
			}
		}
		
		route->rt_mask |= ROUTE_ATTR_METRICS;
	}

	if (tb[RTA_MULTIPATH]) {
		struct rtnl_nexthop *nh;
		struct rtnexthop *rtnh = nla_data(tb[RTA_MULTIPATH]);
		size_t tlen = nla_len(tb[RTA_MULTIPATH]);

		while (tlen >= sizeof(*rtnh) && tlen >= rtnh->rtnh_len) {
			nh = calloc(1, sizeof(*nh));
			if (!nh) {
				err = nl_errno(ENOMEM);
				goto errout;
			}

			nh->rtnh_flags = rtnh->rtnh_flags;
			nh->rtnh_hops = rtnh->rtnh_hops;
			nh->rtnh_ifindex = rtnh->rtnh_ifindex;
			nh->rtnh_mask = (NEXTHOP_HAS_FLAGS | NEXTHOP_HAS_HOPS
					| NEXTHOP_HAS_IFINDEX);
				
			if (rtnh->rtnh_len > sizeof(*rtnh)) {
				struct nlattr *ntb[RTA_MAX + 1];
				nla_parse(ntb, RTA_MAX, (struct nlattr *)
					  RTNH_DATA(rtnh),
					  rtnh->rtnh_len - sizeof(*rtnh),
					  route_policy);

				if (ntb[RTA_GATEWAY]) {
					nh->rtnh_gateway = nla_get_addr(
							ntb[RTA_GATEWAY],
							route->rt_family);
					nh->rtnh_mask = NEXTHOP_HAS_GATEWAY;
				}
			}

			nh->rtnh_next = route->rt_nexthops;
			route->rt_nexthops = nh;

			tlen -= RTNH_ALIGN(rtnh->rtnh_len);
			rtnh = RTNH_NEXT(rtnh);
		}

		route->rt_mask |= ROUTE_ATTR_MULTIPATH;
	}

	/* Not sure if there are any users not using this for fwmark,
	 * allocating for now */
	if (tb[RTA_PROTOINFO]) {
		route->rt_protoinfo = nla_get_data(tb[RTA_PROTOINFO]);
		if (!route->rt_protoinfo)
			goto errout_inherit;
		route->rt_mask |= ROUTE_ATTR_PROTOINFO;
	}

	if (tb[RTA_FLOW]) {
		route->rt_realms = nla_get_u32(tb[RTA_FLOW]);
		route->rt_mask |= ROUTE_ATTR_REALMS;
	}

	if (tb[RTA_CACHEINFO]) {
		struct rta_cacheinfo *ci = nla_data(tb[RTA_CACHEINFO]);

		route->rt_cacheinfo.rtci_clntref  = ci->rta_clntref;
		route->rt_cacheinfo.rtci_last_use = ci->rta_lastuse;
		route->rt_cacheinfo.rtci_expires  = ci->rta_expires;
		route->rt_cacheinfo.rtci_error    = ci->rta_error;
		route->rt_cacheinfo.rtci_used     = ci->rta_used;
		route->rt_cacheinfo.rtci_id       = ci->rta_id;
		route->rt_cacheinfo.rtci_ts       = ci->rta_ts;
		route->rt_cacheinfo.rtci_tsage    = ci->rta_tsage;

		route->rt_mask |= ROUTE_ATTR_CACHEINFO;
	}

	if (tb[RTA_MP_ALGO]) {
		route->rt_mp_algo = nla_get_u32(tb[RTA_MP_ALGO]);
		route->rt_mask |= ROUTE_ATTR_MP_ALGO;
	}

	err = pp->pp_cb((struct nl_object *) route, pp);
	if (err < 0)
		goto errout;

	return P_ACCEPT;

errout_inherit:
	err = nl_get_errno();
errout:
	rtnl_route_put(route);
	return err;
}

static int route_request_update(struct nl_cache *c, struct nl_handle *h)
{
	return nl_rtgen_request(h, RTM_GETROUTE, AF_UNSPEC, NLM_F_DUMP);
}

static int route_dump_brief(struct nl_object *a, struct nl_dump_params *p)
{
	struct rtnl_route *r = (struct rtnl_route *) a;
	struct nl_cache *link_cache;

	link_cache = nl_cache_mngt_require("route/link");

	if (r->rt_mask & ROUTE_ATTR_DST) {
		char dst[INET6_ADDRSTRLEN+5];
		dp_dump(p, "%s ", nl_addr2str(r->rt_dst, dst, sizeof(dst)));
	} else if (r->rt_dst_len)
		dp_dump(p, "0/%u ", r->rt_dst_len);
	else
		dp_dump(p, "default ");

	if (r->rt_mask & ROUTE_ATTR_GATEWAY) {
		char via[INET6_ADDRSTRLEN+5];
		dp_dump(p, "via %s ", nl_addr2str(r->rt_gateway, via, sizeof(via)));
	}

	if (r->rt_mask & ROUTE_ATTR_OIF) {
		if (link_cache) {
			char buf[32];
			dp_dump(p, "dev %s ",
				rtnl_link_i2name(link_cache, r->rt_oif,
						 buf, sizeof(buf)));
		} else
			dp_dump(p, "dev %d ", r->rt_oif);
	}

	if (r->rt_mask & ROUTE_ATTR_SCOPE) {
		char buf[32];
		dp_dump(p, "scope %s ",
			rtnl_scope2str(r->rt_scope, buf, sizeof(buf)));
	}

	if (r->rt_mask & ROUTE_ATTR_FLAGS && r->rt_flags) {
		int flags = r->rt_flags;

		dp_dump(p, "<");
		
#define PRINT_FLAG(f) if (flags & RTNH_F_##f) { \
		flags &= ~RTNH_F_##f; dp_dump(p, #f "%s", flags ? "," : ""); }
		PRINT_FLAG(DEAD);
		PRINT_FLAG(ONLINK);
		PRINT_FLAG(PERVASIVE);
#undef PRINT_FLAG

#define PRINT_FLAG(f) if (flags & RTM_F_##f) { \
		flags &= ~RTM_F_##f; dp_dump(p, #f "%s", flags ? "," : ""); }
		PRINT_FLAG(NOTIFY);
		PRINT_FLAG(CLONED);
		PRINT_FLAG(EQUALIZE);
		PRINT_FLAG(PREFIX);
#undef PRINT_FLAG

		dp_dump(p, ">");
	}

	dp_dump(p, "\n");

	return 1;
}

static int route_dump_full(struct nl_object *a, struct nl_dump_params *p)
{
	struct rtnl_route *r = (struct rtnl_route *) a;
	char buf[128];

	int line = route_dump_brief(a, p);

	dp_dump_line(p, line++, "  ");

	if (r->rt_mask & ROUTE_ATTR_TYPE)
		dp_dump(p, "%s ",
			nl_rtntype2str(r->rt_type, buf, sizeof(buf)));

	if (r->rt_mask & ROUTE_ATTR_TABLE)
		dp_dump(p, "table %s ",
			rtnl_route_table2str(r->rt_table, buf, sizeof(buf)));

	if (r->rt_mask & ROUTE_ATTR_PROTOCOL)
		dp_dump(p, "protocol %u ", r->rt_protocol);

	if (r->rt_mask & ROUTE_ATTR_PRIO)
		dp_dump(p, "metric %u ", r->rt_prio);

	if (r->rt_mask & ROUTE_ATTR_PREF_SRC)
		dp_dump(p, "preferred-src %s ",
			nl_addr2str(r->rt_pref_src, buf, sizeof(buf)));

	if (r->rt_mask & ROUTE_ATTR_TOS)
		dp_dump(p, "dsfield %#x ", r->rt_tos);

	dp_dump(p, "\n");

	return line;
}

static int route_dump_stats(struct nl_object *obj, struct nl_dump_params *p)
{
	struct rtnl_route *route = (struct rtnl_route *) obj;
	int line;

	line = route_dump_full(obj, p);

	if (route->rt_mask & ROUTE_ATTR_CACHEINFO) {
		struct rtnl_rtcacheinfo *ci = &route->rt_cacheinfo;
		dp_dump_line(p, line++, "  used %u error %u refcnt %u\n",
			     ci->rtci_used, ci->rtci_error, ci->rtci_clntref);
		dp_dump_line(p, line++, "  last-use %us expires %us\n",
			     ci->rtci_last_use / nl_get_hz(),
			     ci->rtci_expires / nl_get_hz());
	}

	return line;
}

static int route_dump_xml(struct nl_object *obj, struct nl_dump_params *p)
{
	struct rtnl_route *route = (struct rtnl_route *) obj;
	char buf[128];
	int line = 0;
	
	dp_dump_line(p, line++, "<route>\n");
	dp_dump_line(p, line++, "  <family>%s</family>\n",
		     nl_af2str(route->rt_family, buf, sizeof(buf)));

	if (route->rt_mask & ROUTE_ATTR_DST)
		dp_dump_line(p, line++, "  <dst>%s</dst>\n",
			     nl_addr2str(route->rt_dst, buf, sizeof(buf)));

	if (route->rt_mask & ROUTE_ATTR_DST_LEN)
		dp_dump_line(p, line++, "  <dstlen>%u</dstlen>\n",
			     route->rt_dst_len);

	if (route->rt_mask & ROUTE_ATTR_SRC)
		dp_dump_line(p, line++, "  <src>%s</src>\n",
			     nl_addr2str(route->rt_src, buf, sizeof(buf)));

	if (route->rt_mask & ROUTE_ATTR_SRC_LEN)
		dp_dump_line(p, line++, "  <srclen>%u</srclen>\n",
			     route->rt_src_len);

	if (route->rt_mask & ROUTE_ATTR_GATEWAY)
		dp_dump_line(p, line++, "  <gateway>%s</gateway>\n",
			     nl_addr2str(route->rt_gateway, buf, sizeof(buf)));

	if (route->rt_mask & ROUTE_ATTR_PREF_SRC)
		dp_dump_line(p, line++, "  <prefsrc>%s</prefsrc>\n",
			     nl_addr2str(route->rt_pref_src, buf, sizeof(buf)));

	if (route->rt_mask & ROUTE_ATTR_IIF)
		dp_dump_line(p, line++, "  <iif>%s</iif>\n", route->rt_iif);

	if (route->rt_mask & ROUTE_ATTR_REALMS)
		dp_dump_line(p, line++, "  <realms>%u</realms>\n",
			     route->rt_realms);

	if (route->rt_mask & ROUTE_ATTR_TOS)
		dp_dump_line(p, line++, "  <tos>%u</tos>\n", route->rt_tos);

	if (route->rt_mask & ROUTE_ATTR_TABLE)
		dp_dump_line(p, line++, "  <table>%u</table>\n",
			     route->rt_table);

	if (route->rt_mask & ROUTE_ATTR_SCOPE)
		dp_dump_line(p, line++, "  <scope>%s</scope>\n",
			     rtnl_scope2str(route->rt_scope, buf, sizeof(buf)));

	if (route->rt_mask & ROUTE_ATTR_PRIO)
		dp_dump_line(p, line++, "  <metric>%u</metric>\n",
			     route->rt_prio);

	if (route->rt_mask & ROUTE_ATTR_OIF) {
		struct nl_cache *link_cache;
	
		link_cache = nl_cache_mngt_require("route/link");
		if (link_cache)
			dp_dump_line(p, line++, "  <oif>%s</oif>\n",
				     rtnl_link_i2name(link_cache,
						      route->rt_oif,
						      buf, sizeof(buf)));
		else
			dp_dump_line(p, line++, "  <oif>%u</oif>\n",
				     route->rt_oif);
	}

	if (route->rt_mask & ROUTE_ATTR_TYPE)
		dp_dump_line(p, line++, "  <type>%s</type>\n",
			     nl_rtntype2str(route->rt_type, buf, sizeof(buf)));

	dp_dump_line(p, line++, "</route>\n");

#if 0
	uint8_t			rt_protocol;
	uint32_t		rt_flags;
	uint32_t		rt_metrics[RTAX_MAX];
	uint32_t		rt_metrics_mask;
	struct rtnl_nexthop *	rt_nexthops;
	struct rtnl_rtcacheinfo	rt_cacheinfo;
	struct nl_data *	rt_protoinfo;
	uint32_t		rt_mp_algo;

#endif

	return line;
}

static int route_filter(struct nl_object *obj, struct nl_object *filter)
{
	struct rtnl_route *o = (struct rtnl_route *) obj;
	struct rtnl_route *f = (struct rtnl_route *) filter;

#define REQ(F) (f->rt_mask & ROUTE_ATTR_##F)
#define AVAIL(F) (o->rt_mask & ROUTE_ATTR_##F)
#define _O(F, EXPR) (REQ(F) && (!AVAIL(F) || (EXPR)))
#define _C(F, N) (REQ(F) && (!AVAIL(F) || (o->N != f->N)))
	if (_C(FAMILY,	  rt_family)					||
	    _C(DST_LEN,	  rt_dst_len)					||
	    _C(SRC_LEN,	  rt_src_len)					||
	    _C(TOS,	  rt_tos)					||
	    _C(TABLE,	  rt_table)					||
	    _C(PROTOCOL,  rt_protocol)					||
	    _C(SCOPE,	  rt_scope)					||
	    _C(TYPE,	  rt_type)					||
	    _C(OIF,	  rt_oif)					||
	    _C(PRIO,	  rt_prio)					||
	    _C(REALMS,	  rt_realms)					||
	    _C(MP_ALGO,	  rt_mp_algo)					||
	    _O(DST,	  nl_addr_cmp(o->rt_dst, f->rt_dst))		||
	    _O(SRC,	  nl_addr_cmp(o->rt_src, f->rt_src))		||
	    _O(IIF,	  strcmp(o->rt_iif, f->rt_iif))			||
	    _O(PREF_SRC,  nl_addr_cmp(o->rt_pref_src, f->rt_pref_src))	||
	    _O(GATEWAY,	  nl_addr_cmp(o->rt_gateway, f->rt_gateway))	||
	    _O(FLAGS,	  f->rt_flags ^ (o->rt_flags & f->rt_flag_mask)))
		return 0;

	if (REQ(METRICS)) {
		int i;

		if (!AVAIL(METRICS))
			return 0;

		for (i = 0; i < RTAX_MAX; i++) {
			if (f->rt_metrics_mask & (1 << i)) {
				if (!(o->rt_metrics_mask & (1 << i)) ||
				    f->rt_metrics[i+1] != o->rt_metrics[i+1])
					return 0;
			}
		}
	}
	
#undef REQ
#undef AVAIL
#undef _O
#undef _C

#if 0
#define ROUTE_ATTR_MULTIPATH 0x020000
#define ROUTE_ATTR_CACHEINFO 0x080000
#define ROUTE_ATTR_PROTOINFO 0x200000

	struct rtnl_nexthop *	rt_nexthops;
	struct rtnl_rtcacheinfo	rt_cacheinfo;
	struct nl_data		rt_protoinfo;
#endif

	return 1;

}

/**
 * @name Route Object Allocation/Freeage
 * @{
 */

/**
 * Allocate a new route object
 * @return New route object
 */
struct rtnl_route *rtnl_route_alloc(void)
{
	return (struct rtnl_route *) nl_object_alloc_from_ops(&rtnl_route_ops);
}

/**
 * Give back reference on route object.
 * @arg route		Route object to be given back.
 *
 * Decrements the reference counter and frees the object if the
 * last reference has been released.
 */
void rtnl_route_put(struct rtnl_route *route)
{
	nl_object_put((struct nl_object *) route);
}
/**
 * Free route object.
 * @arg route		Route object to be freed.
 *
 * @note Always use rtnl_route_put() unless you're absolutely sure
 *       that no other user may have a reference on this object.
 */
void rtnl_route_free(struct rtnl_route *route)
{
	nl_object_free((struct nl_object *) route);
}

/** @} */

/**
 * @name Route Cache Management
 * @{
 */

/**
 * Build a route cache including all rout currently configured in the kernel
 * @arg handle		netlink handle
 *
 * Allocates a new cache, initializes it properly and updates it to
 * include all routes currently configured in the kernel.
 *
 * @note The caller is responsible for destroying and freeing the
 *       cache after using it. (nl_cache_destroy_and_free())
 * @return The cache or NULL if an error has occured.
 */
struct nl_cache *rtnl_route_alloc_cache(struct nl_handle *handle)
{
	struct nl_cache *cache = nl_cache_alloc_from_ops(&rtnl_route_ops);

	if (!cache)
		return NULL;

	if (nl_cache_update(handle, cache) < 0) {
		free(cache);
		return NULL;
	}

	return cache;
}

/** @} */

/**
 * @name Attribute Modifications
 * @{
 */

/**
 * Set the table of a route to the specified value
 * @arg route		route to be changed
 * @arg table		new table value
 */
void rtnl_route_set_table(struct rtnl_route *route, int table)
{
	route->rt_table = table;
	route->rt_mask |= ROUTE_ATTR_TABLE;
}

/**
 * Get the table of a route
 * @arg route		route handle
 * @return Table id or -1 if not set
 */
int rtnl_route_get_table(struct rtnl_route *route)
{
	if (route->rt_mask & ROUTE_ATTR_TABLE)
		return route->rt_table;
	else
		return -1;
}

/**
 * Set the scope of a route to the specified value
 * @arg route		route to be changed
 * @arg scope		new scope
 */
void rtnl_route_set_scope(struct rtnl_route *route, int scope)
{
	route->rt_scope = scope;
	route->rt_mask |= ROUTE_ATTR_SCOPE;
}

/**
 * Get the scope of a route
 * @arg route		route handle
 * @return Scope or -1 if not set
 */
int rtnl_route_get_scope(struct rtnl_route *route)
{
	if (route->rt_mask & ROUTE_ATTR_SCOPE)
		return route->rt_scope;
	else
		return -1;
}

/**
 * Set the TOS of a route to the specified value
 * @arg route		route to be changed
 * @arg tos		new TOS value
 */
void rtnl_route_set_tos(struct rtnl_route *route, int tos)
{
	route->rt_tos = tos;
	route->rt_mask |= ROUTE_ATTR_TOS;
}

/**
 * Get the TOS of a route
 * @arg route		route handle
 * @return TOS value or -1 if not set
 */
int rtnl_route_get_tos(struct rtnl_route *route)
{
	if (route->rt_mask & ROUTE_ATTR_TOS)
		return route->rt_tos;
	else
		return -1;
}

/**
 * Set the realms of a route to the specified value
 * @arg route		route to be changed
 * @arg realms		New realms value.
 */
void rtnl_route_set_realms(struct rtnl_route *route, realm_t realms)
{
	route->rt_realms = realms;
	route->rt_mask |= ROUTE_ATTR_REALMS;
}

/**
 * Get realms of route object.
 * @arg route		Route object.
 * @return Realms value or 0 if not set.
 */
realm_t rtnl_route_get_realms(struct rtnl_route *route)
{
	if (route->rt_mask & ROUTE_ATTR_REALMS)
		return route->rt_realms;
	else
		return 0;
}

/**
 * Set the protocol of a route to the specified value
 * @arg route		route to be changed
 * @arg proto		new protocol
 */
void rtnl_route_set_protocol(struct rtnl_route *route, int proto)
{
	route->rt_protocol = proto;
	route->rt_mask |= ROUTE_ATTR_PROTOCOL;
}

/**
 * Get the protocol of a route
 * @arg route		route handle
 * @return Protocol number or -1 if not set
 */
int rtnl_route_get_protocol(struct rtnl_route *route)
{
	if (route->rt_mask & ROUTE_ATTR_PROTOCOL)
		return route->rt_protocol;
	else
		return -1;
}

/**
 * Set the priority of a route to the specified value
 * @arg route		route to be changed
 * @arg prio		new priority
 */
void rtnl_route_set_prio(struct rtnl_route *route, int prio)
{
	route->rt_prio = prio;
	route->rt_mask |= ROUTE_ATTR_PRIO;
}

/**
 * Get the priority of a route
 * @arg route		route handle
 * @return Priority or -1 if not set
 */
int rtnl_route_get_prio(struct rtnl_route *route)
{
	if (route->rt_mask & ROUTE_ATTR_PRIO)
		return route->rt_prio;
	else
		return -1;
}

/**
 * Set the address family of a route to the specified value
 * @arg route		route to be changed
 * @arg family		new address family
 */
void rtnl_route_set_family(struct rtnl_route *route, int family)
{
	route->rt_family = family;
	route->rt_mask |= ROUTE_ATTR_FAMILY;
}

/**
 * Get the address family of a route
 * @arg route		route handle
 * @return Address family or AF_UNSPEC if not set
 */
int rtnl_route_get_family(struct rtnl_route *route)
{
	if (route->rt_mask & ROUTE_ATTR_FAMILY)
		return route->rt_family;
	else
		return AF_UNSPEC;
}

/**
 * Set the destination address prefix length of a route to the specified value
 * @arg route		route to be changed
 * @arg prefix		new destination address prefix
 * @attention The destination address prefix gets overwritten by calls
 *            to rtnl_route_set_dst() rtnl_route_set_dst_str().
 */
void rtnl_route_set_dst_len(struct rtnl_route *route, int prefix)
{
	route->rt_dst_len = prefix;
	route->rt_mask |= ROUTE_ATTR_DST_LEN;
}

/**
 * Get the destination address prefix length of a route
 * @arg route		route handle
 * @return Prefix length or -1 if not set
 */
int rtnl_route_get_dst_len(struct rtnl_route *route)
{
	if (route->rt_mask & ROUTE_ATTR_DST_LEN)
		return route->rt_dst_len;
	else
		return -1;
}

/**
 * Set the source address prefix length of a route to the specified value
 * @arg route		route to be changed
 * @arg prefix		new source address prefix
 * @attention The source address prefix gets overwritten by calls
 *            to rtnl_route_src_dst() rtnl_route_set_src_str().
 */
void rtnl_route_set_src_len(struct rtnl_route *route, int prefix)
{
	route->rt_dst_len = prefix;
	route->rt_mask |= ROUTE_ATTR_SRC_LEN;
}

/**
 * Get the source address prefix length of a route
 * @arg route		route handle
 * @return Prefix length or -1 if not set
 */
int rtnl_route_get_src_len(struct rtnl_route *route)
{
	if (route->rt_mask & ROUTE_ATTR_SRC_LEN)
		return route->rt_src_len;
	else
		return -1;
}

/**
 * Set the type of a route to the specified value
 * @arg route		route to be changed
 * @arg type		new route type
 */
void rtnl_route_set_type(struct rtnl_route *route, int type)
{
	route->rt_type = type;
	route->rt_mask |= ROUTE_ATTR_TYPE;
}

/**
 * Get the type of a route
 * @arg route		route handle
 * @return Type of route or -1 if not set
 */
int rtnl_route_get_type(struct rtnl_route *route)
{
	if (route->rt_mask & ROUTE_ATTR_TYPE)
		return route->rt_type;
	else
		return -1;
}

/**
 * Add flags to a route 
 * @arg route		route to be changed
 * @arg flags		flags to set
 */
void rtnl_route_set_flags(struct rtnl_route *route, unsigned int flags)
{
	route->rt_flag_mask |= flags;
	route->rt_flags |= flags;
	route->rt_mask |= ROUTE_ATTR_FLAGS;
}

/**
 * Remove flags from a route 
 * @arg route		route to be changed
 * @arg flags		flags to unset
 */
void rtnl_route_unset_flags(struct rtnl_route *route, unsigned int flags)
{
	route->rt_flag_mask |= flags;
	route->rt_flags &= ~flags;
	route->rt_mask |= ROUTE_ATTR_FLAGS;
}

/**
 * Get flags of a route
 * @arg route		route handle
 */
unsigned int rtnl_route_get_flags(struct rtnl_route *route)
{
	return route->rt_flags;
}

/**
 * Set a metric of a route to the specified value
 * @arg route		route to be changed
 * @arg metric		metric to be changed (see XXX)
 * @arg value		new metric value
 * @return 0 on sucess or a negative error code
 */
int rtnl_route_set_metric(struct rtnl_route *route, int metric, uint32_t value)
{
	if (metric <= RTAX_MAX || metric < 1)
		return nl_error(EINVAL, "Metric out of range (1..%d)",
		    RTAX_MAX);

	route->rt_metrics[metric - 1] = value;
	route->rt_metrics_mask |= (1 << (metric - 1));

	return 0;
}

/**
 * Unset a metric of a route
 * @arg route		route to be changed
 * @arg metric		metric to be unset (see XXX)
 * @return 0 on sucess or a negative error code
 */
int rtnl_route_unset_metric(struct rtnl_route *route, int metric)
{
	if (metric <= RTAX_MAX || metric < 1)
		return nl_error(EINVAL, "Metric out of range (1..%d)",
		    RTAX_MAX);

	route->rt_metrics_mask &= ~(1 << (metric - 1));

	return 0;
}

/**
 * Get a metric for a route
 * @arg route		route handle
 * @arg metric		metric to get
 * @return The value for the specified metric or UINT_MAX if not set
 */
unsigned int rtnl_route_get_metric(struct rtnl_route *route, int metric)
{
	if (metric <= RTAX_MAX || metric < 1)
		return UINT_MAX;

	if (!(route->rt_metrics_mask & (metric - 1)))
		return UINT_MAX;

	return route->rt_metrics[metric - 1];
}

/**
 * Set the destination address of a route to the specified address
 * @arg route		route to be changed
 * @arg addr		new destination address
 *
 * Assigns the new destination address to the specified \a route,
 * overwrites the destination address length (rtnl_route::rt_dst_len),
 * and sets the route's address family to the new address's family if
 * it is not set already.
 *
 * If a address family has been specified already via either calling
 * rtnl_route_set_family() or by setting one of the other addresses,
 * the specified \a addr is automatically validated against this family
 * and the assignment fails in case of a mismatch.
 * 
 * @return 0 on success or a negative error code.
 */
int rtnl_route_set_dst(struct rtnl_route *route, struct nl_addr *addr)
{
	if (route->rt_mask & ROUTE_ATTR_FAMILY) {
		if (addr->a_family != route->rt_family)
			return nl_error(EINVAL, "Address family mismatch");
	} else
		route->rt_family = addr->a_family;

	if (route->rt_dst)
		nl_addr_put(route->rt_dst);

	nl_addr_get(addr);
	route->rt_dst = addr;
	
	route->rt_mask |= (ROUTE_ATTR_DST|ROUTE_ATTR_FAMILY|ROUTE_ATTR_DST_LEN);

	return 0;
}

/**
 * Get the destination address of a route
 * @arg route		route handle
 * @return Destination address or NULL if not set
 */
struct nl_addr *rtnl_route_get_dst(struct rtnl_route *route)
{
	if (route->rt_mask & ROUTE_ATTR_DST)
		return route->rt_dst;
	else
		return NULL;
}

/**
 * Set the source address of a route to the specified address
 * @arg route		route to be changed
 * @arg addr		new source address
 *
 * Assigns the new source address to the specified \a route,
 * overwrites the source address length (rtnl_route::rt_src_len),
 * and sets the route's address family to the new address's family if
 * it is not set already.
 *
 * If a address family has been specified already via either calling
 * rtnl_route_set_family() or by setting one of the other addresses,
 * the specified \a addr is automatically validated against this family
 * and the assignment fails in case of a mismatch.
 * 
 * @return 0 on success or a negative error code.
 */
int rtnl_route_set_src(struct rtnl_route *route, struct nl_addr *addr)
{
	if (route->rt_mask & ROUTE_ATTR_FAMILY) {
		if (addr->a_family != route->rt_family)
			return nl_error(EINVAL, "Address family mismatch");
	} else
		route->rt_family = addr->a_family;

	if (route->rt_src)
		nl_addr_put(route->rt_src);

	nl_addr_get(addr);
	route->rt_src = addr;
	route->rt_mask |= (ROUTE_ATTR_SRC|ROUTE_ATTR_FAMILY|ROUTE_ATTR_SRC_LEN);

	return 0;
}

/**
 * Get the source address of a route
 * @arg route		route handle
 * @return Source address or NULL if not set
 */
struct nl_addr *rtnl_route_get_src(struct rtnl_route *route)
{
	if (route->rt_mask & ROUTE_ATTR_SRC)
		return route->rt_src;
	else
		return NULL;
}

/**
 * Set the gateway address of a route to the specified address
 * @arg route		route to be changed
 * @arg addr		new gateway address
 *
 * Assigns the new gateway address to the specified \a route,
 * and sets the route's address family to the new address's family if
 * it is not set already.
 *
 * If a address family has been specified already via either calling
 * rtnl_route_set_family() or by setting one of the other addresses,
 * the specified \a addr is automatically validated against this family
 * and the assignment fails in case of a mismatch.
 * 
 * @return 0 on success or a negative error code.
 */
int rtnl_route_set_gateway(struct rtnl_route *route, struct nl_addr *addr)
{
	if (route->rt_mask & ROUTE_ATTR_FAMILY) {
		if (addr->a_family != route->rt_family)
			return nl_error(EINVAL, "Address family mismatch");
	} else
		route->rt_family = addr->a_family;

	if (route->rt_gateway)
		nl_addr_put(route->rt_gateway);

	nl_addr_get(addr);
	route->rt_gateway = addr;
	route->rt_mask |= (ROUTE_ATTR_GATEWAY | ROUTE_ATTR_FAMILY);

	return 0;
}

/**
 * Get the gateway address of a route
 * @arg route		route handle
 * @return Gateway address or NULL if not set
 */
struct nl_addr *rtnl_route_get_gateway(struct rtnl_route *route)
{
	if (route->rt_mask & ROUTE_ATTR_GATEWAY)
		return route->rt_gateway;
	else
		return NULL;
}

/**
 * Set the preferred source address of a route to the specified address
 * @arg route		route to be changed
 * @arg addr		new preferred source address
 *
 * Assigns the new preferred source address to the specified \a route,
 * and sets the route's address family to the new address's family if
 * it is not set already.
 *
 * If a address family has been specified already via either calling
 * rtnl_route_set_family() or by setting one of the other addresses,
 * the specified \a addr is automatically validated against this family
 * and the assignment fails in case of a mismatch.
 * 
 * @return 0 on success or a negative error code.
 */
int rtnl_route_set_pref_src(struct rtnl_route *route, struct nl_addr *addr)
{
	if (route->rt_mask & ROUTE_ATTR_FAMILY) {
		if (addr->a_family != route->rt_family)
			return nl_error(EINVAL, "Address family mismatch");
	} else
		route->rt_family = addr->a_family;

	if (route->rt_pref_src)
		nl_addr_put(route->rt_pref_src);

	nl_addr_get(addr);
	route->rt_pref_src = addr;
	route->rt_mask |= (ROUTE_ATTR_PREF_SRC | ROUTE_ATTR_FAMILY);

	return 0;
}

/**
 * Get the preferred source address of a route
 * @arg route		route handle
 * @return Preferred source address or NULL if not set
 */
struct nl_addr *rtnl_route_get_pref_src(struct rtnl_route *route)
{
	if (route->rt_mask & ROUTE_ATTR_PREF_SRC)
		return route->rt_pref_src;
	else
		return NULL;
}

/**
 * Set the outgoing interface of a route to the specified value
 * @arg route		route to be changed
 * @arg ifindex		interface index of new outoing interface
 */
void rtnl_route_set_oif(struct rtnl_route *route, int ifindex)
{
	route->rt_oif = ifindex;
	route->rt_mask |= ROUTE_ATTR_OIF;
}

/**
 * Get the outgoing interface index of a route
 * @arg route		route handle
 * @return interface index or RTNL_LINK_NOT_FOUND if not set
 */
int rtnl_route_get_oif(struct rtnl_route *route)
{
	if (route->rt_mask & ROUTE_ATTR_OIF)
		return route->rt_oif;
	else
		return RTNL_LINK_NOT_FOUND;
}

/**
 * Set the incoming interface of a route to the specified value
 * @arg route		route to be changed
 * @arg name		interface name of the new incoming interface
 */
void rtnl_route_set_iif(struct rtnl_route *route, const char *name)
{
	strncpy(route->rt_iif, name, sizeof(route->rt_iif) - 1);
	route->rt_mask |= ROUTE_ATTR_IIF;
}

/**
 * Get the incomming interface name of a route
 * @arg route		route handle
 * @return interface name or NULL if not set
 */
char *rtnl_route_get_iif(struct rtnl_route *route)
{
	if (route->rt_mask & ROUTE_ATTR_IIF)
		return route->rt_iif;
	else
		return NULL;
}

#if 0
	struct rtnl_nexthop *	rt_nexthops;
	struct rtnl_rtcacheinfo	rt_cacheinfo;
	struct nl_data		rt_protoinfo;
#endif

/** @} */

/**
 * @name Routing Table Identifier Translations
 * @{
 */

static struct trans_tbl route_tables[] = {
	__ADD(RT_TABLE_UNSPEC, unspec)
	__ADD(RT_TABLE_DEFAULT, default)
	__ADD(RT_TABLE_MAIN, main)
	__ADD(RT_TABLE_LOCAL, local)
};

/**
 * Convert routing table identifier to character string.
 * @arg table		Routing table identifier.
 * @arg buf		Destination buffer
 * @arg size		Size of destination buffer.
 *
 * Converts a routing table identifier to a character string and stores
 * it in the specified destination buffer.
 *
 * @return The destination buffer or the type encoded in hexidecimal
 *         form if the routing table identifier is unknown.
 */
char *rtnl_route_table2str(int table, char *buf, size_t size)
{
	return __type2str(table, buf, size, route_tables,
			  ARRAY_SIZE(route_tables));
}

/**
 * Convert character string to routing table identifier.
 * @arg name		Name of routing table.
 *
 * Converts the provided character string specifying a routing table
 * identifier to the corresponding numeric value.
 *
 * @return Routing table identifier or a negative value if no match was found.
 */
int rtnl_route_str2table(const char *name)
{
	return __str2type(name, route_tables, ARRAY_SIZE(route_tables));
}


/** @} */

static struct nl_cache_ops rtnl_route_ops = {
	.co_name		= "route/route",
	.co_size		= sizeof(struct rtnl_route),
	.co_hdrsize		= sizeof(struct rtmsg),
	.co_msgtypes		= {
					{ RTM_NEWROUTE, "new" },
					{ RTM_DELROUTE, "delete" },
					{ RTM_GETROUTE, "get" },
					{ -1, NULL },
				  },
	.co_protocol		= NETLINK_ROUTE,
	.co_request_update	= route_request_update,
	.co_msg_parser		= route_msg_parser,
	.co_free_data		= route_free_data,
	.co_dump[NL_DUMP_BRIEF]	= route_dump_brief,
	.co_dump[NL_DUMP_FULL]	= route_dump_full,
	.co_dump[NL_DUMP_STATS]	= route_dump_stats,
	.co_dump[NL_DUMP_XML]	= route_dump_xml,
	.co_filter		= route_filter,
};

static void __init route_init(void)
{
	nl_cache_mngt_register(&rtnl_route_ops);
}

static void __exit route_exit(void)
{
	nl_cache_mngt_unregister(&rtnl_route_ops);
}

/** @} */
