/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the libgltf project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#ifndef RENDERSCENE_H
#define RENDERSCENE_H

#include "Shaders.h"
#include "Common.h"
#include "Texture.h"
#include "Camera.h"
#include "LoadScene.h"
#include "FPSCounter.h"
#include "Timer.h"
#include "trackball.h"
#define USE_MSAA       1
#define SSAA           2
#define DEFAULT_VIEW   0
#define LOAD_ONCE      1
#define MAX_BONE_SIZE  480

#define FBOVERT "attribute vec4 vPosition;\n"\
"attribute vec2 texCoord;\n"\
"varying vec2 vTexCoord;\n"\
"void main()\n"\
"{\n"\
"    gl_Position =  vPosition;\n"\
"    vTexCoord = texCoord;\n"\
"}"
#define FBOFRAG "uniform sampler2D RenderTex;\n"\
"varying vec2 vTexCoord;\n"\
"void main()\n"\
"{\n"\
"    gl_FragColor = vec4(texture2D(RenderTex, vTexCoord).rgb, 1);\n"\
"}"
#define INVERSEVERT "attribute vec4 vPosition;\n"\
"attribute vec2 texCoord;\n"\
"varying vec2 vTexCoord;\n"\
"void main()\n"\
"{\n"\
"    gl_Position =  vPosition;\n"\
"    vTexCoord = texCoord;\n"\
"}"
#define INVERSEFRAG "uniform sampler2D RenderTex;\n"\
"varying vec2 vTexCoord;\n"\
"void main()\n"\
"{\n"\
"    gl_FragColor = vec4(texture2D(RenderTex, vTexCoord).rgb, 1.0);\n"\
"}"

namespace libgltf
{

class PrimitiveZ
{
public:
    PrimitiveZ(int tri=0, float z=0.0f): mPrimitiveIndex(tri), mZ(z) {}
    unsigned int mPrimitiveIndex;
    float mZ;
};

class SorterBackToFront
{
public:
    bool operator()(const PrimitiveZ& t1, const PrimitiveZ& t2) const
    {
        return t1.mZ < t2.mZ;
    }
};

class RenderWithFBO
{
public:
    RenderWithFBO();
    int createAndBindFbo(int width, int height);
    void releaseFbo();
    int renderFbo(int srcWidth, int srcHeight);

    void createTextureObj(int width, int height, GLuint& texId);
    void createRenderObj(int width, int height, GLuint& RboId);
    int createFrameBufferObj(GLuint& FboId, GLuint texId, GLuint RboId);
    int createMultiSampleTextureFrameBufObj(GLuint& fboId, GLuint& texId,
                                          GLuint& rboId, int width, int height);
    GLuint loadFboShader(const char* vShader, const char* fShader);
    void setBufferForFbo(GLuint& texCoordBuf, GLuint& vertexBuf,
                         GLfloat* pCoord, GLuint numCoord,
                         GLfloat* pSquare, GLuint numSquare);
    void createAndBindInverseFBO(int width, int height);
    void createBitmapTexture(int width, int height);
    void inverseBitMap(int width, int height);
    void inverseTexture(GLuint proId, GLuint texCoordBuf, GLuint vertexBuf);
    void releaseBitMapFBO();
    void releaseBitmapTexture();
    void releaseMSAAFBO();
    GLuint getMSAAFboId(){ return mMSAAFboId; };
    GLuint getFboId(){ return mFboId; };
    int renderFboTexture();
private:

    GLuint mFboProId;
    GLuint mFboId;
    GLuint mRboId;
    GLuint mTexId;

    GLuint mShotTexId;
    GLuint mRenderTexId;

    GLuint mInverseFboId;
    GLuint mInverseRboId;
    GLuint mInverseTexId;

    GLuint mMSAAFboId;
    GLuint mMSAARboId;
    GLuint mMSAATexId;

    GLuint mVertexBuf;
    GLuint mTexCoordBuf;
};

class RenderPrimitive
{
public:
    Node* getNode();
    void setNode(Node* node);
    Material* getMaterial();
    void setMaterial(Material* pMaterial);

    unsigned int getVerterCount();
    void setVerterCount(unsigned int count);

    unsigned int getIndicesCount();
    void setIndicesCount(unsigned int count);

    unsigned int getVertexBuffer();
    void setVertexBuffer(unsigned int bufferId);

    unsigned int getNormalBuffer();
    void setNormalBuffer(unsigned int bufferId);

    unsigned int getTexCoordBuffer();
    void setTexCoordBuffer(unsigned int bufferId);

    unsigned int getJointBuffer();
    void setJointBuffer(unsigned int bufferId);

    unsigned int getWeightBuffer();
    void setWeightBuffer(unsigned int bufferId);

    unsigned int getIndicesBuffer();
    void setIndicesBuffer(unsigned int bufferId);

    DataType getIndicesDataType();
    void setIndicesDataType(DataType type);

    void copyVertexBufferData(const char* srcBuf, unsigned int bufSize);
    void copyIndiceBufferData(const char* srcBuf, unsigned int bufSize);
    void polyonSorting(glm::mat4& viewMatrix);
    void bindSortedIndicesBuf();

    RenderPrimitive();
    ~RenderPrimitive();
private:
    RenderPrimitive(const RenderPrimitive&);
    RenderPrimitive& operator=(const RenderPrimitive&);

    void sortIndices();

    DataType mIndicesDataType;
    Material* mpMaterial;
    Node* pNode;
    unsigned int mVerterCount;
    unsigned int mIndicesCount;
    unsigned int mVertexBuffer;
    unsigned int mNormalBuffer;
    unsigned int mTexCoordBuffer;
    unsigned int mJointBuffer;
    unsigned int mWeightBuffer;
    unsigned int mIndicesBuffer;
    char* mVertexBufferData;
    char* mIndiceBufferData;
    char* mSortedIndiceBufferData;
    unsigned int mIndiceBufferLen;
    std::vector<float> mEyeSpaceVerts;
    std::vector<PrimitiveZ> mPrimitiveZ;
    bool mIsPolygonSorted;
};

class RenderShader
{
public:
    Technique* getTechnique();
    void setTechnique(Technique* pTechnique);

    void pushRenderPrim(RenderPrimitive* p);
    RenderPrimitive* getRenderPrim(unsigned int i);
    unsigned int getRenderPrimSize();

    RenderShader();
    ~RenderShader();

private:
    RenderShader(const RenderShader&);
    RenderShader& operator=(const RenderShader&);

    std::vector<RenderPrimitive*> mPrimitiveVec;
    Technique* mpTechnique;
};

class RenderScene
{
    struct BindBufferInfo
    {
        unsigned int mBufferId;
        unsigned int mDataCount;
        const char * mSrcData;
        unsigned int mBufferLen;
    };
public:
    // For camera
    glm::vec3* getModelCenterPos();
    void getCameraPos(glm::vec3* Eye, glm::vec3* view, glm::vec3* up);
    double getModelSize();
    void renderFlyCamera(glm::vec3 glPosInfo, double time);
    double getModelSize() const;
    bool renderFlyCamera(const glm::mat4& glPosViewMatrix, double time);
    CPhysicalCamera& getCamera();
    void getCameraIndex(Node* rootNode);

    // For render scene
    glTFHandle* initScene(const std::string& jsonfile, std::vector<glTFFile>& o_glTFFiles);
    int initRender(const std::vector<glTFFile>& inputFiles);
    int prepareRender(glTFViewport* pViewport);
    void render();
    int completeRender();
    void releaseRender();

    // For bitmap
    int prepareRenderBitmap(glTFViewport* pViewport);
    void renderBitmap(double time);
    int completeRenderBitmap(glTFViewport* pViewport,
                              unsigned char* buffer, GLenum format);
    void setBitZoom(unsigned char* Dstbuffer,
                    unsigned char* Srcbuffer,
                    glTFViewport* pViewport,
                    int bufferDepth);

    // For animation
    void startAnimation();
    void stopAnimation();
    void stopAerialView();
    void startAerialView();
    void setAnimTime(double time);
    double getAnimTime();
    void setAnimLoop(bool loop);
    bool getAnimLoop();
    double getAnimDuration();
    bool isAnimPlay();
    void resumeAnim();
    void setTimeForAnim();

    // For model rotation
    void enableRotation();
    void disableRotation();
    bool isRotationEnabled() const;

    void enableTransparency();
    void disableTransparency();
    void initFPS();
    RenderScene();
    ~RenderScene();

private:
    RenderScene(const RenderScene&);
    RenderScene& operator=(const RenderScene&);

    void initOpengl();
    void initNodeTree(Node* pNode, const glm::mat4& matrix,
                      bool parentJointFlag, bool updateFlag);
    int loadScene(const std::vector<glTFFile>& inputFiles);
    void createDefaultCamera();
    void constructShader();
    void constructMesh(const std::string& meshName, Node* pNode);
    void constructPrimitive(const Primitives* pPrim, Node* pNode);
    unsigned int bindAttribute(const Attribute* pAttr);
    unsigned int bindIndices(const Attribute* pAttr);
    void bindAttributeBuffer(const Primitives* pPrimitive,
                             RenderPrimitive* pRP);
    void updateNodeMatrix(Node* pNode, const glm::mat4& matrix, bool flag);
    void updateFlyCamera();
    void updateAnimInfo(Node* pNode);
    bool upLoadTechProperty();
    void upLoadTechPropertyOfJsonFile(Technique* pTech);
    void upLoadTechInfo(unsigned int progId, Technique* pTech);
    void upLoadMatrixInfo(unsigned int progId, RenderPrimitive* pPrimitive);
    void upLoadAnimation(unsigned int progId, RenderPrimitive* pPrimitive);
    void upLoadUniform(unsigned int progId, RenderPrimitive* pPrimitive);
    void upLoadAttribute(unsigned int progId, RenderPrimitive* pPrimitive);
    void drawTriangle(RenderPrimitive* pPrimitive);
    void renderPrimitive(RenderPrimitive* pPrimitive, unsigned int progId);
    void renderShader(RenderShader* pRenderShader);
    void updatePolygonSorting();
    void primitivePolygonSorting(RenderPrimitive* pPrimitive);
    void realRender();
    Node* findNodeByName(Node* pPareNode, const std::string& nodeId);
    Node* findNodeByJoint(Node* pPareNode, const std::string& jointId);
    int initSSAAFrameBuf(glTFViewport* pViewport);
    void calcCameraMoveTime();

    CPhysicalCamera maCamera;
    ParseCamera* cCamera;
    std::vector<std::string> vCameraIndex;
    glm::mat4 mOrbitInitViewMatrix;
    glm::mat4 mWalkthroughInitViewMatrix;
    bool IsAerialMode;
    bool bAerialView;

    double  oldTime;
    glm::mat4 flyInfo;
    double flyTime;
    bool bFlyCamera;
    bool bAnimation;
    Light* pLight;
    glm::mat4* pTempMatrix;

    bool mAnimationPlay;
    bool mAnimationLoop;
    double mCurrentTime;
    double mLastPlaying;
    double mUpdateTimeOut;
    double mDuration;

    float mCameraMoveTime;
    float mPreCameraTime;
    std::vector<RenderShader*> mShaderVec;
    Scene* pScene;
    Parser mLoadJson;
    std::map<std::string, BindBufferInfo> mBindBufferMap;

    static ShaderProgram mShaderProgram;
    glTFViewport mCurrentViewport;
    RenderWithFBO fbo;

    bool mEnableTransparency;
    bool mEnableRotation;
    glm::mat4 mLastModelView;
    bool bIsTimeAvailable;
    bool loadFPSShader();
    void printFPS(glTFViewport* pViewpoit);
    FPSCounter* pFPSCounter;
#if ENABLE_TIMER
    Timer* pTimer;
#endif
#if LOAD_ONCE
    std::string mCurrentImage;
    int mCurrentTextNumber;
#endif
};

} // namespace libgltf

#endif

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
