/*
 * Copyright (C) 2010 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by
 *             Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <eggdbus/eggdbus.h>

#include "zeitgeist-log.h"
#include "zeitgeist-simple-result-set.h"
#include "zeitgeist-eggdbusconversions.h"
#include "eggzeitgeistbindings.h"

/**
 * SECTION:zeitgeist-log
 * @short_description: Primary access point for talking to the Zeitgeist daemon
 * @include: zeitgeist.h
 *
 * #ZeitgeistLog encapsulates the low level access to the Zeitgeist daemon.
 * You can use it to manage the log by inserting and deleting entries as well
 * as do queries on the logged data.
 *
 * It's important to realize that the #ZeitgeistLog class does not expose
 * any API that does synchronous communications with the message bus - 
 * everything is asynchronous. To ease development some of the methods have
 * variants that are "fire and forget" ignoring the normal return value, so
 * that callbacks does not have to be set up.
 */

G_DEFINE_TYPE (ZeitgeistLog, zeitgeist_log, G_TYPE_OBJECT);
#define ZEITGEIST_LOG_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE(obj, ZEITGEIST_TYPE_LOG, ZeitgeistLogPrivate))

typedef struct
{
  /* The connection to the ZG daemon
   * Note: The EggZeitgeistLog is owned by the EggDBusObjectProxy! */
  EggDBusObjectProxy *log_proxy;
  EggZeitgeistLog    *log;

  gboolean            connected;

  /* Hash set of ZeitgeistMonitors we've installed. We actually store
   * a map of (monitor, log) because this is most convenient for use with
   * _zeitgeist_monitor_weak_unref() casted to a GHFunc */
  GHashTable         *monitors;
} ZeitgeistLogPrivate;

/* Property ids */
enum
{
	PROP_0,

	PROP_CONNECTED,

	LAST_PROPERTY
};

static void    _zeitgeist_log_on_name_owner_changed    (ZeitgeistLog    *self);

static void    _zeitgeist_log_on_monitor_destroyed     (ZeitgeistLog     *self,
                                                        ZeitgeistMonitor *monitor);

static void    _zeitgeist_monitor_weak_unref           (ZeitgeistMonitor *monitor,
                                                        ZeitgeistLog     *log);

/* Signature of egg_zeitgeist variants of find_events() and find_event_ids() */
typedef guint (*FindEventsFunc) (EggZeitgeistLog          *instance,
                                 EggDBusCallFlags          call_flags,
                                 EggZeitgeistTimeRange    *time_range,
                                 EggDBusArraySeq          *event_templates,
                                 EggZeitgeistStorageState  storage_state,
                                 guint                     num_events,
                                 EggZeitgeistResultType    result_type,
                                 GCancellable             *cancellable,
                                 GAsyncReadyCallback       callback,
                                 gpointer                  user_data);

/* Signature for egg_zeitgeist_log functions that send an array of event ids */
typedef guint (*SendEventIdsFunc) (EggZeitgeistLog          *instance,
                                   EggDBusCallFlags          call_flags,
                                   EggDBusArraySeq          *event_ids,
                                   GCancellable             *cancellable,
                                   GAsyncReadyCallback       callback,
                                   gpointer                  user_data);

/* Signature for egg_zeitgeist_log functions that does not have arguments  */
typedef guint (*SendVoidFunc) (EggZeitgeistLog          *instance,
                               EggDBusCallFlags          call_flags,
                               GCancellable             *cancellable,
                               GAsyncReadyCallback       callback,
                               gpointer                  user_data);

typedef gboolean (*CollectEventsFunc) (EggZeitgeistLog          *instance,
                                       EggDBusArraySeq         **out_events,
                                       GAsyncResult             *res,
                                       GError                  **error);

typedef gboolean (*CollectIdsFunc) (EggZeitgeistLog          *instance,
                                    EggDBusArraySeq         **out_event_ids,
                                    GAsyncResult             *res,
                                    GError                  **error);

typedef gboolean (*CollectVoidFunc) (EggZeitgeistLog          *instance,
                                     GAsyncResult             *res,
                                     GError                  **error);

static void
zeitgeist_log_init (ZeitgeistLog *self)
{
  ZeitgeistLogPrivate *priv;
  EggDBusConnection   *conn;
  gchar               *name_owner;

  priv = ZEITGEIST_LOG_GET_PRIVATE (self);
  
  /* Reset hash set of monitors */
  priv->monitors = g_hash_table_new (g_direct_hash, g_direct_equal);
  
  /* Set up the connection to the ZG daemon */  
  conn = egg_dbus_connection_get_for_bus (EGG_DBUS_BUS_TYPE_SESSION);
  priv->log_proxy = 
    egg_dbus_connection_get_object_proxy (conn,
                                          "org.gnome.zeitgeist.Engine",
                                          "/org/gnome/zeitgeist/log/activity");
  
  priv->log = EGG_ZEITGEIST_QUERY_INTERFACE_LOG (priv->log_proxy);
  g_object_unref (conn);

  name_owner = egg_dbus_object_proxy_get_name_owner (priv->log_proxy);
  priv->connected = name_owner != NULL;
  if (name_owner)
    {
      g_free (name_owner);
    }

  /* We need to detect if the Zeitgeist daemon leaves/enters the bus */
  g_signal_connect_swapped (priv->log_proxy, "notify::name-owner",
                            G_CALLBACK (_zeitgeist_log_on_name_owner_changed),
                            self);
}

static void
zeitgeist_log_finalize (GObject *object)
{
  ZeitgeistLog *log = ZEITGEIST_LOG (object);
  ZeitgeistLogPrivate *priv;
  
  priv = ZEITGEIST_LOG_GET_PRIVATE (log);

  /* Note: priv->log is owned by priv->log_proxy */
  if (priv->log_proxy)
    {
      g_object_unref (priv->log_proxy);
    }
  
  /* Out list of monitors only holds weak refs to the monitors */
  if (priv->monitors)
    {
      g_hash_table_foreach (priv->monitors,
                            (GHFunc) _zeitgeist_monitor_weak_unref,
                            NULL);
      g_hash_table_unref (priv->monitors);
    }
  
  G_OBJECT_CLASS (zeitgeist_log_parent_class)->finalize (object); 
}

static void
zeitgeist_log_get_property (GObject    *object,
                            guint       prop_id,
                            GValue     *value,
                            GParamSpec *pspec)
{
  ZeitgeistLogPrivate *priv = ZEITGEIST_LOG_GET_PRIVATE (object);

  switch (prop_id)
    {
      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        return;
        break;
    }
}

static void
zeitgeist_log_set_property (GObject      *object,
                            guint         prop_id,
                            const GValue *value,
                            GParamSpec   *pspec)
{
  ZeitgeistLogPrivate *priv = ZEITGEIST_LOG_GET_PRIVATE (object);

  switch (prop_id)
    {
      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        return;
        break;
    }
}


static void
zeitgeist_log_class_init (ZeitgeistLogClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  GParamSpec   *pspec;
  
  object_class->finalize     = zeitgeist_log_finalize;
  object_class->get_property = zeitgeist_log_get_property;
  object_class->set_property = zeitgeist_log_set_property;

  /**
   * ZeitgeistLog:connected:
   *
   * Determines if this Log instance is currently connected to Zeitgeist
   * daemon.
   */
  pspec = g_param_spec_boolean ("connected",
                                "Connected",
                                "Whether this instance is connected to Zeitgeist",
                                FALSE,
                                G_PARAM_READABLE | G_PARAM_STATIC_STRINGS);
  g_object_class_install_property (object_class, PROP_CONNECTED, pspec);

  g_type_class_add_private (object_class, sizeof (ZeitgeistLogPrivate));
}

/* Used to marshal the async callbacks from EggDBus into ones
 * coming from this Log instance */
static void
dispatch_async_callback (GObject      *source_object,
                         GAsyncResult *res,
                         gpointer      user_data)
{ 
  gpointer            *data = (gpointer*) user_data;
  ZeitgeistLog        *self = ZEITGEIST_LOG (data[0]);
  GAsyncReadyCallback  callback = (GAsyncReadyCallback) data[1];
  gpointer             _user_data = data[2];

  if (callback != NULL)
    {
      callback (G_OBJECT (self), res, _user_data);
    }
  
  g_object_unref (self);
  g_free (user_data);
}

/*
 * API BELOW HERE
 */

/**
 * zeitgeist_log_new:
 * Create a new log that interfaces with the default event log of the Zeitgeist
 * daemon.
 *
 * Log instances are not overly expensive for neither client or the Zeitgeist
 * daemon so there's no need to go to lenghts to keep singleton instances
 * around.
 *
 * Returns: A reference to a newly allocated log.
 */
ZeitgeistLog*
zeitgeist_log_new (void)
{
  ZeitgeistLog        *log;

  log = (ZeitgeistLog*) g_object_new (ZEITGEIST_TYPE_LOG,
                                      NULL);

  return log;
}

/* Helper for _finish() functions that returns a GArray of event ids */
static gboolean
_zeitgeist_log_collect_event_ids (CollectIdsFunc       collect_ids_func,
                                  ZeitgeistLog        *self,
                                  GArray             **out_event_ids,
                                  GAsyncResult        *res,
                                  GError             **error)
{
  ZeitgeistLogPrivate   *priv;
  EggDBusArraySeq       *event_ids;
  gint                   len, i;
  guint32                event_id;
  gboolean               result;

  g_return_val_if_fail (ZEITGEIST_IS_LOG (self), FALSE);
  g_return_val_if_fail (out_event_ids != NULL, FALSE);
  g_return_val_if_fail (G_IS_ASYNC_RESULT (res), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = ZEITGEIST_LOG_GET_PRIVATE (self);

  result = collect_ids_func (priv->log,
                             &event_ids,
                             res,
                             error);
  
  if (!result)
    {
      return FALSE;
    }

  len = egg_dbus_array_seq_get_size (event_ids);
  *out_event_ids = g_array_sized_new (FALSE, FALSE, sizeof (guint32), len);
  for (i = 0; i < len; i++)
    {
      event_id = egg_dbus_array_seq_get_fixed (event_ids, i);
      g_array_append_val (*out_event_ids, event_id);
                                
    }
  
  g_object_unref (event_ids);
  return TRUE;  
}

/* Helper for _finish() functions returning an array of events.
 * Set the free_func on @out_events to g_object_unref() */
static gboolean
_zeitgeist_log_collect_events (CollectEventsFunc    collect_events_func,
                               ZeitgeistLog        *self,
                               GPtrArray          **out_events,
                               GAsyncResult        *res,
                               GError             **error)
{
  ZeitgeistLogPrivate   *priv;
  EggDBusArraySeq       *events;
  gboolean               result;

  g_return_val_if_fail (ZEITGEIST_IS_LOG (self), FALSE);
  g_return_val_if_fail (out_events != NULL, FALSE);
  g_return_val_if_fail (G_IS_ASYNC_RESULT (res), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = ZEITGEIST_LOG_GET_PRIVATE (self);

  result = collect_events_func (priv->log,
                                &events,
                                res,
                                error);

  if (!result)
    {
      return FALSE;
    }

  *out_events = _egg_zeitgeist_events_to_zeitgeist_events (events);
  g_ptr_array_set_free_func (*out_events, (GDestroyNotify) g_object_unref);
  
  g_object_unref (events);
  return TRUE;
}

/* Helper for _finish() functions without a return value */
static gboolean
_zeitgeist_log_collect_void (CollectVoidFunc collect_void,
                             ZeitgeistLog        *self,
                             GAsyncResult        *res,
                             GError             **error)
{
  ZeitgeistLogPrivate   *priv;
  EggDBusArraySeq       *events;
  gboolean               result;

  g_return_val_if_fail (ZEITGEIST_IS_LOG (self), FALSE);
  g_return_val_if_fail (G_IS_ASYNC_RESULT (res), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = ZEITGEIST_LOG_GET_PRIVATE (self);

  return collect_void (priv->log, res, error);
}

/**
 * zeitgeist_log_insert_events:
 * @self: The log logging the events
 * @cancellable: To cancel the operation or %NULL
 * @callback: #GAsyncReadyCallback to invoke once the logging operation has
 *            completed. Set to %NULL to ignore the result. In this callback
 *            you can invoke zeitgeist_log_insert_events_finish() to collect
 *            the event ids of the inserted events
 * @user_data: Any user data to pass back to @callback
 * @VarArgs: A list of #ZeitgeistEvent<!-- -->s terminated by a %NULL
 *
 * Asynchrnously send a set of events to the Zeitgeist daemon, requesting they
 * be inserted into the log.
 */
void
zeitgeist_log_insert_events (ZeitgeistLog        *self,
                             GCancellable        *cancellable,
                             GAsyncReadyCallback  callback,
                             gpointer             user_data,
                             ...)
{
  va_list                events;

  va_start (events, user_data);
  zeitgeist_log_insert_events_valist (self, cancellable,
                                      callback, user_data, events);
  va_end (events);
}

/**
 * zeitgeist_log_insert_events_no_reply:
 * @self: The log logging the events
 * @VarArgs: A list of #ZeitgeistEvent<!-- -->s terminated by a %NULL
 *
 * Asynchrnously send a set of events to the Zeitgeist daemon, requesting they
 * be inserted into the log. This method is &quot;fire and forget&quot; and the
 * caller will never know whether the events was succesfully inserted or not.
 *
 * This method is exactly equivalent to calling zeitgeist_log_insert_events()
 * with %NULL set as @cancellable, @callback, and @user_data.
 */
void
zeitgeist_log_insert_events_no_reply (ZeitgeistLog *self,
                                      ...)
{
  va_list                events;

  va_start (events, self);
  zeitgeist_log_insert_events_valist (self, NULL, NULL, NULL, events);
  va_end (events);
}

/**
 * zeitgeist_log_insert_events_valist:
 * @self: The log logging the events
 * @cancellable: To cancel the operation or %NULL
 * @callback: #GAsyncReadyCallback to invoke once the logging operation has
 *            completed. Set to %NULL to ignore the result. In this callback
 *            you can invoke zeitgeist_log_insert_events_finish() to collect
 *            the event ids of the inserted events
 * @user_data: Any user data to pass back to @callback
 * @events: A #GPtrArray of #ZeitgeistEvent<!-- -->s to insert. This method
 *          steals the reference to @events and consumes all floating refs
 *          on the event members. It is assumed that the free_func on @events
 *          is set to g_object_unref().
 *
 * This method is intended for language bindings. If calling this function
 * from C code it's generally more handy to use zeitgeist_log_insert_events()
 * or zeitgeist_log_insert_events_from_ptrarray().
 * 
 * Asynchrnously send a set of events to the Zeitgeist daemon, requesting they
 * be inserted into the log.
 */
void
zeitgeist_log_insert_events_valist (ZeitgeistLog        *self,
                                    GCancellable        *cancellable,
                                    GAsyncReadyCallback  callback,
                                    gpointer             user_data,
                                    va_list              events)
{
  GPtrArray *_events;

  _events = _zeitgeist_events_from_valist (events);
  zeitgeist_log_insert_events_from_ptrarray (self, _events, cancellable,
                                             callback, user_data);
}

/**
 * zeitgeist_log_insert_events_from_ptrarray:
 * @self: The log logging the events
 * @events: A #GPtrArray of #ZeitgeistEvent<!-- -->s to insert. This method
 *          steals the reference to @events and consumes all floating refs
 *          on the event members. It is assumed that the free_func on @events
 *          is set to g_object_unref().
 * @cancellable: To cancel the operation or %NULL
 * @callback: #GAsyncReadyCallback to invoke once the logging operation has
 *            completed. Set to %NULL to ignore the result. In this callback
 *            you can invoke zeitgeist_log_insert_events_finish() to collect
 *            the event ids of the inserted events
 * @user_data: Any user data to pass back to @callback
 *
 * Asynchrnously send a set of events to the Zeitgeist daemon, requesting they
 * be inserted into the log.
 */
void
zeitgeist_log_insert_events_from_ptrarray (ZeitgeistLog        *self,
                                           GPtrArray           *events,
                                           GCancellable        *cancellable,
                                           GAsyncReadyCallback  callback,
                                           gpointer             user_data)
{
  ZeitgeistLogPrivate   *priv;
  EggDBusArraySeq       *_events;
  gpointer              *dispatch_data;

  g_return_if_fail (ZEITGEIST_IS_LOG (self));
  g_return_if_fail (events != NULL);
  g_return_if_fail (cancellable == NULL || G_IS_CANCELLABLE(cancellable));

  priv = ZEITGEIST_LOG_GET_PRIVATE (self);

  /* Own all floating refs on the events */
  g_ptr_array_foreach (events, (GFunc) g_object_ref_sink, NULL);
  
  _events = _zeitgeist_events_to_egg_zeitgeist_events (events);

  dispatch_data = g_new (gpointer, 3);
  dispatch_data[0] = g_object_ref (self);
  dispatch_data[1] = callback;
  dispatch_data[2] = user_data;

  egg_zeitgeist_log_insert_events (priv->log,
                                   EGG_DBUS_CALL_FLAGS_NONE,
                                   _events,
                                   cancellable,
                                   dispatch_async_callback,
                                   dispatch_data);
                                 
  /* Release the events */
  g_ptr_array_unref (events);
  g_object_unref (_events);

}

GArray*
zeitgeist_log_insert_events_finish (ZeitgeistLog        *self,
                                    GAsyncResult        *res,
                                    GError             **error)
{
  GArray *event_ids;

  _zeitgeist_log_collect_event_ids (egg_zeitgeist_log_insert_events_finish,
                                    self, &event_ids, res, error);
  return event_ids;
}

/* Shared helper for dispatching find_events() and find_event_ids() */
static void
_zeitgeist_log_find_events_with_func (FindEventsFunc       find_events_func,
                                      ZeitgeistLog        *self,
                                      ZeitgeistTimeRange  *time_range,
                                      GPtrArray           *event_templates,
                                      ZeitgeistStorageState storage_state,
                                      guint32              num_events,
                                      ZeitgeistResultType  result_type,
                                      GCancellable        *cancellable,
                                      GAsyncReadyCallback  callback,
                                      gpointer             user_data)                                      
{
  ZeitgeistLogPrivate   *priv;
  EggZeitgeistTimeRange *_time_range;
  EggDBusArraySeq       *_event_templates;
  gpointer              *dispatch_data;

  g_return_if_fail (ZEITGEIST_IS_LOG (self));
  g_return_if_fail (ZEITGEIST_IS_TIME_RANGE (time_range));
  g_return_if_fail (event_templates != NULL);
  g_return_if_fail (cancellable == NULL || G_IS_CANCELLABLE(cancellable));

  priv = ZEITGEIST_LOG_GET_PRIVATE (self);

  /* Own all floating refs on the events and the time_range */
  g_ptr_array_foreach (event_templates, (GFunc) g_object_ref_sink, NULL);
  g_object_ref_sink (time_range);
  
  _time_range = _zeitgeist_time_range_to_egg_zeitgeist_time_range (time_range);
  _event_templates = _zeitgeist_events_to_egg_zeitgeist_events (event_templates);

  dispatch_data = g_new (gpointer, 3);
  dispatch_data[0] = g_object_ref (self);
  dispatch_data[1] = callback;
  dispatch_data[2] = user_data;

  find_events_func (priv->log,
                    EGG_DBUS_CALL_FLAGS_NONE,
                    _time_range,
                    _event_templates,
                    (EggZeitgeistStorageState) storage_state,
                    num_events,
                    (EggZeitgeistResultType) result_type,
                    cancellable,
                    dispatch_async_callback,
                    dispatch_data);
                                 
  /* Release the event templates */
  g_ptr_array_foreach (event_templates, (GFunc) g_object_unref, NULL);
  g_ptr_array_unref (event_templates);
  g_object_unref (time_range);
  g_object_unref (_event_templates);
  g_object_unref (_time_range);
}

void
zeitgeist_log_find_events (ZeitgeistLog        *self,
                           ZeitgeistTimeRange  *time_range,
                           GPtrArray           *event_templates,
                           ZeitgeistStorageState storage_state,
                           guint32              num_events,
                           ZeitgeistResultType  result_type,
                           GCancellable        *cancellable,
                           GAsyncReadyCallback  callback,
                           gpointer             user_data)
{
  _zeitgeist_log_find_events_with_func (egg_zeitgeist_log_find_events,
                                        self,
                                        time_range,
                                        event_templates,
                                        storage_state,
                                        num_events,
                                        result_type,
                                        cancellable,
                                        callback,
                                        user_data);
}

/**
 * zeitgeist_log_find_events_finish:
 * @self:
 * @res:
 * @error:
 *
 * Retrieve the result from an asynchronous query started with
 * zeitgeist_log_find_events().
 *
 * Returns: (transfer-full): A newly allocated #ZeitgeistResultSet containing
 *          the #ZeitgeistEvent<!-- -->s
 *          matching the query. You must free the result set with
 *          g_object_unref(). The events held in the result set will
 *          automatically be unreffed when the result set is finalized.
 */
ZeitgeistResultSet*
zeitgeist_log_find_events_finish (ZeitgeistLog        *self,
                                  GAsyncResult        *res,
                                  GError             **error)
{
  GPtrArray *events;
  
  _zeitgeist_log_collect_events (egg_zeitgeist_log_find_events_finish,
                                 self, &events, res, error);
  return _zeitgeist_simple_result_set_new (events, events->len);
}

void
zeitgeist_log_find_event_ids (ZeitgeistLog        *self,
                              ZeitgeistTimeRange  *time_range,
                              GPtrArray           *event_templates,
                              ZeitgeistStorageState storage_state,
                              guint32              num_events,
                              ZeitgeistResultType  result_type,
                              GCancellable        *cancellable,
                              GAsyncReadyCallback  callback,
                              gpointer             user_data)
{
  _zeitgeist_log_find_events_with_func (egg_zeitgeist_log_find_event_ids,
                                        self,
                                        time_range,
                                        event_templates,
                                        storage_state,
                                        num_events,
                                        result_type,
                                        cancellable,
                                        callback,
                                        user_data);
}

GArray*
zeitgeist_log_find_event_ids_finish (ZeitgeistLog        *self,
                                     GAsyncResult        *res,
                                     GError             **error)
{
  GArray *event_ids;
  
  _zeitgeist_log_collect_event_ids (egg_zeitgeist_log_find_event_ids_finish,
                                    self, &event_ids, res, error);
  return event_ids;
}

static void
_zeitgeist_log_send_event_ids (SendEventIdsFunc     send_event_ids,
                               ZeitgeistLog        *self,
                               GArray              *event_ids,
                               GCancellable        *cancellable,
                               GAsyncReadyCallback  callback,
                               gpointer             user_data)
{
  ZeitgeistLogPrivate   *priv;
  EggDBusArraySeq       *_event_ids;
  gpointer              *dispatch_data;
  gint                   i;

  g_return_if_fail (ZEITGEIST_IS_LOG (self));
  g_return_if_fail (event_ids != NULL);
  g_return_if_fail (cancellable == NULL || G_IS_CANCELLABLE(cancellable));

  priv = ZEITGEIST_LOG_GET_PRIVATE (self);

  /* Convert GArray<guint32> to EggDBusArraySeq<guint> */
  _event_ids = egg_dbus_array_seq_new (G_TYPE_UINT, NULL, NULL, NULL);
  for (i = 0; i < event_ids->len; i++)
    {
      egg_dbus_array_seq_add_fixed (_event_ids,
                                    g_array_index (event_ids, guint32, i));
    }

  dispatch_data = g_new (gpointer, 3);
  dispatch_data[0] = g_object_ref (self);
  dispatch_data[1] = callback;
  dispatch_data[2] = user_data;

  send_event_ids (priv->log,
                  EGG_DBUS_CALL_FLAGS_NONE,
                  _event_ids,
                  cancellable,
                  dispatch_async_callback,
                  dispatch_data);
                                 
  /* Release the event ids */
  g_array_unref (event_ids);
  g_object_unref (_event_ids);
}

void
zeitgeist_log_get_events (ZeitgeistLog        *self,
                          GArray              *event_ids,
                          GCancellable        *cancellable,
                          GAsyncReadyCallback  callback,
                          gpointer             user_data)
{
  _zeitgeist_log_send_event_ids (egg_zeitgeist_log_get_events,
                                 self, event_ids, cancellable,
                                 callback, user_data);
}

ZeitgeistResultSet*
zeitgeist_log_get_events_finish (ZeitgeistLog        *self,
                                 GAsyncResult        *res,
                                 GError             **error)
{
  GPtrArray *events;
  
  _zeitgeist_log_collect_events (egg_zeitgeist_log_get_events_finish,
                                 self, &events, res, error);
  return _zeitgeist_simple_result_set_new (events, events->len);
}                          

void
zeitgeist_log_delete_events (ZeitgeistLog        *self,
                             GArray              *event_ids,
                             GCancellable        *cancellable,
                             GAsyncReadyCallback  callback,
                             gpointer             user_data)
{
  _zeitgeist_log_send_event_ids (egg_zeitgeist_log_delete_events,
                                 self, event_ids, cancellable,
                                 callback, user_data);
}

gboolean
zeitgeist_log_delete_events_finish (ZeitgeistLog        *self,
                                    GAsyncResult        *res,
                                    GError             **error)
{
  return _zeitgeist_log_collect_void (egg_zeitgeist_log_delete_events_finish,
                                      self, res, error);
}

void
zeitgeist_log_find_related_uris (ZeitgeistLog        *self,
                                 ZeitgeistTimeRange  *time_range,
                                 GPtrArray           *event_templates,
                                 GPtrArray           *result_event_templates,
                                 ZeitgeistStorageState storage_state,
                                 guint32              num_events,
                                 ZeitgeistResultType  result_type,
                                 GCancellable        *cancellable,
                                 GAsyncReadyCallback  callback,
                                 gpointer             user_data)
{
  ZeitgeistLogPrivate   *priv;
  EggZeitgeistTimeRange *_time_range;
  EggDBusArraySeq       *_event_templates, *_result_event_templates;
  gpointer              *dispatch_data;

  g_return_if_fail (ZEITGEIST_IS_LOG (self));
  g_return_if_fail (ZEITGEIST_IS_TIME_RANGE (time_range));
  g_return_if_fail (event_templates != NULL);
  g_return_if_fail (result_event_templates != NULL);
  g_return_if_fail (cancellable == NULL || G_IS_CANCELLABLE(cancellable));

  priv = ZEITGEIST_LOG_GET_PRIVATE (self);

  /* Own all floating refs on the events and the time_range */
  g_ptr_array_foreach (event_templates, (GFunc) g_object_ref_sink, NULL);
  g_ptr_array_foreach (result_event_templates, (GFunc) g_object_ref_sink, NULL);
  g_object_ref_sink (time_range);
  
  _time_range = _zeitgeist_time_range_to_egg_zeitgeist_time_range (time_range);
  _event_templates = _zeitgeist_events_to_egg_zeitgeist_events (event_templates);
  _result_event_templates = _zeitgeist_events_to_egg_zeitgeist_events (event_templates);

  dispatch_data = g_new (gpointer, 3);
  dispatch_data[0] = g_object_ref (self);
  dispatch_data[1] = callback;
  dispatch_data[2] = user_data;

  egg_zeitgeist_log_find_related_uris (priv->log,
                                       EGG_DBUS_CALL_FLAGS_NONE,
                                       _time_range,
                                       _event_templates,
                                       _result_event_templates,
                                       (EggZeitgeistStorageState) storage_state,
                                       num_events,
                                       (EggZeitgeistResultType) result_type,
                                       cancellable,
                                       dispatch_async_callback,
                                       dispatch_data);
                                 
  /* Release the event templates */
  g_ptr_array_foreach (event_templates, (GFunc) g_object_unref, NULL);
  g_ptr_array_foreach (result_event_templates, (GFunc) g_object_unref, NULL);
  g_ptr_array_unref (event_templates);
  g_ptr_array_unref (result_event_templates);
  g_object_unref (time_range);
  g_object_unref (_event_templates);
  g_object_unref (_result_event_templates);
  g_object_unref (_time_range);
}

gchar**
zeitgeist_log_find_related_uris_finish (ZeitgeistLog    *self,
                                        GAsyncResult    *res,
                                        GError         **error)
{
  ZeitgeistLogPrivate   *priv;
  gboolean               result;
  gchar                **uris;

  g_return_val_if_fail (ZEITGEIST_IS_LOG (self), FALSE);
  g_return_val_if_fail (G_IS_ASYNC_RESULT (res), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = ZEITGEIST_LOG_GET_PRIVATE (self);

  egg_zeitgeist_log_find_related_uris_finish (priv->log,
                                              &uris,
                                              res,
                                              error);

  return uris;
}

static void
_zeitgeist_log_send_void (SendVoidFunc send_void,
                          ZeitgeistLog        *self,
                          GCancellable        *cancellable,
                          GAsyncReadyCallback  callback,
                          gpointer             user_data)
{
  ZeitgeistLogPrivate   *priv;
  gpointer              *dispatch_data;

  g_return_if_fail (ZEITGEIST_IS_LOG (self));
  g_return_if_fail (cancellable == NULL || G_IS_CANCELLABLE(cancellable));

  priv = ZEITGEIST_LOG_GET_PRIVATE (self);

  dispatch_data = g_new (gpointer, 3);
  dispatch_data[0] = g_object_ref (self);
  dispatch_data[1] = callback;
  dispatch_data[2] = user_data;

  send_void (priv->log,
             EGG_DBUS_CALL_FLAGS_NONE,
             cancellable,
             dispatch_async_callback,
             dispatch_data);
}

void
zeitgeist_log_delete_log (ZeitgeistLog        *self,
                          GCancellable        *cancellable,
                          GAsyncReadyCallback  callback,
                          gpointer             user_data)
{
  _zeitgeist_log_send_void (egg_zeitgeist_log_delete_log,
                            self, cancellable, callback, user_data);
}

gboolean
zeitgeist_log_delete_log_finish (ZeitgeistLog        *self,
                                 GAsyncResult        *res,
                                 GError             **error)
{
  _zeitgeist_log_collect_void (egg_zeitgeist_log_delete_log_finish,
                               self, res, error);
}

void
zeitgeist_log_quit (ZeitgeistLog        *self,
                    GCancellable        *cancellable,
                    GAsyncReadyCallback  callback,
                    gpointer             user_data)
{
  _zeitgeist_log_send_void (egg_zeitgeist_log_quit,
                            self, cancellable, callback, user_data);
}

gboolean
zeitgeist_log_quit_finish (ZeitgeistLog        *self,
                           GAsyncResult        *res,
                           GError             **error)
{
  _zeitgeist_log_collect_void (egg_zeitgeist_log_quit_finish,
                               self, res, error);
}

static void
_zeitgeist_log_install_monitor (ZeitgeistLog        *self,
                                ZeitgeistMonitor    *monitor,
                                GCancellable        *cancellable,
                                GAsyncReadyCallback  callback,
                                gpointer             user_data)
{
  ZeitgeistLogPrivate   *priv;
  EggZeitgeistTimeRange *_time_range;
  EggDBusArraySeq       *_event_templates;
  EggDBusConnection     *connection;
  gpointer              *dispatch_data;

  priv = ZEITGEIST_LOG_GET_PRIVATE (self);

  /* Export the monitor, client side, on the session bus. Once the last
   * ref to the monitor is dropped eggdbus will automatically unregister it */
  connection = egg_dbus_object_proxy_get_connection (priv->log_proxy);
  egg_dbus_connection_register_interface (connection,
                                          zeitgeist_monitor_get_path (monitor),
                                          EGG_ZEITGEIST_TYPE_MONITOR,
                                          monitor,
                                          G_TYPE_INVALID);
  
  /* Register the monitor with the ZG daemon */
  _time_range = _zeitgeist_time_range_to_egg_zeitgeist_time_range (
                                    zeitgeist_monitor_get_time_range (monitor));
  _event_templates = _zeitgeist_events_to_egg_zeitgeist_events (
                                     zeitgeist_monitor_get_templates (monitor));

  dispatch_data = g_new (gpointer, 3);
  dispatch_data[0] = g_object_ref (self);
  dispatch_data[1] = callback;
  dispatch_data[2] = user_data;

  egg_zeitgeist_log_install_monitor (priv->log,
                                     EGG_DBUS_CALL_FLAGS_NONE,
                                     zeitgeist_monitor_get_path (monitor),
                                     _time_range,
                                     _event_templates,
                                     cancellable,
                                     dispatch_async_callback,
                                     dispatch_data);
                                 
  /* Release the event templates */
  g_object_unref (_event_templates);
  g_object_unref (_time_range);
}

void
zeitgeist_log_install_monitor (ZeitgeistLog        *self,
                               ZeitgeistMonitor    *monitor,
                               GCancellable        *cancellable,
                               GAsyncReadyCallback  callback,
                               gpointer             user_data)
{
  ZeitgeistLogPrivate   *priv;
  
  g_return_if_fail (ZEITGEIST_IS_LOG (self));
  g_return_if_fail (ZEITGEIST_IS_MONITOR (monitor));
  g_return_if_fail (cancellable == NULL || G_IS_CANCELLABLE(cancellable));

  priv = ZEITGEIST_LOG_GET_PRIVATE (self);
  
  /* Track the monitor so we can reinstate it if the Zeitgeist daemon
   * leaves and reappears on the bus */
  g_object_weak_ref (G_OBJECT (monitor),
                     (GWeakNotify) _zeitgeist_log_on_monitor_destroyed,
                     self);
  g_hash_table_insert (priv->monitors, monitor, self);

  _zeitgeist_log_install_monitor (self, monitor,
                                  cancellable, callback, user_data);
}

gboolean
zeitgeist_log_install_monitor_finish (ZeitgeistLog      *self,
                                      GAsyncResult      *res,
                                      GError           **error)
{
  return _zeitgeist_log_collect_void (egg_zeitgeist_log_install_monitor_finish,
                                      self, res, error);
}

void
zeitgeist_log_remove_monitor (ZeitgeistLog        *self,
                              ZeitgeistMonitor    *monitor,
                              GCancellable        *cancellable,
                              GAsyncReadyCallback  callback,
                              gpointer             user_data)
{
  ZeitgeistLogPrivate   *priv;
  EggDBusConnection     *connection;
  gpointer              *dispatch_data;

  g_return_if_fail (ZEITGEIST_IS_LOG (self));
  g_return_if_fail (ZEITGEIST_IS_MONITOR (monitor));
  g_return_if_fail (cancellable == NULL || G_IS_CANCELLABLE(cancellable));

  priv = ZEITGEIST_LOG_GET_PRIVATE (self);

  /* Retract the monitor from the session bus */
  connection = egg_dbus_object_proxy_get_connection (priv->log_proxy);
  egg_dbus_connection_unregister_interface (connection,
                                            zeitgeist_monitor_get_path (monitor),
                                            EGG_ZEITGEIST_TYPE_MONITOR,
                                            G_TYPE_INVALID);
  
  dispatch_data = g_new (gpointer, 3);
  dispatch_data[0] = g_object_ref (self);
  dispatch_data[1] = callback;
  dispatch_data[2] = user_data;

  egg_zeitgeist_log_remove_monitor (priv->log,
                                    EGG_DBUS_CALL_FLAGS_NONE,
                                    zeitgeist_monitor_get_path (monitor),
                                    cancellable,
                                    dispatch_async_callback,
                                    dispatch_data);
}

gboolean
zeitgeist_log_remove_monitor_finish (ZeitgeistLog       *self,
                                     GAsyncResult       *res,
                                     GError            **error)
{
  return _zeitgeist_log_collect_void (egg_zeitgeist_log_remove_monitor_finish,
                                      self, res, error);
}

/* Helper for dropping a weak ref from a log on a monitor. The method
 * signature makes it suitable for casting to a GHFunc */
static void
_zeitgeist_monitor_weak_unref (ZeitgeistMonitor *monitor,
                               ZeitgeistLog     *log)
{
  g_object_weak_unref (G_OBJECT (monitor),
                       (GWeakNotify) _zeitgeist_log_on_monitor_destroyed,
                       log);
}

/* Called when a monitor is finalized, by virtue of our weak ref */
static void
_zeitgeist_log_on_monitor_destroyed (ZeitgeistLog     *self,
                                     ZeitgeistMonitor *monitor)
{
  ZeitgeistLogPrivate *priv;
  
  priv = ZEITGEIST_LOG_GET_PRIVATE (self);
  g_hash_table_remove (priv->monitors, monitor);
}

/* Called when the Zeitgeist daemon enters/leaves the bus */
static void
_zeitgeist_log_on_name_owner_changed (ZeitgeistLog *self)
{
  ZeitgeistLogPrivate *priv;
  gchar               *name_owner;
  gboolean             has_owner;
  
  priv = ZEITGEIST_LOG_GET_PRIVATE (self);
  g_object_get (priv->log_proxy, "name-owner", &name_owner, NULL);

  has_owner = name_owner != NULL;

  if (name_owner == NULL)
    {
      /* The Zeitgeist daemon disconnected. Unregister the DBus service */
      EggDBusConnection *connection;
      GHashTableIter iter;
      gpointer monitor, dummy;

      connection = egg_dbus_object_proxy_get_connection (priv->log_proxy);

      g_hash_table_iter_init (&iter, priv->monitors);
      while (g_hash_table_iter_next (&iter, &monitor, &dummy))
        {
          egg_dbus_connection_unregister_interface (
              connection,
              zeitgeist_monitor_get_path (ZEITGEIST_MONITOR (monitor)),
              EGG_ZEITGEIST_TYPE_MONITOR,
              G_TYPE_INVALID);
        }
    }
  else
    {
      /* The Zeitgeist daemon entered the bus. Reinstate all active monitors */
      GHashTableIter iter;
      gpointer monitor, dummy;

      g_hash_table_iter_init (&iter, priv->monitors);
      while (g_hash_table_iter_next (&iter, &monitor, &dummy)) 
        {
          _zeitgeist_log_install_monitor (self, ZEITGEIST_MONITOR (monitor), 
                                          NULL, NULL, NULL);
        }
      g_free (name_owner);
    }

  if (has_owner != priv->connected)
    {
      priv->connected = has_owner;
      g_object_notify (G_OBJECT (self), "connected");
    }
}

gboolean
zeitgeist_log_is_connected (ZeitgeistLog *self)
{
  ZeitgeistLogPrivate *priv;

  priv = ZEITGEIST_LOG_GET_PRIVATE (self);
  return priv->connected;
}

